/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_list_result.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref diagnostics::unittest::Test_List_Result
 *
 * $Id: test_list_result.t.cpp,v 1.11 2005/06/23 09:54:27 esdentem Exp $
 *
 * @author Christian Schallhart
 */
#include <diagnostics/unittest.hpp>

#include <diagnostics/unittest/test_system/test_list_result.hpp>

#include <diagnostics/unittest/test_system_exception.hpp>

// used components
#include <diagnostics/util/to_string.hpp>

#define TEST_COMPONENT_NAME Test_List_Result
#define TEST_COMPONENT_NAMESPACE diagnostics::unittest

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;



/**
 * @brief [PRO] fct/abnormal: is_testable_at with LEVEL_TEST
 */
void is_testable_at(Test_Data & test_data)
{
    using ::diagnostics::internal::to_string;
    // cstr
    Test_List_Result t("/path",2,true,true,true,false);
    TEST_ASSERT(t.is_testable_at(LEVEL_PROD)==true);
    TEST_ASSERT(t.is_testable_at(LEVEL_DEBUG)==true);
    TEST_ASSERT(t.is_testable_at(LEVEL_AUDIT)==false);

    TEST_THROWING_BLOCK_ENTER;
    t.is_testable_at(LEVEL_TEST);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
} 


/**
 * @brief [PRO] fct/abnormal: assembly-cstr
 */
void assembly_cstr(Test_Data & test_data)
{
    TEST_THROWING_BLOCK_ENTER;
    Test_List_Result t("path",2,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    Test_List_Result t("/path",0,true,true,true,true);

    TEST_THROWING_BLOCK_ENTER;
    Test_List_Result t("/path",-1,true,true,true,true);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}


/**
 * @brief [PRO] fct/normal: all cstrs, assignment, comparision
 */
void cstr_copy_assign_compare(Test_Data & test_data)
{
    using ::diagnostics::internal::to_string;
    // cstr
    Test_List_Result t("/path",2,true,true,true,true);
    TEST_ASSERT(t.test_case_path()=="/path");
    TEST_ASSERT(t.timeout()==2);
    TEST_ASSERT(t.is_rerunable()==true);
    TEST_ASSERT(test_data.compare("t",to_string(t)));
    // self cmp and assign
    TEST_ASSERT(t==t);
    t=t;
    TEST_ASSERT(t.test_case_path()=="/path");
    TEST_ASSERT(t.timeout()==2);
    TEST_ASSERT(t.is_rerunable()==true);
    TEST_ASSERT(test_data.compare("t",to_string(t)));
    

    // cstr
    Test_List_Result t2("/path1",0,false,true,true,true);
    TEST_ASSERT(t2.test_case_path()=="/path1");
    TEST_ASSERT(t2.timeout()==0);
    TEST_ASSERT(t2.is_rerunable()==false);
    TEST_ASSERT(test_data.compare("t2",to_string(t2)));
    // self cmp and assign
    TEST_ASSERT(t2==t2);
    t2=t2;
    TEST_ASSERT(t2.test_case_path()=="/path1");
    TEST_ASSERT(t2.timeout()==0);
    TEST_ASSERT(t2.is_rerunable()==false);
    TEST_ASSERT(test_data.compare("t2",to_string(t2)));    

    // default cstr
    Test_List_Result t3("/path1",0,false,true,false,true);
    TEST_ASSERT(test_data.compare("t3",to_string(t3)));
    // self cmp and assign
    TEST_ASSERT(t3==t3);
    t3=t3;
    TEST_ASSERT(test_data.compare("t3",to_string(t3)));

    // all different
    TEST_ASSERT(t!=t2);
    TEST_ASSERT(t!=t3);
    TEST_ASSERT(t2!=t3);

    // assign
    t3=t;
    TEST_ASSERT(t==t3);
    TEST_ASSERT(t!=t2);
    TEST_ASSERT(t2!=t3);
    TEST_ASSERT(test_data.compare("t",to_string(t3)));

    // copy cstr
    Test_List_Result t4(t3);
    TEST_ASSERT(test_data.compare("t",to_string(t4)));
    TEST_ASSERT(t4==t3);
}

/**
 * @brief [PRO] fct/normal: assembly cstr with all combination of is_testable_at
 */
void all_combinations(Test_Data & test_data)
{
    int i;
    for(i=0;i<8;++i) {
	Test_List_Result t("/path",2,true,i&(1<<LEVEL_PROD),i&(1<<LEVEL_DEBUG),i&(1<<LEVEL_AUDIT));
	TEST_ASSERT(t.is_testable_at(LEVEL_PROD)==static_cast<bool>(i&(1<<LEVEL_PROD)));
	TEST_ASSERT(t.is_testable_at(LEVEL_DEBUG)==static_cast<bool>(i&(1<<LEVEL_DEBUG)));
	TEST_ASSERT(t.is_testable_at(LEVEL_AUDIT)==static_cast<bool>(i&(1<<LEVEL_AUDIT)));
    }
}

TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&cstr_copy_assign_compare,LEVEL_PROD);
TEST_NORMAL_CASE(&all_combinations,LEVEL_PROD);
TEST_ABNORMAL_CASE(&is_testable_at,LEVEL_PROD);
TEST_ABNORMAL_CASE(&assembly_cstr,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;
// vim:ts=4:sw=4
