/*
 *      fhist - file history and comparison tools
 *      Copyright (C) 2000, 2002, 2004, 2008 Peter Miller
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 3 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program. If not, see
 *      <http://www.gnu.org/licenses/>.
 */

#include <ac/stdio.h>
#include <ac/sys/stat.h>

#include <error_intl.h>
#include <input/private.h>
#include <input/stdin.h>
#include <str.h>


static string_ty *
standard_input(void)
{
        static string_ty *name;
        sub_context_ty  *scp;

        if (!name)
        {
                scp = sub_context_new();
                name = subst_intl(scp, i18n("standard input"));
                sub_context_delete(scp);
        }
        return name;
}


static void
destruct(input_ty *this)
{
}


static long
iread(input_ty *this, void *data, long len)
{
        long            result;

        if (len <= 0)
                return 0;
        result = fread(data, 1, len, stdin);
        if (result <= 0 && ferror(stdin))
        {
                sub_context_ty  *scp;

                scp = sub_context_new();
                sub_errno_set(scp);
                sub_var_set_string(scp, "File_Name", standard_input());
                fatal_intl(scp, i18n("read \"$filename\": $errno"));
                /* NOTREACHED */
        }
        return result;
}


static int
get(input_ty *this)
{
        int             c;

        c = getchar();
        if (c == EOF && ferror(stdin))
        {
                sub_context_ty  *scp;

                scp = sub_context_new();
                sub_errno_set(scp);
                sub_var_set_string(scp, "File_Name", standard_input());
                fatal_intl(scp, i18n("read \"$filename\": $errno"));
                /* NOTREACHED */
        }
        return c;
}


static long
itell(input_ty *this)
{
        return ftell(stdin);
}


static const char *
name(input_ty *this)
{
        return standard_input()->str_text;
}


static long
length(input_ty *this)
{
        struct stat     st;

        if (fstat(fileno(stdin), &st) < 0)
                return -1;
        if (!S_ISREG(st.st_mode))
                return -1;
        return st.st_size;
}


static input_vtbl_ty vtbl =
{
        sizeof(input_ty),
        destruct,
        iread,
        get,
        itell,
        name,
        length,
};


input_ty *
input_stdin(void)
{
        return input_new(&vtbl);
}
