/*
 * Copyright (c) 2001, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package nsk.jdi.ClassPrepareRequest.addClassFilter_rt;

import nsk.share.*;
import nsk.share.jpda.*;
import nsk.share.jdi.*;

import com.sun.jdi.*;
import com.sun.jdi.event.*;
import com.sun.jdi.request.*;

import java.util.*;
import java.io.*;

/**
 * The test for the implementation of an object of the type
 * ClassPrepareRequest.
 *
 * The test checks that results of the method
 * <code>com.sun.jdi.ClassPrepareRequest.addClassFilter_rt()</code>
 * complies with its spec.
 *
 * The test checks up on the following assertion:
 *    Restricts the events generated by this request to be
 *    the preparation  of the given reference type and any subtypes.
 * The cases to test include re-invocations of the method
 * addClassFilter() on the same ClassPrepareRequest object.
 * There are two ClassPrepareRequests to check as follows:
 * (1) For ClassPrepareRequest2, both invocations are with different
 *     ReferenceTypes restricting one ClassPrepare event to two classes.
 *     The test expects no ClassPrepare event will be received.
 * (2) For ClassPrepareRequest1, both invocations are with the same
 *     ReferenceType restricting one ClassPrepare event to one class.
 *     The test expects this ClassPrepare event will be received.
 *
 * The test works as follows.
 * - The debugger resumes the debuggee and waits for the BreakpointEvent.
 * - The debuggee creates class filter objects, to load the Classes to
 *   filter, and invokes the methodForCommunication to be suspended and
 *   to inform the debugger with the event.
 * - Upon getting the BreakpointEvent, the debugger
 *   - gets ReferenceTypes 1&2 for the Classes to filter,
 *   - sets up two ClassPrepareRequests 1&2,
 *   - double restricts ClassPrepareRequest1 to the RefTypes 1 and 1,
 *   - double restricts ClassPrepareRequest2 to the RefTypes 1 and 2,
 *   - resumes debuggee's main thread, and
 *   - waits for the event.
 * - The debuggee creates and starts two threads, thread1 and thread2,
 *   first of them will create an object of sub-Class to be filtered,
 *   whereas second will create an object to be not filtered.
 * - Upon getting the event, the debugger performs the checks required.
 */

public class filter_rt003 extends TestDebuggerType1 {
    public static void main (String argv[]) {
        System.exit(run(argv, System.out) + Consts.JCK_STATUS_BASE);
    }

    public static int run (String argv[], PrintStream out) {
        debuggeeName = "nsk.jdi.ClassPrepareRequest.addClassFilter_rt.filter_rt003a";
        return new filter_rt003().runThis(argv, out);
    }

    private String testedClassName10 =
        "nsk.jdi.ClassPrepareRequest.addClassFilter_rt.filter_rt003aTestClass10";

    private String testedClassName20 =
        "nsk.jdi.ClassPrepareRequest.addClassFilter_rt.filter_rt003aTestClass20";


    protected void testRun() {
        EventRequest  eventRequest1      = null;
        EventRequest  eventRequest2      = null;

        String        property1          = "ClassPrepareRequest1";
        String        property2          = "ClassPrepareRequest2";

        ReferenceType testClassReference10 = null;
        ReferenceType testClassReference20 = null;

        for (int i = 0; ; i++) {

            if (!shouldRunAfterBreakpoint()) {
                vm.resume();
                break;
            }

            display(":::::: case: # " + i);
            switch (i) {

                case 0:
                testClassReference10 = (ReferenceType) vm.classesByName(testedClassName10).get(0);
                testClassReference20 = (ReferenceType) vm.classesByName(testedClassName20).get(0);

                eventRequest1 = setting21ClassPrepareRequest(testClassReference10,
                                             EventRequest.SUSPEND_ALL, property1);

                eventRequest2 = setting21ClassPrepareRequest(testClassReference10,
                                             EventRequest.SUSPEND_ALL, property2);

                ((ClassPrepareRequest) eventRequest1).addClassFilter(testClassReference10);
                ((ClassPrepareRequest) eventRequest2).addClassFilter(testClassReference20);

                display("......waiting for ClassPrepareEvent in expected thread");
                Event newEvent = eventHandler.waitForRequestedEvent(new EventRequest[]{eventRequest1, eventRequest2}, waitTime, true);

                if ( !(newEvent instanceof ClassPrepareEvent)) {
                    setFailedStatus("ERROR: new event is not ClassPrepareEvent");
                } else {

                    String property = (String) newEvent.request().getProperty("number");
                    display("       got new ClassPrepareEvent with propety 'number' == " + property);

                    if ( !property.equals(property1) ) {
                        setFailedStatus("ERROR: property is not : " + property1);
                    }
                }
                vm.resume();
                break;

                default:
                throw new Failure("** default case 1 **");
            }
        }
        return;
    }

    private ClassPrepareRequest setting21ClassPrepareRequest ( ReferenceType  testedClass,
                                                               int            suspendPolicy,
                                                               String         property       ) {
        try {
            display("......setting up ClassPrepareRequest:");
            display("       class: " + testedClass + "; property: " + property);

            ClassPrepareRequest
            cpr = eventRManager.createClassPrepareRequest();
            cpr.putProperty("number", property);
            cpr.addClassFilter(testedClass);
            cpr.setSuspendPolicy(suspendPolicy);

            display("      ClassPrepareRequest has been set up");
            return cpr;
        } catch ( Exception e ) {
            throw new Failure("** FAILURE to set up ClassPrepareRequest **");
        }
    }

}
