/**
 * @brief Class Mixer - controls the volume of the speakers
 *
 * This module contains the class mixer which will define the interface
 * to a sound mixer device. Below the module are the mixer drivers
 * which implement the interface to the specific hardware or kernel
 * interface.
 *
 * Currently two mixer drivers are available:
 *  @li driver_mixer_oss.c  which supports the OSS kernel interface
 *  @li driver_mixer_alsa.c which supports the new ALSA interface
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/class_mixer.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include "systems.h"

#include <pbb.h>
#include <pbbinput.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "class_config.h"
#include "class_mixer.h"
#include "driver_mixer_alsa.h"
#include "driver_mixer_oss.h"

#define SECTION "MODULE MIXER"

struct class_mixer {
	struct driver_mixer *driver;
	unsigned short keyvolup;
	unsigned short modvolup;
	unsigned short keyvoldn;
	unsigned short modvoldn;
	unsigned short keymute;
	unsigned short modmute;
} classdata;


/* keep basic init function for compatibility with old config system */
int
mixer_init ()
{
	struct class_mixer *base = &classdata;

	/* get configuration for this module */
	base->keyvolup	= config_get_keycode (SECTION, "VolumeUpKey", KEY_VOLUMEUP);
	base->modvolup	= config_get_modmask (SECTION, "VolumeUpKey", MOD_NONE);
	base->keyvoldn	= config_get_keycode (SECTION, "VolumeDownKey", KEY_VOLUMEDOWN);
	base->modvoldn	= config_get_modmask (SECTION, "VolumeDownKey", MOD_NONE);
	base->keymute	= config_get_keycode (SECTION, "MuteKey", KEY_MUTE);
	base->modmute	= config_get_modmask (SECTION, "MuteKey", MOD_NONE);

	base->driver = NULL;
#if WITH_ALSA
	if (!base->driver) base->driver = driver_mixer_alsa_init ();
#endif
#if WITH_OSS
	if (!base->driver) base->driver = driver_mixer_oss_init ();
#endif

	if (base->driver) {
		register_function (QUERYQUEUE,  mixer_query);
		register_function (CONFIGQUEUE, mixer_config);
		register_function (KBDQUEUE, mixer_keyboard);
		print_msg (PBB_INFO, _("Initialized: %s\n"), _(base->driver->name));
	} else
		print_msg (PBB_WARN, _("No mixer driver available - check your Kernel configuration.\n"));

	return 0;
}

void
mixer_exit ()
{
	struct class_mixer *base = &classdata;

	if (base->driver)
		base->driver->driver_exit();
}

void
mixer_inform_clients ()
{
	struct class_mixer *base = &classdata;

	if (base->driver->is_muted () == TRUE)
		singletag_to_clients (CHANGEVALUE, TAG_MUTE, 0);
	else
		singletag_to_clients (CHANGEVALUE, TAG_VOLUME,
		        base->driver->get_volume (VALUE_ABS_PERCENT));
}

void
mixer_query (struct tagitem *taglist)
{
	mixer_handle_tags (MODE_QUERY, taglist);
}

void
mixer_config (struct tagitem *taglist)
{
	mixer_handle_tags (MODE_CONFIG, taglist);
}

void
mixer_handle_tags (int cfgure, struct tagitem *taglist)
{
	struct class_mixer *base = &classdata;
	gboolean muted;

	while (taglist->tag != TAG_END) {
		switch (taglist->tag) {
		case TAG_MIXERDRIVER:
			if (cfgure) tagerror(taglist, E_NOWRITE);
			else        taglist->data = (long) base->driver->name;
			break;
		case TAG_VOLUMEUPKEY:
			if (cfgure)	{
				base->keyvolup = taglist->data;
				config_set_keymod (SECTION, "VolumeUpKey", base->keyvolup, base->modvolup);
			} else
				taglist->data = base->keyvolup;
			break;
		case TAG_VOLUMEUPMOD:
			if (cfgure)	{
				base->modvolup = taglist->data;
				config_set_keymod (SECTION, "VolumeUpKey", base->keyvolup, base->modvolup);
			} else
				taglist->data = base->modvolup;
			break;
		case TAG_VOLUMEDOWNKEY:
			if (cfgure)	{
				base->keyvoldn = taglist->data;
				config_set_keymod (SECTION, "VolumeDownKey", base->keyvoldn, base->modvoldn);
			} else
				taglist->data = base->keyvoldn;
			break;
		case TAG_VOLUMEDOWNMOD:
			if (cfgure)	{
				base->modvoldn = taglist->data;
				config_set_keymod (SECTION, "VolumeDownKey", base->keyvoldn, base->modvoldn);
			} else
				taglist->data = base->modvoldn;
			break;
		case TAG_MUTEKEY:
			if (cfgure)	{
				base->keymute = taglist->data;
				config_set_keymod (SECTION, "MuteKey", base->keymute, base->modmute);
			} else
				taglist->data = base->keymute;
			break;
		case TAG_MUTEMOD:
			if (cfgure)	{
				base->modmute = taglist->data;
				config_set_keymod (SECTION, "MuteKey", base->keymute, base->modmute);
			} else
				taglist->data = base->modmute;
			break;
		case TAG_MIXER:
			if (cfgure) base->driver->set_mixer ((char *) taglist->data);
			else        taglist->data = (long) base->driver->get_mixer ();
			break;
		case TAG_MIXERCHANNELS:
			if (cfgure) base->driver->set_channels ((char *)taglist->data);
			else        taglist->data = (long) base->driver->get_channels ();
			break;
		case TAG_VOLUME:
			if (cfgure) {
				base->driver->set_volume (VALUE_ABS_PERCENT, taglist->data);
				mixer_inform_clients ();
			} else
				taglist->data = base->driver->get_volume (VALUE_ABS_PERCENT);
			break;
		case TAG_MUTE:
			muted = base->driver->is_muted ();
			if (cfgure) {
				if (muted == (gboolean) taglist->data) {
					base->driver->set_volume (VALUE_REL, 0);
					mixer_inform_clients ();
				}
			} else
				taglist->data = muted;
			break;
		}
		taglist++;
	}
}

void
mixer_keyboard (struct tagitem *taglist)
{
	struct class_mixer *base = &classdata;
	int code, value, mod, step;

	code = (int) tagfind (taglist, TAG_KEYCODE, 0);
	value = (int) tagfind (taglist, TAG_KEYREPEAT, 0);
	mod = (int) tagfind (taglist, TAG_MODIFIER, 0);

	if (value) {
		if ((code == base->keyvolup) && ((mod & ~MOD_SHIFT) == base->modvolup)) {
			step = (mod & MOD_SHIFT) ? +1 : +10;
		} else if ((code == base->keyvoldn) && ((mod & ~MOD_SHIFT) == base->modvoldn)) {
			step = (mod & MOD_SHIFT) ? -1 : -10;
		} else if ((code == base->keymute) && (mod == base->modmute) && (value == 1)) {
			step = 0;
		} else return;

		/* change volume level */
		if ((step == 0) || (mod & MOD_SHIFT))
			base->driver->set_volume (VALUE_REL, step);  /* mute and fine tuning */
		else
			base->driver->set_volume (VALUE_REL_PERCENT, step); /* normal control */
		mixer_inform_clients ();
	}
}


