// ---------------------------------------------------------------------------
// - Cipher.cpp                                                              -
// - afnix:sec module - base cipher class implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2019 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cipher.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "QuarkZone.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a cipher by name

  Cipher::Cipher (const String& name) : Codec (name) {
  }

  // create a cipher by name and key

  Cipher::Cipher (const String& name, const Key& key) : Codec (name) {
    d_ckey = key;
  }

  // set the cipher key

  void Cipher::setkey (const Key& key) {
    wrlock ();
    try {
      d_ckey = key;
      reset  ();
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the cipher key

  Key Cipher::getkey (void) const {
    rdlock ();
    try {
      Key result = d_ckey;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // normalize a data size

  t_long Cipher::waist (const t_long size) const {
    rdlock ();
    try {
      t_long result = (size < 0) ? 0 : size;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 2;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_SETKEY  = zone.intern ("set-key");
  static const long QUARK_GETKEY  = zone.intern ("get-key");

  // return true if the given quark is defined

  bool Cipher::isquark (const long quark, const bool hflg) const {
    rdlock ();
    try {
      if (zone.exists (quark) == true) {
	unlock ();
	return true;
      }
      bool result = hflg ? Codec::isquark (quark, hflg) : false;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // apply this object with a set of arguments and a quark
  
  Object* Cipher::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nullptr) ? 0 : argv->length ();
    
    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETKEY)  return new Key (getkey ());
    }
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETKEY) {
	Object* obj = argv->get (0);
	Key*    key = dynamic_cast <Key*> (obj);
	if (key != nullptr) {
	  setkey (*key);
	  return nullptr;
	}
      }
    }
    // call the codec method
    return Codec::apply (robj, nset, quark, argv);
  }
}
