/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "DialogMain.h"
//#define _DEBUG_


DialogMain::DialogMain()
: lock_events(FALSE),
  menu_view_elem_sort_author(
        menu_view_group_sort,
        _("Sort by _Author"),
        sigc::bind(sigc::mem_fun(*this, &DialogMain::on_view_sortorder_clicked),
                   GTKBOOKLIST_COLUMN_AUTHORANDTITLE)),
  menu_view_elem_sort_title(
        menu_view_group_sort,
        _("Sort by _Title"),
        sigc::bind(sigc::mem_fun(*this, &DialogMain::on_view_sortorder_clicked),
                   GTKBOOKLIST_COLUMN_TITLE)),
  menu_view_elem_sort_category(
        menu_view_group_sort,
        _("Sort by _Category"),
        sigc::bind(sigc::mem_fun(*this, &DialogMain::on_view_sortorder_clicked),
                   GTKBOOKLIST_COLUMN_CATEGORY)),
  menu_view_elem_sort_readdate(
        menu_view_group_sort,
        _("Sort by Read _Date"),
        sigc::bind(sigc::mem_fun(*this, &DialogMain::on_view_sortorder_clicked),
                   GTKBOOKLIST_COLUMN_READDATE_TIMET)),
  menu_view_elem_sort_rating(
        menu_view_group_sort,
        _("Sort by _Rating"),
        sigc::bind(sigc::mem_fun(*this, &DialogMain::on_view_sortorder_clicked),
                   GTKBOOKLIST_COLUMN_RATING_INT)),
  image_button_add(Gtk::Stock::ADD,
                   Gtk::IconSize(Gtk::ICON_SIZE_SMALL_TOOLBAR)),
  label_button_add(_("Add Book")),
  image_button_delete(Gtk::Stock::DELETE,
                      Gtk::IconSize(Gtk::ICON_SIZE_SMALL_TOOLBAR)),
  label_button_delete(_("Delete Book")),
  image_button_details(Gtk::Stock::JUSTIFY_LEFT,
                       Gtk::IconSize(Gtk::ICON_SIZE_SMALL_TOOLBAR)),
  label_button_details(_("Show Details")),
  table_preview(6, 2),
  label_title(" ", 0, 0.5),
  label_author(" ", 0, 0.5),
  label_summary(" ", 0, 0),
  label_isbn(" ", 0, 0.5),
  label_booklist(" ", 0, 0.5)
{
  // The menubar.
  {
    // The "File" menu.
    menubar.items().push_back(
        Gtk::Menu_Helpers::MenuElem(_("_File"), menu_file));
    menu_file.items().push_back(
        Gtk::Menu_Helpers::StockMenuElem(
                     Gtk::Stock::QUIT,
                     sigc::mem_fun(*this, &DialogMain::on_file_quit_clicked)));
    
    // The "View" menu.
    {
      menubar.items().push_back(
          Gtk::Menu_Helpers::MenuElem(_("_View"), menu_view));
      menu_view.items().push_back(menu_view_elem_sort_author);
      menu_view.items().push_back(menu_view_elem_sort_title);
      menu_view.items().push_back(menu_view_elem_sort_category);
      menu_view.items().push_back(menu_view_elem_sort_readdate);
      menu_view.items().push_back(menu_view_elem_sort_rating);
    }
    
    // The "Help" menu.
    menubar.items().push_back(
        Gtk::Menu_Helpers::MenuElem(_("_Help"), menu_help));
    menu_help.items().push_back(
        Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::ABOUT,
                     sigc::mem_fun(*this, &DialogMain::on_help_about_clicked)));
  }
  
  // The buttons.
  {
    hbox_buttons.set_border_width(6);
    hbox_buttons.set_spacing(12);
    
    // Add book.
    hbox_button_add.set_spacing(3);
    hbox_button_add.pack_start(image_button_add);
    hbox_button_add.pack_start(label_button_add);
    button_add.add(hbox_button_add);
    hbox_buttons.pack_start(button_add, FALSE, FALSE);
    
    // Delete book.
    hbox_button_delete.set_spacing(3);
    hbox_button_delete.pack_start(image_button_delete);
    hbox_button_delete.pack_start(label_button_delete);
    button_delete.add(hbox_button_delete);
    hbox_buttons.pack_start(button_delete, FALSE, FALSE);
    
    // Show book.
    hbox_button_details.set_spacing(3);
    hbox_button_details.pack_start(image_button_details);
    hbox_button_details.pack_start(label_button_details);
    button_details.add(hbox_button_details);
    hbox_buttons.pack_start(button_details, FALSE, FALSE);
    
    hbox_buttons.pack_start(fixed_buttons, TRUE, TRUE);
    button_add.signal_clicked().connect(signal_button_add_clicked);
    button_delete.signal_clicked().connect(signal_button_delete_clicked);
    button_details.signal_clicked().connect(signal_button_details_clicked);
  }
  
  // The book preview.
  {
    table_preview.set_border_width(6);
    label_title.set_line_wrap();
    label_author.set_line_wrap();
    label_summary.set_line_wrap();
    label_expander.set_line_wrap();
    label_isbn.set_line_wrap();
    label_isbn.set_padding(0, 3);
    label_isbn.set_selectable(TRUE);
    fixed_preview.set_size_request(-1, 9);
    
    table_preview.attach(label_title,    0, 1, 0, 1);
    table_preview.attach(label_author,   0, 1, 1, 2);
    table_preview.attach(fixed_preview,  0, 1, 2, 3);
    table_preview.attach(label_summary,  0, 1, 3, 4, Gtk::FILL|Gtk::EXPAND,
                                                     Gtk::FILL|Gtk::EXPAND);
    table_preview.attach(label_expander, 1, 2, 3, 4);
    table_preview.attach(label_isbn,     0, 1, 5, 6);
    
    // Paste some newline characters into the label_expander, so that the
    // area expands properly.
    string foo;
    foo.insert(0, DIALOG_MAIN_SUMMARY_LENGTH / 75, '\n');
    label_expander.set_text(foo);
  }
  
  // The book list.
  {
    label_booklist.set_padding(6, 0);
    scroll_booklist.set_border_width(6);
    scroll_booklist.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
    scroll_booklist.set_shadow_type(Gtk::SHADOW_IN);
    scroll_booklist.add(booklist);
    booklist.signal_book_selected.connect(
              sigc::mem_fun(*this, &DialogMain::on_booklist_selection_changed));
    booklist.signal_changed.connect(
              sigc::mem_fun(*this, &DialogMain::on_booklist_changed));
    booklist.signal_sorting_changed.connect(
              sigc::mem_fun(*this, &DialogMain::on_booklist_sorting_changed));
    on_booklist_changed();
  }
  
  // Pack it all together.
  vbox_main.pack_start(menubar, FALSE, FALSE);
  vbox_main.pack_start(hbox_buttons, FALSE, FALSE);
  vbox_main.pack_start(table_preview, FALSE, FALSE);
  vbox_main.pack_start(label_booklist, FALSE, FALSE);
  vbox_main.pack_start(scroll_booklist, TRUE, TRUE);
  
  add(vbox_main);
  set_size_request(500, 550);
  set_title(_("Book Organizer"));
  show_all();
}


DialogMain::~DialogMain()
{
}


void DialogMain::update_preview(Book *book)
{
  label_title.set_markup("<span weight=\"bold\" size=\"large\">"
                       + book->get_title()
                       + "</span>");
  label_author.set_text(book->get_author());
  label_summary.set_text(book->get_summary(DIALOG_MAIN_SUMMARY_LENGTH));
  label_isbn.set_markup(book->get_isbn() == "" ?
                        _("Unknown ISBN")
                      : "<i>ISBN " + book->get_isbn() + "</i>");
}


void DialogMain::on_file_quit_clicked(void)
{
  this->hide();
}


void DialogMain::on_view_sortorder_clicked(int column)
{
  if (lock_events)
    return;
#ifdef _DEBUG_
  printf("DialogMain::on_view_sortorder_clicked(): %i\n", column);
#endif
  lock_events = TRUE;
  booklist.set_sorting(column);
  lock_events = FALSE;
}


void DialogMain::on_booklist_sorting_changed(void)
{
  if (lock_events)
    return;
#ifdef _DEBUG_
  printf("DialogMain::on_booklist_sorting_changed(): Called.\n");
#endif
  Glib::RefPtr<Gtk::MenuItem> item;
  switch (booklist.get_sorting()) {
  case GTKBOOKLIST_COLUMN_AUTHORANDTITLE:
    item = menu_view_elem_sort_author.get_child();
    break;
  
  case GTKBOOKLIST_COLUMN_TITLE:
    item = menu_view_elem_sort_title.get_child();
    break;
  
  case GTKBOOKLIST_COLUMN_CATEGORY:
    item = menu_view_elem_sort_category.get_child();
    break;
  
  case GTKBOOKLIST_COLUMN_READDATE_TIMET:
    item = menu_view_elem_sort_readdate.get_child();
    break;
  
  case GTKBOOKLIST_COLUMN_RATING_INT:
    item = menu_view_elem_sort_rating.get_child();
    break;
  
  default:
    g_warning("DialogMain::on_booklist_sorting_changed(): Unknown sorting.\n");
    break;
  }
  
  lock_events = TRUE;
  item->activate();
  lock_events = FALSE;
}


void DialogMain::on_help_about_clicked(void)
{
  aboutbox.run();
  aboutbox.hide();
}


void DialogMain::on_booklist_selection_changed(Book* book)
{
  char title[200];
  if (book)
    snprintf(title, 200, _("%s - Book Organizer"),
                         book->get_title().c_str());
  else
    snprintf(title, 200, _("Book Organizer"));
  set_title(title);
}


void DialogMain::on_booklist_changed(void)
{
  char str[1000];
  char count[1000];
  snprintf(str,   1000, "<b>%s</b>", _("Booklist "));
  snprintf(count, 1000, _("(%i books in the list):"), booklist.get_numitems());
  string text = str;
  text.append(count);
  label_booklist.set_markup(text);
}
