//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Instrument/DistributionPlot.cpp
//! @brief     Implements class DistributionPlot
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Instrument/DistributionPlot.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Descriptor/DistributionItems.h"
#include "GUI/View/Info/CautionSign.h"
#include "Param/Distrib/Distributions.h"
#include "Param/Distrib/ParameterSample.h"

DistributionPlot::DistributionPlot(QWidget* parent)
    : QWidget(parent)
    , m_plot(new QCustomPlot)
    , m_distItem(nullptr)
    , m_label(new QLabel)
    , m_resetAction(new QAction(this))
    , m_cautionSign(new CautionSign(this))
{
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

    m_resetAction->setText("Reset View");
    connect(m_resetAction, &QAction::triggered, this, &DistributionPlot::resetView);

    m_label->setAlignment(Qt::AlignVCenter | Qt::AlignLeft);
    m_label->setStyleSheet("background-color:white;");
    m_label->setContentsMargins(3, 3, 3, 3);

    auto* mainLayout = new QVBoxLayout;
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->setSpacing(0);
    mainLayout->addWidget(m_plot, 1);
    mainLayout->addWidget(m_label);
    m_plot->setAttribute(Qt::WA_NoMousePropagation, false);
    setLayout(mainLayout);

    setStyleSheet("background-color:white;");
    connect(m_plot, &QCustomPlot::mousePress, this, &DistributionPlot::onMousePress);
    connect(m_plot, &QCustomPlot::mouseMove, this, &DistributionPlot::onMouseMove);
}

void DistributionPlot::setDistItem(DistributionItem* distItem)
{
    ASSERT(distItem);
    if (m_distItem == distItem)
        return;

    m_distItem = distItem;
    plotItem();
}

void DistributionPlot::plotItem()
{
    init_plot();

    if (!m_distItem->is<DistributionNoneItem>()) {
        try {
            plot_distributions();
        } catch (const std::exception& ex) {
            init_plot();
            QString message = QString("Wrong parameters\n\n") + (QString::fromStdString(ex.what()));
            m_cautionSign->setCautionMessage(message);
        }
    }

    m_plot->replot();
}

//! Generates label with current mouse position.

void DistributionPlot::onMouseMove(QMouseEvent* event)
{
    QPoint point = event->pos();
    double xPos = m_plot->xAxis->pixelToCoord(point.x());
    double yPos = m_plot->yAxis->pixelToCoord(point.y());

    if (m_plot->xAxis->range().contains(xPos) && m_plot->yAxis->range().contains(yPos)) {
        QString text = QString("[x:%1, y:%2]").arg(xPos).arg(yPos);
        m_label->setText(text);
    }
}

void DistributionPlot::onMousePress(QMouseEvent* event)
{
    if (event->button() == Qt::RightButton) {
        QPoint point = event->globalPos();
        QMenu menu;
        menu.addAction(m_resetAction);
        menu.exec(point);
    }
}

//! Reset zoom range to initial state.

void DistributionPlot::resetView()
{
    m_plot->xAxis->setRange(m_xRange);
    m_plot->replot();
}

//! Clears all plottables, resets axes to initial state.

void DistributionPlot::init_plot()
{
    m_cautionSign->clear();

    m_plot->clearGraphs();
    m_plot->clearItems();
    m_plot->clearPlottables();
    m_plot->setInteractions(QCP::iRangeDrag | QCP::iRangeZoom | QCP::iSelectAxes
                            | QCP::iSelectLegend | QCP::iSelectPlottables);
    m_plot->yAxis->setLabel("probability");
    m_plot->yAxis->setTickLabels(false);
    m_plot->xAxis2->setVisible(true);
    m_plot->yAxis2->setVisible(true);
    m_plot->xAxis2->setTickLabels(false);
    m_plot->yAxis2->setTickLabels(false);
    m_plot->xAxis2->setTicks(false);
    m_plot->yAxis2->setTicks(false);

    m_plot->yAxis->setRange({0, 1.1});
    setPlotRange({-1, 1});
}

void DistributionPlot::plot_distributions()
{
    ASSERT(!m_distItem->is<DistributionNoneItem>());

    auto distrib = m_distItem->createDistribution();

    //... Plot function graph
    std::vector<std::pair<double, double>> graph = distrib->plotGraph();
    double max_y = 0;
    for (size_t i = 0; i < graph.size(); ++i)
        max_y = std::max(max_y, graph[i].second);

    QVector<double> xFunc(graph.size());
    QVector<double> yFunc(graph.size());
    for (size_t i = 0; i < graph.size(); ++i) {
        xFunc[i] = graph[i].first;
        yFunc[i] = graph[i].second / max_y;
    }

    setPlotRange({xFunc.first(), xFunc.last()});

    m_plot->addGraph();
    m_plot->graph(0)->setData(xFunc, yFunc);

    //... Plot bars to represent weighted sampling points
    std::vector<ParameterSample> samples = distrib->distributionSamples();
    size_t N = samples.size();
    max_y = 0;
    for (size_t i = 0; i < N; ++i)
        max_y = std::max(max_y, samples[i].weight);

    QVector<double> xBar(N);
    QVector<double> yBar(N);
    for (size_t i = 0; i < N; ++i) {
        xBar[i] = samples[i].value;
        yBar[i] = samples[i].weight / max_y;
    }

    // use rational function to set bar width:
    // - at low N, a constant fraction of the plot range
    // - at large N, decreasing with N^-1
    double barWidth = (xFunc.last() - xFunc.first()) / (30 + 3 * N * N / (10 + N));

    auto* bars = new QCPBars(m_plot->xAxis, m_plot->yAxis);
    bars->setWidth(barWidth);
    bars->setData(xBar, yBar);
}

void DistributionPlot::setPlotRange(const QPair<double, double>& xRange)
{
    m_xRange = QCPRange(xRange.first, xRange.second);
    m_plot->xAxis->setRange(m_xRange);
}

void DistributionPlot::plotVerticalLine(double xMin, double yMin, double xMax, double yMax,
                                        const QColor& color)
{
    auto* line = new QCPItemLine(m_plot);

    QPen pen(color, 1, Qt::DashLine);
    line->setPen(pen);
    line->setSelectable(true);

    line->start->setCoords(xMin, yMin);
    line->end->setCoords(xMax, yMax);
}

void DistributionPlot::setXAxisName(const QString& xAxisName)
{
    m_plot->xAxis->setLabel(xAxisName);
}

void DistributionPlot::setShowMouseCoords(bool b)
{
    m_label->setVisible(b);
}
