/*
 *  Copyright (C) 2010  Giorgio Wicklein <g.wicklein@giowisys.com>
 *
 *  This file is part of DaemonFS.
 *
 *  DaemonFS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  DaemonFS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with DaemonFS.  If not, see <http://www.gnu.org/licenses/>.
 */

//-----------------------------------------------------------------------------
// Hearders
//-----------------------------------------------------------------------------

#include <QTableWidgetItem>
#include <QFileInfo>
#include <QDateTime>
#include <QFileIconProvider>

#include "eventswidget.h"
#include "ui_eventswidget.h"
#include "mainwindow.h"
#include "../modules/filewatcherengine.h"


//-----------------------------------------------------------------------------
// Public
//-----------------------------------------------------------------------------

EventsWidget::EventsWidget(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::EventsWidget)
{
    ui->setupUi(this);

    ui->eventsTable->horizontalHeader()->setResizeMode(0, QHeaderView::ResizeToContents);
    ui->eventsTable->horizontalHeader()->setResizeMode(1, QHeaderView::Stretch);
    ui->eventsTable->horizontalHeader()->setResizeMode(2, QHeaderView::Stretch);
    ui->eventsTable->setIconSize(QSize(32,32));

    connect(ui->backbutton, SIGNAL(clicked()),
            this, SLOT(backButtonSlot()));
    connect(ui->stopButton, SIGNAL(clicked()),
            this, SLOT(stopButtonSlot()));
    connect(ui->hideButton, SIGNAL(clicked()),
           this, SIGNAL(hideSignal()));
}

EventsWidget::~EventsWidget()
{
    delete ui;
}

void EventsWidget::addEvent(QString path)
{
    QString action;
    QString lastModified;

    Q_ASSERT(mainWindow != 0);

    if (mainWindow->fswatch->isNew(path)) {
        action = tr("content modified");
    }
    else if (mainWindow->fswatch->isDeleted(path)) {
        action = tr("deleted");
    }
    else {
        action = tr("modified");
    }

    //TODO better date format
    lastModified = QFileInfo(path).lastModified().toString();

    //TODO: workaround... is this really needed?
    if (lastModified.isEmpty())
        action = tr("deleted");

    addToTable(path, action, lastModified);
}


//-----------------------------------------------------------------------------
// Public slots
//-----------------------------------------------------------------------------

void EventsWidget::addToTableSlot(QString path)
{
    //TODO improve time format string
    addToTable(path, tr("created"), QFileInfo(path).lastModified().toString());
}


//-----------------------------------------------------------------------------
// Private slots
//-----------------------------------------------------------------------------

void EventsWidget::stopButtonSlot()
{
    ui->backbutton->setEnabled(true);
    ui->stopButton->setEnabled(false);
    emit stopButtonClicked();
}

void EventsWidget::backButtonSlot()
{
    ui->stopButton->setEnabled(true);
    ui->backbutton->setEnabled(false);
    //ui->eventsTable->clearContents(); crash! :-(
    emit backButtonClicked();
}


//-----------------------------------------------------------------------------
// Private
//-----------------------------------------------------------------------------

void EventsWidget::addToTable(QString path, QString action, QString lastModified)
{
    QTableWidgetItem *pathItem;
    QTableWidgetItem *actionItem;
    QTableWidgetItem *lastModifiedItem;
    bool pathAlreadyAdded = false;
    int row_found;

    //tray message
    QString message;
    message.append(QFileInfo(path).fileName() + tr(" "));

    //avoid duplicates
    for (int i = 0; i < ui->eventsTable->rowCount(); i++) {
        pathItem = ui->eventsTable->item(i, 0);
        if (pathItem->text() == path) {
            lastModifiedItem = ui->eventsTable->item(i, 2);
            actionItem = ui->eventsTable->item(i, 1);
            pathAlreadyAdded = true;
            row_found = i;
        }
    }

    if (!pathAlreadyAdded) {
        pathItem = new QTableWidgetItem();
        actionItem = new QTableWidgetItem();
        lastModifiedItem = new QTableWidgetItem();

        pathItem->setText(path);
        pathItem->setIcon(QFileIconProvider().icon(QFileInfo(path)));
        actionItem->setTextAlignment(Qt::AlignCenter);
        lastModifiedItem->setTextAlignment(Qt::AlignCenter);

        lastModifiedItem->setText(lastModified);
        actionItem->setText(action);

        ui->eventsTable->setRowCount(ui->eventsTable->rowCount() + 1);
        ui->eventsTable->setItem(ui->eventsTable->rowCount() - 1, 0, pathItem);
        ui->eventsTable->setItem(ui->eventsTable->rowCount() - 1, 1, actionItem);
        ui->eventsTable->setItem(ui->eventsTable->rowCount() - 1, 2, lastModifiedItem);
    } else {
        lastModifiedItem->setText(lastModified);
        actionItem->setText(action);
    }

    message.append(action + tr("..."));
    emit trayMessage(message);
}
