/*
 * $Id: chip_gen_vga.c,v 1.26 2012-02-22 09:27:19 siflkres Exp $ 
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fixme.h"
#include "glue-main.h"
#include "glue-log.h"
#include "glue-shm.h"
#include "glue-suspend.h"

#include "chip_gen_vga.h"

#define SZ_VIDEO_MEMORY		(8*1024*1024) /* Maximum size. */

#define PA_VGABIOS		0x000c0000
#define SZ_VGABIOS		(64*1024) /* Maximum size of ROM */

#define PA_VIDEO_MEMORY_LOW	0x000a0000
#define SZ_VIDEO_MEMORY_LOW	(128*1024)

#define PA_VIDEO_MEMORY_HIGH	(0x40000000)
#define SZ_VIDEO_MEMORY_HIGH	SZ_VIDEO_MEMORY

/*
 * With a given VSYNC rate, a total refresh rate of
 *
 *         VSYNC * (REFRESH_CYCLES + 1) Hz
 *
 * is resulting.
 */

#define VSYNC 5
#define REFRESH_CYCLES 25

struct cpssp {
	/*
	 * Ports
	 */
	struct sig_video *port_video;
	struct sig_cs *port_rom;

	/*
	 * State
	 */
	unsigned int state_power;
	enum {
		VGA,
		VESA_FAUM,
		VBE_BOCHS
	} mode;

	unsigned char colregs[256][3];

	unsigned long sz_vga_bios;
	unsigned char *vga_memory;

#define STATE
#define NAME		vesa
#define NAME_(x)	vesa_ ## x
#define SNAME		"vesa"
#include "arch_vesa_faum.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef STATE

#define STATE
#define NAME		vga
#define NAME_(x)	vga_ ## x
#define SNAME		"vga"
#include "arch_vga.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef STATE
};

/* ----------------------------------------------------------------- */
/* Generic part.                                                     */
/* ----------------------------------------------------------------- */

static inline __attribute__((always_inline)) unsigned char
video_readb(struct cpssp * cpssp, unsigned long pa)
{
	return cpssp->vga_memory[pa];
}

static inline __attribute__((always_inline)) void
video_writeb(struct cpssp * cpssp, unsigned long pa, unsigned char val)
{
	cpssp->vga_memory[pa] = val;
}

static inline __attribute__((always_inline)) unsigned char
video_col_get(struct cpssp * cpssp, unsigned int nr, unsigned int rgb)
{
	return cpssp->colregs[nr][rgb];
}

static inline __attribute__((always_inline)) void
video_col_set(struct cpssp * cpssp, unsigned int nr, unsigned int rgb, unsigned char val)
{
	cpssp->colregs[nr][rgb] = val;
}

static inline __attribute__((always_inline)) int 
video_map(
	struct cpssp * cpssp,
	unsigned long pa,
	char **haddr_mr_p,
	char **haddr_mw_p
)
{
	*haddr_mr_p =
	*haddr_mw_p = cpssp->vga_memory + pa;
	return 0;
}

/* ----------------------------------------------------------------- */
/* Special parts.                                                    */
/* ----------------------------------------------------------------- */

#define ARCH_VGA_PORT	0x3c0

#define BEHAVIOUR
#define NAME		vesa
#define NAME_(x)	vesa_ ## x
#define SNAME		"vesa"
#include "arch_vesa_faum.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef BEHAVIOUR

#define BEHAVIOUR
#define NAME		vga
#define NAME_(x)	vga_ ## x
#define SNAME		"vga"
#include "arch_vga.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef BEHAVIOUR

/* ----------------------------------------------------------------- */
/* Generic part.                                                     */
/* ----------------------------------------------------------------- */

static int
chip_gen_vga_inb(void *_cpssp, unsigned char *valuep, unsigned short port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (vga_inb(cpssp, valuep, port) == 0) {
		return 0;
	}
	return 1;
}

static int
chip_gen_vga_inw(void *_cpssp, unsigned short *valuep, unsigned short port)
{
	return 1;
}

static int
chip_gen_vga_outb(void *_cpssp, unsigned char value, unsigned short port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (vga_outb(cpssp, value, port) == 0) {
		return 0;
	}
	return 1;
}

static int
chip_gen_vga_outw(void *_cpssp, unsigned short value, unsigned short port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (vesa_outw(cpssp, value, port) == 0
	 || vga_outw(cpssp, value, port) == 0) {
		return 0;
	}
	return 1;
}

static int
chip_gen_vga_read(void *_cpssp, unsigned long pa, void *to, int len)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->state_power) {
		switch (cpssp->mode) {
		case VGA:
			if (vga_read(cpssp, pa, to, len) == 0) {
				return 0;
			}
			break;
		case VESA_FAUM:
			if (vesa_read(cpssp, pa, to, len) == 0) {
				return 0;
			}
			break;
		case VBE_BOCHS:
			assert(0);	/* FIXME VOSSI */
			break;
		default:
			assert(0);
		}

		if (PA_VGABIOS <= pa
				&& pa <= PA_VGABIOS + cpssp->sz_vga_bios - 1) {
			/*
			 * VGA BIOS
			 *
			 * Map directly.
			 */
			return 1;

		} else {
			/* Don't know... */
			return 1;
		}
	} else {
		return 1;
	}
}

static int
chip_gen_vga_readb(void *_cpssp, uint32_t addr, uint8_t *valp)
{
	return chip_gen_vga_read(_cpssp, addr, valp, 1);
}

static int
chip_gen_vga_readw(void *_cpssp, uint32_t addr, uint16_t *valp)
{
	return chip_gen_vga_read(_cpssp, addr, valp, 2);
}


static int
chip_gen_vga_write(void *_cpssp, unsigned long pa, const void *from, int len)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->state_power) {
		switch (cpssp->mode) {
		case VGA:
			if (vga_write(cpssp, pa, from, len) == 0) {
				return 0;
			}
			break;
		case VESA_FAUM:
			if (vesa_write(cpssp, pa, from, len) == 0) {
				return 0;
			}
			break;
		case VBE_BOCHS:
			assert(0);	/* FIXME VOSSI */
			break;
		default:
			assert(0);
		}

		if (PA_VGABIOS <= pa
				&& pa <= PA_VGABIOS + cpssp->sz_vga_bios - 1) {
			/*
			 * VGA BIOS
			 *
			 * Map directly.
			 */
			return 1;

		} else {
			/* Don't know... */
			return 1;
		}
	} else {
		return 1;
	}
}

static int
chip_gen_vga_writeb(void *_cpssp, uint32_t addr, uint8_t val)
{
	return chip_gen_vga_write(_cpssp, addr, &val, 1);
}

static int
chip_gen_vga_writew(void *_cpssp, uint32_t addr, uint16_t val)
{
	return chip_gen_vga_write(_cpssp, addr, &val, 2);
}

static int
chip_gen_vga_map(
	void *_cpssp,
	unsigned long pa,
	char **haddr_mr_p,
	char **haddr_mw_p
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (vga_map(cpssp, pa, haddr_mr_p, haddr_mw_p) == 0) {
		return 0;
	}
	if (vesa_map(cpssp, pa, haddr_mr_p, haddr_mw_p) == 0) {
		return 0;
	}
	if (PA_VGABIOS <= pa
	 && pa <= PA_VGABIOS + cpssp->sz_vga_bios - 1) {
		/*
		 * VGA BIOS
		 *
		 * Map directly.
		 */
		return sig_cs_map(
				cpssp->port_rom,
				cpssp,
				pa - PA_VGABIOS,
				haddr_mr_p,
				haddr_mw_p);

	} else {
		/* Don't know... */
		return 1;
	}
}

static void
chip_gen_vga_power_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (val) {
		memset(cpssp->vga_memory, 0, SZ_VIDEO_MEMORY);
	}

	cpssp->state_power = val;
}

static void
chip_gen_vga_n_reset_set(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->mode = VGA;
}

static void
do_video(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->state_power) {
		switch (cpssp->mode) {
		case VGA:
			/* VGA Mode */
			vga_gen(cpssp, cpssp->port_video);
			break;

		case VESA_FAUM:
			/* VESA Mode */
			vesa_gen(cpssp, cpssp->port_video);
			break;

		case VBE_BOCHS:
			/* Bochs VBE Mode */
			assert(0);	/* FIXME VOSSI */
			break;

		default:
			assert(0);	/* Mustn't happen. */
		}
	} else {
		/* Power Off */
		sig_video_no_sync(cpssp->port_video, cpssp);
	}

	time_call_after(TIME_HZ / ((REFRESH_CYCLES + 1) * VSYNC), do_video, cpssp);
}

void *
chip_gen_vga_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_boolean *port_power,
	struct sig_boolean *port_reset_hash_,
	struct sig_isa_bus_main *port_bus,
	struct sig_video *port_video,
	struct sig_cs *port_rom
)
{
	static const struct sig_boolean_funcs power_funcs = {
		.set = chip_gen_vga_power_set,
	};
	static const struct sig_boolean_funcs reset_hash__funcs = {
		.set = chip_gen_vga_n_reset_set,
	};
	static const struct sig_isa_bus_main_funcs bus_funcs = {
		inb:	chip_gen_vga_inb,
		inw:	chip_gen_vga_inw,
		outb:	chip_gen_vga_outb,
		outw:	chip_gen_vga_outw,

		readb:	chip_gen_vga_readb,
		readw:	chip_gen_vga_readw,
		writeb:	chip_gen_vga_writeb,
		writew:	chip_gen_vga_writew,
		map:	chip_gen_vga_map,
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);
	cpssp->vga_memory = shm_palloc(SZ_VIDEO_MEMORY, 0);
	assert(cpssp->vga_memory);

	cpssp->sz_vga_bios = SZ_VGABIOS;

	vga_init(cpssp);
	vesa_init(cpssp);

	/* Out */
	/* Call */
	sig_isa_bus_main_connect(port_bus, cpssp, &bus_funcs);

	cpssp->port_video = port_video;

	cpssp->port_rom = port_rom;

	/* In */
	cpssp->state_power = 0;
	sig_boolean_connect_in(port_power, cpssp, &power_funcs);

	sig_boolean_connect_in(port_reset_hash_, cpssp, &reset_hash__funcs);

	time_call_after(TIME_HZ / ((REFRESH_CYCLES + 1) * VSYNC), do_video, cpssp);

	return cpssp;
}

void
chip_gen_vga_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_pfree(cpssp->vga_memory);
	shm_free(cpssp);
}

void
chip_gen_vga_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
	
	if (fwrite(cpssp->vga_memory, SZ_VIDEO_MEMORY, 1, fComp) != 1){
		fprintf(stderr, "fwrite error\n");
	}
}

void
chip_gen_vga_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
	
	if (fread(cpssp->vga_memory, SZ_VIDEO_MEMORY, 1, fComp) != 1){
		fprintf(stderr, "fread error\n");
	}
}
