/*******************************************************************
 * libfaxophone                                                    *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file faxophone.h
 * \brief faxophone main header
 */

#ifndef FAXOPHONE_H
#define FAXOPHONE_H

/* CAPI headers */
#include <capi20.h>

#ifdef __linux__
#include <linux/capi.h>
#else

typedef unsigned short uint16_t;
typedef unsigned int uint32_t;

typedef struct capi_profile {
	uint16_t ncontroller;
	uint16_t nbchannel;
	uint32_t goptions;
	uint32_t support1;
	uint32_t support2;
	uint32_t support3;
	uint32_t reserved[6];
	uint32_t manu[5];
} capi_profile;
#endif

/* GTK */
#include <gtk/gtk.h>

/* C */
#include <ctype.h>
#include <unistd.h>

/* Faxophone */
#include <faxophone/debug.h>

#include <sndfile.h>

#define CAPI_CONNECTIONS		5
/* Packet size */
#define CAPI_PACKETS			160
/* Packer buffer count */
#define CAPI_BUFFERCNT			6
/* max. B-Channels */
#define CAPI_BCHANNELS			2

#define isdnLock()	do { g_mutex_lock( psSession -> psIsdnMutex ); } while ( 0 );
#define isdnUnlock()	do { g_mutex_unlock( psSession -> psIsdnMutex ); } while ( 0 );

#if GLIB_CHECK_VERSION(2, 31, 0)
#undef CREATE_THREAD
#define CREATE_THREAD(name, func, data) g_thread_new(name, func, data)
#else
#undef CREATE_THREAD
#define CREATE_THREAD(name, func, data) g_thread_create(func, data, FALSE, NULL)
#endif

enum eState {
	STATE_IDLE = 0,
	STATE_CONNECT_REQ,
	STATE_CONNECT_WAIT,
	STATE_CONNECT_ACTIVE,
	STATE_CONNECT_B3_WAIT,
	STATE_CONNECTED,
	STATE_DISCONNECT_B3_REQ,
	STATE_DISCONNECT_B3_WAIT,
	STATE_DISCONNECT_ACTIVE,
	STATE_DISCONNET_WAIT,
	STATE_RINGING,
	STATE_INCOMING_WAIT,
	STATE_MAXSTATE
};

enum eSession {
	SESSION_NONE,
	SESSION_FAX,
	SESSION_PHONE
};

#define RECORDING_BUFSIZE	32768
#define RECORDING_JITTER	200

enum eRecording {
	RECORDING_LOCAL,
	RECORDING_REMOTE
};

struct sRecordChannel {
	unsigned long nPosition;
	short anBuffer[ RECORDING_BUFSIZE ];
};

struct sRecorder {
	SNDFILE *psFile;
	char *pnFileName;

	unsigned long nStartTime;
	struct sRecordChannel sLocal;
	struct sRecordChannel sRemote;
	unsigned long nLastWrite;
};

struct sCapiConnection {
	enum eState nState;
	enum eSession nType;

	unsigned int nId;
	unsigned int nController;
	unsigned long int nPlci;
	unsigned long int nNcci;
	unsigned int nCapiCode;
	char *pnSource;
	char *pnTarget;
	void *pPrivate;
	int nEarlyB3;
	int nHold;
	time_t nConnectTime;
	int nMute;
	int nRecording;
	struct sRecorder sRecorder;

	void (*Data)( struct sCapiConnection *psConnection, _cmsg sCapiMessage );
	void (*Clean)( struct sCapiConnection *psConnection );
};

struct sSession;

struct sSessionHandlers {
	int (*AudioOpen)( void );
	int (*AudioInput)( unsigned char *pnBuf, unsigned int nLen );
	int (*AudioOutput)( unsigned char *pnBuf, unsigned int nLen );
	int (*AudioClose)( void );

	void (*Connected)( struct sCapiConnection *psConnection );
	void (*Disconnected)( struct sCapiConnection *psConnection );
	void (*Ring)( struct sCapiConnection *psConnection );

	void (*Code)( struct sCapiConnection *psConnection, int nCode );

	void (*Status)( struct sCapiConnection *psConnection, int nErrorCode );
};

struct sSession {
	GMutex *psIsdnMutex;

	struct sCapiConnection asConnection[ CAPI_CONNECTIONS ];
	int nApplId;
	int nMessageNumber;
	int nInputThreadState;

	struct sSessionHandlers *psHandlers;
};

struct sCapiConnection *capiGetFreeConnection( void );
struct sCapiConnection *capiCall( unsigned, const char *, const char *, unsigned, unsigned, unsigned );
void capiSendDtmfCode( struct sCapiConnection *psConnection, unsigned char nCode );
void capiHangup( struct sCapiConnection *psConnection );
int capiPickup( struct sCapiConnection *psConnection, int nType );

struct sSession *faxophoneGetSession( void );
struct sSession *faxophoneInit( struct sSessionHandlers *psHandlers, const char *pnHost );
int faxophoneClose( void );

#endif
