################################################################################
# BSD LICENSE
#
# Copyright(c) 2019-2023 Intel Corporation. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
#   * Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#   * Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#   * Neither the name of Intel Corporation nor the names of its
#     contributors may be used to endorse or promote products derived
#     from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
################################################################################

"""
REST API module
Exceptions
"""


from werkzeug.exceptions import HTTPException, ServiceUnavailable


class RestError(HTTPException):
    """
    RestError exception base class
    """


    def __init__(self, code, description):
        HTTPException.__init__(self)
        self.code = code
        self.description = description


class NotFound(RestError):
    """
    NotFound exception
    """


    def __init__(self, description="Not Found"):
        RestError.__init__(self, 404, description)


class BadRequest(RestError):
    """
    BadRequest exception
    """

    def __init__(self, description="Bad Request"):
        RestError.__init__(self, 400, description)


class InternalError(RestError):
    """
    InternalError exception
    """

    def __init__(self, description="Internal Server Error"):
        RestError.__init__(self, 500, description)


class MethodNotAllowed(RestError):
    """
    Method Not Allowed exception
    """

    def __init__(self, description="Method Not Allowed"):
        RestError.__init__(self, 405, description)


class Reconfiguring(ServiceUnavailable):
    """
    App Qos is reconfiguring
    """
    def __init__(self):
        desc = "App QoS is reconfiguring, please try again later"
        ServiceUnavailable.__init__(self, description=desc, retry_after=1)
