///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file NearestNeighborList.h
 * \brief Contains the definition of the AtomViz::NearestNeighborList class.
 */

#ifndef __NEAREST_NEIGHBOR_LIST_H
#define __NEAREST_NEIGHBOR_LIST_H

#include <core/Core.h>
#include <core/reference/RefTarget.h>
#include <core/utilities/ProgressIndicator.h>
#include <core/gui/properties/PropertiesEditor.h>
#include <atomviz/AtomViz.h>

namespace AtomViz {

class AtomsObject;		// defined in AtomsObject.h

/**
 * \brief This class can calculate and store the list of nearest neighbors
 *        for each of the atoms in an AtomsObject.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT NearestNeighborList : public RefTarget
{
public:

	struct NeighborListAtom;		// defined below

	// Stores the pointers to the neighbor atoms of an atom.
	typedef QVarLengthArray<NeighborListAtom*, 16> NeighborList;

	// A internal atom structure.
	struct NeighborListAtom {
		/// The nearest neighbors of the atom.
		NeighborList neighbors;
		/// The next atom in the linked list used for binning.
		NeighborListAtom* nextInBin;
		/// The index of the atom.
		int index;
		/// The wrapped positions of the atoms.
		Point3 pos;

		/// Returns true if the given atom is in the list of near-neighbors of this atom.
		bool hasNeighbor(const NeighborListAtom* other) const {
			const NeighborListAtom* const* a = neighbors.constData();
			for(int c = neighbors.size(); c--; ++a)
				if(*a == other) return true;
			return false;
		}
	};

public:

	/// \brief Default constructor.
	NearestNeighborList(bool isLoading = false);

	/// \brief Build the nearest neighbor list.
	/// \param atoms The atoms object for which the neighbor list should be built.
	/// \param noProgressIndicator If set to \c true then no progress indicator is created to
	///                            show the build progress. In this case the building cannot be canceled by the user.
	/// \return \c false when the operation has been canceled by the user;
	///         \c true on success.
	/// \throw Exception on error.
	bool build(AtomsObject* atoms, bool noProgressIndicator = false);

	/// \brief Returns the list of neighbor lists.
	/// \return The returned vector contains the neighbor list for each of the atoms.
	const vector<NeighborListAtom>& neighborLists() const { return atoms; }

	/// \brief Returns whether the neighbor list has been built.
	/// \return \c true if the nearest-neighbor lists have been built via build();
	///         \c false otherwise.
	/// \sa clear()
	bool isValid() const { return !atoms.empty(); }

	/// \brief Resets the nearest-neighbor lists.
	void clear() { atoms.clear(); }

	/// \brief Returns the cutoff radius used to build the nearest neighbor lists.
	/// \return The cutoff radius in world units.
	/// \sa setNearestNeighborCutoff()
	FloatType nearestNeighborCutoff() const { return _nearestNeighborCutoff; }

	/// \brief Sets the cutoff radius used to build the nearest neighbor lists.
	/// \param newCutoff The new cutoff radius in world units.
	/// \undoable
	/// \sa nearestNeighborCutoff()
	void setNearestNeighborCutoff(FloatType newCutoff) { _nearestNeighborCutoff = newCutoff; }

public:

	Q_PROPERTY(FloatType nearestNeighborCutoff READ nearestNeighborCutoff WRITE setNearestNeighborCutoff)

private:

	/// This helper class is used to split up the computation into small
	/// operations that can be performed on multiple processors in parallel.
	class Kernel {
	public:
		// Constructor that takes references to the input and output arrays.
		Kernel(const QVector<NeighborListAtom*>& _bins, const array<int,3>& _binDim, const AffineTransformation& _simCell, const array<bool,3>& _pbc, FloatType _cutoffRadiusSquared, const Vector3I& _offset) : bins(_bins), binDim(_binDim), simCell(_simCell), pbc(_pbc), cutoffRadiusSquared(_cutoffRadiusSquared), offset(_offset) {}

		// The actual kernel function that is called by the Qt concurrent framework for each neighbor list bin.
		void operator()(const Point3I& bin);

	private:
		/// The current bin cell offset.
		const Vector3I offset;
		/// Input bin array that contains all atoms.
		const QVector<NeighborListAtom*>& bins;
		/// The dimensions of the bin array.
		const array<int,3> binDim;
		/// The simulation cell dimensions
		const AffineTransformation simCell;
		/// The squared cutoff radius
		const FloatType cutoffRadiusSquared;
		/// Periodic boundary condition flags
		const array<bool,3> pbc;
	};

	/// The neighbor lists for each atom.
	vector<NeighborListAtom> atoms;

	/// Controls the cutoff radius for the nearest neighbor list building.
	PropertyField<FloatType> _nearestNeighborCutoff;

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(NearestNeighborList)
	DECLARE_PROPERTY_FIELD(_nearestNeighborCutoff)
};

/**
 * \brief A properties editor for the NearestNeighborList class.
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT NearestNeighborListEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

protected Q_SLOTS:

	/// Is called when the user has selected an item in the cutoff presets box.
	void onSelectNearestNeighborPreset(int index);

	/// Stores the current cutoff radius in the application settings
	/// so it can be used as default value for new modifiers.
	void memorizeCutoff();

private:

	/// The selection box with the presets for the nearest-neighbor cutoff.
	QComboBox* nearestNeighborPresetsBox;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(NearestNeighborListEditor)
};

};	// End of namespace AtomViz

#endif // __NEAREST_NEIGHBOR_LIST_H
