///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file MainFrame.h 
 * \brief Contains the definition of the Core::MainFrame class. 
 */
 
#ifndef __OVITO_MAINFRAME_H
#define __OVITO_MAINFRAME_H

#include <core/Core.h>
#include <core/rendering/FrameBufferWindow.h>

namespace Core {

class ViewportPanel;	// defined in ViewportPanel.h
class MainMenu;			// defined in MainMenu.h
class CommandPanel;		// defined in CommandPanel.h
class FrameBufferWindow;// defined in FrameBufferWindow.h

/// \def MAIN_FRAME
/// \brief The global instance of the Core::MainFrame class.
/// 
/// Always use this macro to access the Core::MainFrame class instance.
///
/// Please not that in non-gui mode no instance of the main window class is created.
/// It is then illegal to use the MAIN_FRAME macro.
#define MAIN_FRAME		MainFrame::getSingletonInstance()

/**
 * \brief The main window of the application.
 * 
 * There is a global instance of this class creating on application startup. 
 * You can access the instance of this class using the \c MAIN_FRAME macro.
 * 
 * Please not that in non-gui mode no instance of the main window class is created.
 * It is then illegal to use the \c MAIN_FRAME macro.
 * 
 * \author Alexander Stukowski
 * 
 * \sa ApplicationManager::guiMode()
 */
class CORE_DLLEXPORT MainFrame : public QMainWindow
{
	Q_OBJECT
	
public:

	/// \brief Returns the one and only instance of this class.
	/// \return The global instance of the MainFrame class.
	/// \note Generally you should use the MAIN_FRAME macro to access the MainFrame instance instead
	///       of this method.
	inline static MainFrame* getSingletonInstance() {
		OVITO_ASSERT_MSG(QApplication::type() == QApplication::GuiClient, "MainFrame", "No main window available in non-gui mode."); 
		OVITO_ASSERT_MSG(_singletonInstance, "MainFrame::getSingletonInstance", "MainFrame class is not initialized yet.");
		return _singletonInstance;
	}

	/// Returns the viewport panel containing all viewports.
	ViewportPanel* viewportPanel() const { OVITO_ASSERT(_viewportPanel != NULL); return _viewportPanel; }

	/// Returns the menu of the main window.
	MainMenu* mainMenu() const { return (MainMenu*)menuBar(); }

	/// Returns the main toolbar of the window.
	QToolBar* mainToolbar() const { return _mainToolbar; }
	
	/// Returns the command panel.
	CommandPanel* commandPanel() const { return _commandPanel; }
	
	/// Returns the status bar of the main window.
	QStatusBar* statusBar() const { return _statusBar; }

	/// Returns the frame buffer window showing the rendered image.
	FrameBufferWindow* frameBufferWindow() const { return _frameBufferWindow; }

	/// Returns the recommended size for this widget.
	virtual QSize sizeHint() const { return QSize(1024,768); }

	/// \brief Opens the external Qt Assistant program to display online help.
	/// \param topic_identifier The unique identifier of the help topic to show.
	/// \return The process of the external Qt Assistant.
	/// \throw Exception on error.
	QProcess* showQtAssistant(const char* topic_identifier = NULL);	
	
private:

	/// Private constructor.
	/// This is a singleton class; creating additional instances is not allowed.
	MainFrame(const QString& title);

	/// The upper main toolbar.
	QToolBar* _mainToolbar;
	
	/// The panel containing the viewports
	ViewportPanel* _viewportPanel;

	/// The command panel.
	CommandPanel* _commandPanel;
	
	/// The internal status bar widget.
	QStatusBar* _statusBar;
	
	/// The frame buffer window showing the rendered image.
	QPointer<FrameBufferWindow> _frameBufferWindow;

	/// The Qt-Assistant used to display online help.
	QProcess* _qtAssistant;

	/// Creates and fills the main toolbar.
	void createMainToolbar();

private:

	/// The global instance of this window class.
	static MainFrame* _singletonInstance;

protected:

	/// Is called when the user closes the window.
	virtual void closeEvent(QCloseEvent* event);
	
	/// Is called when the window receives an event.
	virtual bool event(QEvent *event);

	friend class ApplicationManager;
};


};

#endif // __OVITO_MAINFRAME_H
