///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __OVITO_COLOR_CONTROLLER_UI_H
#define __OVITO_COLOR_CONTROLLER_UI_H

#include <core/Core.h>
#include "ParameterUI.h"
#include "../ColorPickerWidget.h"

namespace Core {

/******************************************************************************
* Allows the user to edit a color parameter in the properties panel.
******************************************************************************/
class CORE_DLLEXPORT ColorControllerUI : public ReferenceParameterUI
{
public:
	/// Constructor.
	ColorControllerUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField);
	
	/// Destructor, that releases all GUI controls.
	virtual ~ColorControllerUI();
	    
	/// This returns the color picker widget managed by this parameter UI.
	ColorPickerWidget* colorPicker() const { return _colorPicker; }

	/// This returns a label for the color picker managed by this ColorControllerUI.
	/// The text of the label widget is taken from the description text stored along 
	/// with the parameter reference field.
	QLabel* label() const { return _label; }

	/// This method is called when parameter object has been assigned to the reference field of the editable object 
	/// this parameter UI is bound to. It is also called when the editable object itself has
	/// been replaced in the editor. This implementation of the method automatically
	/// disables the UI widgets when the reference field of the edited object does not contain a
	/// controller.
	virtual void resetUI();
	
	/// This method updates the displayed value of the parameter UI.
	virtual void updateUI();	
	
	/// Sets the enabled state of the UI.
	virtual void setEnabled(bool enabled);
	
	/// Sets the What's This helper text for the label and the color picker.
	void setWhatsThis(const QString& text) const { 
		if(label()) label()->setWhatsThis(text); 
		if(colorPicker()) colorPicker()->setWhatsThis(text); 
	}
	
public:
	
	Q_PROPERTY(QLabel* label READ label)		
	Q_PROPERTY(QWidget* colorPicker READ colorPicker)		
	
public Q_SLOTS:
	
	/// Is called when the user has changed the color.
	void onColorPickerChanged();

protected:

	/// The color picker control of the UI component.
	QPointer<ColorPickerWidget> _colorPicker;

	/// The label of the UI component.
	QPointer<QLabel> _label;
	
private:

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(ColorControllerUI)
};

};

#endif // __OVITO_COLOR_CONTROLLER_UI_H
