#include <gtest/gtest.h>

#include <vector>
#include <algorithm>
#include <stdlib.h>

#include <stk_topology/topology.hpp>
#include <stk_mesh/base/BulkData.hpp>
#include <stk_mesh/base/MetaData.hpp>
#include <stk_mesh/base/Comm.hpp>
#include <stk_mesh/base/GetEntities.hpp>
#include <stk_mesh/base/FEMHelpers.hpp>
#include <stk_mesh/baseImpl/MeshImplUtils.hpp>
#include <stk_mesh/base/FieldTraits.hpp>
#include <stk_mesh/base/Field.hpp>

#include <stk_mesh/base/SkinMesh.hpp>
#include <stk_mesh/base/CreateFaces.hpp>
#include <stk_mesh/base/ElemElemGraph.hpp>
#include <stk_mesh/base/ElemElemGraphImpl.hpp>

#include <stk_util/parallel/Parallel.hpp>
#include <stk_util/parallel/ParallelVectorConcat.hpp>
#include <stk_util/parallel/ParallelComm.hpp>
#include <stk_util/environment/WallTime.hpp>
#include <stk_util/environment/memory_util.hpp>
#include <stk_util/parallel/CommSparse.hpp>
#include <stk_util/environment/ReportHandler.hpp>

#include <stk_io/IossBridge.hpp>
#include <stk_io/StkMeshIoBroker.hpp>   // for StkMeshIoBroker

#include <stk_unit_test_utils/ioUtils.hpp>
#include <stk_unit_test_utils/getOption.h>

#include "UnitTestElementDeathUtils.hpp"
#include "stk_unit_test_utils/unittestMeshUtils.hpp"
#include <stk_unit_tests/stk_mesh/SetupKeyholeMesh.hpp>

#include <stk_mesh/fixtures/QuadFixture.hpp>  // for QuadFixture
#include <stk_mesh/fixtures/heterogeneous_mesh.hpp>
#include <stk_mesh/fixtures/degenerate_mesh.hpp>

#include "BulkDataElementGraphTester.hpp"
#include "ElementGraphTester.hpp"

namespace {

using namespace stk::mesh::impl;
using namespace stk::mesh;

// Not to be used with ElemElemGraph or ElemElemGraphTester class.
bool is_valid_graph_element(const impl::ElementGraph &elem_graph, stk::mesh::impl::LocalId elem_id);

// Not to be used with ElemElemGraph or ElemElemGraphTester class.
int check_connectivity(const impl::ElementGraph &elem_graph, const impl::SidesForElementGraph &via_sides,
                       stk::mesh::impl::LocalId element_id1, stk::mesh::impl::LocalId element_id2);

// Not to be used with ElemElemGraph or ElemElemGraphTester class.
int get_side_from_element1_to_element2(const impl::ElementGraph &elem_graph,
                                       const impl::SidesForElementGraph &via_sides,
                                       stk::mesh::impl::LocalId element1_local_id,
                                       stk::mesh::impl::LocalId other_element_id);

void create_faces_using_graph(BulkDataElementGraphTester& bulkData, stk::mesh::Part& part);

ElemElemGraphTester create_base_1x1x4_elem_graph(stk::ParallelMachine &comm, stk::mesh::BulkData &bulk);

void setup_node_sharing(stk::mesh::BulkData &mesh, const std::vector< std::vector<unsigned> > & shared_nodeIDs_and_procs );

ElemElemGraphTester test_add_elements_to_pre_existing_graph_and_mesh(stk::mesh::BulkData &bulkData);

void test_add_elements_to_empty_graph(stk::mesh::BulkData::AutomaticAuraOption auto_aura_option)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    int pSize = stk::parallel_machine_size(comm);

    if (pSize <= 4)
    {
        stk::mesh::MetaData meta(3);
        stk::mesh::BulkData bulkData(meta, comm, auto_aura_option);
        int nProc  = stk::parallel_machine_size(comm);
        int myProc = stk::parallel_machine_rank(comm);

        ElemElemGraphTester elem_graph(bulkData);

        EXPECT_EQ(0u, elem_graph.size());

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        std::vector<unsigned> counts;
        stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
        unsigned numLocallyOwnedElems = counts[stk::topology::ELEM_RANK];

        if(myProc != 0)
        {
            EXPECT_EQ(4u/nProc, numLocallyOwnedElems);
        }
        else
        {
            EXPECT_EQ((4u/nProc + 4u%nProc), numLocallyOwnedElems);
        }

        stk::mesh::EntityVector elements_to_add;
        stk::mesh::Selector selector(bulkData.mesh_meta_data().locally_owned_part());
        bulkData.get_entities(stk::topology::ELEMENT_RANK, selector, elements_to_add);

        elem_graph.add_elements(elements_to_add);

        stk::mesh::Entity elem1 = bulkData.get_entity(stk::topology::ELEM_RANK, 1);
        stk::mesh::Entity elem2 = bulkData.get_entity(stk::topology::ELEM_RANK, 2);
        stk::mesh::Entity elem3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
        stk::mesh::Entity elem4 = bulkData.get_entity(stk::topology::ELEM_RANK, 4);

        // We know that element 1 has 1 connection
        //                      2 has 2 connections
        //                      3 has 2 connections
        //                      4 has 1 connection

        bool ownedElem1 = false;
        bool ownedElem2 = false;
        bool ownedElem3 = false;
        bool ownedElem4 = false;

        unsigned numEdges = 0;
        if(bulkData.is_valid(elem1) && bulkData.bucket(elem1).owned())
        {
            numEdges += 1;
            ownedElem1 = true;
        }
        if(bulkData.is_valid(elem2) && bulkData.bucket(elem2).owned())
        {
            numEdges += 2;
            ownedElem2 = true;
        }

        if(bulkData.is_valid(elem3) && bulkData.bucket(elem3).owned())
        {
            numEdges += 2;
            ownedElem3 = true;
        }
        if(bulkData.is_valid(elem4) && bulkData.bucket(elem4).owned())
        {
            numEdges += 1;
            ownedElem4 = true;
        }

        unsigned numParallelEdges = 0;
        if(ownedElem1)
        {
            if(!ownedElem2)
            {
                numParallelEdges++;
            }
        }
        if(ownedElem2)
        {
            if(!ownedElem1)
            {
                numParallelEdges++;
            }
            if(!ownedElem3)
            {
                numParallelEdges++;
            }
        }
        if(ownedElem3)
        {
            if(!ownedElem2)
            {
                numParallelEdges++;
            }
            if(!ownedElem4)
            {
                numParallelEdges++;
            }
        }
        if(ownedElem4)
        {
            if(!ownedElem3)
            {
                numParallelEdges++;
            }
        }

        EXPECT_EQ(elements_to_add.size(), elem_graph.size());
        EXPECT_EQ(numEdges, elem_graph.num_edges());
        EXPECT_EQ(numParallelEdges, elem_graph.num_parallel_edges());

        if (ownedElem1)
        {
            EXPECT_EQ(5, elem_graph.check_connectivity(1, 2));
            EXPECT_EQ(-1, elem_graph.check_connectivity(1, 3));
            EXPECT_EQ(-1, elem_graph.check_connectivity(1, 4));
        }
        if (ownedElem2)
        {
            EXPECT_EQ(4, elem_graph.check_connectivity(2, 1));
            EXPECT_EQ(5, elem_graph.check_connectivity(2, 3));
            EXPECT_EQ(-1, elem_graph.check_connectivity(2, 4));
        }
        if (ownedElem3)
        {
            EXPECT_EQ(4, elem_graph.check_connectivity(3, 2));
            EXPECT_EQ(5, elem_graph.check_connectivity(3, 4));
            EXPECT_EQ(-1, elem_graph.check_connectivity(3, 1));
        }
        if (ownedElem4)
        {
            EXPECT_EQ(4, elem_graph.check_connectivity(4, 3));
            EXPECT_EQ(-1, elem_graph.check_connectivity(4, 1));
            EXPECT_EQ(-1, elem_graph.check_connectivity(4, 2));
        }
    }
}

TEST(ElementGraph, add_elements_to_empty_graph_aura_on)
{
    test_add_elements_to_empty_graph(stk::mesh::BulkData::AUTO_AURA);
}

TEST(ElementGraph, add_elements_to_empty_graph_aura_off)
{
    test_add_elements_to_empty_graph(stk::mesh::BulkData::NO_AUTO_AURA);
}

void test_similarity_on_both_procs(const std::vector<stk::mesh::EntityId> &ids_in_use, stk::ParallelMachine comm)
{
    std::vector<stk::mesh::EntityId> global_ids_in_use;
    stk::parallel_vector_concat(comm, ids_in_use, global_ids_in_use);
    std::sort(global_ids_in_use.begin(), global_ids_in_use.end());

    std::vector<stk::mesh::EntityId>::iterator iter = std::unique(global_ids_in_use.begin(), global_ids_in_use.end());
    size_t num_unique = iter - global_ids_in_use.begin();
    EXPECT_GE(num_unique, ids_in_use.size());
}

ElemElemGraphTester test_add_elements_to_pre_existing_graph_and_mesh(stk::mesh::BulkData &bulkData)
{
    const stk::ParallelMachine comm = bulkData.parallel();
    const int p_size  = bulkData.parallel_size();
    const int p_rank = bulkData.parallel_rank();

    ThrowRequire(2 == p_size);

    stk::mesh::MetaData &meta = bulkData.mesh_meta_data();

    stk::mesh::Part * hexPart = &meta.declare_part_with_topology("hex_part", stk::topology::HEX_8);

    stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x2", bulkData, comm);

    std::vector<unsigned> counts;
    stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);

    ElemElemGraphTester elem_graph(bulkData);

    EXPECT_EQ(1u, elem_graph.size());
    EXPECT_EQ(1u, counts[stk::topology::ELEM_RANK]);

    const std::vector<size_t> numHex{1,1};
    stk::mesh::EntityIdVector hexNodeIDs[] {
        { 8, 7, 16, 15, 12, 11, 18, 17 },
        { 4, 3, 14, 13,  8,  7, 16, 15 }
    };
    stk::mesh::EntityIdVector hexElemIDs {3, 4};

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0,  3, 1 },  // proc 0
        { 0,  4, 1 },
        { 0, 11, 1 },
        { 0, 12, 1 },
        { 0, 15, 1 },
        { 0, 16, 1 },
        { 1,  3, 0 },  // proc 1
        { 1,  4, 0 },
        { 1, 11, 0 },
        { 1, 12, 0 },
        { 1, 15, 0 },
        { 1, 16, 0 },
    };

    bulkData.modification_begin();
    stk::mesh::declare_element(bulkData, *hexPart, hexElemIDs[p_rank], hexNodeIDs[p_rank]);
    setup_node_sharing(bulkData, shared_nodeIDs_and_procs );
    bulkData.modification_end();

    stk::mesh::EntityVector elements_to_add;

    if (0 == p_rank)
    {
        stk::mesh::Entity elem3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
        elements_to_add.push_back(elem3);
    }
    else
    {
        stk::mesh::Entity elem4 = bulkData.get_entity(stk::topology::ELEM_RANK, 4);
        elements_to_add.push_back(elem4);
    }

    elem_graph.add_elements(elements_to_add);

    if (0 == p_rank)
    {
        EXPECT_EQ(5, elem_graph.check_connectivity(1, 2));
        EXPECT_EQ(2, elem_graph.check_connectivity(1, 4));
        EXPECT_EQ(0, elem_graph.check_connectivity(3, 2));
        EXPECT_EQ(4, elem_graph.check_connectivity(3, 4));
    }
    else
    {
        EXPECT_EQ(4, elem_graph.check_connectivity(2, 1));
        EXPECT_EQ(2, elem_graph.check_connectivity(2, 3));
        EXPECT_EQ(0, elem_graph.check_connectivity(4, 1));
        EXPECT_EQ(5, elem_graph.check_connectivity(4, 3));
    }
    EXPECT_EQ(2u, elem_graph.size());
    EXPECT_EQ(4u, elem_graph.num_edges());
    EXPECT_EQ(4u, elem_graph.num_parallel_edges());

    std::vector<stk::mesh::EntityId> ids = elem_graph.get_copy_of_all_ids();
    std::sort(ids.begin(), ids.end());
    std::vector<stk::mesh::EntityId>::iterator iter = std::unique(ids.begin(), ids.end());
    bool ids_are_unique = iter == ids.end();
    EXPECT_TRUE(ids_are_unique);

    stk::mesh::impl::ParallelGraphInfo p_graph_info = elem_graph.get_parallel_graph_info();
    std::vector<stk::mesh::EntityId> chosen_ids;
    stk::mesh::impl::ParallelGraphInfo::iterator iter1 = p_graph_info.begin();
    for(;iter1!=p_graph_info.end();++iter1)
    {
        chosen_ids.push_back(iter1->second.m_chosen_side_id);
    }

    std::sort(chosen_ids.begin(), chosen_ids.end());
    iter = std::unique(chosen_ids.begin(), chosen_ids.end());
    ids_are_unique = iter == chosen_ids.end();
    EXPECT_TRUE(ids_are_unique);

    test_similarity_on_both_procs(chosen_ids, bulkData.parallel());

    return elem_graph;
}

TEST(ElementGraph, add_elements_to_pre_existing_graph_and_mesh_aura_on)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    if (2 == stk::parallel_machine_size(comm))
    {
        const unsigned spatialDim = 3;
        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::BulkData bulkData(meta, comm, stk::mesh::BulkData::AUTO_AURA);

        test_add_elements_to_pre_existing_graph_and_mesh(bulkData);
    }
}

TEST(ElementGraph, add_elements_to_pre_existing_graph_and_mesh_aura_off)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    if (2 == stk::parallel_machine_size(comm))
    {
        const unsigned spatialDim = 3;
        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::BulkData bulkData(meta, comm, stk::mesh::BulkData::NO_AUTO_AURA);

        test_add_elements_to_pre_existing_graph_and_mesh(bulkData);
    }
}

void test_delete_elements_from_graph(ElemElemGraphTester &elem_graph, std::vector<stk::mesh::EntityId> &ids_to_delete)
{
    std::set<stk::mesh::EntityId> currentElements = {1,2,3,4};
    stk::mesh::BulkData &bulkData = elem_graph.get_bulk_data();

    stk::mesh::impl::DeletedElementInfoVector elements_to_delete;
    for (size_t i = 0; i < ids_to_delete.size(); ++i)
    {
        stk::mesh::EntityId id = ids_to_delete[i];
        currentElements.erase(id);
        stk::mesh::Entity elem = bulkData.get_entity(stk::topology::ELEM_RANK, id);
        if (bulkData.is_valid(elem) && bulkData.bucket(elem).owned())
        {
            elements_to_delete.push_back({elem, id, bulkData.bucket(elem).topology().is_shell()});
        }
    }

    bulkData.modification_begin();
    for (stk::mesh::impl::DeletedElementInfo elem : elements_to_delete)
    {
        bulkData.destroy_entity(elem.entity);
        stk::mesh::Entity elemCheck = bulkData.get_entity(stk::topology::ELEM_RANK, elem.identifier);
        EXPECT_FALSE(bulkData.is_valid(elemCheck));
    }
    bulkData.modification_end();

    elem_graph.delete_elements(elements_to_delete);

    std::vector<unsigned> counts;
    stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
    unsigned numLocallyOwnedElems = counts[stk::topology::ELEM_RANK];

    EXPECT_EQ(numLocallyOwnedElems, elem_graph.size());

    unsigned numEdges = 0;
    unsigned numParallelEdges = 0;
    for (stk::mesh::EntityId elem_id : currentElements)
    {
        stk::mesh::Entity elem = bulkData.get_entity(stk::topology::ELEM_RANK, elem_id);

        if (!bulkData.is_valid(elem) || !bulkData.bucket(elem).owned())
        {
            continue;
        }

        stk::mesh::EntityId leftNeighbor = elem_id - 1;
        if (currentElements.find(leftNeighbor) != currentElements.end())
        {
            EXPECT_EQ(4, elem_graph.check_connectivity(elem_id, leftNeighbor));
            ++numEdges;

            stk::mesh::Entity leftElem = bulkData.get_entity(stk::topology::ELEM_RANK, leftNeighbor);
            bool ownedLeftNeighbor = bulkData.is_valid(leftElem) && bulkData.bucket(leftElem).owned();
            if(!ownedLeftNeighbor)
            {
                numParallelEdges++;
            }
        }

        stk::mesh::EntityId rightNeighbor = elem_id + 1;
        if (currentElements.find(rightNeighbor) != currentElements.end())
        {
            EXPECT_EQ(5, elem_graph.check_connectivity(elem_id, rightNeighbor));
            ++numEdges;

            stk::mesh::Entity rightElem = bulkData.get_entity(stk::topology::ELEM_RANK, rightNeighbor);
            bool ownedRightNeighbor = bulkData.is_valid(rightElem) && bulkData.bucket(rightElem).owned();
            if(!ownedRightNeighbor)
            {
                numParallelEdges++;
            }
        }
    }
    EXPECT_EQ(numEdges, elem_graph.num_edges());
    EXPECT_EQ(numParallelEdges, elem_graph.num_parallel_edges());
}

void test_element_graph_delete_elements_from_graph(stk::mesh::BulkData::AutomaticAuraOption auto_aura_option)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    int pSize = stk::parallel_machine_size(comm);

    // Test designed for 1, 2, or 4 procs.
    if (pSize > 4)
    {
        return;
    }

    {
        stk::mesh::MetaData meta;
        stk::mesh::BulkData bulkData(meta, comm, auto_aura_option);

        ElemElemGraphTester elem_graph = create_base_1x1x4_elem_graph(comm, bulkData);
        std::vector<stk::mesh::EntityId> ids_to_delete = {1, 4};
        test_delete_elements_from_graph(elem_graph, ids_to_delete);
    }

    {
        stk::mesh::MetaData meta;
        stk::mesh::BulkData bulkData(meta, comm, auto_aura_option);

        ElemElemGraphTester elem_graph = create_base_1x1x4_elem_graph(comm, bulkData);
        std::vector<stk::mesh::EntityId> ids_to_delete = {2, 3};
        test_delete_elements_from_graph(elem_graph, ids_to_delete);
    }

    {
        stk::mesh::MetaData meta;
        stk::mesh::BulkData bulkData(meta, comm, auto_aura_option);

        ElemElemGraphTester elem_graph = create_base_1x1x4_elem_graph(comm, bulkData);
        std::vector<stk::mesh::EntityId> ids_to_delete = {1, 3};
        test_delete_elements_from_graph(elem_graph, ids_to_delete);
    }
}

TEST(ElementGraph, delete_elements_from_graph_aura_on)
{
    test_element_graph_delete_elements_from_graph(stk::mesh::BulkData::AUTO_AURA);
}

TEST(ElementGraph, delete_elements_from_graph_aura_off)
{
    test_element_graph_delete_elements_from_graph(stk::mesh::BulkData::NO_AUTO_AURA);
}

TEST(ElementGraph, add_and_delete_elements_from_graph_serial)
{
    MPI_Comm comm = MPI_COMM_WORLD;

    if(stk::parallel_machine_size(comm) == 1)
    {
        stk::mesh::MetaData meta;
        stk::mesh::BulkData bulkData(meta, comm);

        ElemElemGraphTester elem_graph(bulkData);

        EXPECT_EQ(0u, elem_graph.size());
        EXPECT_EQ(0u, elem_graph.num_edges());
        EXPECT_EQ(0u, elem_graph.num_parallel_edges());

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        std::vector<unsigned> counts;
        stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
        int numLocallyOwnedElems = counts[stk::topology::ELEM_RANK];
        EXPECT_EQ(4, numLocallyOwnedElems);

        stk::mesh::Entity elem1 = bulkData.get_entity(stk::topology::ELEM_RANK, 1);
        stk::mesh::Entity elem2 = bulkData.get_entity(stk::topology::ELEM_RANK, 2);
        stk::mesh::Entity elem3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
        stk::mesh::Entity elem4 = bulkData.get_entity(stk::topology::ELEM_RANK, 4);

        stk::mesh::EntityVector elements_to_add;
        elements_to_add.push_back(elem1);
        elements_to_add.push_back(elem2);
        elements_to_add.push_back(elem3);
        elements_to_add.push_back(elem4);

        for (unsigned i=0; i<elements_to_add.size(); i++)
        {
        	EXPECT_TRUE(bulkData.is_valid(elements_to_add[i]));
        	EXPECT_EQ(0, bulkData.parallel_owner_rank(elements_to_add[i]));
        }

        elem_graph.add_elements(elements_to_add);

        EXPECT_EQ(4u, elem_graph.size());
        EXPECT_EQ(6u, elem_graph.num_edges());
        EXPECT_EQ(0u, elem_graph.num_parallel_edges());

        EXPECT_EQ(5, elem_graph.check_local_connectivity(elem1, elem2));
        EXPECT_EQ(4, elem_graph.check_local_connectivity(elem2, elem1));
        EXPECT_EQ(5, elem_graph.check_local_connectivity(elem2, elem3));
        EXPECT_EQ(4, elem_graph.check_local_connectivity(elem3, elem2));
        EXPECT_EQ(5, elem_graph.check_local_connectivity(elem3, elem4));
        EXPECT_EQ(4, elem_graph.check_local_connectivity(elem4, elem3));
        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem1, elem3));
        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem2, elem4));

        stk::mesh::impl::DeletedElementInfoVector elems_to_delete;
        elems_to_delete.push_back({elem2, 2, false});
        elems_to_delete.push_back({elem3, 3, false});

        elem_graph.delete_elements(elems_to_delete);

        EXPECT_EQ(2u, elem_graph.size());
        EXPECT_EQ(0u, elem_graph.num_edges());
        EXPECT_EQ(0u, elem_graph.num_parallel_edges());

        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem1, elem2));
        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem2, elem1));
        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem2, elem3));
        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem3, elem2));
        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem3, elem4));
        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem4, elem3));
        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem1, elem3));
        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem2, elem4));

        elements_to_add.clear();
        elements_to_add.push_back(elem2);
        elements_to_add.push_back(elem3);

        elem_graph.add_elements(elements_to_add);

        EXPECT_EQ(4u, elem_graph.size());
        EXPECT_EQ(6u, elem_graph.num_edges());
        EXPECT_EQ(0u, elem_graph.num_parallel_edges());

        EXPECT_EQ(5, elem_graph.check_local_connectivity(elem1, elem2));
        EXPECT_EQ(4, elem_graph.check_local_connectivity(elem2, elem1));
        EXPECT_EQ(5, elem_graph.check_local_connectivity(elem2, elem3));
        EXPECT_EQ(4, elem_graph.check_local_connectivity(elem3, elem2));
        EXPECT_EQ(5, elem_graph.check_local_connectivity(elem3, elem4));
        EXPECT_EQ(4, elem_graph.check_local_connectivity(elem4, elem3));
        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem1, elem3));
        EXPECT_EQ(-1, elem_graph.check_local_connectivity(elem2, elem4));

        elems_to_delete.clear();
        elems_to_delete.push_back({elem4, 4, false});
        elems_to_delete.push_back({elem2, 2, false});
        elems_to_delete.push_back({elem1, 1, false});
        elems_to_delete.push_back({elem3, 3, false});

        elem_graph.delete_elements(elems_to_delete);

        EXPECT_EQ(0u, elem_graph.size());
        const stk::mesh::impl::ElementGraph &e_graph = elem_graph.get_element_graph();
        for(unsigned i=0; i<e_graph.size(); ++i)
        {
            EXPECT_EQ(0u, e_graph[i].size());
        }

        const stk::mesh::impl::SidesForElementGraph &via_sides = elem_graph.get_via_sides();
        for(unsigned i=0; i<via_sides.size(); ++i)
        {
            EXPECT_EQ(0u, via_sides[i].size());
        }
    }
}

TEST(ElementGraph, HexAddShellSerial)
{
    //  ID.proc
    //
    //          3.0------------7.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      4.0------------8.0  |
    //       |   |          |   |
    //       |   |   1.0    |2.0|
    //       |   |          |   |
    //       |  2.0---------|--6.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------5.0
    //                      ^
    //                      |
    //                       ---- Added single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size != 1) {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5, 6, 7, 8 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 2 };

    // Build the base hex mesh
    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    mesh.modification_end();

    // Initialize the graph based on the existing hex mesh
    ElemElemGraphTester elem_graph(mesh);
    EXPECT_EQ(1u, elem_graph.size());

    // Add a shell
    mesh.modification_begin();
    stk::mesh::EntityVector added_shells;
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        added_shells.push_back( stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]) );
    }
    mesh.modification_end();

    elem_graph.add_elements(added_shells);

    EXPECT_EQ(2u, elem_graph.size());
    EXPECT_EQ(0u, elem_graph.num_parallel_edges());

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity shell2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);

    // Connectivity for Hex Element 1
    EXPECT_EQ(1u,     elem_graph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,      elem_graph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(shell2, elem_graph.get_connected_element(hex1, 0));
    EXPECT_TRUE(elem_graph.is_connected_elem_locally_owned(hex1, 0));

    // Connectivity for Shell Element 2
    EXPECT_EQ(1u,             elem_graph.get_num_connected_elems(shell2));
    EXPECT_EQ(1,              elem_graph.get_side_id_to_connected_element(shell2, 0));
    EXPECT_EQ(hex1, elem_graph.get_connected_element(shell2, 0));
    EXPECT_TRUE(elem_graph.is_connected_elem_locally_owned(shell2, 0));

    EXPECT_EQ(0u, elem_graph.num_parallel_edges());
}

TEST( ElementGraph, HexDelShellSerial )
{
    //  ID.proc
    //
    //          3.0------------7.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      4.0------------8.0  |
    //       |   |          |   |
    //       |   |   1.0    |2.0|
    //       |   |          |   |
    //       |  2.0---------|--6.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------5.0
    //                      ^
    //                      |
    //                       ---- Deleting shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size > 1)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 2 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity shell2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);

    stk::mesh::impl::DeletedElementInfoVector elements_to_delete;
    elements_to_delete.push_back({shell2, 2, mesh.bucket(shell2).topology().is_shell()});

    elemElemGraph.delete_elements(elements_to_delete);

    mesh.modification_begin();
    mesh.destroy_entity(shell2);
    mesh.modification_end();

    // Connectivity for Hex Element 1
    EXPECT_EQ(0u, elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());

}

TEST( ElementGraph, HexDelShellHexSerial )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.0  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.0    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.0
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.0
    //                      ^
    //                      |
    //                       ---- Deleted shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size > 1)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    stk::mesh::impl::DeletedElementInfoVector elements_to_delete;
    elements_to_delete.push_back({shell3, 3, mesh.bucket(shell3).topology().is_shell()});

    elemElemGraph.delete_elements(elements_to_delete);

    mesh.modification_begin();
    mesh.destroy_entity(shell3);
    mesh.modification_end();

    // Connectivity for Hex Element 1
    EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,    elemElemGraph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(hex1, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

    // Connectivity for Hex Element 2
    EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(hex2));
    EXPECT_EQ(4,    elemElemGraph.get_side_id_to_connected_element(hex2, 0));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(hex2, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));

    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, DISABLED_HexAddShellAddShellSerial )
{
    //  ID.proc
    //
    //          3.0------------7.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      4.0------------8.0  |
    //       |   |          |   |
    //       |   |   1.0    |2.0|
    //       |   |          |3.0|
    //       |  2.0---------|--6.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------5.0
    //                      ^
    //                      |
    //                       ---- Two stacked shell elements

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size > 1)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 },
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 2, 3 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    mesh.modification_begin();
    stk::mesh::EntityVector added_shells;
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        added_shells.push_back( stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]) );
    }
    mesh.modification_end();

    elemElemGraph.add_elements(added_shells);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity shell2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    // Connectivity for Hex Element 1
    EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 1));
    EXPECT_EQ(shell2, elemElemGraph.get_connected_element(hex1, 0));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

    // Connectivity for Shell Element 2
    EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(shell2));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell2, 0));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell2, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell2, 0));

    // Connectivity for Shell Element 3
    EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(shell3));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));

    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, HexAddShellHexSerial )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.0  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.0    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.0
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.0
    //                      ^
    //                      |
    //                       ---- Added single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size > 1)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    mesh.modification_begin();
    stk::mesh::EntityVector added_shells;
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        added_shells.push_back( stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]) );
    }
    mesh.modification_end();

    elemElemGraph.add_elements(added_shells);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    // Connectivity for Hex Element 1
    EXPECT_EQ(1u,     elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

    // Connectivity for Hex Element 2
    EXPECT_EQ(1u,     elemElemGraph.get_num_connected_elems(hex2));
    EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 0));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex2, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));

    // Connectivity for Shell Element 3
    EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
    EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(shell3, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));

    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, DISABLED_HexAddShellAddShellHexSerial )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.0  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.0    |   |
    //       |   |          |4.0|          |   |
    //       |  2.0---------|--6.0---------|-10.0
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.0
    //                      ^
    //                      |
    //                       ---- Added two stacked shell elements

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size > 1)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 },
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3, 4 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    mesh.modification_begin();
    stk::mesh::EntityVector added_shells;
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        added_shells.push_back( stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]) );
    }
    mesh.modification_end();

    elemElemGraph.add_elements(added_shells);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);
    const Entity shell4 = mesh.get_entity(stk::topology::ELEM_RANK, 4);

    // Connectivity for Hex Element 1
    EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 1));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 0));
    EXPECT_EQ(shell4, elemElemGraph.get_connected_element(hex1, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

    // Connectivity for Hex Element 2
    EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex2));
    EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 0));
    EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 1));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex2, 0));
    EXPECT_EQ(shell4, elemElemGraph.get_connected_element(hex2, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));

    // Connectivity for Shell Element 3
    EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
    EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(shell3, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));

    // Connectivity for Shell Element 4
    EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell4));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell4, 0));
    EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell4, 1));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell4, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(shell4, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell4, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell4, 1));

    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

stk::mesh::Entity get_element_side(stk::mesh::BulkData& bulkData, stk::mesh::Entity element, stk::mesh::ConnectivityOrdinal side_ordinal)
{
    stk::mesh::Entity side = stk::mesh::Entity();
    stk::mesh::EntityRank side_rank = bulkData.mesh_meta_data().side_rank();

    unsigned elem_num_faces = bulkData.num_connectivity(element, side_rank);
    const stk::mesh::Entity * elem_sides = bulkData.begin(element, side_rank);
    const stk::mesh::ConnectivityOrdinal * elem_ord_it = bulkData.begin_ordinals(element, side_rank);
    for (unsigned i=0 ; i<elem_num_faces ; ++i)
    {
        if (elem_ord_it[i] == static_cast<unsigned>(side_ordinal))
        {
            side = elem_sides[i];
            break;
        }
    }

    return side;
}

void test_parallel_graph_info(const ElementGraph& elem_graph, const ParallelGraphInfo& parallel_graph_info,
        LocalId this_element, LocalId other_element, int other_proc, int other_side_ord, int permutation)
{
    ParallelGraphInfo::const_iterator iter1 = parallel_graph_info.find(std::make_pair(this_element, other_element));
    ASSERT_TRUE(iter1 != parallel_graph_info.end());

    for(size_t i=0;i<elem_graph.size();++i)
    {
        const std::vector<LocalId>& conn_elements = elem_graph[i];
        for(size_t j=0;j<conn_elements.size();++j)
        {
            if(conn_elements[j]==-1*other_element && static_cast<LocalId>(i) == this_element)
            {
                ParallelGraphInfo::const_iterator iter = parallel_graph_info.find(std::make_pair(this_element, other_element));

                ASSERT_TRUE(iter != parallel_graph_info.end());
                EXPECT_EQ(other_proc, iter->second.m_other_proc);
                EXPECT_EQ(other_side_ord, iter->second.m_other_side_ord);
                EXPECT_EQ(permutation, iter->second.m_permutation);
            }
        }
    }
}

template<class T>
void print_graph(const std::string &title, int proc_id, T& elem_graph)
{
    std::ostringstream os;

    os << title << " for processor " << proc_id << std::endl;
    for (size_t i=0;i<elem_graph.size();++i)
    {
        os << "Element " << i << "::\t";
        for (size_t j=0;j<elem_graph[i].size();++j)
        {
            os << elem_graph[i][j] << "\t";
        }
        os << std::endl;
    }
    std::cerr << os.str();
}


void create_faces_using_graph(BulkDataElementGraphTester& bulkData, stk::mesh::Part& part)
{
    double wall_time_start = stk::wall_time();

    std::vector<unsigned> counts;
    stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
    int numElems = counts[stk::topology::ELEM_RANK];

    stk::mesh::EntityVector local_id_to_element_entity(numElems, Entity());
    std::vector<stk::topology> element_topologies(numElems);
    impl::set_local_ids_and_fill_element_entities_and_topologies(bulkData, local_id_to_element_entity, element_topologies);

    ElemElemGraphTester elemElemGraph(bulkData);

    ElementGraph & elem_graph = elemElemGraph.get_element_graph();
    SidesForElementGraph & via_sides = elemElemGraph.get_via_sides();
    stk::mesh::impl::ParallelGraphInfo & parallel_graph_info = elemElemGraph.get_parallel_graph_info();

    double graph_time = stk::wall_time() - wall_time_start;
    wall_time_start = stk::wall_time();

    stk::mesh::EntityRank side_rank = bulkData.mesh_meta_data().side_rank();
    stk::mesh::PartVector parts {&part};
    //BeginDocExample4

    bulkData.modification_begin();

    std::vector<stk::mesh::sharing_info> shared_modified;

    for(size_t i = 0; i < elem_graph.size(); ++i)
    {
        const std::vector<LocalId>& connected_elements = elem_graph[i];
        stk::mesh::Entity element1 = local_id_to_element_entity[i];

        LocalId this_element = i;

        for(size_t j = 0; j < connected_elements.size(); ++j)
        {
            if(this_element < connected_elements[j] && connected_elements[j] >= 0)
            {
                stk::mesh::EntityId face_global_id = impl::get_element_side_multiplier() * bulkData.identifier(element1) + via_sides[i][j];
                if ( impl::is_id_already_in_use_locally(bulkData, side_rank, face_global_id) )
                {

                }
                stk::mesh::Entity face = stk::mesh::impl::get_or_create_face_at_element_side(bulkData, element1, via_sides[i][j],
                        face_global_id, stk::mesh::PartVector(1,&part));

                const stk::mesh::Entity* side_nodes = bulkData.begin_nodes(face);
                unsigned num_side_nodes = bulkData.num_nodes(face);
                stk::mesh::EntityVector side_nodes_vec(side_nodes, side_nodes + num_side_nodes);

                stk::mesh::Entity element2 = local_id_to_element_entity[connected_elements[j]];
                std::pair<stk::mesh::ConnectivityOrdinal, stk::mesh::Permutation> ord_and_perm = stk::mesh::get_ordinal_and_permutation(bulkData, element2, stk::topology::FACE_RANK, side_nodes_vec);
                bulkData.declare_relation(element2, face, ord_and_perm.first, ord_and_perm.second);
            }
            else if(connected_elements[j] < 0)
            {
                LocalId other_element = -1 * connected_elements[j];
                ParallelGraphInfo::const_iterator iter = parallel_graph_info.find(std::make_pair(this_element, other_element));
                ThrowRequireMsg( iter != parallel_graph_info.end(), "Program error. Contact sierra-help@sandia.gov for support.");
                int other_proc = iter->second.m_other_proc;
                int other_side = iter->second.m_other_side_ord;

                int this_proc = bulkData.parallel_rank();
                int owning_proc = this_proc < other_proc ? this_proc : other_proc;

                stk::mesh::EntityId face_global_id = 0;
                stk::mesh::Permutation perm;
                if(owning_proc == this_proc)
                {
                    stk::mesh::EntityId id = bulkData.identifier(element1);
                    face_global_id = impl::get_element_side_multiplier() * id + via_sides[i][j];
                    perm = static_cast<stk::mesh::Permutation>(0);
                }
                else
                {
                    face_global_id = impl::get_element_side_multiplier() * other_element + other_side;
                    perm = static_cast<stk::mesh::Permutation>(iter->second.m_permutation);
                }

                stk::mesh::ConnectivityOrdinal side_ord = static_cast<stk::mesh::ConnectivityOrdinal>(via_sides[i][j]);

                std::string msg = "Program error. Contact sierra-help@sandia.gov for support.";

                ThrowRequireMsg(!impl::is_id_already_in_use_locally(bulkData, side_rank, face_global_id), msg);
                ThrowRequireMsg(!impl::does_side_exist_with_different_permutation(bulkData, element1, side_ord, perm), msg);
                ThrowRequireMsg(!impl::does_element_side_exist(bulkData, element1, side_ord), msg);

                stk::mesh::Entity face = impl::connect_side_to_element(bulkData, element1, face_global_id, side_ord, perm, parts);

                shared_modified.push_back(stk::mesh::sharing_info(face, other_proc, owning_proc));
            }
        }

        std::vector<ElementSidePair> element_side_pairs;
        stk::mesh::impl::add_element_side_pairs_for_unused_sides(i, element_topologies[i], via_sides[i], element_side_pairs);

        for(size_t j = 0; j < element_side_pairs.size(); j++)
        {
            stk::mesh::EntityId face_global_id = impl::get_element_side_multiplier() * bulkData.identifier(element1) + element_side_pairs[j].second;
            stk::mesh::impl::get_or_create_face_at_element_side(bulkData, element1, element_side_pairs[j].second,
                    face_global_id, stk::mesh::PartVector(1,&part));

        }
    }

    double start_mod_end = stk::wall_time();
    bulkData.my_modification_end_for_entity_creation(shared_modified);

    //EndDocExample4

    double mod_end_time = stk::wall_time() - start_mod_end;

    double create_faces_time = stk::wall_time() - wall_time_start;

    if(bulkData.parallel_rank() == 0)
    {
        std::cerr << "graph time: " << graph_time << std::endl;
        std::cerr << "create faces time: " << create_faces_time << std::endl;
        std::cerr << "mod end time: " << mod_end_time << std::endl;
    }
}

//BeginDocExample1
std::vector<ElementSidePair>
skin_mesh(const SidesForElementGraph &via_side, const std::vector<stk::topology> &element_topologies)
{
    std::vector<ElementSidePair> element_side_pairs;

    size_t num_elems = via_side.size();
    for(size_t i=0; i<num_elems; ++i)
    {
        const std::vector<int>& internal_sides = via_side[i];
        stk::mesh::impl::add_element_side_pairs_for_unused_sides(i, element_topologies[i], internal_sides, element_side_pairs);
    }
    return element_side_pairs;
}


stk::mesh::EntityVector get_killed_elements(stk::mesh::BulkData& bulkData, const int killValue, const stk::mesh::Part& active)
{
    stk::mesh::EntityVector killedElements;
    const stk::mesh::BucketVector& buckets = bulkData.buckets(stk::topology::ELEMENT_RANK);
    for(size_t b = 0; b < buckets.size(); ++b)
    {
        const stk::mesh::Bucket &bucket = *buckets[b];
        if(bucket.owned() && bucket.member(active))
        {
            for(size_t e = 0; e < bucket.size(); ++e)
            {
                stk::mesh::Entity entity = bucket[e];
                bool should_element_be_killed = bulkData.identifier(entity) < static_cast<stk::mesh::EntityId>(killValue);
                if(bulkData.bucket(entity).member(active) && should_element_be_killed == true)
                {
                    killedElements.push_back(bucket[e]);
                }
            }
        }
    }
    return killedElements;
}


TEST(ElementGraph, check_graph_connectivity)
{
    // element0 --> element1 --> element2
    ElementGraph elem_graph = {
            {1},
            {0,2},
            {1}
    };

    SidesForElementGraph via_side = {
            {4},
            {1,5},
            {3}
    };

    EXPECT_EQ(4, check_connectivity(elem_graph, via_side, 0, 1));
    EXPECT_EQ(1, check_connectivity(elem_graph, via_side, 1, 0));
    EXPECT_EQ(5, check_connectivity(elem_graph, via_side, 1, 2));
    EXPECT_EQ(3, check_connectivity(elem_graph, via_side, 2, 1));

    EXPECT_EQ(-1, check_connectivity(elem_graph, via_side, 0, 2));
    EXPECT_EQ(-1, check_connectivity(elem_graph, via_side, 2, 0));
    EXPECT_EQ(-1, check_connectivity(elem_graph, via_side, 3, 0));
    EXPECT_EQ(-1, check_connectivity(elem_graph, via_side, 0, 3));
    EXPECT_EQ(-1, check_connectivity(elem_graph, via_side, 0, 0));
}

TEST(ElementGraph, skin_mesh_using_graph)
{
    // element0 --> element1 --> element2
    ElementGraph elem_graph = {
            {1},
            {0,2},
            {1}
    };

    SidesForElementGraph via_side = {
            {4},
            {1,5},
            {3}
    };

    std::vector<stk::topology> element_topologies{
        stk::topology::HEXAHEDRON_8,
        stk::topology::HEXAHEDRON_8,
        stk::topology::HEXAHEDRON_8
    };

    std::vector<ElementSidePair> element_side_pairs = skin_mesh(via_side, element_topologies);

    std::vector<ElementSidePair>gold_element_side_pairs{
        {0,0},
        {0,1},
        {0,2},
        {0,3},
        {0,5},
        {1,0},
        {1,2},
        {1,3},
        {1,4},
        {2,0},
        {2,1},
        {2,2},
        {2,4},
        {2,5}
    };

    ASSERT_EQ(gold_element_side_pairs.size(), element_side_pairs.size());

    for (size_t i=0;i<gold_element_side_pairs.size();++i)
    {
        std::vector<ElementSidePair >::iterator iter = std::find(element_side_pairs.begin(), element_side_pairs.end(), gold_element_side_pairs[i]);
        EXPECT_TRUE(iter != element_side_pairs.end()) << "gold elem-side-pair=" << gold_element_side_pairs[i].first << ", " << gold_element_side_pairs[i].second;
    }
}

TEST(ElementGraph, create_element_graph_serial)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    std::vector<double> wall_times;
    wall_times.reserve(10);
    std::vector<std::string> msgs;
    msgs.reserve(10);

    std::vector<size_t> mem_usage;

    wall_times.push_back(stk::wall_time());
    msgs.push_back("program-start");
    mem_usage.push_back(stk::get_memory_usage_now());

    if(stk::parallel_machine_size(comm) == 1)
    {
        stk::mesh::MetaData meta;
        stk::mesh::BulkData bulkData(meta, comm);
        std::ostringstream os;

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after mesh-read");
        mem_usage.push_back(stk::get_memory_usage_now());

        std::vector<unsigned> counts;
        stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
        int numElems = counts[stk::topology::ELEM_RANK];
        EXPECT_EQ(4, numElems);

        ElemElemGraphTester elemElemGraph(bulkData);

        size_t expectedNumElems = counts[stk::topology::ELEM_RANK];
        ASSERT_EQ(expectedNumElems, elemElemGraph.get_graph_size());

        ElementGraph & elem_graph = elemElemGraph.get_element_graph();
        SidesForElementGraph & via_sides = elemElemGraph.get_via_sides();

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after ElemElemGraphTester constructor");
        mem_usage.push_back(stk::get_memory_usage_now());

        int left_side_id = 4;
        int right_side_id = 5;

        for(size_t i=0; i<elem_graph.size(); ++i)
        {
            const std::vector<LocalId>& conn_elements = elem_graph[i];
            if (i == 0)
            {
                ASSERT_EQ(1u, conn_elements.size());
                EXPECT_EQ(1, conn_elements[0]);
                EXPECT_EQ(right_side_id, via_sides[i][0]);
            }
            else if (i == elem_graph.size() - 1)
            {
                LocalId second_to_last_element_index = elem_graph.size() - 2;
                ASSERT_EQ(1u, conn_elements.size());
                EXPECT_EQ(second_to_last_element_index, conn_elements[0]);
                EXPECT_EQ(left_side_id, via_sides[i][0]);
            }
            else
            {
                ASSERT_EQ(2u, conn_elements.size());
                LocalId element_to_the_left = i-1;
                LocalId element_to_the_right = i+1;
                EXPECT_EQ(element_to_the_left, conn_elements[0]);
                EXPECT_EQ(element_to_the_right, conn_elements[1]);
                EXPECT_EQ(left_side_id, via_sides[i][0]);
                EXPECT_EQ(right_side_id, via_sides[i][1]);
            }
        }

        EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());

        if (stk::parallel_machine_rank(comm) == 0)
        {
            for(size_t i=0;i<wall_times.size();++i)
            {
                std::cerr << "Wall time " << msgs[i] << ":\t" << wall_times[i] - wall_times[0] << std::endl;
            }

            for(size_t i=0;i<mem_usage.size();++i)
            {
                std::cerr << "Memory usage " << msgs[i] << ":\t" << mem_usage[i] - mem_usage[0] << std::endl;
            }
        }
    }
}

TEST(ElementGraph, create_element_graph_parallel)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    std::vector<double> wall_times;
    wall_times.reserve(10);
    std::vector<std::string> msgs;
    msgs.reserve(10);

    std::vector<size_t> mem_usage;

    wall_times.push_back(stk::wall_time());
    msgs.push_back("program-start");
    mem_usage.push_back(stk::get_memory_usage_now());

    if(stk::parallel_machine_size(comm) == 2)
    {
        stk::mesh::MetaData meta;
        stk::mesh::BulkData bulkData(meta, comm);

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        std::vector<unsigned> counts;
        stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
        int numLocallyOwnedElems = counts[stk::topology::ELEM_RANK];
        EXPECT_EQ(2, numLocallyOwnedElems);

        ElemElemGraphTester elemElemGraph(bulkData);

        size_t expectedNumElems = counts[stk::topology::ELEM_RANK];
        ASSERT_EQ(expectedNumElems, elemElemGraph.get_graph_size());

        ElementGraph & elem_graph = elemElemGraph.get_element_graph();
        SidesForElementGraph & via_sides = elemElemGraph.get_via_sides();

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after ElemElemGraphTester constructor");
        mem_usage.push_back(stk::get_memory_usage_now());

        LocalId element_to_test_local_id = std::numeric_limits<LocalId>::max();
        int side_id = -1;
        int left_side_id = 4;
        int right_side_id = 5;

        EXPECT_EQ(2u, elem_graph.size());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());

        if (stk::parallel_machine_rank(comm) == 0)
        {
            element_to_test_local_id = 1;
            side_id = right_side_id;
        }
        else
        {
            element_to_test_local_id = 0;
            side_id = left_side_id;
        }

        for(size_t i=0;i<elem_graph.size();++i)
        {
            if (static_cast<LocalId>(i) == element_to_test_local_id)
            {
                // Element on parallel boundary
                ASSERT_EQ(2u, elem_graph[i].size());
                ASSERT_EQ(2u, via_sides[i].size());
                ASSERT_GE(-1, elem_graph[i][1]);
                ASSERT_EQ(side_id, via_sides[i][1]);
            }
            else
            {
                EXPECT_EQ(1u, elem_graph[i].size());
                EXPECT_EQ(1u, via_sides[i].size());
            }
        }

        if (stk::parallel_machine_rank(comm) == 0)
        {
            for(size_t i=0;i<wall_times.size();++i)
            {
                std::cerr << "Wall time " << msgs[i] << ":\t" << wall_times[i] - wall_times[0] << std::endl;
            }

            for(size_t i=0;i<mem_usage.size();++i)
            {
                std::cerr << "Memory usage " << msgs[i] << ":\t" << mem_usage[i] - mem_usage[0] << std::endl;
            }
        }
    }
}

TEST(ElementGraph, skin_mesh_using_element_graph_serial)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    std::vector<double> wall_times;
    wall_times.reserve(10);
    std::vector<std::string> msgs;
    msgs.reserve(10);

    std::vector<size_t> mem_usage;

    wall_times.push_back(stk::wall_time());
    msgs.push_back("program-start");
    mem_usage.push_back(stk::get_memory_usage_now());

    if(stk::parallel_machine_size(comm) == 1)
    {
        std::string dimension = unitTestUtils::getOption("--zdim", "3");
        const int zdim = std::atoi(dimension.c_str());

        unsigned spatialDim = 3;
        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::Part& skin_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
        stk::io::put_io_part_attribute(skin_part);
        BulkDataElementGraphTester bulkData(meta, comm);
        std::ostringstream os;
        os << "generated:" << zdim << "x" << zdim << "x" << zdim;
        bool check_results = false;

        std::string filename = os.str();

        stk::unit_test_util::fill_mesh_using_stk_io(filename, bulkData, comm);

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after mesh-read");
        mem_usage.push_back(stk::get_memory_usage_now());

        std::vector<unsigned> counts;
        stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
        int numElems = counts[stk::topology::ELEM_RANK];
        if ( check_results )
        {
            EXPECT_EQ(zdim, numElems);
        }

        ElemElemGraphTester elemElemGraph(bulkData);

        size_t expectedNumElems = counts[stk::topology::ELEM_RANK];

        ElementGraph & elem_graph = elemElemGraph.get_element_graph();
        SidesForElementGraph & via_sides = elemElemGraph.get_via_sides();

        if ( check_results )
        {
            ASSERT_EQ(expectedNumElems, elemElemGraph.get_graph_size());
        }

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after ElemElemGraphTester constructor");
        mem_usage.push_back(stk::get_memory_usage_now());

        stk::mesh::EntityVector local_id_to_element_entity(numElems, Entity());
        std::vector<stk::topology> element_topologies(numElems);
        impl::set_local_ids_and_fill_element_entities_and_topologies(bulkData, local_id_to_element_entity, element_topologies);

        std::vector<ElementSidePair> elem_side_pairs = skin_mesh(via_sides, element_topologies);

        bulkData.modification_begin();

        for(size_t face_index=0; face_index<elem_side_pairs.size(); ++face_index)
        {
            stk::mesh::Entity element = local_id_to_element_entity[elem_side_pairs[face_index].first];
            stk::mesh::EntityId face_global_id = face_index + 1;
            stk::mesh::declare_element_side(bulkData, face_global_id, element, elem_side_pairs[face_index].second, &skin_part);
        }

        stk::mesh::Selector element_selector = bulkData.mesh_meta_data().locally_owned_part();
        bulkData.my_internal_modification_end_for_skin_mesh(stk::topology::FACE_RANK, stk::mesh::impl::MeshModification::MOD_END_SORT, element_selector, NULL);

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after create-faces");
        mem_usage.push_back(stk::get_memory_usage_now());

        stk::unit_test_util::write_mesh_using_stk_io("out.exo", bulkData, bulkData.parallel());

        if ( check_results )
        {
            int left_side_id = 4;
            int right_side_id = 5;

            for(size_t i=0; i<elem_graph.size(); ++i)
            {
                const std::vector<LocalId>& conn_elements = elem_graph[i];
                if (i == 0)
                {
                    ASSERT_EQ(1u, conn_elements.size());
                    EXPECT_EQ(1, conn_elements[0]);
                    EXPECT_EQ(right_side_id, via_sides[i][0]);
                }
                else if (i == elem_graph.size() - 1)
                {
                    LocalId second_to_last_element_index = elem_graph.size() - 2;
                    ASSERT_EQ(1u, conn_elements.size());
                    EXPECT_EQ(second_to_last_element_index, conn_elements[0]);
                    EXPECT_EQ(left_side_id, via_sides[i][0]);
                }
                else
                {
                    ASSERT_EQ(2u, conn_elements.size());
                    LocalId element_to_the_left = i-1;
                    LocalId element_to_the_right = i+1;
                    EXPECT_EQ(element_to_the_left, conn_elements[0]);
                    EXPECT_EQ(element_to_the_right, conn_elements[1]);
                    EXPECT_EQ(left_side_id, via_sides[i][0]);
                    EXPECT_EQ(right_side_id, via_sides[i][1]);
                }
            }

            EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
        }

        if (stk::parallel_machine_rank(comm) == 0)
        {
            for(size_t i=0;i<wall_times.size();++i)
            {
                std::cerr << "Wall time " << msgs[i] << ":\t" << wall_times[i] - wall_times[0] << std::endl;
            }

            for(size_t i=0;i<mem_usage.size();++i)
            {
                std::cerr << "Memory usage " << msgs[i] << ":\t" << mem_usage[i] - mem_usage[0] << std::endl;
            }
        }
    }
}

void expect_connected_to_local_elem_id(stk::mesh::BulkData &bulkData,
                                       ElemElemGraphTester &elem_graph,
                                       stk::mesh::Entity elem,
                                       size_t connectedIndex,
                                       stk::mesh::EntityId connectedId)
{
    stk::mesh::EntityId elemId = bulkData.identifier(elem);
    ASSERT_TRUE(elem_graph.is_connected_elem_locally_owned(elem, connectedIndex))
            << "elem " << elemId
            << " expected local elem " << connectedId;
    EXPECT_EQ(connectedId, bulkData.identifier(elem_graph.get_connected_element(elem, connectedIndex)))
            << "elem " << elemId;
}

void expect_connected_to_remote_elem_id(stk::mesh::BulkData &bulkData,
                                       ElemElemGraphTester &elem_graph,
                                       stk::mesh::Entity elem,
                                       size_t connectedIndex,
                                       stk::mesh::EntityId connectedId)
{
    stk::mesh::EntityId elemId = bulkData.identifier(elem);
    ASSERT_TRUE(!elem_graph.is_connected_elem_locally_owned(elem, connectedIndex))
            << "elem " << elemId
            << " expected remote elem " << connectedId;
    EXPECT_EQ(connectedId, elem_graph.get_entity_id_of_remote_element(elem, connectedIndex))
            << "elem " << elemId;
}

void change_entity_owner_hex_test_2_procs_move_from_1_to_0(bool aura_on)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    int proc = stk::parallel_machine_rank(comm);

    if(stk::parallel_machine_size(comm) == 2)
    {
        stk::mesh::MetaData meta;
        stk::mesh::BulkData::AutomaticAuraOption aura_option = stk::mesh::BulkData::AUTO_AURA;
        if (!aura_on)
        {
            aura_option = stk::mesh::BulkData::NO_AUTO_AURA;
        }
        stk::mesh::BulkData bulkData(meta, comm, aura_option);

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        std::vector<unsigned> counts;
        stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
        int numLocallyOwnedElems = counts[stk::topology::ELEM_RANK];
        EXPECT_EQ(2, numLocallyOwnedElems);

        ElemElemGraphTester elemGraph(bulkData);

        std::vector< std::pair< stk::mesh::Entity, int > > elem_proc_pairs_to_move;

        if (proc == 0)
        {
            stk::mesh::Entity elem2 = bulkData.get_entity(stk::topology::ELEM_RANK, 2);
            ASSERT_EQ(2u, elemGraph.get_num_connected_elems(elem2));
            expect_connected_to_local_elem_id(bulkData, elemGraph, elem2, 0, 1);
            expect_connected_to_remote_elem_id(bulkData, elemGraph, elem2, 1, 3);
            EXPECT_EQ(5, elemGraph.get_side_from_element1_to_remote_element2(elem2, 3));

            const impl::parallel_info &parInfo = elemGraph.get_parallel_edge_info(elem2, 3);
            EXPECT_EQ(1, parInfo.m_other_proc);
            EXPECT_EQ(4, parInfo.m_other_side_ord);
            EXPECT_EQ(4, parInfo.m_permutation);
            EXPECT_TRUE(parInfo.m_in_body_to_be_skinned);
            EXPECT_FALSE(parInfo.m_is_air);
            EXPECT_EQ(1u, parInfo.m_chosen_side_id);
        }
        else
        {
            stk::mesh::Entity elem3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
            ASSERT_EQ(2u, elemGraph.get_num_connected_elems(elem3));
            expect_connected_to_local_elem_id(bulkData, elemGraph, elem3, 0, 4);
            expect_connected_to_remote_elem_id(bulkData, elemGraph, elem3, 1, 2);
            EXPECT_EQ(4, elemGraph.get_side_from_element1_to_remote_element2(elem3, 2));

            const impl::parallel_info &parInfo = elemGraph.get_parallel_edge_info(elem3, 2);
            EXPECT_EQ(0, parInfo.m_other_proc);
            EXPECT_EQ(5, parInfo.m_other_side_ord);
            EXPECT_EQ(4, parInfo.m_permutation);
            EXPECT_TRUE(parInfo.m_in_body_to_be_skinned);
            EXPECT_FALSE(parInfo.m_is_air);
            EXPECT_EQ(1u, parInfo.m_chosen_side_id);

            int other_proc = 0;
            elem_proc_pairs_to_move.push_back(std::make_pair(elem3, other_proc));
        }

        change_entity_owner(bulkData, elemGraph, elem_proc_pairs_to_move);

        if (proc == 0)
        {
            stk::mesh::Entity elem2 = bulkData.get_entity(stk::topology::ELEM_RANK, 2);
            ASSERT_EQ(2u, elemGraph.get_num_connected_elems(elem2));
            expect_connected_to_local_elem_id(bulkData, elemGraph, elem2, 0, 1);
            expect_connected_to_local_elem_id(bulkData, elemGraph, elem2, 1, 3);

            stk::mesh::Entity elem3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
            ASSERT_EQ(2u, elemGraph.get_num_connected_elems(elem3));
            expect_connected_to_remote_elem_id(bulkData, elemGraph, elem3, 0, 4);
            EXPECT_EQ(5, elemGraph.get_side_from_element1_to_remote_element2(elem3, 4));
            expect_connected_to_local_elem_id(bulkData, elemGraph, elem3, 1, 2);

            const impl::parallel_info &parInfo = elemGraph.get_parallel_edge_info(elem3, 4);
            EXPECT_EQ(1, parInfo.m_other_proc);
            EXPECT_EQ(4, parInfo.m_other_side_ord);
            EXPECT_EQ(4, parInfo.m_permutation);
            EXPECT_TRUE(parInfo.m_in_body_to_be_skinned);
            EXPECT_FALSE(parInfo.m_is_air);
            EXPECT_EQ(17u, parInfo.m_chosen_side_id);
        }
        else
        {
            stk::mesh::Entity elem4 = bulkData.get_entity(stk::topology::ELEM_RANK, 4);
            ASSERT_EQ(1u, elemGraph.get_num_connected_elems(elem4));
            expect_connected_to_remote_elem_id(bulkData, elemGraph, elem4, 0, 3);
            EXPECT_EQ(4, elemGraph.get_side_from_element1_to_remote_element2(elem4, 3));

            const impl::parallel_info &parInfo = elemGraph.get_parallel_edge_info(elem4, 3);
            EXPECT_EQ(0, parInfo.m_other_proc);
            EXPECT_EQ(5, parInfo.m_other_side_ord);
            EXPECT_EQ(4, parInfo.m_permutation);
            EXPECT_TRUE(parInfo.m_in_body_to_be_skinned);
            EXPECT_FALSE(parInfo.m_is_air);
            EXPECT_EQ(17u, parInfo.m_chosen_side_id);
        }
    }
}

void change_entity_owner_hex_test_2_procs(bool aura_on)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    int proc = stk::parallel_machine_rank(comm);

    if(stk::parallel_machine_size(comm) == 2)
    {
        stk::mesh::MetaData meta;
        stk::mesh::BulkData::AutomaticAuraOption aura_option = stk::mesh::BulkData::AUTO_AURA;
        if (!aura_on)
        {
            aura_option = stk::mesh::BulkData::NO_AUTO_AURA;
        }
        stk::mesh::BulkData bulkData(meta, comm, aura_option);

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        std::vector<unsigned> counts;
        stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
        int numLocallyOwnedElems = counts[stk::topology::ELEM_RANK];
        EXPECT_EQ(2, numLocallyOwnedElems);

        ElemElemGraphTester elem_graph(bulkData);

        // Create a vector of the elements to be moved
        std::vector <stk::mesh::Entity> elems_to_move;

        stk::mesh::EntityId elem_2_id = 2;
        std::vector< std::pair< stk::mesh::Entity, int > > elem_proc_pairs_to_move;
        stk::mesh::Entity elem_2 = bulkData.get_entity(stk::topology::ELEM_RANK, elem_2_id);

        if (proc == 0)
        {
            int side_from_elem2_to_elem3 = elem_graph.get_side_from_element1_to_remote_element2(elem_2, stk::mesh::EntityId(3));
            int side_from_elem2_to_elem1 = elem_graph.get_side_from_element1_to_locally_owned_element2(elem_2, bulkData.get_entity(stk::topology::ELEM_RANK,1));

            EXPECT_EQ(5, side_from_elem2_to_elem3);
            EXPECT_EQ(4, side_from_elem2_to_elem1);

            elems_to_move.push_back(elem_2);

            int other_proc = 1;
            for (unsigned i=0; i<elems_to_move.size(); i++)
            {
                EXPECT_TRUE(bulkData.is_valid(elems_to_move[i]));
                EXPECT_EQ(0, bulkData.parallel_owner_rank(elems_to_move[i]));
                elem_proc_pairs_to_move.push_back(std::make_pair(elems_to_move[i], other_proc));
            }
        }

        change_entity_owner(bulkData, elem_graph, elem_proc_pairs_to_move);

        elem_2 = bulkData.get_entity(stk::topology::ELEM_RANK, elem_2_id);

        if (proc == 0)
        {
            stk::mesh::Entity elem_1 = bulkData.get_entity(stk::topology::ELEM_RANK, 1);
            impl::parallel_info &p_info = elem_graph.get_parallel_edge_info(elem_1, stk::mesh::EntityId(2));
            int other_side_ord = p_info.m_other_side_ord;
            EXPECT_EQ(4, other_side_ord);
            stk::mesh::EntityId chosen_face_id = 2;
            EXPECT_EQ(chosen_face_id, p_info.m_chosen_side_id);

            ASSERT_THROW(elem_graph.get_parallel_edge_info(elem_2, stk::mesh::EntityId(3)), std::logic_error);
        }

        if (proc == 1)
        {
            EXPECT_TRUE(bulkData.is_valid(elem_2));
            EXPECT_EQ(1, bulkData.parallel_owner_rank(elem_2));

            EXPECT_EQ(2u, elem_graph.get_num_connected_elems(elem_2));

            stk::mesh::Entity elem = elem_graph.get_connected_element(elem_2, 1);
            ASSERT_TRUE(elem_graph.is_connected_elem_locally_owned(elem_2, 1));
            EXPECT_EQ(3u, bulkData.identifier(elem));

            stk::mesh::EntityId connected_elem_global_id = elem_graph.get_entity_id_of_remote_element(elem_2, 0);
            ASSERT_FALSE(elem_graph.is_connected_elem_locally_owned(elem_2, 0));
            EXPECT_EQ(1u, connected_elem_global_id);

            stk::mesh::Entity elem_3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
            int side_from_elem2_to_elem3 = elem_graph.get_side_from_element1_to_locally_owned_element2(elem_2, elem_3);
            EXPECT_EQ(5, side_from_elem2_to_elem3);

            int side_from_elem2_to_elem1 = elem_graph.get_side_from_element1_to_remote_element2(elem_2, stk::mesh::EntityId(1));
            EXPECT_EQ(4, side_from_elem2_to_elem1);

            impl::parallel_info &elem_2_to_1_p_info = elem_graph.get_parallel_edge_info(elem_2, stk::mesh::EntityId(1));
            int other_side_ord = elem_2_to_1_p_info.m_other_side_ord;
            EXPECT_EQ(5, other_side_ord);
            stk::mesh::EntityId chosen_face_id = 2;
            EXPECT_EQ(chosen_face_id, elem_2_to_1_p_info.m_chosen_side_id);

            ASSERT_THROW(elem_graph.get_parallel_edge_info(elem_3, stk::mesh::EntityId(2)), std::logic_error);
        }

        EXPECT_EQ(1u, elem_graph.num_parallel_edges());
    }
}


TEST(ElementGraph, test_change_entity_owner_2_procs_hex_mesh_with_aura)
{
    bool aura_on = true;
    change_entity_owner_hex_test_2_procs(aura_on);
}

TEST(ElementGraph, test_change_entity_owner_2_procs_hex_mesh_without_aura)
{
    bool aura_on = false;
    change_entity_owner_hex_test_2_procs(aura_on);
}


TEST(ElementGraph, test_change_entity_owner_2_procs_move_from_1_to_0_hex_mesh_with_aura)
{
    bool aura_on = true;
    change_entity_owner_hex_test_2_procs_move_from_1_to_0(aura_on);
}

TEST(ElementGraph, test_change_entity_owner_2_procs_move_from_1_to_0_hex_mesh_without_aura)
{
    bool aura_on = false;
    change_entity_owner_hex_test_2_procs_move_from_1_to_0(aura_on);
}

void change_entity_owner_then_death_hex_test_2_procs(bool aura_on)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    int proc = stk::parallel_machine_rank(comm);

    if(stk::parallel_machine_size(comm) == 2)
    {
        unsigned spatial_dim = 3;
        stk::mesh::MetaData meta(spatial_dim, stk::mesh::entity_rank_names());

        stk::mesh::Part& faces_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
        stk::mesh::Part& active = meta.declare_part("active", stk::topology::ELEMENT_RANK);
        stk::mesh::PartVector boundary_mesh_parts { &faces_part };

        stk::mesh::BulkData::AutomaticAuraOption aura_option = stk::mesh::BulkData::AUTO_AURA;
        if (!aura_on)
        {
            aura_option = stk::mesh::BulkData::NO_AUTO_AURA;
        }
        stk::mesh::BulkData bulkData(meta, comm, aura_option);

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        stk::unit_test_util::put_mesh_into_part(bulkData, active);

        std::vector<unsigned> counts;
        stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
        int numLocallyOwnedElems = counts[stk::topology::ELEM_RANK];
        EXPECT_EQ(2, numLocallyOwnedElems);

        ElemElemGraphTester elem_graph(bulkData);

        // Create a vector of the elements to be moved
        std::vector <stk::mesh::Entity> elems_to_move;

        stk::mesh::EntityId elem_2_id = 2;
        std::vector< std::pair< stk::mesh::Entity, int > > elem_proc_pairs_to_move;
        stk::mesh::Entity elem_2 = bulkData.get_entity(stk::topology::ELEM_RANK, elem_2_id);

        if (proc == 0)
        {
            int side_from_elem2_to_elem3 = elem_graph.get_side_from_element1_to_remote_element2(elem_2, stk::mesh::EntityId(3));
            int side_from_elem2_to_elem1 = elem_graph.get_side_from_element1_to_locally_owned_element2(elem_2, bulkData.get_entity(stk::topology::ELEM_RANK,1));

            EXPECT_EQ(5, side_from_elem2_to_elem3);
            EXPECT_EQ(4, side_from_elem2_to_elem1);

            elems_to_move.push_back(elem_2);

            int other_proc = 1;
            for (unsigned i=0; i<elems_to_move.size(); i++)
            {
                EXPECT_TRUE(bulkData.is_valid(elems_to_move[i]));
                EXPECT_EQ(0, bulkData.parallel_owner_rank(elems_to_move[i]));
                elem_proc_pairs_to_move.push_back(std::make_pair(elems_to_move[i], other_proc));
            }
        }

        change_entity_owner(bulkData, elem_graph, elem_proc_pairs_to_move, &active);

        elem_2 = bulkData.get_entity(stk::topology::ELEM_RANK, elem_2_id);

        stk::mesh::EntityVector killedElements;
        std::vector<stk::mesh::PartVector> add_parts, remove_parts;
        if (proc == 1)
        {
            killedElements.push_back(elem_2);
            add_parts.push_back(stk::mesh::PartVector());
            remove_parts.push_back(stk::mesh::PartVector{&active});
        }

        bulkData.batch_change_entity_parts(killedElements, add_parts, remove_parts);
        boundary_mesh_parts.push_back(&active);

        process_killed_elements(bulkData, elem_graph, killedElements, active, boundary_mesh_parts, &boundary_mesh_parts);

        if (proc == 1)
        {
            EXPECT_TRUE(bulkData.is_valid(elem_2));
            EXPECT_EQ(1, bulkData.parallel_owner_rank(elem_2));
            EXPECT_FALSE(bulkData.bucket(elem_2).member(active));

            EXPECT_EQ(2u, elem_graph.get_num_connected_elems(elem_2));

            stk::mesh::Entity elem = elem_graph.get_connected_element(elem_2, 1);
            ASSERT_TRUE(elem_graph.is_connected_elem_locally_owned(elem_2, 1));
            EXPECT_EQ(3u, bulkData.identifier(elem));

            stk::mesh::EntityId connected_elem_global_id = elem_graph.get_entity_id_of_remote_element(elem_2, 0);
            ASSERT_FALSE(elem_graph.is_connected_elem_locally_owned(elem_2, 0));
            EXPECT_EQ(1u, connected_elem_global_id);

            stk::mesh::Entity elem_3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
            int side_from_elem2_to_elem3 = elem_graph.get_side_from_element1_to_locally_owned_element2(elem_2, elem_3);
            EXPECT_EQ(5, side_from_elem2_to_elem3);

            int side_from_elem2_to_elem1 = elem_graph.get_side_from_element1_to_remote_element2(elem_2, stk::mesh::EntityId(1));
            EXPECT_EQ(4, side_from_elem2_to_elem1);

            impl::parallel_info &elem_2_to_1_p_info = elem_graph.get_parallel_edge_info(elem_2, stk::mesh::EntityId(1));
            int other_side_ord = elem_2_to_1_p_info.m_other_side_ord;
            EXPECT_EQ(5, other_side_ord);
        }
        if (proc == 0)
        {
            stk::mesh::Entity elem_1 = bulkData.get_entity(stk::topology::ELEM_RANK, 1);
            impl::parallel_info &elem1_to_elem2_info = elem_graph.get_parallel_edge_info(elem_1, stk::mesh::EntityId(2));

            EXPECT_FALSE(elem1_to_elem2_info.m_in_body_to_be_skinned);

            EXPECT_EQ(4, elem1_to_elem2_info.m_other_side_ord);

            ASSERT_THROW(elem_graph.get_parallel_edge_info(elem_2, stk::mesh::EntityId(3)), std::logic_error);
       }

        EXPECT_EQ(1u, elem_graph.num_parallel_edges());
    }
}

TEST(ElementGraph, test_change_entity_owner_and_death_hex_mesh_2_procs_with_aura)
{
    bool aura_on = true;
    change_entity_owner_then_death_hex_test_2_procs(aura_on);
}

TEST(ElementGraph, test_change_entity_owner_and_death_hex_mesh_2_procs_without_aura)
{
    bool aura_on = false;
    change_entity_owner_then_death_hex_test_2_procs(aura_on);
}

void change_entity_owner_hex_test_4_procs(bool aura_on)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    int proc = stk::parallel_machine_rank(comm);
    std::vector<double> wall_times;
    wall_times.reserve(10);
    std::vector<std::string> msgs;
    msgs.reserve(10);

    std::vector<size_t> mem_usage;

    wall_times.push_back(stk::wall_time());
    msgs.push_back("program-start");
    mem_usage.push_back(stk::get_memory_usage_now());

    if(stk::parallel_machine_size(comm) == 4)
    {
        stk::mesh::MetaData meta;
        stk::mesh::BulkData::AutomaticAuraOption aura_option = stk::mesh::BulkData::AUTO_AURA;
        if (!aura_on)
        {
            aura_option = stk::mesh::BulkData::NO_AUTO_AURA;
        }
        stk::mesh::BulkData bulkData(meta, comm, aura_option);

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after mesh-read");
        mem_usage.push_back(stk::get_memory_usage_now());

        std::vector<unsigned> counts;
        stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
        int numLocallyOwnedElems = counts[stk::topology::ELEM_RANK];
        EXPECT_EQ(1, numLocallyOwnedElems);

        ElemElemGraphTester elem_graph(bulkData);

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after fill-graph");
        mem_usage.push_back(stk::get_memory_usage_now());

        // Create a vector of the elements to be moved
        std::vector <stk::mesh::Entity> elems_to_move;

        stk::mesh::EntityId elem_global_id = 2;
        std::vector< std::pair< stk::mesh::Entity, int > > elem_proc_pairs_to_move;
        stk::mesh::Entity elem_to_move = bulkData.get_entity(stk::topology::ELEM_RANK, elem_global_id);
        if (proc == 1)
        {
            elems_to_move.push_back(elem_to_move);

            int other_proc = 2;
            for (unsigned i=0; i<elems_to_move.size(); i++)
            {
                EXPECT_TRUE(bulkData.is_valid(elems_to_move[i]));
                EXPECT_EQ(1, bulkData.parallel_owner_rank(elems_to_move[i]));
                elem_proc_pairs_to_move.push_back(std::make_pair(elems_to_move[i], other_proc));
            }
        }

        change_entity_owner(bulkData, elem_graph, elem_proc_pairs_to_move);

        elem_to_move = bulkData.get_entity(stk::topology::ELEM_RANK, elem_global_id);

        if (proc == 2)
        {
            EXPECT_TRUE(bulkData.is_valid(elem_to_move));
            EXPECT_EQ(2, bulkData.parallel_owner_rank(elem_to_move));

            EXPECT_EQ(2u, elem_graph.get_num_connected_elems(elem_to_move));

            stk::mesh::Entity elem = elem_graph.get_connected_element(elem_to_move, 1);
            ASSERT_TRUE(elem_graph.is_connected_elem_locally_owned(elem_to_move, 1));
            EXPECT_EQ(3u, bulkData.identifier(elem));

            stk::mesh::EntityId connected_elem_global_id = elem_graph.get_entity_id_of_remote_element(elem_to_move, 0);
            ASSERT_FALSE(elem_graph.is_connected_elem_locally_owned(elem_to_move, 0));
            EXPECT_EQ(1u, connected_elem_global_id);

            stk::mesh::Entity elem_3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
            ASSERT_THROW(elem_graph.get_parallel_edge_info(elem_3, stk::mesh::EntityId(2)), std::logic_error);

            EXPECT_EQ(2u, elem_graph.num_parallel_edges());
        }
        else if (proc == 0)
        {
            stk::mesh::Entity elem_1 = bulkData.get_entity(stk::topology::ELEM_RANK, 1);
            impl::parallel_info &elem_1_to_2_p_info = elem_graph.get_parallel_edge_info(elem_1, stk::mesh::EntityId(2));
            EXPECT_EQ(2, elem_1_to_2_p_info.m_other_proc);
            EXPECT_EQ(1u, elem_graph.num_parallel_edges());
        }
        else if (proc == 1)
        {
            EXPECT_EQ(0u, elem_graph.size());
            EXPECT_EQ(0u, elem_graph.num_edges());
            EXPECT_EQ(0u, elem_graph.num_parallel_edges());
        }
        else if (proc == 3)
        {
            EXPECT_EQ(1u, elem_graph.num_parallel_edges());
        }
    }
}

TEST(ElementGraph, test_change_entity_owner_4_procs_hex_mesh_with_aura)
{
    bool aura_on = true;
    change_entity_owner_hex_test_4_procs(aura_on);
}

TEST(ElementGraph, test_change_entity_owner_4_procs_hex_mesh_without_aura)
{
    bool aura_on = false;
    change_entity_owner_hex_test_4_procs(aura_on);
}

void setup_hex_shell_hex_mesh(stk::mesh::BulkData& bulkData)
{
//
//                proc 0               proc 1           proc 2
//
//               block_1          |   block_2  |      block_3
//
//          3---------------7        7            7-------------11
//          /|             /|       /|           /|             /|
//         / |            / |      / |          / |            / |
//        /  |           /  |     /  |         /  |           /  |
//       4--------------8   |    8   |        8--------------12  |
//       |   |          |   |    |   |        |   |          |   |
//       |   |   1      |   |    | 2 |        |   |   3      |   |
//       |   |          |   |    |   |        |   |          |   |
//       |   2----------|---6    |   6        |   6----------|---10
//       |  /           |  /     |  /         |  /           |  /
//       | /            | /      | /          | /            | /
//       |/             |/       |/           |/             |/
//       1--------------5        5            5--------------9

    stk::mesh::MetaData& meta = bulkData.mesh_meta_data();
    unsigned spatial_dimension = 3;
    meta.initialize(spatial_dimension, stk::mesh::entity_rank_names());

    stk::mesh::Field<double>& field = meta.declare_field<stk::mesh::Field<double> >(stk::topology::NODE_RANK, "field1");
    stk::mesh::put_field(field, meta.universal_part());

    stk::mesh::Part& block_1 = meta.declare_part_with_topology("block_1", stk::topology::HEX_8);
    stk::mesh::Part& block_2 = meta.declare_part_with_topology("block_2", stk::topology::SHELL_QUAD_4);
    stk::mesh::Part& block_3 = meta.declare_part_with_topology("block_3", stk::topology::HEX_8);
    meta.commit();

    bulkData.modification_begin();

    stk::mesh::EntityIdVector elem1_nodes {1, 2, 3, 4, 5, 6, 7, 8};
    stk::mesh::EntityIdVector elem2_nodes {5, 6, 7, 8};
    stk::mesh::EntityIdVector elem3_nodes {5, 6, 7, 8, 9, 10, 11, 12};

    stk::mesh::EntityId elemId = 1;
    if (bulkData.parallel_rank() == 0) {
        stk::mesh::declare_element(bulkData, block_1, elemId, elem1_nodes);
        stk::mesh::Entity node5 = bulkData.get_entity(stk::topology::NODE_RANK, 5);
        stk::mesh::Entity node6 = bulkData.get_entity(stk::topology::NODE_RANK, 6);
        stk::mesh::Entity node7 = bulkData.get_entity(stk::topology::NODE_RANK, 7);
        stk::mesh::Entity node8 = bulkData.get_entity(stk::topology::NODE_RANK, 8);
        bulkData.add_node_sharing(node5, 1);
        bulkData.add_node_sharing(node6, 1);
        bulkData.add_node_sharing(node7, 1);
        bulkData.add_node_sharing(node8, 1);
        bulkData.add_node_sharing(node5, 2);
        bulkData.add_node_sharing(node6, 2);
        bulkData.add_node_sharing(node7, 2);
        bulkData.add_node_sharing(node8, 2);
    }
    else if (bulkData.parallel_rank() == 1) {
        elemId = 2;
        stk::mesh::declare_element(bulkData, block_2, elemId, elem2_nodes);
        stk::mesh::Entity node5 = bulkData.get_entity(stk::topology::NODE_RANK, 5);
        stk::mesh::Entity node6 = bulkData.get_entity(stk::topology::NODE_RANK, 6);
        stk::mesh::Entity node7 = bulkData.get_entity(stk::topology::NODE_RANK, 7);
        stk::mesh::Entity node8 = bulkData.get_entity(stk::topology::NODE_RANK, 8);
        bulkData.add_node_sharing(node5, 0);
        bulkData.add_node_sharing(node6, 0);
        bulkData.add_node_sharing(node7, 0);
        bulkData.add_node_sharing(node8, 0);
        bulkData.add_node_sharing(node5, 2);
        bulkData.add_node_sharing(node6, 2);
        bulkData.add_node_sharing(node7, 2);
        bulkData.add_node_sharing(node8, 2);
    }
    else if (bulkData.parallel_rank() == 2) {
        elemId = 3;
        stk::mesh::declare_element(bulkData, block_3, elemId, elem3_nodes);
        stk::mesh::Entity node5 = bulkData.get_entity(stk::topology::NODE_RANK, 5);
        stk::mesh::Entity node6 = bulkData.get_entity(stk::topology::NODE_RANK, 6);
        stk::mesh::Entity node7 = bulkData.get_entity(stk::topology::NODE_RANK, 7);
        stk::mesh::Entity node8 = bulkData.get_entity(stk::topology::NODE_RANK, 8);
        bulkData.add_node_sharing(node5, 0);
        bulkData.add_node_sharing(node6, 0);
        bulkData.add_node_sharing(node7, 0);
        bulkData.add_node_sharing(node8, 0);
        bulkData.add_node_sharing(node5, 1);
        bulkData.add_node_sharing(node6, 1);
        bulkData.add_node_sharing(node7, 1);
        bulkData.add_node_sharing(node8, 1);
    }

    bulkData.modification_end();
}

void change_entity_owner_hex_shell_hex_test_3_procs(bool aura_on)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    int proc = stk::parallel_machine_rank(comm);
    if(stk::parallel_machine_size(comm) == 3)
    {
        stk::mesh::MetaData meta;
        stk::mesh::BulkData::AutomaticAuraOption aura_option = stk::mesh::BulkData::AUTO_AURA;
        if (!aura_on)
        {
            aura_option = stk::mesh::BulkData::NO_AUTO_AURA;
        }
        stk::mesh::BulkData bulkData(meta, comm, aura_option);

        setup_hex_shell_hex_mesh(bulkData);

        ElemElemGraphTester elem_graph(bulkData);

        const Entity hex1   = bulkData.get_entity(stk::topology::ELEM_RANK, 1);
        const Entity hex3   = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
        const Entity shell2 = bulkData.get_entity(stk::topology::ELEM_RANK, 2);

        if (proc == 0) {
            // Connectivity for Hex Element 1
            EXPECT_EQ(1u, elem_graph.get_num_connected_elems(hex1));
            EXPECT_EQ(5,  elem_graph.get_side_id_to_connected_element(hex1, 0));
            EXPECT_EQ(2u, elem_graph.get_entity_id_of_remote_element(hex1, 0));
            EXPECT_FALSE(elem_graph.is_connected_elem_locally_owned(hex1, 0));
            EXPECT_EQ(1u, elem_graph.num_edges());
            EXPECT_EQ(1u, elem_graph.num_parallel_edges());
        }
        else if (proc == 1) {
            // Connectivity for Shell Element 2
            EXPECT_EQ(2u, elem_graph.get_num_connected_elems(shell2));
            EXPECT_EQ(0,  elem_graph.get_side_id_to_connected_element(shell2, 0));
            EXPECT_EQ(1,  elem_graph.get_side_id_to_connected_element(shell2, 1));
            EXPECT_EQ(3u, elem_graph.get_entity_id_of_remote_element(shell2, 0));
            EXPECT_EQ(1u, elem_graph.get_entity_id_of_remote_element(shell2, 1));
            EXPECT_FALSE(elem_graph.is_connected_elem_locally_owned(shell2, 0));
            EXPECT_FALSE(elem_graph.is_connected_elem_locally_owned(shell2, 1));
            EXPECT_EQ(2u, elem_graph.num_edges());
            EXPECT_EQ(2u, elem_graph.num_parallel_edges());
        }
        else if (proc == 2) {
            // Connectivity for Hex Element 3
            EXPECT_EQ(1u, elem_graph.get_num_connected_elems(hex3));
            EXPECT_EQ(4,  elem_graph.get_side_id_to_connected_element(hex3, 0));
            EXPECT_EQ(2u, elem_graph.get_entity_id_of_remote_element(hex3, 0));
            EXPECT_FALSE(elem_graph.is_connected_elem_locally_owned(hex3, 0));
            EXPECT_EQ(1u, elem_graph.num_edges());
            EXPECT_EQ(1u, elem_graph.num_parallel_edges());
        }

        stk::mesh::EntityId elem_to_move_global_id = 2;
        std::vector< std::pair< stk::mesh::Entity, int > > elem_proc_pairs_to_move;
        stk::mesh::Entity elem_to_move = bulkData.get_entity(stk::topology::ELEM_RANK, elem_to_move_global_id);

        if (proc == 1)
        {
            int destination_proc = 2;
            elem_proc_pairs_to_move.push_back(std::make_pair(elem_to_move, destination_proc));
        }

        change_entity_owner(bulkData, elem_graph, elem_proc_pairs_to_move);

        std::vector<unsigned> counts;
        stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
        int numLocallyOwnedElemsInMesh = counts[stk::topology::ELEM_RANK];

        size_t size_of_elem_graph = elem_graph.size();

        if (proc == 0)
        {
            EXPECT_EQ(1, numLocallyOwnedElemsInMesh);
            EXPECT_EQ(1u, size_of_elem_graph);

            stk::mesh::Entity elem_1 = bulkData.get_entity(stk::topology::ELEM_RANK, 1);
            impl::parallel_info& elem1_to_elem2_info = elem_graph.get_parallel_edge_info(elem_1, stk::mesh::EntityId(2));
            EXPECT_EQ(2, elem1_to_elem2_info.m_other_proc);
            EXPECT_EQ(1u, elem_graph.num_edges());
            EXPECT_EQ(1u, elem_graph.num_parallel_edges());
        }
        if (proc == 1)
        {
            EXPECT_EQ(0, numLocallyOwnedElemsInMesh);
            EXPECT_EQ(0u, size_of_elem_graph);

            stk::mesh::Entity elem_2 = bulkData.get_entity(stk::topology::ELEM_RANK, 2);
            ASSERT_THROW(elem_graph.get_parallel_edge_info(elem_2, stk::mesh::EntityId(1)), std::logic_error);

            ASSERT_THROW(elem_graph.get_parallel_edge_info(elem_2, stk::mesh::EntityId(3)), std::logic_error);
            EXPECT_EQ(0u, elem_graph.num_edges());
            EXPECT_EQ(0u, elem_graph.num_parallel_edges());
        }
        if (proc == 2)
        {
            EXPECT_EQ(2, numLocallyOwnedElemsInMesh);
            EXPECT_EQ(2u, size_of_elem_graph);

            stk::mesh::Entity elem_2 = bulkData.get_entity(stk::topology::ELEM_RANK, 2);
            impl::parallel_info& elem2_to_elem1_info = elem_graph.get_parallel_edge_info(elem_2, stk::mesh::EntityId(1));
            EXPECT_EQ(0, elem2_to_elem1_info.m_other_proc);
            EXPECT_EQ(3u, elem_graph.num_edges());
            EXPECT_EQ(1u, elem_graph.num_parallel_edges());
        }
    }
}

TEST(ElementGraph, test_change_entity_owner_3_procs_hex_shell_hex_mesh_with_aura)
{
    bool aura_on = true;
    change_entity_owner_hex_shell_hex_test_3_procs(aura_on);
}

TEST(ElementGraph, test_change_entity_owner_3_procs_hex_shell_hex_mesh_without_aura)
{
    bool aura_on = false;
    change_entity_owner_hex_shell_hex_test_3_procs(aura_on);
}

TEST(ElementGraph, skin_mesh_using_element_graph_parallel)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    std::vector<double> wall_times;
    wall_times.reserve(10);
    std::vector<std::string> msgs;
    msgs.reserve(10);

    std::vector<size_t> mem_usage;

    wall_times.push_back(stk::wall_time());
    msgs.push_back("program-start");
    mem_usage.push_back(stk::get_memory_usage_now());

    unsigned nProc = stk::parallel_machine_size(comm);
    if(nProc <= 2)
    {
        unsigned spatialDim = 3;
        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::Part& skin_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
        stk::io::put_io_part_attribute(skin_part);
        BulkDataElementGraphTester bulkData(meta, comm);

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after mesh-read");
        mem_usage.push_back(stk::get_memory_usage_now());

        unsigned num_locally_owned_elems = stk::mesh::count_selected_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData.buckets(stk::topology::ELEM_RANK));

        stk::mesh::EntityVector local_id_to_element_entity(num_locally_owned_elems, Entity());
        std::vector<stk::topology> element_topologies(num_locally_owned_elems);
        impl::set_local_ids_and_fill_element_entities_and_topologies(bulkData, local_id_to_element_entity, element_topologies);

        ElemElemGraphTester elemElemGraph(bulkData);

        ASSERT_EQ(num_locally_owned_elems, elemElemGraph.get_graph_size());

        SidesForElementGraph & via_sides = elemElemGraph.get_via_sides();

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after ElemElemGraphTester constructor");
        mem_usage.push_back(stk::get_memory_usage_now());

        std::vector<ElementSidePair> elem_side_pairs = skin_mesh(via_sides, element_topologies);

        bulkData.modification_begin();

        int offset_per_proc = 40*bulkData.parallel_rank();

        for(size_t face_index=0; face_index<elem_side_pairs.size(); ++face_index)
        {
            stk::mesh::Entity element = local_id_to_element_entity[elem_side_pairs[face_index].first];
            stk::mesh::EntityId face_global_id = face_index + 1 + offset_per_proc;
            stk::mesh::declare_element_side(bulkData, face_global_id, element, elem_side_pairs[face_index].second, &skin_part);
        }

        stk::mesh::Selector element_selector = bulkData.mesh_meta_data().locally_owned_part();
        bulkData.my_internal_modification_end_for_skin_mesh(stk::topology::FACE_RANK, stk::mesh::impl::MeshModification::MOD_END_SORT, element_selector, NULL);

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after create-faces");
        mem_usage.push_back(stk::get_memory_usage_now());

        stk::unit_test_util::write_mesh_using_stk_io("out.exo", bulkData, bulkData.parallel());

        std::vector<size_t> counts;
        stk::mesh::Selector skin = skin_part;
        stk::mesh::comm_mesh_counts(bulkData, counts, &skin);

        size_t num_faces = counts[stk::topology::FACE_RANK];
        EXPECT_EQ(18u, num_faces);

        EXPECT_EQ(6u/nProc, elemElemGraph.num_edges());
        EXPECT_EQ((nProc-1), elemElemGraph.num_parallel_edges());

        if (stk::parallel_machine_rank(comm) == 0)
        {
            for(size_t i=0;i<wall_times.size();++i)
            {
                std::cerr << "Wall time " << msgs[i] << ":\t" << wall_times[i] - wall_times[0] << std::endl;
            }

            for(size_t i=0;i<mem_usage.size();++i)
            {
                std::cerr << "Memory usage " << msgs[i] << ":\t" << mem_usage[i] - mem_usage[0] << std::endl;
            }
        }
    }
}

TEST(ElementSide, get_or_create_element_side_with_permutation)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;
    if(stk::parallel_machine_size(comm) == 1)
    {
        unsigned spatialDim = 3;
        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::Part& new_faces_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
        stk::mesh::PartVector face_parts {&new_faces_part};
        stk::io::put_io_part_attribute(new_faces_part);
        BulkDataElementGraphTester bulkData(meta, comm);

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x3", bulkData, comm);

        //////////////// Make first side

        bulkData.modification_begin();

        //get_or_create_element_side_with_permutation(bulkData, element);
        stk::mesh::Entity element1 = bulkData.get_entity(stk::topology::ELEM_RANK, 1);
        stk::mesh::EntityId side_global_id = 11;
        stk::mesh::EntityRank side_rank = bulkData.mesh_meta_data().side_rank();
        stk::mesh::Permutation side_permutation = static_cast<stk::mesh::Permutation>(4);
        stk::mesh::ConnectivityOrdinal side_ordinal = static_cast<stk::mesh::ConnectivityOrdinal>(1);

        EXPECT_FALSE(impl::is_id_already_in_use_locally(bulkData, side_rank, side_global_id));
        EXPECT_FALSE(impl::does_side_exist_with_different_permutation(bulkData, element1, side_ordinal, side_permutation));
        EXPECT_FALSE(impl::does_element_side_exist(bulkData, element1, side_ordinal));

        impl::connect_side_to_element(bulkData, element1, side_global_id, side_ordinal, side_permutation, face_parts);

        bulkData.modification_end();

        stk::mesh::Entity side1 = get_element_side(bulkData, element1, side_ordinal);
        EXPECT_TRUE(bulkData.is_valid(side1));

        EXPECT_TRUE(impl::is_id_already_in_use_locally(bulkData, side_rank, side_global_id));

        stk::mesh::Permutation side_permutation1 = static_cast<stk::mesh::Permutation>(0);
        EXPECT_TRUE(impl::does_side_exist_with_different_permutation(bulkData, element1, side_ordinal, side_permutation1));

        size_t num_sides = stk::mesh::count_selected_entities(new_faces_part, bulkData.buckets(side_rank));
        EXPECT_EQ(1u, num_sides);
    }
}

TEST(ElementGraph, test_parallel_graph_info_data_structure)
{
    if(stk::parallel_machine_size(MPI_COMM_WORLD) == 1)
    {
        ElementGraph elem_graph {
                {1},
                {0,-3},
        };

        SidesForElementGraph via_side {
                {4},
                {1,5},
        };


        ParallelGraphInfo parallel_graph_info;
        int other_proc = 1;
        int other_side_ord = 2;
        LocalId local_element = 1;
        LocalId other_element = 3;
        int permutation = 0;
        stk::mesh::EntityId chosen_face_id = 1;

        const bool inActivePart = true;
        parallel_graph_info.insert(std::make_pair(std::make_pair(local_element, other_element), parallel_info(other_proc, other_side_ord, permutation,
                chosen_face_id, stk::topology::INVALID_TOPOLOGY, inActivePart)));

        size_t num_elems_this_proc = elem_graph.size();
        EXPECT_EQ(2u, num_elems_this_proc);

        test_parallel_graph_info(elem_graph, parallel_graph_info, local_element, other_element, other_proc, other_side_ord, permutation);
    }
}

TEST(ElementGraph, test_parallel_graph_info_with_parallel_element_graph)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    if(stk::parallel_machine_size(comm) == 2)
    {
        stk::mesh::MetaData meta;
        stk::mesh::BulkData bulkData(meta, comm);

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        std::vector<unsigned> counts;
        stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
        size_t numLocallyOwnedElems = counts[stk::topology::ELEM_RANK];

        stk::mesh::EntityVector local_id_to_element_entity(numLocallyOwnedElems, Entity());
        std::vector<stk::topology> element_topologies(numLocallyOwnedElems);
        impl::set_local_ids_and_fill_element_entities_and_topologies(bulkData, local_id_to_element_entity, element_topologies);

        ElemElemGraphTester elemElemGraph(bulkData);

        ASSERT_EQ(numLocallyOwnedElems, elemElemGraph.get_graph_size());

        ElementGraph & elem_graph = elemElemGraph.get_element_graph();

        ParallelGraphInfo & parallel_graph_info = elemElemGraph.get_parallel_graph_info();

        if(stk::parallel_machine_rank(comm)==0)
        {
            LocalId local_element = 1;
            LocalId other_element = 3;
            int other_proc = 1;
            int other_side_ord = 4; // 4 left, 5 right
            int permutation = 4;

            test_parallel_graph_info(elem_graph, parallel_graph_info, local_element, other_element, other_proc, other_side_ord, permutation);
        }
        else
        {
            LocalId local_element = 0;
            LocalId other_element = 2;
            int other_proc = 0;
            int other_side_ord = 5; // 4 left, 5 right
            int permutation = 4;

            test_parallel_graph_info(elem_graph, parallel_graph_info, local_element, other_element, other_proc, other_side_ord, permutation);
        }

        EXPECT_EQ(3u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
}

TEST(ElementGraph, create_faces_using_element_graph_parallel)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    std::vector<double> wall_times;
    wall_times.reserve(10);
    std::vector<std::string> msgs;
    msgs.reserve(10);

    std::vector<size_t> mem_usage;

    wall_times.push_back(stk::wall_time());
    msgs.push_back("program-start");
    mem_usage.push_back(stk::get_memory_usage_now());

    if(stk::parallel_machine_size(comm) <= 2)
    {
        unsigned spatialDim = 3;
        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::Part& new_faces_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
        stk::io::put_io_part_attribute(new_faces_part);
        BulkDataElementGraphTester bulkData(meta, comm);

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after mesh-read");
        mem_usage.push_back(stk::get_memory_usage_now());

        create_faces_using_graph(bulkData, new_faces_part);

        const stk::mesh::BucketVector& sharedNodeBuckets = bulkData.get_buckets(stk::topology::NODE_RANK, meta.globally_shared_part());
        for(size_t bucket_index=0; bucket_index<sharedNodeBuckets.size(); ++bucket_index)
        {
            const stk::mesh::Bucket& bucket = *sharedNodeBuckets[bucket_index];
            EXPECT_TRUE(bucket.member(new_faces_part));
        }

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after create-faces");
        mem_usage.push_back(stk::get_memory_usage_now());

        std::vector<size_t> entity_counts;
        stk::mesh::comm_mesh_counts(bulkData, entity_counts);
        stk::mesh::EntityRank side_rank = meta.side_rank();
        unsigned num_faces = entity_counts[side_rank];
        EXPECT_EQ(21u, num_faces);

        stk::unit_test_util::write_mesh_using_stk_io("out.exo", bulkData, bulkData.parallel());

        if (stk::parallel_machine_rank(comm) == 0)
        {
            for(size_t i=0;i<wall_times.size();++i)
            {
                std::cerr << "Wall time " << msgs[i] << ":\t" << wall_times[i] - wall_times[0] << std::endl;
            }

            for(size_t i=0;i<mem_usage.size();++i)
            {
                std::cerr << "Memory usage " << msgs[i] << ":\t" << mem_usage[i] - mem_usage[0] << std::endl;
            }
        }
    }
}

TEST(ElementGraph, create_faces_using_element_graph_parallel_block_membership)
{
    MPI_Comm comm = MPI_COMM_WORLD;
    std::vector<double> wall_times;
    wall_times.reserve(10);
    std::vector<std::string> msgs;
    msgs.reserve(10);

    std::vector<size_t> mem_usage;

    wall_times.push_back(stk::wall_time());
    msgs.push_back("program-start");
    mem_usage.push_back(stk::get_memory_usage_now());

    if(stk::parallel_machine_size(comm) <= 2)
    {
        unsigned spatialDim = 3;
        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::Part& new_faces_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);

        stk::mesh::Part& block_2 = meta.declare_part_with_topology("block_2", stk::topology::HEX_8);
        EXPECT_EQ(stk::topology::ELEM_RANK, block_2.primary_entity_rank());

        stk::io::put_io_part_attribute(new_faces_part);
        BulkDataElementGraphTester bulkData(meta, comm);

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after mesh-read");
        mem_usage.push_back(stk::get_memory_usage_now());

        stk::mesh::Part& block_1 = *meta.get_part("block_1");

        bulkData.modification_begin();

        if (bulkData.parallel_rank() == 1)
        {
            //Move elem 3 from block_1 to block_2 so that the boundary between block_1 and block_2
            //will coincide with the proc boundary, and the shared face between elems 2 & 3
            //should have both block_1 and block_2.
            stk::mesh::Entity elem3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
            ASSERT_TRUE(bulkData.is_valid(elem3));
            stk::mesh::PartVector add_parts(1, &block_2);
            stk::mesh::PartVector rem_parts(1, &block_1);
            bulkData.change_entity_parts(elem3, add_parts, rem_parts);
        }

        bulkData.modification_end();

        create_faces_using_graph(bulkData, new_faces_part);

        wall_times.push_back(stk::wall_time());
        msgs.push_back("after create-faces");
        mem_usage.push_back(stk::get_memory_usage_now());

        std::vector<size_t> entity_counts;
        stk::mesh::comm_mesh_counts(bulkData, entity_counts);
        stk::mesh::EntityRank side_rank = bulkData.mesh_meta_data().side_rank();
        unsigned num_faces = entity_counts[side_rank];
        EXPECT_EQ(21u, num_faces);

        const stk::mesh::BucketVector& sharedFaceBuckets = bulkData.get_buckets(stk::topology::FACE_RANK, meta.globally_shared_part());
        if (bulkData.parallel_size() == 2)
        {
            ASSERT_EQ(1u, sharedFaceBuckets.size());

            const stk::mesh::Bucket& bucket = *sharedFaceBuckets[0];
            ASSERT_EQ(1u, bucket.size());
            EXPECT_TRUE(bucket.member(block_2));
            EXPECT_TRUE(bucket.member(block_1));
        }

        if (stk::parallel_machine_rank(comm) == 0)
        {
            for(size_t i=0;i<wall_times.size();++i)
            {
                std::cerr << "Wall time " << msgs[i] << ":\t" << wall_times[i] - wall_times[0] << std::endl;
            }

            for(size_t i=0;i<mem_usage.size();++i)
            {
                std::cerr << "Memory usage " << msgs[i] << ":\t" << mem_usage[i] - mem_usage[0] << std::endl;
            }
        }
    }
}

TEST(ElementGraph, compare_performance_skin_mesh)
{
    MPI_Comm comm = MPI_COMM_WORLD;

    //wall_times.push_back(stk::wall_time());
    // mem_usage.push_back(stk::get_memory_usage_now());

    std::string dimension = unitTestUtils::getOption("--zdim", "none");

    int xdim = 3;
    if ( dimension != "none")
    {
        xdim = std::atoi(dimension.c_str());
    }


    int ydim = xdim;
    int zdim = xdim * stk::parallel_machine_size(comm);

    std::string filename = stk::unit_test_util::get_name_of_generated_mesh(xdim, ydim, zdim, "|nodeset:zZ");

    {
        unsigned spatialDim = 3;
        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::Part& skin_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
        stk::io::put_io_part_attribute(skin_part);
        BulkDataElementGraphTester bulkData(meta, comm);

        stk::unit_test_util::fill_mesh_using_stk_io(filename, bulkData, comm);

        {
            double wall_time_start = stk::wall_time();

            stk::mesh::PartVector parts(1, &skin_part);
            stk::mesh::skin_mesh(bulkData, parts);

            double elapsed_time = stk::wall_time() - wall_time_start;

            if (stk::parallel_machine_rank(comm) == 0)
            {
                std::cerr << "STK time: " << elapsed_time << std::endl;
            }
        }
    }

    {
        unsigned spatialDim = 3;
        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::Part& skin_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
        stk::io::put_io_part_attribute(skin_part);
        BulkDataElementGraphTester bulkData(meta, comm);

        stk::unit_test_util::fill_mesh_using_stk_io(filename, bulkData, comm);

        {
            double wall_time_start = stk::wall_time();

            unsigned num_locally_owned_elems = stk::mesh::count_selected_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData.buckets(stk::topology::ELEM_RANK));

            stk::mesh::EntityVector local_id_to_element_entity(num_locally_owned_elems, Entity());
            std::vector<stk::topology> element_topologies(num_locally_owned_elems);
            impl::set_local_ids_and_fill_element_entities_and_topologies(bulkData, local_id_to_element_entity, element_topologies);

            ElemElemGraphTester elemElemGraph(bulkData);

            SidesForElementGraph & via_sides = elemElemGraph.get_via_sides();

            std::vector<ElementSidePair> elem_side_pairs = skin_mesh(via_sides, element_topologies);

            bulkData.modification_begin();

            int offset_per_proc = 100000*bulkData.parallel_rank();

            for(size_t face_index=0; face_index<elem_side_pairs.size(); ++face_index)
            {
                stk::mesh::Entity element = local_id_to_element_entity[elem_side_pairs[face_index].first];
                stk::mesh::EntityId face_global_id = face_index + 1 + offset_per_proc;
                stk::mesh::declare_element_side(bulkData, face_global_id, element, elem_side_pairs[face_index].second, &skin_part);
            }

            stk::mesh::Selector element_selector = bulkData.mesh_meta_data().locally_owned_part();
            bulkData.my_internal_modification_end_for_skin_mesh(stk::topology::FACE_RANK, stk::mesh::impl::MeshModification::MOD_END_SORT, element_selector, NULL);

            double elapsed_time = stk::wall_time() - wall_time_start;

            stk::unit_test_util::write_mesh_using_stk_io("out.exo", bulkData, bulkData.parallel());

            std::vector<size_t> counts;
            stk::mesh::comm_mesh_counts(bulkData, counts);

            if (stk::parallel_machine_rank(comm) == 0)
            {
                std::cerr << "Element graph time: " << elapsed_time << std::endl;
                std::cerr << "Total # of elements: " << counts[stk::topology::ELEM_RANK] << std::endl;
            }
        }
    }
}

TEST(ElementGraph, compare_performance_create_faces)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    std::string dimension = unitTestUtils::getOption("--zdim", "none");

    int xdim = 3;
    if ( dimension != "none")
    {
        xdim = std::atoi(dimension.c_str());
    }

    int ydim = xdim;
    int zdim = xdim * stk::parallel_machine_size(comm);

    std::string filename = stk::unit_test_util::get_name_of_generated_mesh(xdim, ydim, zdim, "|nodeset:zZ");

    {
        unsigned spatialDim = 3;
        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::Part& faces_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
        stk::io::put_io_part_attribute(faces_part);
        BulkDataElementGraphTester bulkData(meta, comm);

        stk::unit_test_util::fill_mesh_using_stk_io(filename, bulkData, comm);

        {
            double wall_time_start = stk::wall_time();

            stk::mesh::PartVector parts(1, &faces_part);
            stk::mesh::create_faces(bulkData);

            double elapsed_time = stk::wall_time() - wall_time_start;

            if (stk::parallel_machine_rank(comm) == 0)
            {
                std::cerr << "STK time: " << elapsed_time << std::endl;
            }
        }
    }

    {
        unsigned spatialDim = 3;
        stk::mesh::MetaData meta(spatialDim);
        bool force_no_induce = true;
        stk::mesh::Part& faces_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4, force_no_induce);
        stk::io::put_io_part_attribute(faces_part);
        BulkDataElementGraphTester bulkData(meta, comm);

        stk::unit_test_util::fill_mesh_using_stk_io(filename, bulkData, comm);

        {
            double wall_time_start = stk::wall_time();

            create_faces_using_graph(bulkData, faces_part);

            double elapsed_time = stk::wall_time() - wall_time_start;

            stk::unit_test_util::write_mesh_using_stk_io("out.exo", bulkData, bulkData.parallel());

            std::vector<size_t> counts;
            stk::mesh::comm_mesh_counts(bulkData, counts);

            if (stk::parallel_machine_rank(comm) == 0)
            {
                std::cerr << "Element graph time: " << elapsed_time << std::endl;
                std::cerr << "Total # of elements: " << counts[stk::topology::ELEM_RANK] << std::endl;
            }
        }
    }
}

TEST(ElementGraph, make_items_inactive)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    unsigned nProc = stk::parallel_machine_size(comm);

    if(nProc <= 2)
    {
        unsigned spatialDim = 3;

        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::Part& faces_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
        stk::mesh::PartVector boundary_mesh_parts { &faces_part };
        stk::io::put_io_part_attribute(faces_part);
        ElementDeathUtils::ElementDeathBulkDataTester bulkData(meta, comm, stk::mesh::BulkData::AUTO_AURA);

        stk::mesh::Part& active = meta.declare_part("active"); // can't specify rank, because it gets checked against size of rank_names

        ASSERT_TRUE(active.primary_entity_rank() == stk::topology::INVALID_RANK);

        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        stk::unit_test_util::put_mesh_into_part(bulkData, active);

        ElemElemGraphTester graph(bulkData);

        size_t num_gold_edges =  6/bulkData.parallel_size();
        ASSERT_EQ(num_gold_edges, graph.num_edges());
        ASSERT_EQ((nProc-1), graph.num_parallel_edges());

        stk::mesh::EntityVector deactivated_elems;

        stk::mesh::Entity elem1 = bulkData.get_entity(stk::topology::ELEM_RANK, 1);
        if (bulkData.is_valid(elem1) && bulkData.bucket(elem1).owned() )
        {
            deactivated_elems.push_back(elem1);
        }
        stk::mesh::Entity elem3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
        if (bulkData.is_valid(elem3) && bulkData.bucket(elem3).owned())
        {
            deactivated_elems.push_back(elem3);
        }

        ElementDeathUtils::deactivate_elements(deactivated_elems, bulkData, active);
        bulkData.modification_begin();
        bulkData.my_de_induce_unranked_part_from_nodes(deactivated_elems, active);

        for(size_t i=0; i<deactivated_elems.size(); ++i)
        {
            EXPECT_FALSE(bulkData.bucket(deactivated_elems[i]).member(active));
        }

        const stk::mesh::BucketVector& all_node_buckets = bulkData.buckets(stk::topology::NODE_RANK);
        for(size_t i=0; i<all_node_buckets.size(); ++i)
        {
            const stk::mesh::Bucket& bucket = *all_node_buckets[i];
            for(size_t node_index=0; node_index<bucket.size(); ++node_index)
            {
                stk::mesh::EntityId id = bulkData.identifier(bucket[node_index]);
                if (id >=1 && id <= 4)
                {
                    EXPECT_FALSE(bucket.member(active));
                }
                else
                {
                    EXPECT_TRUE(bucket.member(active)) << "for node id " << id << std::endl;
                }
            }
        }

        bulkData.modification_end();
        ASSERT_EQ((nProc-1), graph.num_parallel_edges());
   }
}

TEST(ElementGraph, test_element_death)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    if(stk::parallel_machine_size(comm) <= 2)
    {
        std::string dimension = unitTestUtils::getOption("--zdim", "none");

        //IO error when this is <4.  Shared face being attached to the wrong element
        int xdim = 4;
        if(dimension != "none")
        {
            xdim = std::atoi(dimension.c_str());
        }

        int ydim = xdim;
        int zdim = xdim; //  * stk::parallel_machine_size(comm);

        std::string filename = stk::unit_test_util::get_name_of_generated_mesh(xdim, ydim, zdim, "|nodeset:zZ");

        {
            unsigned spatialDim = 3;
            stk::mesh::MetaData meta(spatialDim);
            stk::mesh::Part& faces_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
            stk::mesh::PartVector boundary_mesh_parts { &faces_part };
            stk::io::put_io_part_attribute(faces_part);
            BulkDataElementGraphTester bulkData(meta, comm);

            stk::mesh::Part& active = meta.declare_part("active", stk::topology::ELEMENT_RANK);
            stk::unit_test_util::fill_mesh_using_stk_io(filename, bulkData, comm);
            stk::unit_test_util::write_mesh_using_stk_io("orig.exo", bulkData, bulkData.parallel());

            double start_graph = stk::wall_time();

            ASSERT_TRUE(meta.get_part("block_1") != NULL);

            stk::mesh::Part& block_1 = *meta.get_part("block_1");

            stk::unit_test_util::put_mesh_into_part(bulkData, active);

            std::ostringstream os;
            os << "Proc id: " << bulkData.parallel_rank() << std::endl;

            ElemElemGraphTester elementGraph(bulkData);

            double elapsed_graph_time = stk::wall_time() - start_graph;
            os << "Time to create graph: " << elapsed_graph_time << std::endl;

            stk::mesh::EntityRank side_rank = meta.side_rank();

            int num_time_steps = xdim * ydim * zdim;
            double elapsed_death_time = 0;

            boundary_mesh_parts.push_back(&active);

            for(int i = 0; i < num_time_steps; ++i)
            {
                stk::mesh::EntityVector killedElements = get_killed_elements(bulkData, i, active);
                stk::unit_test_util::move_killed_elements_out_of_parts(bulkData, killedElements, {&block_1, &active});
                double start_time = stk::wall_time();
                process_killed_elements(bulkData, elementGraph, killedElements, active, boundary_mesh_parts, &boundary_mesh_parts);
                elapsed_death_time += (stk::wall_time() - start_time);
            }

            stk::mesh::Selector sel = block_1;
            std::vector<size_t> counts1;
            stk::mesh::comm_mesh_counts(bulkData, counts1, &sel);

            size_t num_active = counts1[stk::topology::ELEM_RANK];

            stk::mesh::Selector sel2 = faces_part;
            stk::mesh::comm_mesh_counts(bulkData, counts1, &sel2);

            size_t num_faces = counts1[side_rank];

            EXPECT_EQ(2u, num_active);
            EXPECT_EQ(5u, num_faces);

            if(stk::parallel_machine_rank(comm) == 0)
            {
                os << "Total time: " << elapsed_death_time << std::endl;
                os << "Total # of alive elements: " << num_active << std::endl;
                os << "Total # of faces: " << num_faces << std::endl;
            }

            std::cerr << os.str();
            stk::unit_test_util::write_mesh_using_stk_io("out.exo", bulkData, bulkData.parallel());
        }
    }
}


class ElementDeathRestartTest
{
public:
    typedef stk::mesh::Field<double> ActiveFieldType;

    ElementDeathRestartTest() {
    }

    virtual ~ElementDeathRestartTest() {
        delete elementGraph;
        delete bulkData;
        delete metaData;
    }

    void load_without_restart()
    {
        initializeObjects();
        std::string filename = stk::unit_test_util::get_name_of_generated_mesh(1, 1, 2, "|sideset:zZ");
        stk::unit_test_util::fill_mesh_using_stk_io(filename, *bulkData, bulkData->parallel());
        stk::unit_test_util::put_mesh_into_part(*bulkData, *activePart);
        createElemElemGraph();
    }

    void read_restart_file(const std::string& filename)
    {
        stk::io::StkMeshIoBroker stkIo(bulkData->parallel());
        stkIo.set_bulk_data(*bulkData);
        stkIo.add_mesh_database(filename, stk::io::READ_RESTART);
        stkIo.create_input_mesh();
        stkIo.populate_bulk_data();
        stk::io::MeshField ioActiveField(*deathStatusField);
        stkIo.add_input_field(ioActiveField);
        stkIo.read_defined_input_fields(restartTime);
    }

    void load_with_restart(const std::string& filename)
    {
        initializeObjects();
        read_restart_file(filename);

        stk::mesh::EntityVector deactivatedElements = set_active_part_from_field();
        createElemElemGraph();

        stk::mesh::process_killed_elements(*bulkData, *elementGraph, deactivatedElements, *activePart, partsForCreatingSides, NULL);
    }

    void initializeObjects()
    {
        stk::ParallelMachine comm = MPI_COMM_WORLD;
        unsigned spatialDim = 3;
        metaData = new stk::mesh::MetaData(spatialDim);
        stk::io::put_io_part_attribute(metaData->universal_part());
        deathStatusField = &metaData->declare_field<ActiveFieldType>(stk::topology::ELEM_RANK,deathStatusFieldName);
        int zeroInitialValue = 0;
        stk::mesh::put_field(*deathStatusField,metaData->universal_part(),&zeroInitialValue);

        activePart = &metaData->declare_part("active");
        deathPart = &metaData->declare_part("death_1", metaData->side_rank());
        const bool forceNoInduce = true;
        sidesCreatedDuringDeath = &metaData->declare_part("sides_created_during_death", metaData->side_rank(),forceNoInduce);

        partsForCreatingSides.push_back(activePart);
        partsForCreatingSides.push_back(sidesCreatedDuringDeath);
        partsForCreatingSides.push_back(deathPart);
        boundaryMeshParts.push_back(deathPart);
        activePartVector.push_back(activePart);

        bulkData = new stk::mesh::BulkData(*metaData, comm, stk::mesh::BulkData::NO_AUTO_AURA);
    }

    void createElemElemGraph()
    {
        elementGraph = new stk::mesh::ElemElemGraph(*bulkData, *activePart);
    }

    void kill_element(int globalId)
    {
        stk::mesh::Entity element = bulkData->get_entity(stk::topology::ELEM_RANK,globalId);
        stk::mesh::EntityVector elementsToKill;
        if (bulkData->is_valid(element))
        {
            elementsToKill.push_back(element);
        }
        stk::unit_test_util::move_killed_elements_out_of_parts(*bulkData, elementsToKill, activePartVector);
        process_killed_elements(*bulkData, *elementGraph, elementsToKill, *activePart, partsForCreatingSides, &boundaryMeshParts);
    }

    void verify_mesh_before_death() const
    {
        stk::mesh::Entity element1 = bulkData->get_entity(stk::topology::ELEM_RANK,1);
        if (bulkData->is_valid(element1)) {
            EXPECT_EQ(bulkData->num_faces(element1), 1u);
            stk::mesh::Entity side4OfElement1 = bulkData->begin_faces(element1)[0];
            EXPECT_TRUE(bulkData->is_valid(side4OfElement1));
            EXPECT_EQ(bulkData->begin_face_ordinals(element1)[0], 4u);
            EXPECT_FALSE(bulkData->bucket(side4OfElement1).member(*sidesCreatedDuringDeath));
            EXPECT_TRUE(bulkData->bucket(side4OfElement1).member(*activePart));
        }
    }

    void verify_mesh_after_killing_element_1() const
    {
        stk::mesh::Entity element1 = bulkData->get_entity(stk::topology::ELEM_RANK,1);
        if (bulkData->is_valid(element1)) {
            EXPECT_FALSE( bulkData->bucket(element1).member(*activePart) );
            EXPECT_EQ(bulkData->num_faces(element1), 2u);
            const stk::mesh::Entity* sides = bulkData->begin_faces(element1);
            const stk::mesh::ConnectivityOrdinal* sideOrdinals = bulkData->begin_face_ordinals(element1);
            for (size_t sideI=0 ; sideI<2u ; ++sideI) {
                stk::mesh::Entity side = sides[sideI];
                int sideOrdinal = sideOrdinals[sideI];
                if (sideOrdinal == 5) {
                    // Face between element1 and element2
                    EXPECT_TRUE(bulkData->bucket(side).member(*activePart));
                    EXPECT_TRUE(bulkData->bucket(side).member(*sidesCreatedDuringDeath));
                }
                else {
                    // Side from generated mesh
                    EXPECT_FALSE(bulkData->bucket(side).member(*activePart));
                    EXPECT_FALSE(bulkData->bucket(side).member(*sidesCreatedDuringDeath));
                }
            }
        }
    }

    void verify_mesh_after_killing_element_2() const
    {
        stk::mesh::Entity element1 = bulkData->get_entity(stk::topology::ELEM_RANK,1);
        if (bulkData->is_valid(element1)) {
            EXPECT_EQ(bulkData->num_faces(element1), 1u);
            stk::mesh::Entity side = bulkData->begin_faces(element1)[0];
            stk::mesh::ConnectivityOrdinal sideOrdinalForSideFromGeneratedMesh = bulkData->begin_face_ordinals(element1)[0];
            EXPECT_EQ(sideOrdinalForSideFromGeneratedMesh, 4u);
            EXPECT_FALSE(bulkData->bucket(side).member(*activePart));
            EXPECT_FALSE(bulkData->bucket(side).member(*sidesCreatedDuringDeath));
        }

        stk::mesh::Entity element2 = bulkData->get_entity(stk::topology::ELEM_RANK,2);
        if (bulkData->is_valid(element2)) {
            EXPECT_FALSE( bulkData->bucket(element2).member(*activePart) );
            EXPECT_EQ(bulkData->num_faces(element2), 1u);
            stk::mesh::Entity side = bulkData->begin_faces(element2)[0];
            stk::mesh::ConnectivityOrdinal sideOrdinalForSideFromGeneratedMesh = bulkData->begin_face_ordinals(element2)[0];
            EXPECT_EQ(sideOrdinalForSideFromGeneratedMesh, 5u);
            EXPECT_FALSE(bulkData->bucket(side).member(*activePart));
            EXPECT_FALSE(bulkData->bucket(side).member(*sidesCreatedDuringDeath));
        }
    }

    stk::mesh::EntityVector set_active_part_from_field()
    {
        stk::mesh::EntityVector deactivatedElements;
        stk::mesh::EntityVector entities;
        std::vector<stk::mesh::PartVector> addParts;
        std::vector<stk::mesh::PartVector> removeParts;
        const stk::mesh::BucketVector& elements = bulkData->buckets(stk::topology::ELEM_RANK);
        const int deadElementStatus = 1;
        for (size_t bucketI=0 ; bucketI<elements.size() ; ++bucketI) {
            stk::mesh::Bucket& bucket = *(elements[bucketI]);
            for (size_t elementI=0 ; elementI<bucket.size() ; ++elementI) {
                stk::mesh::Entity element = bucket[elementI];
                double* deathStatus = stk::mesh::field_data(*deathStatusField,element);
                if (static_cast<int>(deathStatus[0]) == deadElementStatus) {
                    entities.push_back(element);
                    addParts.push_back({});
                    removeParts.push_back({activePart});
                    deactivatedElements.push_back(element);
                } else {
                    entities.push_back(element);
                    addParts.push_back({activePart});
                    removeParts.push_back({});
                }
            }
        }
        bulkData->batch_change_entity_parts( entities, addParts, removeParts);
        return deactivatedElements;
    }

    void set_active_field_from_part()
    {
        const stk::mesh::BucketVector& elements = bulkData->buckets(stk::topology::ELEM_RANK);
        for (size_t bucketI=0 ; bucketI<elements.size() ; ++bucketI) {
            stk::mesh::Bucket& bucket = *(elements[bucketI]);
            for (size_t elementI=0 ; elementI<bucket.size() ; ++elementI) {
                stk::mesh::Entity element = bucket[elementI];
                double* deathStatus = stk::mesh::field_data(*deathStatusField,element);
                deathStatus[0] = !bulkData->bucket(element).member(*activePart);
            }
        }
    }

    void write_restart_file(std::string restartFileName)
    {
        stk::mesh::FieldBase* deathStatusFieldBase = deathStatusField;
        set_active_field_from_part();

        stk::io::StkMeshIoBroker stkIo(bulkData->parallel());
        stkIo.set_bulk_data(*bulkData);
        size_t fileHandle = stkIo.create_output_mesh(restartFileName, stk::io::WRITE_RESTART);
        stkIo.write_output_mesh(fileHandle);
        stkIo.add_field(fileHandle, *deathStatusFieldBase);
        stkIo.begin_output_step(fileHandle, restartTime);
        stkIo.write_defined_output_fields(fileHandle);
        stkIo.end_output_step(fileHandle);
    }
public:
    stk::mesh::MetaData* metaData;
    stk::mesh::BulkData* bulkData;
    stk::mesh::ElemElemGraph* elementGraph;
    stk::mesh::Part* activePart;
    stk::mesh::PartVector activePartVector;
    stk::mesh::PartVector partsForCreatingSides;
    stk::mesh::PartVector boundaryMeshParts;
    stk::mesh::Part* deathPart;
    stk::mesh::Part* sidesCreatedDuringDeath;
    ActiveFieldType* deathStatusField;
    std::string deathStatusFieldName = "death_status";
    double restartTime = 1.0;

};


TEST(ElementDeath, test_element_death_without_restart)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;
    if(stk::parallel_machine_size(comm) <= 2)
    {
        ElementDeathRestartTest elementDeathTest;
        elementDeathTest.load_without_restart();
        elementDeathTest.verify_mesh_before_death();
        elementDeathTest.kill_element(1);
        elementDeathTest.verify_mesh_after_killing_element_1();
        elementDeathTest.kill_element(2);
        elementDeathTest.verify_mesh_after_killing_element_2();
    }
}

TEST(ElementDeath, test_element_death_with_restart)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;
    if(stk::parallel_machine_size(comm) <= 2)
    {
        {
            ElementDeathRestartTest elementDeathTest;
            elementDeathTest.load_without_restart();
            elementDeathTest.verify_mesh_before_death();
            elementDeathTest.kill_element(1);
            elementDeathTest.verify_mesh_after_killing_element_1();
            elementDeathTest.write_restart_file("elemDeathRestartFile.exo");
        }
        {
            ElementDeathRestartTest elementDeathTest;
            elementDeathTest.load_with_restart("elemDeathRestartFile.exo");
            elementDeathTest.verify_mesh_after_killing_element_1();
            elementDeathTest.kill_element(2);
            elementDeathTest.verify_mesh_after_killing_element_2();
        }
    }
}

//EndDocExample1

TEST( ElementGraph, HexHexHexSerial )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0-----------15.0
    //          /|             /|             /|             /|
    //         / |            / |            / |            / |
    //        /  |           /  |           /  |           /  |
    //      4.0------------8.0-----------12.0-----------16.0  |
    //       |   |          |   |          |   |          |   |
    //       |   |   1.0    |   |   2.0    |   |   3.0    |   |
    //       |   |          |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.0---------|-14.0
    //       |  /           |  /           |  /           |  /
    //       | /            | /            | /            | /
    //       |/             |/             |/             |/
    //      1.0------------5.0------------9.0-----------13.0
    //

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 1u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart = &meta.declare_part_with_topology("hex_part", stk::topology::HEX_8);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1,  2,  3,  4,  5,  6,  7,  8 },
        { 5,  6,  7,  8,  9, 10, 11, 12 },
        { 9, 10, 11, 12, 13, 14, 15, 16 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2, 3 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 0, 0 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1 = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity hex3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    // Connectivity for Hex Element 1
    EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,    elemElemGraph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(hex1, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

    // Connectivity for Hex Element 2
    EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(hex2));
    EXPECT_EQ(4,    elemElemGraph.get_side_id_to_connected_element(hex2, 0));
    EXPECT_EQ(5,    elemElemGraph.get_side_id_to_connected_element(hex2, 1));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(hex2, 0));
    EXPECT_EQ(hex3, elemElemGraph.get_connected_element(hex2, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));

    // Connectivity for Hex Element 3
    EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(hex3));
    EXPECT_EQ(4,    elemElemGraph.get_side_id_to_connected_element(hex3, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(hex3, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex3, 0));

    EXPECT_EQ(4u, elemElemGraph.num_edges());
    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, HexShellSerial )
{
    //  ID.proc
    //
    //          3.0------------7.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      4.0------------8.0  |
    //       |   |          |   |
    //       |   |   1.0    |2.0|
    //       |   |          |   |
    //       |  2.0---------|--6.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------5.0
    //                      ^
    //                      |
    //                       ---- Single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size > 1)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 2 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity shell2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);

    // Connectivity for Hex Element 1
    EXPECT_EQ(1u,     elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(shell2, elemElemGraph.get_connected_element(hex1, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

    // Connectivity for Shell Element 2
    EXPECT_EQ(1u,             elemElemGraph.get_num_connected_elems(shell2));
    EXPECT_EQ(1,              elemElemGraph.get_side_id_to_connected_element(shell2, 0));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell2, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell2, 0));

    EXPECT_EQ(2u, elemElemGraph.num_edges());
    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, AdjacentHexShellSerial )
{
    //  ID.proc
    //
    //         12.0-----------11.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      9.0-----------10.0  |
    //       |   |          |   |
    //       |   |   2.0    |4.0|
    //       |   |          |   |
    //       |  8.0---------|--7.0
    //       |  /|          |  /|
    //       | / |          | / |
    //       |/  |          |/  |
    //      5.0------------6.0  |
    //       |   |          |   |
    //       |   |   1.0    |3.0|
    //       |   |          |   |
    //       |  4.0---------|--3.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------2.0
    //                      ^
    //                      |
    //                       ---- Single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size > 1)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 2, 3,  7,  6 },
        { 6, 7, 11, 10 },
    };
    stk::mesh::EntityId shellElemIDs[] = { 3, 4 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);
    const Entity shell4 = mesh.get_entity(stk::topology::ELEM_RANK, 4);

    // Connectivity for Hex Element 1
    EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(1,      elemElemGraph.get_side_id_to_connected_element(hex1, 1));
    EXPECT_EQ(hex2,   elemElemGraph.get_connected_element(hex1, 0));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

    // Connectivity for Hex Element 2
    EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex2));
    EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 0));
    EXPECT_EQ(1,      elemElemGraph.get_side_id_to_connected_element(hex2, 1));
    EXPECT_EQ(hex1,   elemElemGraph.get_connected_element(hex2, 0));
    EXPECT_EQ(shell4, elemElemGraph.get_connected_element(hex2, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));

    // Connectivity for Shell Element 3
    EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(shell3));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));

    // Connectivity for Shell Element 4
    EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(shell4));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell4, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(shell4, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell4, 0));

    EXPECT_EQ(6u, elemElemGraph.num_edges());
    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, DISABLED_HexShellShellSerial )
{
    //  ID.proc
    //
    //          3.0------------7.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      4.0------------8.0  |
    //       |   |          |   |
    //       |   |   1.0    |2.0|
    //       |   |          |3.0|
    //       |  2.0---------|--6.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------5.0
    //                      ^
    //                      |
    //                       ---- Two stacked shell elements

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size > 1)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 },
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 2, 3 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity shell2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    // Connectivity for Hex Element 1
    EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 1));
    EXPECT_EQ(shell2, elemElemGraph.get_connected_element(hex1, 0));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

    // Connectivity for Shell Element 2
    EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(shell2));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell2, 0));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell2, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell2, 0));

    // Connectivity for Shell Element 3
    EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(shell3));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));

    EXPECT_EQ(4u, elemElemGraph.num_edges());
    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, HexShellHexSerial )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.0  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.0    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.0
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.0
    //                      ^
    //                      |
    //                       ---- Single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size > 1)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    // Connectivity for Hex Element 1
    EXPECT_EQ(1u,     elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

    // Connectivity for Hex Element 2
    EXPECT_EQ(1u,     elemElemGraph.get_num_connected_elems(hex2));
    EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 0));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex2, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));

    // Connectivity for Shell Element 3
    EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
    EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(shell3, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));

    EXPECT_EQ(4u, elemElemGraph.num_edges());
    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, DISABLED_HexShellShellHexSerial )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.0  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.0    |   |
    //       |   |          |4.0|          |   |
    //       |  2.0---------|--6.0---------|-10.0
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.0
    //                      ^
    //                      |
    //                       ---- Two stacked shell elements

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size > 1)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 },
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3, 4 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);
    const Entity shell4 = mesh.get_entity(stk::topology::ELEM_RANK, 4);

    // Connectivity for Hex Element 1
    EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 1));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 0));
    EXPECT_EQ(shell4, elemElemGraph.get_connected_element(hex1, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

    // Connectivity for Hex Element 2
    EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex2));
    EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 0));
    EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 1));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex2, 0));
    EXPECT_EQ(shell4, elemElemGraph.get_connected_element(hex2, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));

    // Connectivity for Shell Element 3
    EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
    EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(shell3, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));

    // Connectivity for Shell Element 4
    EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell4));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell4, 0));
    EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell4, 1));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell4, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(shell4, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell4, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell4, 1));

    EXPECT_EQ(8u, elemElemGraph.num_edges());
    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, DISABLED_HexShellReversedShellHexSerial )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.0  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.0    |   |
    //       |   |          |4.0|          |   |
    //       |  2.0---------|--6.0---------|-10.0
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.0
    //                      ^
    //                      |
    //                       ---- Two stacked shell elements

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    int p_size = stk::parallel_machine_size(pm);

    if(p_size > 1)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 },
        { 5, 8, 7, 6 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3, 4 };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);
    const Entity shell4 = mesh.get_entity(stk::topology::ELEM_RANK, 4);

    // Connectivity for Hex Element 1
    EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex1));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
    EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 1));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 0));
    EXPECT_EQ(shell4, elemElemGraph.get_connected_element(hex1, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

    // Connectivity for Hex Element 2
    EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex2));
    EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 0));
    EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 1));
    EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex2, 0));
    EXPECT_EQ(shell4, elemElemGraph.get_connected_element(hex2, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));

    // Connectivity for Shell Element 3
    EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
    EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(shell3, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));

    // Connectivity for Shell Element 4
    EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell4));
    EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell4, 0));
    EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell4, 1));
    EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell4, 0));
    EXPECT_EQ(hex2, elemElemGraph.get_connected_element(shell4, 1));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell4, 0));
    EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell4, 1));

    EXPECT_EQ(8u, elemElemGraph.num_edges());
    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

void setup_node_sharing(stk::mesh::BulkData &mesh, const std::vector< std::vector<unsigned> > & shared_nodeIDs_and_procs )
{
    const unsigned p_rank = mesh.parallel_rank();

    for (size_t nodeIdx = 0, end = shared_nodeIDs_and_procs.size(); nodeIdx < end; ++nodeIdx) {
        if (p_rank == shared_nodeIDs_and_procs[nodeIdx][0]) {
            stk::mesh::EntityId nodeID = shared_nodeIDs_and_procs[nodeIdx][1];
            int sharingProc = shared_nodeIDs_and_procs[nodeIdx][2];
            stk::mesh::Entity node = mesh.get_entity(stk::topology::NODE_RANK, nodeID);
            mesh.add_node_sharing(node, sharingProc);
        }
    }
}

TEST( ElementGraph, Hex0Hex0Hex1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0-----------15.1
    //          /|             /|             /|             /|
    //         / |            / |            / |            / |
    //        /  |           /  |           /  |           /  |
    //      4.0------------8.0-----------12.0-----------16.1  |
    //       |   |          |   |          |   |          |   |
    //       |   |   1.0    |   |   2.0    |   |   3.1    |   |
    //       |   |          |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.0---------|-14.1
    //       |  /           |  /           |  /           |  /
    //       | /            | /            | /            | /
    //       |/             |/             |/             |/
    //      1.0------------5.0------------9.0-----------13.1
    //

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart = &meta.declare_part_with_topology("hex_part", stk::topology::HEX_8);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1,  2,  3,  4,  5,  6,  7,  8 },
        { 5,  6,  7,  8,  9, 10, 11, 12 },
        { 9, 10, 11, 12, 13, 14, 15, 16 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2, 3 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 0, 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0,  9, 1 },  // proc 0
        { 0, 10, 1 },
        { 0, 11, 1 },
        { 0, 12, 1 },
        { 1,  9, 0 },  // proc 1
        { 1, 10, 0 },
        { 1, 11, 0 },
        { 1, 12, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1 = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity hex3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,    elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(hex2, elemElemGraph.get_connected_element(hex1, 0));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

        // Connectivity for Hex Element 2
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,    elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(5,    elemElemGraph.get_side_id_to_connected_element(hex2, 1));
        EXPECT_EQ(hex1, elemElemGraph.get_connected_element(hex2, 0));
        EXPECT_EQ(3u,   elemElemGraph.get_entity_id_of_remote_element(hex2, 1));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));

        EXPECT_EQ(3u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Hex Element 3
        EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(hex3));
        EXPECT_EQ(4,    elemElemGraph.get_side_id_to_connected_element(hex3, 0));
        EXPECT_EQ(2u,   elemElemGraph.get_entity_id_of_remote_element(hex3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex3, 0));

        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, Hex0Hex1Hex0Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0-----------15.0
    //          /|             /|             /|             /|
    //         / |            / |            / |            / |
    //        /  |           /  |           /  |           /  |
    //      4.0------------8.0-----------12.0-----------16.0  |
    //       |   |          |   |          |   |          |   |
    //       |   |   1.0    |   |   2.1    |   |   3.0    |   |
    //       |   |          |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.0---------|-14.0
    //       |  /           |  /           |  /           |  /
    //       | /            | /            | /            | /
    //       |/             |/             |/             |/
    //      1.0------------5.0------------9.0-----------13.0
    //

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart = &meta.declare_part_with_topology("hex_part", stk::topology::HEX_8);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1,  2,  3,  4,  5,  6,  7,  8 },
        { 5,  6,  7,  8,  9, 10, 11, 12 },
        { 9, 10, 11, 12, 13, 14, 15, 16 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2, 3 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 1, 0 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0,  5, 1 },  // proc 0
        { 0,  6, 1 },
        { 0,  7, 1 },
        { 0,  8, 1 },
        { 0,  9, 1 },
        { 0, 10, 1 },
        { 0, 11, 1 },
        { 0, 12, 1 },
        { 1,  5, 0 },  // proc 1
        { 1,  6, 0 },
        { 1,  7, 0 },
        { 1,  8, 0 },
        { 1,  9, 0 },
        { 1, 10, 0 },
        { 1, 11, 0 },
        { 1, 12, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1 = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity hex3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

        // Connectivity for Hex Element 3
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex3));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex3, 0));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(hex3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex3, 0));
    }
    else if (p_rank == 1) {
        // Connectivity for Hex Element 2
        EXPECT_EQ(2u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex2, 1));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex2, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));
    }

    EXPECT_EQ(2u, elemElemGraph.num_edges());
    EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, Hex0Hex1Hex2Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.1-----------15.2
    //          /|             /|             /|             /|
    //         / |            / |            / |            / |
    //        /  |           /  |           /  |           /  |
    //      4.0------------8.0-----------12.1-----------16.2  |
    //       |   |          |   |          |   |          |   |
    //       |   |   1.0    |   |   2.1    |   |   3.2    |   |
    //       |   |          |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.1---------|-14.2
    //       |  /           |  /           |  /           |  /
    //       | /            | /            | /            | /
    //       |/             |/             |/             |/
    //      1.0------------5.0------------9.1-----------13.2
    //

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 3u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart = &meta.declare_part_with_topology("hex_part", stk::topology::HEX_8);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1,  2,  3,  4,  5,  6,  7,  8 },
        { 5,  6,  7,  8,  9, 10, 11, 12 },
        { 9, 10, 11, 12, 13, 14, 15, 16 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2, 3 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 1, 2 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0,  5, 1 },  // proc 0
        { 0,  6, 1 },
        { 0,  7, 1 },
        { 0,  8, 1 },
        { 1,  5, 0 },  // proc 1
        { 1,  6, 0 },
        { 1,  7, 0 },
        { 1,  8, 0 },
        { 1,  9, 2 },
        { 1, 10, 2 },
        { 1, 11, 2 },
        { 1, 12, 2 },
        { 2,  9, 1 },  // proc 2
        { 2, 10, 1 },
        { 2, 11, 1 },
        { 2, 12, 1 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1 = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity hex3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Hex Element 2
        EXPECT_EQ(2u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex2, 1));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex2, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));
        EXPECT_EQ(2u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 2) {
        // Connectivity for Hex Element 3
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex3));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex3, 0));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(hex3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex3, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, Hex0Shell1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      4.0------------8.0  |
    //       |   |          |   |
    //       |   |   1.0    |2.1|
    //       |   |          |   |
    //       |  2.0---------|--6.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------5.0
    //                      ^
    //                      |
    //                       ---- Single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 2 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity shell2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_EQ(1,  elemElemGraph.get_owning_proc_id_of_remote_element(hex1, 2));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
    }
    else if (p_rank == 1) {
        // Connectivity for Shell Element 2
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(shell2));
        EXPECT_EQ(1,  elemElemGraph.get_side_id_to_connected_element(shell2, 0));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(shell2, 0));
        EXPECT_EQ(0,  elemElemGraph.get_owning_proc_id_of_remote_element(shell2, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell2, 0));
    }

    EXPECT_EQ(1u, elemElemGraph.num_edges());
    EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, Hex0DelShell1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      4.0------------8.0  |
    //       |   |          |   |
    //       |   |   1.0    |2.1|
    //       |   |          |   |
    //       |  2.0---------|--6.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------5.0
    //                      ^
    //                      |
    //                       ---- Deleted single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 2 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity shell2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);

    stk::mesh::impl::DeletedElementInfoVector elements_to_delete;
    if (p_rank == 1) {
        elements_to_delete.push_back({shell2, 2, mesh.bucket(shell2).topology().is_shell()});
    }

    elemElemGraph.delete_elements( elements_to_delete );

    mesh.modification_begin();
    if (p_rank == 1) {
        mesh.destroy_entity(shell2);
    }
    mesh.modification_end();

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(0u, elemElemGraph.num_edges());
        EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
        EXPECT_EQ(0u, elemElemGraph.get_num_connected_elems(hex1));
    }
    else if (p_rank == 1) {
        EXPECT_EQ(0u, elemElemGraph.num_edges());
        EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
        EXPECT_EQ(0u, elemElemGraph.size());
    }

    EXPECT_EQ(0u, elemElemGraph.num_edges());
    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, Hex0AddShell1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      4.0------------8.0  |
    //       |   |          |   |
    //       |   |   1.0    |2.1|
    //       |   |          |   |
    //       |  2.0---------|--6.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------5.0
    //                      ^
    //                      |
    //                       ---- Add a single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5, 6, 7, 8 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 2 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    EXPECT_EQ(0u, elemElemGraph.num_edges());
    EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());

    mesh.modification_begin();
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs);
    mesh.modification_end();

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity shell2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);


    stk::mesh::EntityVector elements_to_add;
    if (p_rank == 1) {
        elements_to_add.push_back(shell2);
    }
    elemElemGraph.add_elements(elements_to_add);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_EQ(1,  elemElemGraph.get_owning_proc_id_of_remote_element(hex1, 2));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
    }
    else if (p_rank == 1) {
        // Connectivity for Shell Element 2
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(shell2));
        EXPECT_EQ(1,  elemElemGraph.get_side_id_to_connected_element(shell2, 0));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(shell2, 0));
        EXPECT_EQ(0,  elemElemGraph.get_owning_proc_id_of_remote_element(shell2, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell2, 0));
    }
}


TEST( ElementGraph, Hex0AddShell0Hex1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.1
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.1  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.1    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.1
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.1
    //                      ^
    //                      |
    //                       ---- Add a Single shell element here
    //
    //      side 0 of the shell points right to element 2
    //      side 1 of the shell points left to element 1

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 0 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();


    ElemElemGraphTester elemElemGraph(mesh);

    EXPECT_EQ(1u, elemElemGraph.num_edges());
    EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());

    mesh.modification_begin();
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    mesh.modification_end();


    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    stk::mesh::EntityVector addVector;
    if (0 == p_rank) {
        addVector.push_back(shell3);
    }
    elemElemGraph.add_elements(addVector);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u,     elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 0));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

        // Connectivity for Shell Element 3
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
        EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
        EXPECT_EQ(2u,   elemElemGraph.get_entity_id_of_remote_element(shell3, 1));
        EXPECT_TRUE (elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));
        EXPECT_EQ(3u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Hex Element 2
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, Hex0AddShell1Hex2Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.2
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.2  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.1|   2.2    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.2
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.2
    //                      ^
    //                      |
    //                       ---- Add a Single shell element here

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 3u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 2 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 0, 5, 2 },  // proc 0
        { 0, 6, 2 },
        { 0, 7, 2 },
        { 0, 8, 2 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 },
        { 1, 5, 2 },
        { 1, 6, 2 },
        { 1, 7, 2 },
        { 1, 8, 2 },
        { 2, 5, 1 },  // proc 2
        { 2, 6, 1 },
        { 2, 7, 1 },
        { 2, 8, 1 },
        { 2, 5, 0 },  // proc 2
        { 2, 6, 0 },
        { 2, 7, 0 },
        { 2, 8, 0 }
    };
    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    if (1 == p_rank) {
        mesh.declare_entity(stk::topology::NODE_RANK, 5);
        mesh.declare_entity(stk::topology::NODE_RANK, 6);
        mesh.declare_entity(stk::topology::NODE_RANK, 7);
        mesh.declare_entity(stk::topology::NODE_RANK, 8);
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    if (0 == p_rank || 2 == p_rank) {
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    else {
        EXPECT_EQ(0u, elemElemGraph.num_edges());
        EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
    }

    mesh.modification_begin();
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    mesh.modification_end();

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    stk::mesh::EntityVector addVector;
    if (1 == p_rank) {
        addVector.push_back(shell3);
    }

    elemElemGraph.add_elements(addVector);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Shell Element 3
        EXPECT_EQ(2u, elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(1,  elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(0,  elemElemGraph.get_side_id_to_connected_element(shell3, 1));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(shell3, 0));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(shell3, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));
        EXPECT_EQ(2u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 2) {
        // Connectivity for Hex Element 2
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, Hex0Shell1AddHex2Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.2
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.2  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.1|   2.2    |   |  <-- Add this hex on the right
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.2
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.2
    //                      ^
    //                      |
    //                       ---- Single shell element here

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 3u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 2 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 0, 5, 2 },
        { 0, 6, 2 },
        { 0, 7, 2 },
        { 0, 8, 2 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 },
        { 1, 5, 2 },
        { 1, 6, 2 },
        { 1, 7, 2 },
        { 1, 8, 2 },
        { 2, 5, 1 },  // proc 2
        { 2, 6, 1 },
        { 2, 7, 1 },
        { 2, 8, 1 },
        { 2, 5, 0 },
        { 2, 6, 0 },
        { 2, 7, 0 },
        { 2, 8, 0 }
    };
    mesh.modification_begin();
    if (hexElemOwningProc[0] == p_rank) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[0], hexNodeIDs[0]);
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    if (2 == p_rank) {
        mesh.declare_entity(stk::topology::NODE_RANK, 5);
        mesh.declare_entity(stk::topology::NODE_RANK, 6);
        mesh.declare_entity(stk::topology::NODE_RANK, 7);
        mesh.declare_entity(stk::topology::NODE_RANK, 8);
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    if (0 == p_rank || 1 == p_rank) {
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    else {
        EXPECT_EQ(0u, elemElemGraph.num_edges());
        EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
    }

    mesh.modification_begin();
    if (hexElemOwningProc[1] == p_rank) {
        stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[1], hexNodeIDs[1]);
    }
    mesh.modification_end();

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    stk::mesh::EntityVector addVector;
    if (2 == p_rank) {
        addVector.push_back(hex2);
    }

    elemElemGraph.add_elements(addVector);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Shell Element 3
        EXPECT_EQ(2u, elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(1,  elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(0,  elemElemGraph.get_side_id_to_connected_element(shell3, 1));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(shell3, 0));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(shell3, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));
        EXPECT_EQ(2u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 2) {
        // Connectivity for Hex Element 2
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, AdjacentHex0Shell1Parallel )
{
    //  ID.proc
    //
    //         12.0-----------11.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      9.0-----------10.0  |
    //       |   |          |   |
    //       |   |   2.0    |4.1|
    //       |   |          |   |
    //       |  8.0---------|--7.0
    //       |  /|          |  /|
    //       | / |          | / |
    //       |/  |          |/  |
    //      5.0------------6.0  |
    //       |   |          |   |
    //       |   |   1.0    |3.1|
    //       |   |          |   |
    //       |  4.0---------|--3.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------2.0
    //                      ^
    //                      |
    //                       ---- Two adjacent shell elements

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 0 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 2, 3,  7,  6 },
        { 6, 7, 11, 10 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3, 4 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1, 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0,  2, 1 },  // proc 0
        { 0,  3, 1 },
        { 0,  7, 1 },
        { 0,  6, 1 },
        { 0, 11, 1 },
        { 0, 10, 1 },
        { 1,  2, 0 },  // proc 1
        { 1,  3, 0 },
        { 1,  7, 0 },
        { 1,  6, 0 },
        { 1, 11, 0 },
        { 1, 10, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);
    const Entity shell4 = mesh.get_entity(stk::topology::ELEM_RANK, 4);

    if (p_rank == 0u) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,    elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(hex1, 1));
        EXPECT_EQ(hex2, elemElemGraph.get_connected_element(hex1, 0));
        EXPECT_EQ(3u,   elemElemGraph.get_entity_id_of_remote_element(hex1, 1));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

        // Connectivity for Hex Element 2
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,    elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(hex2, 1));
        EXPECT_EQ(hex1, elemElemGraph.get_connected_element(hex2, 0));
        EXPECT_EQ(4u,   elemElemGraph.get_entity_id_of_remote_element(hex2, 1));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));
        EXPECT_EQ(4u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1u) {
        // Connectivity for Shell Element 3
        EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(1u,   elemElemGraph.get_entity_id_of_remote_element(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));

        // Connectivity for Shell Element 4
        EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(shell4));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell4, 0));
        EXPECT_EQ(2u,   elemElemGraph.get_entity_id_of_remote_element(shell4, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell4, 0));
        EXPECT_EQ(2u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, DISABLED_Hex0Shell0Shell1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      4.0------------8.0  |
    //       |   |          |   |
    //       |   |   1.0    |2.0|
    //       |   |          |3.1|
    //       |  2.0---------|--6.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------5.0
    //                      ^
    //                      |
    //                       ---- Two stacked shells

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 },
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 2, 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 0, 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity shell2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(shell2, elemElemGraph.get_connected_element(hex1, 0));
        EXPECT_EQ(3u,     elemElemGraph.get_entity_id_of_remote_element(hex1, 1));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

        // Connectivity for Shell Element 2
        EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(shell2));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell2, 0));
        EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell2, 0));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(shell2, 0));
        EXPECT_EQ(3u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Shell Element 3
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(1,  elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, DISABLED_Hex0Shell1Shell1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0
    //          /|             /|
    //         / |            / |
    //        /  |           /  |
    //      4.0------------8.0  |
    //       |   |          |   |
    //       |   |   1.0    |2.1|
    //       |   |          |3.1|
    //       |  2.0---------|--6.0
    //       |  /           |  /
    //       | /            | /
    //       |/             |/
    //      1.0------------5.0
    //                      ^
    //                      |
    //                       ---- Two stacked shells

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 },
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 2, 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1, 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity shell2 = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(2u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 1));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex1, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));
        EXPECT_EQ(2u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Shell Element 2
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(shell2));
        EXPECT_EQ(1,  elemElemGraph.get_side_id_to_connected_element(shell2, 0));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(shell2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell2, 0));

        // Connectivity for Shell Element 3
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(1,  elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_EQ(2u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, Hex0Shell0Hex1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.1
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.1  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.1    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.1
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.1
    //                      ^
    //                      |
    //                       ---- Single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 0 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u,     elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 0));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

        // Connectivity for Shell Element 3
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
        EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
        EXPECT_EQ(2u,   elemElemGraph.get_entity_id_of_remote_element(shell3, 1));
        EXPECT_TRUE (elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));
        EXPECT_EQ(3u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Hex Element 2
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
}

void test_chosen_face_ids(ElemElemGraphTester & elemElemGraph, stk::mesh::BulkData & mesh)
{
    stk::mesh::impl::ParallelGraphInfo p_graph_info = elemElemGraph.get_parallel_graph_info();
    std::vector<stk::mesh::EntityId> chosen_ids;
    stk::mesh::impl::ParallelGraphInfo::iterator pgraph_iter = p_graph_info.begin();
    for(;pgraph_iter!=p_graph_info.end();++pgraph_iter)
    {
        chosen_ids.push_back(pgraph_iter->second.m_chosen_side_id);
    }
    std::sort(chosen_ids.begin(), chosen_ids.end());
    auto unique_iter = std::unique(chosen_ids.begin(), chosen_ids.end());
    bool ids_are_unique = unique_iter == chosen_ids.end();
    EXPECT_TRUE(ids_are_unique);

    test_similarity_on_both_procs(chosen_ids, mesh.parallel());
}

TEST( ElementGraph, Hex0DelShell0Hex1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.1
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.1  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.1    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.1
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.1
    //                      ^
    //                      |
    //                       ---- Deleted single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 0 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    stk::mesh::impl::DeletedElementInfoVector elements_to_delete;
    if (p_rank == 0) {
        elements_to_delete.push_back({shell3, 3, mesh.bucket(shell3).topology().is_shell()});
    }

    mesh.modification_begin();
    if (p_rank == 0) {
        mesh.destroy_entity(shell3);
    }
    mesh.modification_end();

    elemElemGraph.delete_elements( elements_to_delete );

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_EQ(1,  elemElemGraph.get_owning_proc_id_of_remote_element(hex1, 2));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
    }
    else if (p_rank == 1) {
        // Connectivity for Hex Element 2
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_EQ(0,  elemElemGraph.get_owning_proc_id_of_remote_element(hex2, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
    }
    test_chosen_face_ids(elemElemGraph, mesh);
}

TEST( ElementGraph, Hex0DelShell1Hex2Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.2
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.2  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.1|   2.2    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.2
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.2
    //                      ^
    //                      |
    //                       ---- Delete this single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 3u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 2 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 0, 5, 2 },
        { 0, 6, 2 },
        { 0, 7, 2 },
        { 0, 8, 2 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 },
        { 1, 5, 2 },
        { 1, 6, 2 },
        { 1, 7, 2 },
        { 1, 8, 2 },
        { 2, 5, 0 },  // proc 2
        { 2, 6, 0 },
        { 2, 7, 0 },
        { 2, 8, 0 },
        { 2, 5, 1 },
        { 2, 6, 1 },
        { 2, 7, 1 },
        { 2, 8, 1 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    stk::mesh::impl::DeletedElementInfoVector elements_to_delete;
    if (p_rank == 1) {
        elements_to_delete.push_back({shell3, 3, mesh.bucket(shell3).topology().is_shell()});
    }

    elemElemGraph.delete_elements( elements_to_delete );

    mesh.modification_begin();
    if (p_rank == 1) {
        mesh.destroy_entity(shell3);
    }
    mesh.modification_end();

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_EQ(2,  elemElemGraph.get_owning_proc_id_of_remote_element(hex1, 2));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Shell Element 3
        EXPECT_EQ(0u, elemElemGraph.num_edges());
        EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
        EXPECT_EQ(0u, elemElemGraph.size());
        EXPECT_EQ(0u, elemElemGraph.num_edges());
        EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 2) {
        // Connectivity for Hex Element 2
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_EQ(0,  elemElemGraph.get_owning_proc_id_of_remote_element(hex2, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    test_chosen_face_ids(elemElemGraph, mesh);
}

TEST( ElementGraph, Hex0Shell1Hex2Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.2
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.2  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.1|   2.2    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.2
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.2
    //                      ^
    //                      |
    //                       ---- Single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 3u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 2 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 0, 5, 2 },
        { 0, 6, 2 },
        { 0, 7, 2 },
        { 0, 8, 2 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 },
        { 1, 5, 2 },
        { 1, 6, 2 },
        { 1, 7, 2 },
        { 1, 8, 2 },
        { 2, 5, 0 },  // proc 2
        { 2, 6, 0 },
        { 2, 7, 0 },
        { 2, 8, 0 },
        { 2, 5, 1 },
        { 2, 6, 1 },
        { 2, 7, 1 },
        { 2, 8, 1 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Shell Element 3
        EXPECT_EQ(2u, elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(0,  elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(1,  elemElemGraph.get_side_id_to_connected_element(shell3, 1));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(shell3, 0));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(shell3, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));
        EXPECT_EQ(2u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 2) {
        // Connectivity for Hex Element 2
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_EQ(1u, elemElemGraph.num_edges());
        EXPECT_EQ(1u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, Hex0Shell1Hex0Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.0  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.1|   2.0    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.0
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.0
    //                      ^
    //                      |
    //                       ---- Single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 0 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

        // Connectivity for Hex Element 2
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
    }
    else if (p_rank == 1) {
        // Connectivity for Shell Element 3
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
        EXPECT_EQ(2u,   elemElemGraph.get_entity_id_of_remote_element(shell3, 0));
        EXPECT_EQ(1u,   elemElemGraph.get_entity_id_of_remote_element(shell3, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));
    }

    EXPECT_EQ(2u, elemElemGraph.num_edges());
    EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, Hex0AddShell1Hex0Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.0  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.1|   2.0    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.0
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.0
    //                      ^
    //                      |
    //                       ---- Single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 0 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    if (1 == p_rank) {
        mesh.declare_entity(stk::topology::NODE_RANK, 5);
        mesh.declare_entity(stk::topology::NODE_RANK, 6);
        mesh.declare_entity(stk::topology::NODE_RANK, 7);
        mesh.declare_entity(stk::topology::NODE_RANK, 8);
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    mesh.modification_begin();
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    mesh.modification_end();

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    stk::mesh::EntityVector addVector;
    if (1 == p_rank) {
        addVector.push_back(shell3);
    }
    elemElemGraph.add_elements(addVector);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

        // Connectivity for Hex Element 2
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
    }
    else if (p_rank == 1) {
        // Connectivity for Shell Element 3
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
        EXPECT_EQ(1u,   elemElemGraph.get_entity_id_of_remote_element(shell3, 0));
        EXPECT_EQ(2u,   elemElemGraph.get_entity_id_of_remote_element(shell3, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));
    }

    EXPECT_EQ(2u, elemElemGraph.num_edges());
    EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, Hex0DelShell1Hex0Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.0
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.0  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.1|   2.0    |   |
    //       |   |          |   |          |   |
    //       |  2.0---------|--6.0---------|-10.0
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.0
    //                      ^
    //                      |
    //                       ---- Deleted single shell element

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 0 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3 };
    stk::mesh::EntityId shellElemOwningProc[] = { 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    EXPECT_EQ(2u, elemElemGraph.num_edges());
    EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);

    stk::mesh::impl::DeletedElementInfoVector elements_to_delete;
    if (p_rank == 1) {
        elements_to_delete.push_back({shell3, 3, mesh.bucket(shell3).topology().is_shell()});
    }

    elemElemGraph.delete_elements( elements_to_delete );

    mesh.modification_begin();
    if (p_rank == 1) {
        mesh.destroy_entity(shell3);
    }
    mesh.modification_end();

    if (p_rank == 0) {
        EXPECT_EQ(2u, elemElemGraph.num_edges());
        EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());

        // Connectivity for Hex Element 1
        EXPECT_EQ(1u,   elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,    elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(hex2, elemElemGraph.get_connected_element(hex1, 0));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));

        // Connectivity for Hex Element 2
        EXPECT_EQ(1u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(hex1, elemElemGraph.get_connected_element(hex2, 0));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
    }
    else if (p_rank == 1) {
        EXPECT_EQ(0u, elemElemGraph.size());
        EXPECT_EQ(0u, elemElemGraph.num_edges());
        EXPECT_EQ(0u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, DISABLED_Hex0Shell0Shell0Hex1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.1
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.1  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.1    |   |
    //       |   |          |4.0|          |   |
    //       |  2.0---------|--6.0---------|-10.1
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.1
    //                      ^
    //                      |
    //                       ---- Two stacked shells

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 },
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3, 4 };
    stk::mesh::EntityId shellElemOwningProc[] = { 0, 0 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);
    const Entity shell4 = mesh.get_entity(stk::topology::ELEM_RANK, 4);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 1));
        EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 0));
        EXPECT_EQ(shell4, elemElemGraph.get_connected_element(hex1, 1));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

        // Connectivity for Shell Element 3
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
        EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
        EXPECT_EQ(2u,   elemElemGraph.get_entity_id_of_remote_element(shell3, 1));
        EXPECT_TRUE (elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));

        // Connectivity for Shell Element 4
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell4));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell4, 0));
        EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell4, 1));
        EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell4, 0));
        EXPECT_EQ(2u,   elemElemGraph.get_entity_id_of_remote_element(shell4, 1));
        EXPECT_TRUE (elemElemGraph.is_connected_elem_locally_owned(shell4, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell4, 1));

        EXPECT_EQ(6u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Hex Element 2
        EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 1));
        EXPECT_EQ(3u,   elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_EQ(4u,   elemElemGraph.get_entity_id_of_remote_element(hex2, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));

        EXPECT_EQ(2u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, DISABLED_Hex0Shell0Shell1Hex1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.1
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.1  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.1    |   |
    //       |   |          |4.1|          |   |
    //       |  2.0---------|--6.0---------|-10.1
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.1
    //                      ^
    //                      |
    //                       ---- Two stacked shells

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 },
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3, 4 };
    stk::mesh::EntityId shellElemOwningProc[] = { 0, 1 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);
    const Entity shell4 = mesh.get_entity(stk::topology::ELEM_RANK, 4);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 1));
        EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 0));
        EXPECT_EQ(4u,     elemElemGraph.get_entity_id_of_remote_element(hex1, 1));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

        // Connectivity for Shell Element 3
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
        EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
        EXPECT_EQ(2u,   elemElemGraph.get_entity_id_of_remote_element(shell3, 1));
        EXPECT_TRUE (elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));
    }
    else if (p_rank == 1) {
        // Connectivity for Shell Element 4
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell4));
        EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell4, 0));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell4, 1));
        EXPECT_EQ(hex2, elemElemGraph.get_connected_element(shell4, 0));
        EXPECT_EQ(1u,   elemElemGraph.get_entity_id_of_remote_element(shell4, 1));
        EXPECT_TRUE (elemElemGraph.is_connected_elem_locally_owned(shell4, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell4, 1));

        // Connectivity for Hex Element 2
        EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(4,      elemElemGraph.get_side_id_to_connected_element(hex2, 1));
        EXPECT_EQ(shell4, elemElemGraph.get_connected_element(hex2, 0));
        EXPECT_EQ(3u,     elemElemGraph.get_entity_id_of_remote_element(hex2, 1));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));
    }

    EXPECT_EQ(4u, elemElemGraph.num_edges());
    EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
}

TEST( ElementGraph, DISABLED_Hex0Shell0ReversedShell0Hex1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.0-----------11.1
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.0-----------12.1  |
    //       |   |          |   |          |   |
    //       |   |   1.0    |3.0|   2.1    |   |
    //       |   |          |4.0|          |   |
    //       |  2.0---------|--6.0---------|-10.1
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.0------------9.1
    //                      ^
    //                      |
    //                       ---- Two stacked shells, opposite orientation

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 0, 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 },
        { 5, 8, 7, 6 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3, 4 };
    stk::mesh::EntityId shellElemOwningProc[] = { 0, 0 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);
    const Entity shell4 = mesh.get_entity(stk::topology::ELEM_RANK, 4);

    if (p_rank == 0) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(2u,     elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(5,      elemElemGraph.get_side_id_to_connected_element(hex1, 1));
        EXPECT_EQ(shell3, elemElemGraph.get_connected_element(hex1, 0));
        EXPECT_EQ(shell4, elemElemGraph.get_connected_element(hex1, 1));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_TRUE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

        // Connectivity for Shell Element 3
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell3, 1));
        EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell3, 0));
        EXPECT_EQ(2u,   elemElemGraph.get_entity_id_of_remote_element(shell3, 1));
        EXPECT_TRUE (elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));

        // Connectivity for Shell Element 4
        EXPECT_EQ(2u,   elemElemGraph.get_num_connected_elems(shell4));
        EXPECT_EQ(0,    elemElemGraph.get_side_id_to_connected_element(shell4, 0));
        EXPECT_EQ(1,    elemElemGraph.get_side_id_to_connected_element(shell4, 1));
        EXPECT_EQ(hex1, elemElemGraph.get_connected_element(shell4, 0));
        EXPECT_EQ(2u,   elemElemGraph.get_entity_id_of_remote_element(shell4, 1));
        EXPECT_TRUE (elemElemGraph.is_connected_elem_locally_owned(shell4, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell4, 1));
        EXPECT_EQ(6u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
    else if (p_rank == 1) {
        // Connectivity for Hex Element 2
        EXPECT_EQ(2u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 1));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_EQ(4u, elemElemGraph.get_entity_id_of_remote_element(hex2, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));
        EXPECT_EQ(2u, elemElemGraph.num_edges());
        EXPECT_EQ(2u, elemElemGraph.num_parallel_edges());
    }
}

TEST( ElementGraph, DISABLED_Hex1Shell0Shell0Hex1Parallel )
{
    //  ID.proc
    //
    //          3.0------------7.1-----------11.1
    //          /|             /|             /|
    //         / |            / |            / |
    //        /  |           /  |           /  |
    //      4.0------------8.1-----------12.1  |
    //       |   |          |   |          |   |
    //       |   |   1.1    |3.0|   2.1    |   |
    //       |   |          |4.0|          |   |
    //       |  2.0---------|--6.1---------|-10.1
    //       |  /           |  /           |  /
    //       | /            | /            | /
    //       |/             |/             |/
    //      1.0------------5.1------------9.1
    //                      ^
    //                      |
    //                       ---- Two stacked shells

    stk::ParallelMachine pm = MPI_COMM_WORLD;
    unsigned p_size = stk::parallel_machine_size(pm);
    unsigned p_rank = stk::parallel_machine_rank(pm);

    if(p_size != 2u)
    {
        return;
    }

    const unsigned spatialDim = 3;
    stk::mesh::MetaData meta(spatialDim);
    stk::mesh::BulkData mesh(meta, pm);

    stk::mesh::Part * hexPart   = &meta.declare_part_with_topology("hex_part",   stk::topology::HEX_8);
    stk::mesh::Part * shellPart = &meta.declare_part_with_topology("shell_part", stk::topology::SHELL_QUAD_4);
    meta.commit();

    std::vector<stk::mesh::EntityIdVector> hexNodeIDs {
        { 1, 2, 3, 4, 5,  6,  7,  8 },
        { 5, 6, 7, 8, 9, 10, 11, 12 }
    };
    stk::mesh::EntityId hexElemIDs[] = { 1, 2 };
    stk::mesh::EntityId hexElemOwningProc[] = { 1, 1 };

    std::vector<stk::mesh::EntityIdVector> shellNodeIDs {
        { 5, 6, 7, 8 },
        { 5, 6, 7, 8 }
    };
    stk::mesh::EntityId shellElemIDs[] = { 3, 4 };
    stk::mesh::EntityId shellElemOwningProc[] = { 0, 0 };

    // list of triplets: (owner-proc, shared-nodeID, sharing-proc)
    std::vector< std::vector<unsigned> > shared_nodeIDs_and_procs
    {
        { 0, 5, 1 },  // proc 0
        { 0, 6, 1 },
        { 0, 7, 1 },
        { 0, 8, 1 },
        { 1, 5, 0 },  // proc 1
        { 1, 6, 0 },
        { 1, 7, 0 },
        { 1, 8, 0 }
    };

    mesh.modification_begin();
    for (size_t i = 0; i < hexNodeIDs.size(); ++i) {
        if (hexElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *hexPart, hexElemIDs[i], hexNodeIDs[i]);
        }
    }
    for (size_t i = 0; i < shellNodeIDs.size(); ++i) {
        if (shellElemOwningProc[i] == p_rank) {
            stk::mesh::declare_element(mesh, *shellPart, shellElemIDs[i], shellNodeIDs[i]);
        }
    }
    setup_node_sharing(mesh, shared_nodeIDs_and_procs );
    mesh.modification_end();

    ElemElemGraphTester elemElemGraph(mesh);

    const Entity hex1   = mesh.get_entity(stk::topology::ELEM_RANK, 1);
    const Entity hex2   = mesh.get_entity(stk::topology::ELEM_RANK, 2);
    const Entity shell3 = mesh.get_entity(stk::topology::ELEM_RANK, 3);
    const Entity shell4 = mesh.get_entity(stk::topology::ELEM_RANK, 4);

    if (p_rank == 0) {
        // Connectivity for Shell Element 3
        EXPECT_EQ(2u, elemElemGraph.get_num_connected_elems(shell3));
        EXPECT_EQ(0,  elemElemGraph.get_side_id_to_connected_element(shell3, 0));
        EXPECT_EQ(1,  elemElemGraph.get_side_id_to_connected_element(shell3, 1));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(shell3, 0));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(shell3, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell3, 1));

        // Connectivity for Shell Element 4
        EXPECT_EQ(2u, elemElemGraph.get_num_connected_elems(shell4));
        EXPECT_EQ(0,  elemElemGraph.get_side_id_to_connected_element(shell4, 0));
        EXPECT_EQ(1,  elemElemGraph.get_side_id_to_connected_element(shell4, 1));
        EXPECT_EQ(2u, elemElemGraph.get_entity_id_of_remote_element(shell4, 0));
        EXPECT_EQ(1u, elemElemGraph.get_entity_id_of_remote_element(shell4, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell4, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(shell4, 1));
    }
    else if (p_rank == 1) {
        // Connectivity for Hex Element 1
        EXPECT_EQ(2u, elemElemGraph.get_num_connected_elems(hex1));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 0));
        EXPECT_EQ(5,  elemElemGraph.get_side_id_to_connected_element(hex1, 1));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex1, 0));
        EXPECT_EQ(4u, elemElemGraph.get_entity_id_of_remote_element(hex1, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex1, 1));

        // Connectivity for Hex Element 2
        EXPECT_EQ(2u, elemElemGraph.get_num_connected_elems(hex2));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 0));
        EXPECT_EQ(4,  elemElemGraph.get_side_id_to_connected_element(hex2, 1));
        EXPECT_EQ(3u, elemElemGraph.get_entity_id_of_remote_element(hex2, 0));
        EXPECT_EQ(4u, elemElemGraph.get_entity_id_of_remote_element(hex2, 1));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 0));
        EXPECT_FALSE(elemElemGraph.is_connected_elem_locally_owned(hex2, 1));
    }

    EXPECT_EQ(4u, elemElemGraph.num_edges());
    EXPECT_EQ(4u, elemElemGraph.num_parallel_edges());
}

void test_add_element_to_graph_with_element_death(stk::mesh::BulkData::AutomaticAuraOption autoAuraOption)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;
    int pSize = stk::parallel_machine_size(comm);
    int pRank = stk::parallel_machine_rank(comm);

    if(2 == pSize)
    {
        stk::mesh::MetaData meta(3);

        stk::mesh::Part& faces_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
        stk::mesh::PartVector boundary_mesh_parts {&faces_part};
        stk::io::put_io_part_attribute(faces_part);

        stk::mesh::Part& active = meta.declare_part("active"); // can't specify rank, because it gets checked against size of rank_names

        stk::mesh::BulkData bulkData(meta, comm, autoAuraOption);

        ElemElemGraphTester graph = test_add_elements_to_pre_existing_graph_and_mesh(bulkData);

        stk::unit_test_util::put_mesh_into_part(bulkData, active);

        stk::mesh::EntityVector deactivated_elems;

        if (0 == pRank)
        {
            deactivated_elems.push_back(bulkData.get_entity(stk::topology::ELEM_RANK, 1));
        }
        else
        {
            deactivated_elems.push_back(bulkData.get_entity(stk::topology::ELEM_RANK, 4));
        }

        std::vector<size_t> entity_counts;
        stk::mesh::comm_mesh_counts(bulkData, entity_counts);
        ASSERT_TRUE(entity_counts[stk::topology::FACE_RANK] == 0);

        boundary_mesh_parts.push_back(&active);

        ElementDeathUtils::deactivate_elements(deactivated_elems, bulkData,  active);

        stk::mesh::process_killed_elements(bulkData, graph, deactivated_elems, active, boundary_mesh_parts, &boundary_mesh_parts);

        if (0 == pRank)
        {
            stk::mesh::EntityId elem1Id = 1;
            stk::mesh::EntityId elem2Id = 2;
            stk::mesh::Entity face_between_elem1_and_elem2 = ElementDeathUtils::get_face_between_element_ids(graph, bulkData, elem1Id, elem2Id);

            ASSERT_TRUE(bulkData.is_valid(face_between_elem1_and_elem2));
            EXPECT_TRUE(bulkData.bucket(face_between_elem1_and_elem2).member(active));

            stk::mesh::Entity elem1 = bulkData.get_entity(stk::topology::ELEM_RANK, 1);
            stk::mesh::Entity elem3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
            EXPECT_FALSE(bulkData.bucket(elem1).member(active));
            EXPECT_TRUE(bulkData.bucket(elem3).member(active));
        }
        else
        {
            stk::mesh::EntityId elem3Id = 3;
            stk::mesh::EntityId elem4Id = 4;
            stk::mesh::Entity face_between_elem3_and_elem4 = ElementDeathUtils::get_face_between_element_ids(graph, bulkData, elem3Id, elem4Id);

            ASSERT_TRUE(bulkData.is_valid(face_between_elem3_and_elem4));
            EXPECT_TRUE(bulkData.bucket(face_between_elem3_and_elem4).member(active));

            stk::mesh::Entity elem2 = bulkData.get_entity(stk::topology::ELEM_RANK, 2);
            stk::mesh::Entity elem4 = bulkData.get_entity(stk::topology::ELEM_RANK, 4);
            EXPECT_TRUE(bulkData.bucket(elem2).member(active));
            EXPECT_FALSE(bulkData.bucket(elem4).member(active));
        }

        // bulkData.dump_all_mesh_info(std::cout, true);
        stk::mesh::comm_mesh_counts(bulkData, entity_counts);
        ASSERT_TRUE(entity_counts[stk::topology::FACE_RANK] == 2);

        EXPECT_EQ(4u, graph.num_edges());
        EXPECT_EQ(4u, graph.num_parallel_edges());
    }
}

TEST(ElementGraph, add_element_to_graph_with_element_death_aura_on)
{
    test_add_element_to_graph_with_element_death(stk::mesh::BulkData::AUTO_AURA);
}

TEST(ElementGraph, add_element_to_graph_with_element_death_aura_off)
{
    test_add_element_to_graph_with_element_death(stk::mesh::BulkData::NO_AUTO_AURA);
}

void test_delete_element_from_graph_with_element_death(stk::mesh::BulkData::AutomaticAuraOption autoAuraOption)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;
    int pSize = stk::parallel_machine_size(comm);
    int pRank = stk::parallel_machine_rank(comm);

    if(3 == pSize)
    {
        const int procRank = stk::parallel_machine_rank(comm);
        unsigned spatialDim = 3;

        stk::mesh::MetaData meta(spatialDim);

        stk::mesh::Part& faces_part = meta.declare_part_with_topology("surface_5", stk::topology::QUAD_4);
        stk::mesh::PartVector boundary_mesh_parts {&faces_part};
        stk::io::put_io_part_attribute(faces_part);

        stk::mesh::Part& active = meta.declare_part("active"); // can't specify rank, because it gets checked against size of rank_names

        stk::mesh::BulkData bulkData(meta, comm, autoAuraOption);
        stk::unit_test_util::fill_mesh_using_stk_io("generated:3x1x3", bulkData, comm);

        bulkData.modification_begin();
        if (procRank == 0) {
            stk::mesh::Entity element1 = bulkData.get_entity(stk::topology::ELEM_RANK,1);
            stk::mesh::Entity element3 = bulkData.get_entity(stk::topology::ELEM_RANK,3);
            bulkData.destroy_entity(element1);
            bulkData.destroy_entity(element3);
        }
        if (procRank == 2) {
            stk::mesh::Entity element7 = bulkData.get_entity(stk::topology::ELEM_RANK,7);
            stk::mesh::Entity element9 = bulkData.get_entity(stk::topology::ELEM_RANK,9);
            bulkData.destroy_entity(element7);
            bulkData.destroy_entity(element9);
        }
        bulkData.modification_end();

        ElemElemGraphTester graph(bulkData);

        stk::unit_test_util::put_mesh_into_part(bulkData, active);

        stk::mesh::EntityVector deactivated_elems;

        if (1 == pRank)
        {
            deactivated_elems.push_back(bulkData.get_entity(stk::topology::ELEM_RANK, 5));
        }

        std::vector<size_t> entity_counts;
        stk::mesh::comm_mesh_counts(bulkData, entity_counts);
        ASSERT_TRUE(entity_counts[stk::topology::FACE_RANK] == 0);

        boundary_mesh_parts.push_back(&active);

        ElementDeathUtils::deactivate_elements(deactivated_elems, bulkData,  active);

        stk::mesh::process_killed_elements(bulkData, graph, deactivated_elems, active, boundary_mesh_parts, &boundary_mesh_parts);

        stk::mesh::comm_mesh_counts(bulkData, entity_counts);

        ASSERT_EQ(4u, entity_counts[stk::topology::FACE_RANK]);

        if (0 == pRank)
        {
            stk::mesh::EntityId elem2Id = 2;
            stk::mesh::EntityId elem5Id = 5;
            stk::mesh::Entity face_between_elem2_and_elem5 = ElementDeathUtils::get_face_between_element_ids(graph, bulkData, elem2Id, elem5Id);

            ASSERT_TRUE(bulkData.is_valid(face_between_elem2_and_elem5));
            EXPECT_TRUE(bulkData.bucket(face_between_elem2_and_elem5).member(active));

            stk::mesh::Entity elem2 = bulkData.get_entity(stk::topology::ELEM_RANK, 2);
            EXPECT_TRUE(bulkData.bucket(elem2).member(active));

            EXPECT_EQ(1u, graph.num_edges());
            EXPECT_EQ(1u, graph.num_parallel_edges());
        }
        else if(1 == pRank)
        {
            stk::mesh::EntityId elem4Id = 4;
            stk::mesh::EntityId elem5Id = 5;
            stk::mesh::EntityId elem6Id = 6;
            stk::mesh::Entity face_between_elem4_and_elem5 = ElementDeathUtils::get_face_between_element_ids(graph, bulkData, elem4Id, elem5Id);
            stk::mesh::Entity face_between_elem6_and_elem5 = ElementDeathUtils::get_face_between_element_ids(graph, bulkData, elem6Id, elem5Id);

            ASSERT_TRUE(bulkData.is_valid(face_between_elem4_and_elem5));
            EXPECT_TRUE(bulkData.bucket(face_between_elem4_and_elem5).member(active));

            ASSERT_TRUE(bulkData.is_valid(face_between_elem6_and_elem5));
            EXPECT_TRUE(bulkData.bucket(face_between_elem6_and_elem5).member(active));

            stk::mesh::Entity elem4 = bulkData.get_entity(stk::topology::ELEM_RANK, 4);
            stk::mesh::Entity elem5 = bulkData.get_entity(stk::topology::ELEM_RANK, 5);
            stk::mesh::Entity elem6 = bulkData.get_entity(stk::topology::ELEM_RANK, 6);

            EXPECT_TRUE(bulkData.bucket(elem4).member(active));
            EXPECT_TRUE(bulkData.bucket(elem6).member(active));
            EXPECT_FALSE(bulkData.bucket(elem5).member(active));

            EXPECT_EQ(6u, graph.num_edges());
            EXPECT_EQ(2u, graph.num_parallel_edges());
        }
        else if (2 == pRank)
        {
            stk::mesh::EntityId elem8Id = 8;
            stk::mesh::EntityId elem5Id = 5;
            stk::mesh::Entity face_between_elem8_and_elem5 = ElementDeathUtils::get_face_between_element_ids(graph, bulkData, elem8Id, elem5Id);

            ASSERT_TRUE(bulkData.is_valid(face_between_elem8_and_elem5));
            EXPECT_TRUE(bulkData.bucket(face_between_elem8_and_elem5).member(active));

            stk::mesh::Entity elem8 = bulkData.get_entity(stk::topology::ELEM_RANK, 8);
            EXPECT_TRUE(bulkData.bucket(elem8).member(active));

            EXPECT_EQ(1u, graph.num_edges());
            EXPECT_EQ(1u, graph.num_parallel_edges());
        }
    }
}


TEST(ElementGraph, delete_element_from_graph_with_element_death_aura_on)
{
    test_delete_element_from_graph_with_element_death(stk::mesh::BulkData::AUTO_AURA);
}

TEST(ElementGraph, delete_element_from_graph_with_element_death_aura_off)
{
    test_delete_element_from_graph_with_element_death(stk::mesh::BulkData::NO_AUTO_AURA);
}

ElemElemGraphTester create_base_1x1x4_elem_graph(stk::ParallelMachine &comm, stk::mesh::BulkData &bulkData)
{
    unsigned nProc  = stk::parallel_machine_size(comm);
    unsigned myProc = stk::parallel_machine_rank(comm);

    ThrowRequire(nProc <= 4);

    stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

    std::vector<unsigned> counts;
    stk::mesh::count_entities(bulkData.mesh_meta_data().locally_owned_part(), bulkData, counts);
    unsigned numLocallyOwnedElems = counts[stk::topology::ELEM_RANK];

    if(myProc != 0)
    {
        EXPECT_EQ(4u/nProc, numLocallyOwnedElems);
    }
    else
    {
        EXPECT_EQ((4u/nProc + 4u%nProc), numLocallyOwnedElems);
    }

    ElemElemGraphTester elem_graph(bulkData);

    stk::mesh::Entity elem1 = bulkData.get_entity(stk::topology::ELEM_RANK, 1);
    stk::mesh::Entity elem2 = bulkData.get_entity(stk::topology::ELEM_RANK, 2);
    stk::mesh::Entity elem3 = bulkData.get_entity(stk::topology::ELEM_RANK, 3);
    stk::mesh::Entity elem4 = bulkData.get_entity(stk::topology::ELEM_RANK, 4);

    // We know that element 1 has 1 connection
    //                      2 has 2 connections
    //                      3 has 2 connections
    //                      4 has 1 connection

    bool ownedElem1 = false;
    bool ownedElem2 = false;
    bool ownedElem3 = false;
    bool ownedElem4 = false;

    unsigned numEdges = 0;

    if(bulkData.is_valid(elem1) && bulkData.bucket(elem1).owned())
    {
        numEdges += 1;
        ownedElem1 = true;
    }
    if(bulkData.is_valid(elem2) && bulkData.bucket(elem2).owned())
    {
        numEdges += 2;
        ownedElem2 = true;
    }

    if(bulkData.is_valid(elem3) && bulkData.bucket(elem3).owned())
    {
        numEdges += 2;
        ownedElem3 = true;
    }
    if(bulkData.is_valid(elem4) && bulkData.bucket(elem4).owned())
    {
        numEdges += 1;
        ownedElem4 = true;
    }

    unsigned numParallelEdges = 0;
    for(unsigned elem_id=1; elem_id<=4; ++elem_id)
    {
        stk::mesh::Entity elem = bulkData.get_entity(stk::topology::ELEM_RANK, elem_id);

        if (!bulkData.is_valid(elem) || !bulkData.bucket(elem).owned())
        {
            continue;
        }

        stk::mesh::EntityId leftNeighbor = elem_id - 1;
        stk::mesh::Entity leftElem = bulkData.get_entity(stk::topology::ELEM_RANK, leftNeighbor);
        bool ownedLeftNeighbor = bulkData.is_valid(leftElem) && bulkData.bucket(leftElem).owned();
        if(!ownedLeftNeighbor && (elem_id > 1))
        {
            numParallelEdges++;
        }

        stk::mesh::EntityId rightNeighbor = elem_id + 1;
        stk::mesh::Entity rightElem = bulkData.get_entity(stk::topology::ELEM_RANK, rightNeighbor);
        bool ownedRightNeighbor = bulkData.is_valid(rightElem) && bulkData.bucket(rightElem).owned();
        if(!ownedRightNeighbor && (elem_id < 4))
        {
            numParallelEdges++;
        }
    }

    EXPECT_EQ(numLocallyOwnedElems, elem_graph.size());
    EXPECT_EQ(numEdges, elem_graph.num_edges());
    EXPECT_EQ(numParallelEdges, elem_graph.num_parallel_edges());

    if (ownedElem1)
    {
        EXPECT_EQ(5, elem_graph.check_connectivity(1, 2));
        EXPECT_EQ(-1, elem_graph.check_connectivity(1, 3));
        EXPECT_EQ(-1, elem_graph.check_connectivity(1, 4));

    }
    if (ownedElem2)
    {
        EXPECT_EQ(4, elem_graph.check_connectivity(2, 1));
        EXPECT_EQ(5, elem_graph.check_connectivity(2, 3));
        EXPECT_EQ(-1, elem_graph.check_connectivity(2, 4));
    }
    if (ownedElem3)
    {
        EXPECT_EQ(4, elem_graph.check_connectivity(3, 2));
        EXPECT_EQ(5, elem_graph.check_connectivity(3, 4));
        EXPECT_EQ(-1, elem_graph.check_connectivity(3, 1));
    }
    if (ownedElem4)
    {
        EXPECT_EQ(4, elem_graph.check_connectivity(4, 3));
        EXPECT_EQ(-1, elem_graph.check_connectivity(4, 1));
        EXPECT_EQ(-1, elem_graph.check_connectivity(4, 2));
    }

    return elem_graph;
}

// element ids / proc_id:
// |-------|-------|-------|
// |       |       |       |
// |  1/0  |  4/2  |  7/2  |
// |       |       |       |
// |-------|-------|-------|
// |       |       |       |
// |  2/0  |  5/1  |  8/2  |
// |       |       |       |
// |-------|-------|-------|
// |       |       |       |
// |  3/0  |  6/2  |  9/2  |
// |       |       |       |
// |-------|-------|-------|


TEST(ElementGraph, TestKeyHoleSimilarProblemAInParallel)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    if(stk::parallel_machine_size(comm) == 3)
    {
        const int procRank = stk::parallel_machine_rank(comm);
        unsigned spatialDim = 3;

        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::BulkData bulkData(meta, comm, stk::mesh::BulkData::NO_AUTO_AURA);
        stk::unit_test_util::fill_mesh_using_stk_io("generated:3x1x3", bulkData, comm);

        stk::mesh::EntityProcVec elementProcChanges;
        if (procRank == 1) {
            elementProcChanges.push_back(stk::mesh::EntityProc(bulkData.get_entity(stk::topology::ELEM_RANK,4),2));
            elementProcChanges.push_back(stk::mesh::EntityProc(bulkData.get_entity(stk::topology::ELEM_RANK,6),2));
        }
        bulkData.change_entity_owner(elementProcChanges);

        ElemElemGraphTester graph(bulkData);
        if (procRank == 0) {
            stk::mesh::Entity local_element = bulkData.get_entity(stk::topology::ELEM_RANK,2);
            ASSERT_TRUE(bulkData.bucket(local_element).owned());
            ASSERT_EQ(3u, graph.get_num_connected_elems(local_element));

            EXPECT_EQ( 3, graph.get_side_id_to_connected_element(local_element,0));
            EXPECT_EQ( 1, graph.get_side_id_to_connected_element(local_element,1));
            EXPECT_EQ( 5, graph.get_side_id_to_connected_element(local_element,2));

            EXPECT_TRUE(graph.is_connected_elem_locally_owned(local_element, 0));
            EXPECT_TRUE(graph.is_connected_elem_locally_owned(local_element, 1));
            EXPECT_FALSE(graph.is_connected_elem_locally_owned(local_element, 2));

            EXPECT_EQ( 1u, bulkData.identifier(graph.get_connected_element(local_element, 0)));
            EXPECT_EQ( 3u, bulkData.identifier(graph.get_connected_element(local_element, 1)));
            EXPECT_EQ( 5u, graph.get_entity_id_of_remote_element(local_element, 2));

            EXPECT_EQ( 1, graph.get_owning_proc_id_of_remote_element(local_element, 5));

            EXPECT_EQ(7u, graph.num_edges());
            EXPECT_EQ(3u, graph.num_parallel_edges());
        }
        if (procRank == 1) {
            stk::mesh::Entity local_element = bulkData.get_entity(stk::topology::ELEM_RANK,5);
            ASSERT_TRUE(bulkData.bucket(local_element).owned());
            size_t numConnectedElems = graph.get_num_connected_elems(local_element);
            ASSERT_EQ(4u, numConnectedElems);

            EXPECT_EQ( 4, graph.get_side_id_to_connected_element(local_element,0));
            EXPECT_EQ( 3, graph.get_side_id_to_connected_element(local_element,1));
            EXPECT_EQ( 1, graph.get_side_id_to_connected_element(local_element,2));
            EXPECT_EQ( 5, graph.get_side_id_to_connected_element(local_element,3));

            EXPECT_FALSE(graph.is_connected_elem_locally_owned(local_element, 0));
            EXPECT_FALSE(graph.is_connected_elem_locally_owned(local_element, 1));
            EXPECT_FALSE(graph.is_connected_elem_locally_owned(local_element, 2));
            EXPECT_FALSE(graph.is_connected_elem_locally_owned(local_element, 3));

            EXPECT_EQ( 2u, graph.get_entity_id_of_remote_element(local_element, 0));
            EXPECT_EQ( 4u, graph.get_entity_id_of_remote_element(local_element, 1));
            EXPECT_EQ( 6u, graph.get_entity_id_of_remote_element(local_element, 2));
            EXPECT_EQ( 8u, graph.get_entity_id_of_remote_element(local_element, 3));

            EXPECT_EQ( 0, graph.get_owning_proc_id_of_remote_element(local_element, 2));
            EXPECT_EQ( 2, graph.get_owning_proc_id_of_remote_element(local_element, 8));
            EXPECT_EQ( 2, graph.get_owning_proc_id_of_remote_element(local_element, 4));
            EXPECT_EQ( 2, graph.get_owning_proc_id_of_remote_element(local_element, 6));

            EXPECT_EQ(4u, graph.num_edges());
            EXPECT_EQ(4u, graph.num_parallel_edges());
        }
        if (procRank == 2) {
            EXPECT_EQ(13u, graph.num_edges());
            EXPECT_EQ( 5u, graph.num_parallel_edges());
        }
    }
}

// element ids / proc_id:
// |-------|-------|-------|
// |       |       |       |
// |  1/0  |  4/2  |  7/2  |
// |       |       |       |
// |-------|-------|-------|
// |       |       |       |
// |  2/0  |  n/a  |  8/2  |
// |       |       |       |
// |-------|-------|-------|
// |       |       |       |
// |  3/0  |  6/2  |  9/2  |
// |       |       |       |
// |-------|-------|-------|
// The element in the middle has been deleted

TEST(ElementGraph, TestKeyHoleSimilarProblemBInParallel)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    if(stk::parallel_machine_size(comm) == 3)
    {
        const int procRank = stk::parallel_machine_rank(comm);
        unsigned spatialDim = 3;

        stk::mesh::MetaData meta(spatialDim);
        stk::mesh::BulkData bulkData(meta, comm);
        stk::unit_test_util::fill_mesh_using_stk_io("generated:3x1x3", bulkData, comm);

        stk::mesh::EntityProcVec elementProcChanges;
        if (procRank == 1) {
            elementProcChanges.push_back(stk::mesh::EntityProc(bulkData.get_entity(stk::topology::ELEM_RANK,4),2));
            elementProcChanges.push_back(stk::mesh::EntityProc(bulkData.get_entity(stk::topology::ELEM_RANK,6),2));
        }
        bulkData.change_entity_owner(elementProcChanges);
        bulkData.modification_begin();
        if (procRank == 1) {
            stk::mesh::Entity local_element5 = bulkData.get_entity(stk::topology::ELEM_RANK,5);
            bulkData.destroy_entity(local_element5);
        }
        bulkData.modification_end();

        ElemElemGraphTester graph(bulkData);
        if (procRank == 0) {
            EXPECT_EQ(2u, graph.get_num_connected_elems(bulkData.get_entity(stk::topology::ELEM_RANK,1)));
            EXPECT_EQ(2u, graph.get_num_connected_elems(bulkData.get_entity(stk::topology::ELEM_RANK,2)));
            EXPECT_EQ(2u, graph.get_num_connected_elems(bulkData.get_entity(stk::topology::ELEM_RANK,3)));
            EXPECT_EQ(6u, graph.num_edges());
            EXPECT_EQ(2u, graph.num_parallel_edges());
        }
        if (procRank == 1) {
            EXPECT_EQ(0u, graph.size());
            EXPECT_EQ(0u, graph.num_edges());
            EXPECT_EQ(0u, graph.num_parallel_edges());
        }
        if (procRank == 2) {
            EXPECT_EQ(2u, graph.get_num_connected_elems(bulkData.get_entity(stk::topology::ELEM_RANK,4)));
            EXPECT_EQ(2u, graph.get_num_connected_elems(bulkData.get_entity(stk::topology::ELEM_RANK,6)));
            EXPECT_EQ(2u, graph.get_num_connected_elems(bulkData.get_entity(stk::topology::ELEM_RANK,7)));
            EXPECT_EQ(2u, graph.get_num_connected_elems(bulkData.get_entity(stk::topology::ELEM_RANK,8)));
            EXPECT_EQ(2u, graph.get_num_connected_elems(bulkData.get_entity(stk::topology::ELEM_RANK,9)));
            EXPECT_EQ(10u, graph.num_edges());
            EXPECT_EQ( 2u, graph.num_parallel_edges());
        }
    }
}

void test_parallel_uniqueness(const std::vector<stk::mesh::EntityId> &ids_in_use, const std::vector<stk::mesh::EntityId>& requested_ids, stk::ParallelMachine comm)
{
    std::vector<stk::mesh::EntityId> global_ids_in_use;
    stk::parallel_vector_concat(comm, ids_in_use, global_ids_in_use);
    std::sort(global_ids_in_use.begin(), global_ids_in_use.end());

    std::vector<stk::mesh::EntityId> global_requested_ids;
    stk::parallel_vector_concat(comm, requested_ids, global_requested_ids);
    std::sort(global_requested_ids.begin(), global_requested_ids.end());

    std::vector<stk::mesh::EntityId> intersection;

    std::set_intersection(global_ids_in_use.begin(), global_ids_in_use.end(),
            global_requested_ids.begin(), global_requested_ids.end(),
            std::back_inserter(intersection));

    EXPECT_TRUE(intersection.empty());
}

TEST(ElemGraph, test_id_reservation)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;
    if(stk::parallel_machine_size(comm)==2)
    {
        unsigned spatialDim = 3;

        stk::mesh::MetaData meta(spatialDim);
        BulkDataElementGraphTester bulkData(meta, comm);
        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

        std::vector<stk::mesh::EntityId> ids_in_use = bulkData.my_internal_get_ids_in_use_this_proc_for_locally_owned(stk::topology::ELEM_RANK);

        size_t num_ids_requested_per_proc = 10;
        std::vector<stk::mesh::EntityId> requested_ids;
        bulkData.generate_new_ids(stk::topology::ELEM_RANK, num_ids_requested_per_proc, requested_ids);

        test_parallel_uniqueness(ids_in_use, requested_ids, comm);

        std::vector<stk::mesh::EntityId> requested_ids_again;
        bulkData.generate_new_ids_given_reserved_ids(stk::topology::ELEM_RANK, num_ids_requested_per_proc, requested_ids, requested_ids_again);

        test_parallel_uniqueness(ids_in_use, requested_ids_again, comm);
        test_parallel_uniqueness(requested_ids, requested_ids_again, comm);
    }
}

bool is_valid_graph_element(const impl::ElementGraph &elem_graph, stk::mesh::impl::LocalId elem_id)
{
    stk::mesh::impl::LocalId max_elem_id = static_cast<stk::mesh::impl::LocalId>(elem_graph.size());
    return (elem_id >= 0 && elem_id < max_elem_id);
}

int check_connectivity(const impl::ElementGraph &elem_graph, const impl::SidesForElementGraph &via_sides,
                       stk::mesh::impl::LocalId element_id1, stk::mesh::impl::LocalId element_id2)
{
    int side=-1;
    if (is_valid_graph_element(elem_graph, element_id1) && is_valid_graph_element(elem_graph, element_id2)) {
        side = get_side_from_element1_to_element2(elem_graph, via_sides, element_id1, element_id2);
    }
    return side;
}

int get_side_from_element1_to_element2(const impl::ElementGraph &elem_graph,
                                       const impl::SidesForElementGraph &via_sides,
                                       stk::mesh::impl::LocalId element1_local_id,
                                       stk::mesh::impl::LocalId other_element_id)
{
    int side = -1;
    const std::vector<stk::mesh::impl::LocalId>& conn_elements = elem_graph[element1_local_id];

    std::vector<stk::mesh::impl::LocalId>::const_iterator iter = std::find(conn_elements.begin(), conn_elements.end(), other_element_id);
    if ( iter != conn_elements.end() )
    {
        int64_t index = iter - conn_elements.begin();
        side = via_sides[element1_local_id][index];
    }
    return side;
}
//EndDocExample1

TEST(ElemGraph, test_initial_graph_creation_with_deactivated_elements)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;
    if(stk::parallel_machine_size(comm)==2)
    {
        stk::mesh::MetaData meta(3);
        stk::mesh::Part &activePart = meta.declare_part("active");
        BulkDataElementGraphTester bulkData(meta, comm);
        stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);
        stk::unit_test_util::put_mesh_into_part(bulkData, activePart);

        bulkData.modification_begin();
        if(bulkData.parallel_rank() == 0)
        {
            stk::mesh::Entity elem2 = bulkData.get_entity(stk::topology::ELEMENT_RANK, 2);
            bulkData.change_entity_parts(elem2, {}, {&activePart});
        }
        bulkData.modification_end();

        stk::mesh::ElemElemGraph graph(bulkData, activePart);
        if (bulkData.parallel_rank() == 0) {
            stk::mesh::Entity elem1 = bulkData.get_entity(stk::topology::ELEM_RANK,1);
            stk::mesh::Entity elem2 = bulkData.get_entity(stk::topology::ELEM_RANK,2);


            ASSERT_EQ(1u, graph.get_num_connected_elems(elem1));

            EXPECT_TRUE(graph.is_connected_elem_locally_owned(elem1, 0));
            EXPECT_EQ(elem2, graph.get_connected_element(elem1, 0));


            ASSERT_EQ(2u, graph.get_num_connected_elems(elem2));

            EXPECT_TRUE(graph.is_connected_elem_locally_owned(elem2, 0));
            EXPECT_EQ(elem1, graph.get_connected_element(elem2, 0));

            EXPECT_TRUE(!graph.is_connected_elem_locally_owned(elem2, 1));
            stk::mesh::EntityId remoteElemId = 3;
            EXPECT_EQ(remoteElemId, graph.get_entity_id_of_remote_element(elem2, 1));
            stk::mesh::impl::parallel_info &parallelInfo = graph.get_parallel_edge_info(elem2, remoteElemId);
            EXPECT_EQ(1, parallelInfo.m_other_proc);
            EXPECT_TRUE(parallelInfo.m_in_body_to_be_skinned);
        }
        else if (bulkData.parallel_rank() == 1) {
            stk::mesh::Entity elem3 = bulkData.get_entity(stk::topology::ELEM_RANK,3);
            stk::mesh::Entity elem4 = bulkData.get_entity(stk::topology::ELEM_RANK,4);


            ASSERT_EQ(2u, graph.get_num_connected_elems(elem3));

            EXPECT_TRUE(graph.is_connected_elem_locally_owned(elem3, 0));
            EXPECT_EQ(elem4, graph.get_connected_element(elem3, 0));

            EXPECT_TRUE(!graph.is_connected_elem_locally_owned(elem3, 1));
            stk::mesh::EntityId remoteElemId = 2;
            EXPECT_EQ(remoteElemId, graph.get_entity_id_of_remote_element(elem3, 1));
            stk::mesh::impl::parallel_info &parallelInfo = graph.get_parallel_edge_info(elem3, remoteElemId);
            EXPECT_EQ(0, parallelInfo.m_other_proc);
            EXPECT_TRUE(!parallelInfo.m_in_body_to_be_skinned);


            ASSERT_EQ(1u, graph.get_num_connected_elems(elem4));

            EXPECT_TRUE(graph.is_connected_elem_locally_owned(elem4, 0));
            EXPECT_EQ(elem3, graph.get_connected_element(elem4, 0));
        }
    }
}

stk::mesh::PartVector get_skin_parts(stk::mesh::MetaData &meta)
{
    stk::mesh::PartVector skin_parts;
    skin_parts.push_back(meta.get_part("skin"));
    skin_parts.push_back(meta.get_part("active"));
    return skin_parts;
}

void make_2_hex_mesh_with_element1_inactive(stk::mesh::BulkData& bulkData)
{
    stk::mesh::MetaData &meta = bulkData.mesh_meta_data();
    meta.declare_part_with_topology("skin", stk::topology::QUAD_4);
    stk::mesh::Part& active = meta.declare_part("active");
    stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x2", bulkData, bulkData.parallel());

    stk::mesh::EntityVector entitiesToMakeActive;
    std::vector<stk::mesh::PartVector> add_parts;
    std::vector<stk::mesh::PartVector> rm_parts;

    stk::mesh::EntityRank rank = stk::topology::ELEM_RANK;
    stk::mesh::Entity element1 = bulkData.get_entity(rank, 1);

    if(bulkData.is_valid(element1) && bulkData.bucket(element1).owned())
    {
        entitiesToMakeActive.push_back(element1);
        add_parts.push_back(stk::mesh::PartVector(1, &active));
        rm_parts.push_back(stk::mesh::PartVector());
    }
    bulkData.batch_change_entity_parts(entitiesToMakeActive, add_parts, rm_parts);
}

void test_num_faces_on_this_element(const stk::mesh::BulkData& bulkData, stk::mesh::EntityId id, size_t gold_num_faces_this_elem)
{
    stk::mesh::Entity element = bulkData.get_entity(stk::topology::ELEM_RANK, id);
    if(bulkData.is_valid(element))
    {
        unsigned num_faces_this_elem = bulkData.num_faces(element);
        EXPECT_EQ(gold_num_faces_this_elem, num_faces_this_elem);
    }
}

void test_num_faces_per_element(const stk::mesh::BulkData& bulkData, const std::vector<size_t>& gold_num_faces_per_elem)
{
    for(size_t i=0;i<gold_num_faces_per_elem.size();++i)
    {
        stk::mesh::EntityId element_id = i+1;
        test_num_faces_on_this_element(bulkData, element_id, gold_num_faces_per_elem[i]);
    }
}

void add_element_to_block(stk::mesh::BulkData& bulkData, stk::mesh::Entity element, stk::mesh::Part& block)
{
    bulkData.change_entity_parts(element, {&block}, {});
}

stk::mesh::PartVector create_element_blocks(stk::mesh::MetaData& meta, stk::topology top)
{
    stk::mesh::PartVector blocks;
    blocks.push_back(&meta.declare_part_with_topology("block_2", top));
    blocks.push_back(&meta.declare_part_with_topology("block_3", top));
    blocks.push_back(&meta.declare_part_with_topology("block_4", top));
    return blocks;
}

stk::mesh::EntityVector get_elements(stk::mesh::BulkData& bulkData)
{
    stk::mesh::EntityVector elements;
    stk::mesh::EntityRank rank = stk::topology::ELEM_RANK;
    elements.push_back(bulkData.get_entity(rank, 2));
    elements.push_back(bulkData.get_entity(rank, 3));
    elements.push_back(bulkData.get_entity(rank, 4));
    return elements;
}

void add_elements_to_various_blocks(stk::mesh::BulkData& bulkData, stk::mesh::PartVector& blocks, stk::mesh::EntityVector& elements)
{
    bulkData.modification_begin();
    for(size_t i=0;i<elements.size();++i)
    {
        if (bulkData.is_valid(elements[i]) && bulkData.bucket(elements[i]).owned())
        {
            add_element_to_block(bulkData, elements[i], *blocks[i]);
        }
    }
    bulkData.modification_end();
}

std::vector<size_t> get_num_faces_per_elements(const stk::mesh::BulkData& bulkData, const stk::mesh::EntityVector& elements)
{
    std::vector<size_t> num_sides_on_element(elements.size(),0);
    for(size_t i=0;i<elements.size();++i)
    {
        if (bulkData.is_valid(elements[i]))
        {
            num_sides_on_element[i] = bulkData.num_sides(elements[i]);
        }
    }
    return num_sides_on_element;
}

void skin_block_and_test(stk::mesh::BulkData& bulkData, const std::vector<size_t>& num_gold_faces_per_element, const size_t gold_num_faces, const stk::mesh::Part& block, const stk::mesh::PartVector &skin_parts, const stk::mesh::EntityVector& elements)
{
    ElementDeathUtils::skin_part(bulkData, block, skin_parts);

    std::vector<size_t> num_faces_on_element = get_num_faces_per_elements(bulkData, elements);
    for(size_t i=0; i<elements.size(); ++i)
    {
        if (bulkData.is_valid(elements[i]))
        {
            EXPECT_EQ(num_faces_on_element[i], num_gold_faces_per_element[i]);
        }
    }

    std::vector<size_t> mesh_counts;
    stk::mesh::comm_mesh_counts(bulkData, mesh_counts);
    EXPECT_EQ(gold_num_faces, mesh_counts[bulkData.mesh_meta_data().side_rank()]);
}


//////////////////////////// 3D

void test_skinning_by_adding_skin_of_block_4(stk::mesh::BulkData& bulkData, const stk::mesh::Part& block_4, const stk::mesh::PartVector& skin_parts, const stk::mesh::EntityVector &elements)
{
    std::vector<size_t> gold_num_faces_per_element = { 1, 6, 6, 6 };
    size_t gold_num_total_faces = 16u;
    skin_block_and_test(bulkData, gold_num_faces_per_element, gold_num_total_faces, block_4, skin_parts, elements);
}

void test_skinning_by_adding_skin_of_block_1(stk::mesh::BulkData& bulkData, const stk::mesh::Part& block_1, const stk::mesh::PartVector& skin_parts, const stk::mesh::EntityVector &elements)
{
    std::vector<size_t> gold_num_faces_per_element = { 6, 6, 6, 6 };
    size_t gold_num_total_faces = 21u;
    skin_block_and_test(bulkData, gold_num_faces_per_element, gold_num_total_faces, block_1, skin_parts, elements);
}

void test_skinning_of_block_2(stk::mesh::BulkData& bulkData, const stk::mesh::Part& block_2, const stk::mesh::PartVector& skin_parts, const stk::mesh::EntityVector &elements)
{
    std::vector<size_t> gold_num_faces_per_element = { 1, 6, 1, 0 };
    size_t gold_num_total_faces = 6u;
    skin_block_and_test(bulkData, gold_num_faces_per_element, gold_num_total_faces, block_2, skin_parts, elements);
}

void test_skinning_by_adding_skin_of_block_3(stk::mesh::BulkData& bulkData, const stk::mesh::Part& block_3, const stk::mesh::PartVector& skin_parts, const stk::mesh::EntityVector &elements)
{
    std::vector<size_t> gold_num_faces_per_element = { 1, 6, 6, 1 };
    size_t gold_num_total_faces = 11u;
    skin_block_and_test(bulkData, gold_num_faces_per_element, gold_num_total_faces, block_3, skin_parts, elements);
}

////////////////////////////// 2D

void test_skinning_by_adding_skin_of_block_4_2D(stk::mesh::BulkData& bulkData, const stk::mesh::Part& block_4, const stk::mesh::PartVector& skin_parts, const stk::mesh::EntityVector &elements)
{
    std::vector<size_t> gold_num_faces_per_element = { 1, 4, 4, 4 };
    size_t gold_num_total_faces = 10u;
    skin_block_and_test(bulkData, gold_num_faces_per_element, gold_num_total_faces, block_4, skin_parts, elements);
}

void test_skinning_by_adding_skin_of_block_1_2D(stk::mesh::BulkData& bulkData, const stk::mesh::Part& block_1, const stk::mesh::PartVector& skin_parts, const stk::mesh::EntityVector &elements)
{
    std::vector<size_t> gold_num_faces_per_element = { 4, 4, 4, 4 };
    size_t gold_num_total_faces = 13u;
    skin_block_and_test(bulkData, gold_num_faces_per_element, gold_num_total_faces, block_1, skin_parts, elements);
}

void test_skinning_of_block_2_2D(stk::mesh::BulkData& bulkData, const stk::mesh::Part& block_2, const stk::mesh::PartVector& skin_parts, const stk::mesh::EntityVector &elements)
{
    std::vector<size_t> gold_num_faces_per_element = { 1, 4, 1, 0 };
    size_t gold_num_total_faces = 4u;
    skin_block_and_test(bulkData, gold_num_faces_per_element, gold_num_total_faces, block_2, skin_parts, elements);
}

void test_skinning_by_adding_skin_of_block_3_2D(stk::mesh::BulkData& bulkData, const stk::mesh::Part& block_3, const stk::mesh::PartVector& skin_parts, const stk::mesh::EntityVector &elements)
{
    std::vector<size_t> gold_num_faces_per_element = { 1, 4, 4, 1 };
    size_t gold_num_total_faces = 7u;
    skin_block_and_test(bulkData, gold_num_faces_per_element, gold_num_total_faces, block_3, skin_parts, elements);
}

TEST(ElementGraph, skin_exposed_boundary)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

     if(stk::parallel_machine_size(comm) <= 2)
     {
         unsigned spatialDim = 3;

         stk::mesh::MetaData meta(spatialDim);
         stk::mesh::BulkData bulkData(meta, comm);
         make_2_hex_mesh_with_element1_inactive(bulkData);
         stk::mesh::PartVector skin_parts = get_skin_parts(meta);
         ElementDeathUtils::skin_boundary(bulkData, *meta.get_part("active"), skin_parts);
         test_num_faces_per_element(bulkData, {5u, 0u});
         std::vector<size_t> global_mesh_counts;
         stk::mesh::comm_mesh_counts(bulkData, global_mesh_counts);
         EXPECT_EQ(5u, global_mesh_counts[meta.side_rank()]);
     }
}

TEST(ElementGraph, skin_part)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

     if(stk::parallel_machine_size(comm) <= 2)
     {
         unsigned spatialDim = 3;

         stk::mesh::MetaData meta(spatialDim);
         stk::mesh::BulkData bulkData(meta, comm);
         make_2_hex_mesh_with_element1_inactive(bulkData);
         stk::mesh::PartVector skin_parts = get_skin_parts(meta);
         ElementDeathUtils::skin_part(bulkData, *meta.get_part("active"), skin_parts);
         test_num_faces_per_element(bulkData, {6u, 1u});
         std::vector<size_t> global_mesh_counts;
         stk::mesh::comm_mesh_counts(bulkData, global_mesh_counts);
         EXPECT_EQ(6u, global_mesh_counts[meta.side_rank()]);
     }
}

TEST(ElementGraph, skin_part_3_blocks)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

     if(stk::parallel_machine_size(comm) <= 2)
     {
         unsigned spatialDim = 3;

         stk::mesh::MetaData meta(spatialDim);
         stk::mesh::BulkData bulkData(meta, comm);
         stk::mesh::Part& skin = meta.declare_part_with_topology("skin", stk::topology::QUAD_4);

         stk::mesh::PartVector blocks = create_element_blocks(meta, stk::topology::HEX_8);

         stk::mesh::Part& active = meta.declare_part("active");
         stk::unit_test_util::fill_mesh_using_stk_io("generated:1x1x4", bulkData, comm);

         stk::mesh::EntityVector elements = get_elements(bulkData);
         add_elements_to_various_blocks(bulkData, blocks, elements);

         elements.insert(elements.begin(), bulkData.get_entity(stk::topology::ELEM_RANK, 1));
         blocks.insert(blocks.begin(), meta.get_part("block_1"));

         test_skinning_of_block_2(bulkData, *blocks[1], {&active, &skin}, elements);
         test_skinning_by_adding_skin_of_block_3(bulkData, *blocks[2], {&active, &skin}, elements);
         test_skinning_by_adding_skin_of_block_4(bulkData, *blocks[3], {&active, &skin}, elements);
         test_skinning_by_adding_skin_of_block_1(bulkData, *blocks[0], {&active, &skin}, elements);
     }
}

TEST(ElementGraph, skin_part_3_blocks_2D)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    if(stk::parallel_machine_size(comm) <= 4)
    {
        const unsigned X = 4, Y = 1;
        bool auraOn = true;
        stk::mesh::fixtures::QuadFixture fixture(comm, X, Y, auraOn);

        stk::mesh::MetaData & meta = fixture.m_meta;
        stk::mesh::Part& skin = meta.declare_part_with_topology("skin", stk::topology::LINE_2);

        stk::mesh::PartVector blocks = create_element_blocks(meta, stk::topology::QUAD_4);
        stk::mesh::Part& active = meta.declare_part("active");

        fixture.m_meta.commit();
        fixture.generate_mesh();

        stk::mesh::BulkData & bulkData = fixture.m_bulk_data;

        stk::mesh::EntityVector elements = get_elements(bulkData);
        add_elements_to_various_blocks(bulkData, blocks, elements);

        elements.insert(elements.begin(), bulkData.get_entity(stk::topology::ELEM_RANK, 1));
        blocks.insert(blocks.begin(), meta.get_part("quad_part")); // because quad fixture doesn't do block_1

        test_skinning_of_block_2_2D(bulkData, *blocks[1], {&active, &skin}, elements);
        test_skinning_by_adding_skin_of_block_3_2D(bulkData, *blocks[2], {&active, &skin}, elements);
        test_skinning_by_adding_skin_of_block_4_2D(bulkData, *blocks[3], {&active, &skin}, elements);
        test_skinning_by_adding_skin_of_block_1_2D(bulkData, *blocks[0], {&active, &skin}, elements);
    }
}

TEST(ElementGraph, heterogeneous_mesh)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    if(stk::parallel_machine_size(comm) <= 17)
    {
        std::string fileName("hetero.g");
        {
            stk::mesh::MetaData meta_data(3);
            stk::mesh::fixtures::VectorFieldType & node_coord =
            meta_data.declare_field<stk::mesh::fixtures::VectorFieldType>(stk::topology::NODE_RANK, "coordinates");
            stk::mesh::put_field( node_coord , meta_data.universal_part() , 3);

            stk::mesh::fixtures::heterogeneous_mesh_meta_data( meta_data , node_coord );
            meta_data.commit();

            stk::mesh::BulkData bulk_data( meta_data, MPI_COMM_SELF, stk::mesh::BulkData::NO_AUTO_AURA );
            stk::mesh::fixtures::heterogeneous_mesh_bulk_data( bulk_data , node_coord );
            if (stk::parallel_machine_rank(comm) == 0)
            {
                stk::unit_test_util::write_mesh_using_stk_io(fileName, bulk_data, comm);
            }
        }
        stk::mesh::MetaData meta_data(3);

        for(int i=1;i<=17;i++)
        {
            std::ostringstream os;
            os << "skin_" << i;
            std::string part_name = os.str();
            stk::mesh::Part &tmp = meta_data.declare_part(part_name, meta_data.side_rank());
            stk::io::put_io_part_attribute(tmp);
        }

        stk::mesh::Part& skin = meta_data.declare_part("skin", meta_data.side_rank());

        stk::mesh::BulkData bulk_data( meta_data, comm, stk::mesh::BulkData::NO_AUTO_AURA );
        stk::unit_test_util::read_from_serial_file_and_decompose(fileName, bulk_data, "RIB");
        EXPECT_NO_FATAL_FAILURE(ElementDeathUtils::skin_boundary(bulk_data, meta_data.locally_owned_part(), {&skin}));
        std::vector<size_t> mesh_counts;
        stk::mesh::comm_mesh_counts(bulk_data, mesh_counts);
        EXPECT_EQ(23u, mesh_counts[meta_data.side_rank()]);

        std::vector<std::pair<stk::mesh::EntityId, int>> id_and_num_faces = {
                {7, 1},
                {8, 1},
                {9, 1},
                {10, 2},
                {11, 1},
                {4, 2},
                {5, 2},
                {6, 2},
                {1, 3},
                {2, 2},
                {3, 4},
                {15, 0},
                {16, 0},
                {17, 1},
                {12, 0},
                {13, 0},
                {14, 1}
        };

        bulk_data.modification_begin();

        for(size_t i=0;i<id_and_num_faces.size();++i)
        {
            stk::mesh::EntityId id = id_and_num_faces[i].first;
            int gold_num_faces = id_and_num_faces[i].second;
            stk::mesh::Entity elem = bulk_data.get_entity(stk::topology::ELEM_RANK, id);
            if(bulk_data.is_valid(elem))
            {
                int num_faces = bulk_data.num_sides(elem);
                const stk::mesh::Entity *faces = bulk_data.begin_faces(elem);
                for(int j=0;j<num_faces;++j)
                {
                    std::ostringstream os;
                    os << "skin_" << bulk_data.identifier(elem);
                    stk::mesh::PartVector add_parts;
                    add_parts.push_back(meta_data.get_part(os.str()));
                    bulk_data.change_entity_parts(faces[j], add_parts, {});

                }
                EXPECT_EQ(gold_num_faces, num_faces) << "element " << id << " has topology " << bulk_data.bucket(elem).topology() << " with num faces " << num_faces << " not same as gold value " << gold_num_faces << std::endl;
            }
        }

        bulk_data.modification_begin();

        //stk::unit_test_util::write_mesh_using_stk_io("heter.g", bulk_data, comm);
    }
}

TEST(ElementGraph, degenerate_mesh)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    if(stk::parallel_machine_size(comm) <= 2)
    {
        std::string fileName("degenerate.g");
        {
            stk::mesh::MetaData meta_data(3);
            stk::mesh::fixtures::VectorFieldType & node_coord =
                    meta_data.declare_field<stk::mesh::fixtures::VectorFieldType>(stk::topology::NODE_RANK, "coordinates");
            stk::mesh::put_field(node_coord, meta_data.universal_part(), 3);

            stk::mesh::fixtures::degenerate_mesh_meta_data(meta_data, node_coord);
            meta_data.commit();

            stk::mesh::BulkData bulk_data(meta_data, MPI_COMM_SELF, stk::mesh::BulkData::NO_AUTO_AURA);
            stk::mesh::fixtures::degenerate_mesh_bulk_data(bulk_data, node_coord);
            if(stk::parallel_machine_rank(comm) == 0)
            {
                stk::unit_test_util::write_mesh_using_stk_io(fileName, bulk_data, comm);
            }
        }
        stk::mesh::MetaData meta_data(3);
        stk::mesh::Part &skin = meta_data.declare_part("skin", meta_data.side_rank());
        stk::io::put_io_part_attribute(skin);
        stk::mesh::BulkData bulk_data(meta_data, comm, stk::mesh::BulkData::NO_AUTO_AURA);
        stk::unit_test_util::read_from_serial_file_and_decompose(fileName, bulk_data, "RIB");
        EXPECT_NO_FATAL_FAILURE(ElementDeathUtils::skin_boundary(bulk_data, meta_data.locally_owned_part(), {&skin}));
        std::vector<size_t> mesh_counts;
        stk::mesh::comm_mesh_counts(bulk_data, mesh_counts);
        EXPECT_EQ(10u, mesh_counts[meta_data.side_rank()]);
    }
}

TEST(ElementGraph, two_wedge_sandwich_with_quad_shell)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;

    if(stk::parallel_machine_size(comm) <= 3)
    {
        std::string fileName("made_up.g");
        {
            stk::mesh::MetaData meta_data(3);
            stk::mesh::fixtures::VectorFieldType & node_coord =
                    meta_data.declare_field<stk::mesh::fixtures::VectorFieldType>(stk::topology::NODE_RANK, "coordinates");
            stk::mesh::put_field(node_coord, meta_data.universal_part(), 3);

            stk::io::put_io_part_attribute(meta_data.declare_part_with_topology("wedges", stk::topology::WEDGE_6));
            stk::io::put_io_part_attribute(meta_data.declare_part_with_topology("quad_shells", stk::topology::SHELL_QUAD_4));

            meta_data.commit();

            stk::mesh::BulkData bulk_data(meta_data, MPI_COMM_SELF, stk::mesh::BulkData::NO_AUTO_AURA);

            static const char method[] =
                    "stk_mesh::fixtures::heterogenous_mesh_bulk_data";
            ////////////////////////////////

            bulk_data.modification_begin();

            stk::mesh::Part & wedge_block = *meta_data.get_part("wedges", method);
            stk::mesh::Part & quad_shell_block = *meta_data.get_part("quad_shells", method);

            unsigned elem_id = 1;

            unsigned number_wedge = 2;

            stk::mesh::EntityIdVector wedge_node_ids[2] = {
                    {1, 7, 3, 2, 8, 4},
                    {7, 5, 3, 8, 6, 4}
            };

            for(unsigned i = 0; i < number_wedge; ++i, ++elem_id)
            {
                stk::mesh::declare_element(bulk_data, wedge_block, elem_id, wedge_node_ids[i]);
            }

            stk::mesh::EntityIdVector shell_quad_node_ids = {3, 4, 8, 7};

            stk::mesh::declare_element(bulk_data, quad_shell_block, elem_id, shell_quad_node_ids);

            const unsigned node_count = 8;
            static const double node_coord_data[node_count][3] = {
                    {0, 0, 0}, {0, 0, 1}, {1, 0, 0}, {1, 0, 1},
                    {1, 1, 0}, {1, 1, 1}, {0, 1, 0}, {0, 1, 1}};

            for(unsigned i = 0; i < node_count; ++i)
            {
                stk::mesh::Entity const node = bulk_data.get_entity(stk::topology::NODE_RANK, i + 1);

                double * const coord = stk::mesh::field_data(node_coord, node);

                coord[0] = node_coord_data[i][0];
                coord[1] = node_coord_data[i][1];
                coord[2] = node_coord_data[i][2];
            }

            bulk_data.modification_end();
            ////////////////////////////////

            if(stk::parallel_machine_rank(comm) == 0)
            {
                stk::unit_test_util::write_mesh_using_stk_io(fileName, bulk_data, comm);
            }
        }
        stk::mesh::MetaData meta_data(3);
        stk::mesh::Part &skin = meta_data.declare_part("skin", meta_data.side_rank());
        stk::io::put_io_part_attribute(skin);
        stk::mesh::BulkData bulk_data(meta_data, comm, stk::mesh::BulkData::NO_AUTO_AURA);
        stk::unit_test_util::read_from_serial_file_and_decompose(fileName, bulk_data, "RIB");
        EXPECT_NO_FATAL_FAILURE(ElementDeathUtils::skin_boundary(bulk_data, meta_data.locally_owned_part(), {&skin}));
        std::vector<size_t> mesh_counts;
        stk::mesh::comm_mesh_counts(bulk_data, mesh_counts);
        EXPECT_EQ(8u, mesh_counts[meta_data.side_rank()]);

        std::vector<std::pair<stk::mesh::EntityId, int>> id_and_num_faces = {
                {1, 4},
                {2, 4},
                {3, 0}
        };

        for(size_t i = 0; i < id_and_num_faces.size(); ++i)
        {
            stk::mesh::EntityId id = id_and_num_faces[i].first;
            int gold_num_faces = id_and_num_faces[i].second;
            stk::mesh::Entity elem = bulk_data.get_entity(stk::topology::ELEM_RANK, id);
            if(bulk_data.is_valid(elem))
            {
                int num_faces = bulk_data.num_sides(elem);
                EXPECT_EQ(gold_num_faces, num_faces)<< "element " << id << " has topology " << bulk_data.bucket(elem).topology() << " with num faces " << num_faces << " not same as gold value " << gold_num_faces << std::endl;
            }
        }
    }
}

TEST(ElementGraph, RefinedQuad)
{
    stk::ParallelMachine comm = MPI_COMM_WORLD;
    if (stk::parallel_machine_size(comm) <= 2)
    {
        const int spatialDim = 2;
        stk::mesh::MetaData meta(spatialDim);

        stk::mesh::Part &quad_part = meta.declare_part_with_topology("Quads", stk::topology::QUADRILATERAL_4_2D);
        stk::mesh::Part &skin = meta.declare_part_with_topology("Edges", stk::topology::LINE_2);
        stk::io::put_io_part_attribute(skin);
        stk::mesh::PartVector skin_parts = {&skin};
        stk::mesh::Part &active = meta.declare_part("active");

        stk::mesh::Field<double,stk::mesh::Cartesian> & node_coord = meta.declare_field<stk::mesh::Field<double,stk::mesh::Cartesian>>(stk::topology::NODE_RANK, "coordinates");
        stk::mesh::put_field(node_coord, meta.universal_part(), 3);
        stk::io::put_io_part_attribute(quad_part);
        meta.commit();

        stk::mesh::BulkData mesh(meta, comm, stk::mesh::BulkData::NO_AUTO_AURA);

        mesh.modification_begin();

        std::vector<std::vector<stk::mesh::EntityId>> elem_this_proc = {
                {1, 2, 3, 4, 5, 6},
                {7}
        };

        std::vector<stk::mesh::EntityId> ids = {1, 2, 3, 4, 5, 6, 7};
        std::vector<std::vector<stk::mesh::EntityId> > connectivity = {
                {1, 2, 3, 4},
                {1, 2, 6, 5},
                {2, 3, 7, 6},
                {8, 7, 3, 4},
                {1, 5, 8, 4},
                {5, 6, 7, 8},
                {10, 1, 4, 9}
        };

        bool running_2_procs = mesh.parallel_size() > 1;

        int my_proc_id = mesh.parallel_rank();
        std::vector<stk::mesh::EntityId> elems_this_proc;
        if(running_2_procs)
        {
            elems_this_proc = elem_this_proc[my_proc_id];
        }
        else
        {
            elems_this_proc = ids;
        }

        for(size_t i = 0; i < elems_this_proc.size(); ++i)
        {
            int index = static_cast<int>(elems_this_proc[i])-1;
            stk::mesh::Entity element = stk::mesh::declare_element(mesh, quad_part, elems_this_proc[i], connectivity[index]);
            if(ids[index]!=1)
            {
                mesh.change_entity_parts(element, {&active}, {});
            }
        }

        std::vector<stk::mesh::EntityId> nodes = { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 };

        std::vector<std::vector<stk::mesh::EntityId>> nodes_this_proc = {
                {1, 2, 3, 4, 5, 6, 7, 8},
                {1, 4, 9, 10}
        };

        std::vector<std::vector<double> > node_coords = {
                {0, 0}, {3, 0}, {3, 3}, {0, 3}, {1, 1}, {2, 1}, {2, 2}, {1, 2}, {-3, 3}, {-3, 0}
        };

        std::vector<stk::mesh::EntityId> nodes_to_create;
        if(running_2_procs)
        {
            nodes_to_create = nodes_this_proc[my_proc_id];
        }
        else
        {
            nodes_to_create = nodes;
        }

        for(unsigned i = 0; i < nodes_to_create.size(); ++i)
        {
            stk::mesh::Entity const node = mesh.get_entity(stk::topology::NODE_RANK, nodes_to_create[i]);
            if(running_2_procs && (nodes_to_create[i]==1 || nodes_to_create[i]==4))
            {
                mesh.add_node_sharing(node,1-my_proc_id);
            }

            double * const coord = stk::mesh::field_data(node_coord, node);
            int index = static_cast<int>(nodes_to_create[i])-1;

            coord[0] = node_coords[index][0];
            coord[1] = node_coords[index][1];
        }

        mesh.modification_end();

        stk::mesh::Selector active_sel = active;
        stk::mesh::Selector air = !active;

        stk::mesh::ElemElemGraph elem_elem_graph(mesh, active_sel, &air);
        elem_elem_graph.skin_mesh(skin_parts);

        std::vector<size_t> mesh_counts;
        stk::mesh::comm_mesh_counts(mesh, mesh_counts);
        EXPECT_EQ(10u, mesh_counts[stk::topology::NODE_RANK]);
        EXPECT_EQ(7u, mesh_counts[stk::topology::ELEM_RANK]);
        EXPECT_EQ(6u, mesh_counts[meta.side_rank()]);

        stk::unit_test_util::write_mesh_using_stk_io("refined.g", mesh, comm);

        std::vector<std::pair<stk::mesh::EntityId, int>> id_and_num_faces = {
                {1, 3},
                {2, 1},
                {3, 1},
                {4, 1},
                {5, 0},
                {6, 0},
                {7, 3}
        };

        for(size_t i = 0; i < id_and_num_faces.size(); ++i)
        {
            stk::mesh::EntityId id = id_and_num_faces[i].first;
            int gold_num_faces = id_and_num_faces[i].second;
            stk::mesh::Entity elem = mesh.get_entity(stk::topology::ELEM_RANK, id);
            if(mesh.is_valid(elem))
            {
                int num_faces = mesh.num_sides(elem);
                EXPECT_EQ(gold_num_faces, num_faces)<< "element " << id << " has topology " << mesh.bucket(elem).topology() << " with num faces " << num_faces << " not same as gold value " << gold_num_faces << std::endl;
            }
        }
    }
}

} // namespace
