/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

//*****************************************************************************
//
// Class:			UNSSoapServer
// Description:		Run a stand-alone soap server
//
//*****************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "UNSSoapServer.h"
#ifdef _WIN32
	#include "UNSMessageFile.h"
#else
	#include "UNSMessageFileLinux.h"
#endif
#include "soapStub.h"
#include "SoapBinding.h"

extern "C" {
  #include "httpda.h"
}

UNSEventLogger* UNSSoapServer::m_Logger = NULL;
string			UNSSoapServer::m_userid	= "user";
string			UNSSoapServer::m_passwd	= "user";
static char UNS_AUTH_REALM[] = "Intel AMT UNS";
//*****************************************************************************
// Name			: UNSSoapServer
// Description	: Constructor - initializes soap structures
//*****************************************************************************
UNSSoapServer::UNSSoapServer():
	m_socket	(0),
	m_port		(0),
	m_init		(false)
{
	// Initialize soap.
	m_soap = soap_new();
	soap_init(m_soap);
	m_soap->accept_timeout = UNS_SERVER_TIMEOUT;
	if (!m_soap->namespaces) m_soap->namespaces = namespaces;
	if (soap_register_plugin(m_soap, http_da))
		soap_print_fault(m_soap, stderr); // failed to register
}

//*****************************************************************************
// Name			: UNSSoapServer
// Description	: Destructor - kill soap.
//*****************************************************************************
UNSSoapServer::~UNSSoapServer(void)
{
	if (m_soap != NULL)
	{
		soap_destroy(m_soap);
		soap_end(m_soap);
		soap_done(m_soap);
		soap_del(m_soap);
	}
}

//*****************************************************************************
// Name			: UNSSoapServer::Bind
// Description	: Bind host.
//*****************************************************************************
UNS_STATUS UNSSoapServer::Bind (const char *host)
{
	// Check if we are not already bind.
	if (m_init && m_socket != 0 && m_port != 0) return UNS_STATUS_SUCCESS;

	struct	sockaddr	name;
	socklen_t		namelen	= sizeof(name) ;

	// Bind to free port.
	m_socket = soap_bind(m_soap, host, 0, BACKLOG);
	if (!soap_valid_socket(m_socket)) return UNS_STATUS_BIND_ERROR;

	// Get the port name.
	if (getsockname( m_socket, &name, &namelen) != 0)
		return UNS_STATUS_BIND_ERROR;

	if (name.sa_family != AF_INET)
		return UNS_STATUS_BIND_ERROR; // Support only IPv4

	m_port = ntohs (((sockaddr_in*)(&name))->sin_port);

	// << Debug log
	printf("Server bind success, Port is: %d\n",m_port);
	// Debug log >>

	m_init = true;
	return UNS_STATUS_SUCCESS;
}

//*****************************************************************************
// Name			: UNSSoapServer::RunServer
// Description	: Listen on socket for one time interval.
//*****************************************************************************
void UNSSoapServer::RunServer	()
{
	SOAP_SOCKET  s; // master and slave sockets

	s = soap_accept(m_soap);
	if (!soap_valid_socket(s))
	{
		if (m_soap->errnum != 0) // If not timeout log error message.
			printf("Accept failed\n");
		return;
	}

	// << Debug log
	printf("accepted connection from IP=%d.%d.%d.%d socket=%u\n",
		(int)(m_soap->ip >> 24)&0xFF, (int)(m_soap->ip >> 16)&0xFF, (int)(m_soap->ip >> 8)&0xFF,
		(int)(m_soap->ip)&0xFF,s);
	// Debug log >>

	if (soap_serve(m_soap) != SOAP_OK)
	{
		// << Debug log
		printf("Failed to process request\n");
		// Debug log >>
	}
	soap_destroy(m_soap);	// clean up class instances
	soap_end(m_soap);		// clean up everything and close socket

	return;
}

//*****************************************************************************
// Name			: UNSSoapServer::GetPort
// Description	: Get this server port, 0 if not bi
//*****************************************************************************
int	UNSSoapServer::GetPort ()
{
	return m_init ?	m_port : 0;
}

//*****************************************************************************
// Name			: UNSSoapServer::IsInit
// Description	: Return true iff the server is already initialized (i.e we have
//                a master socket)
//*****************************************************************************
bool	UNSSoapServer::IsInit ()
{
	return m_init;
}
//*****************************************************************************
// Name			: UNSSoapServer::GetMasterSocket
// Description	: Get this server master socket.
//*****************************************************************************
SOAP_SOCKET	UNSSoapServer::GetMasterSocket	()
{
	return m_init ?	m_socket : 0;
}

//*****************************************************************************
// Name			: UNSSoapServer::SetCredentials
// Description	: Set the m_stop flag and stop the server.
//*****************************************************************************
void UNSSoapServer::SetCredentials(const string userid,
								   const string passwd)
{
	m_userid = userid;
	m_passwd = passwd;
}

//*****************************************************************************
// Name			: UNSSoapServer::SetLogger
// Description	: Set the m_stop flag and stop the server.
//*****************************************************************************
void UNSSoapServer::SetLogger(UNSEventLogger *logger)
{
	m_Logger = logger;
}

void UNSSoapServer::printEvent(emi__EventLogRecordType * event)
{
	printf ("EventSensorType\t\t: 0x02%x\n"	, event->EventSensorType);
	printf ("EventType\t\t: 0x%02x\n"		, event->EventType);
	printf ("EventOffset\t\t: 0x%02x\n"		, event->EventOffset);
	printf ("EventSourceType\t\t: 0x02%x\n"	, event->EventSourceType);
	printf ("EventSeverity\t\t: 0x%02x\n"	, event->EventSeverity);
	printf ("SensorNumber\t\t: 0x%02x\n"	, event->SensorNumber);
	printf ("Entity\t\t\t: 0x%02x\n"			, event->Entity);
	printf ("EventSensorType\t\t: 0x%02x\n"	, event->EventSensorType);
	printf ("EntityInstance\t\t: 0x%02x\n"	, event->EntityInstance);
	printf ("EventData\t\t: 0x");
	for (unsigned int i=0; i < event->EventData->Byte.size(); i++)
	{
		printf ("%02x", event->EventData->Byte[i]);
	}
	printf ("\n");
}

//*****************************************************************************
// Name			: UNSSoapServer::HandlePetAlert
// Description	: Parse the given pet alert and record the appropriate event
//				  to the Event Log.
// Params		: request - soap request (pet alert).
// Returns		: true if alert was parsed successfully, false otherwise.
//*****************************************************************************
bool UNSSoapServer::HandlePetAlert(_emc__SoapAlertRequest *request)
{
	bool res = true;					// Return status.

	// Go over alerts vector and parse each message.
	std::vector<emi__EventLogRecordType * >::iterator itr;
	for (	itr = request->SoapAlertRecords.begin();
			itr != request->SoapAlertRecords.end(); itr++)
	{
		printEvent(*itr);
		if (!IsEventLegal(*itr)) {
			// log that an illegal event was received
			// open: return error or skip to next alert record in vector?
			printf ("Not a legal User Notification Event\n");
			res = false;
			continue;
		}

		// Copy the data vector to a local vector.
		EventType event;
		if (GetEventType((*itr)->EventData->Byte, event)) {
			m_Logger->LogInfoEvent(event.category , event.id);
			printf ("Received legal alert: 0x%x 0x%lx\n",event.category, event.id);
		}
		else {
			printf ("Unknown event: 0x%x 0x%lx\n",event.category, event.id);
			res = false;
			// Log some kind of error message.
		}
	}
	return res;
}

//*****************************************************************************
// Name			: UNSSoapServer::IsEventLegal
// Description	: Check the event for constant fields.
// Params		: event of type emi__EventLogRecordType
// Returns		: true iff event is legal
//*****************************************************************************
bool UNSSoapServer::IsEventLegal (emi__EventLogRecordType * event)
{
	// Check for constant fields.
	if (event->EventSensorType 	!= UNS_EVENT_SENSOR_TYPE	||
		event->EventType		!= UNS_EVENT_TYPE			||
		event->EventOffset		!= UNS_EVENT_OFFSET			||
		event->EventSourceType	!= UNS_EVENT_SOURCE_TYPE	||
		event->EventSeverity	!= UNS_EVENT_SEVERITY		||
		event->SensorNumber		!= UNS_EVENT_SENSOR_NUMBER	||
		event->Entity			!= UNS_EVENT_ENTITY)
		return false;

	if ((event->EntityInstance != 0x00) &&
		(event->EntityInstance != 0x61) &&
		(event->EntityInstance != 0x62))
		return false;

	if ((event->EventData->Byte.size() < 3)  ||
		(event->EventData->Byte[0] != 0xAA) ||
		(event->EventData->Byte[1] & 0x07) != 0)
		return false;

	return true;
}

//*****************************************************************************
// Name			: UNSSoapServer::GetEventID
// Description	: Return the event category and ID according to the event data.
// Params		: data - a vector of 8 bytes
//*****************************************************************************
bool UNSSoapServer::GetEventType (vector<unsigned char> &data, EventType &event)
{
	unsigned char reason = data[1];
	unsigned char code	 = data[2];

	switch (reason)
	{
	case UNS_REASON_CIRCUIT_BREAKER:
		event.category = UNS_CIRCUIT_BREAKER;
		switch (code) {
			case UNS_CODE_NETWORK_TRAFFIC_TX_CEASED:
				event.id = NETWORK_TRAFFIC_TX_CEASED;
				break;
			case UNS_CODE_NETWORK_CONNECTIVITY_TX_REDUCED:
				event.id = NETWORK_CONNECTIVITY_TX_REDUCED;
				break;
			case UNS_CODE_NETWORK_TRAFFIC_RX_CEASED:
				event.id = NETWORK_TRAFFIC_RX_CEASED;
				break;
			case UNS_CODE_NETWORK_CONNECTIVITY_RX_REDUCED:
				event.id = NETWORK_CONNECTIVITY_RX_REDUCED;
				break;
			default:
				return false;
		}
		break;
	case UNS_REASON_WLAN:
		event.category = UNS_WLAN;
		switch (code) {
			case UNS_CODE_WLAN_PROFILE_INSUFFICIENT:
				event.id = WLAN_PROFILE_INSUFFICIENT;
				break;
			case UNS_CODE_WLAN_SECURITY_INSUFFICIENT:
				event.id = WLAN_SECURITY_INSUFFICIENT;
				break;
			case UNS_CODE_WLAN_SESSION_ESTABLISHED:
				event.id = WLAN_SESSION_ESTABLISHED;
				break;
			case UNS_CODE_WLAN_SESSION_ENDED:
				event.id = WLAN_SESSION_ENDED;
				break;
			default:
				return false;
		}
		break;
	case UNS_REASON_REMOTE_DIAGNOSTIC:
		event.category = UNS_REMOTE_DIAGNOSTIC;
		switch (code) {
			case UNS_CODE_REMOTE_SOL_STARTED:
				event.id = REMOTE_SOL_STARTED;
				break;
			case UNS_CODE_REMOTE_SOL_ENDED:
				event.id = REMOTE_SOL_ENDED;
				break;
			case UNS_CODE_REMOTE_IDER_STARTED:
				event.id = REMOTE_IDER_STARTED;
				break;
			case UNS_CODE_REMOTE_IDER_ENDED:
				event.id = REMOTE_IDER_ENDED;
				break;
			default:
				return false;
		}
		break;
	default:
		return false;
	}

	return true;
}


//*****************************************************************************
// Name			: __emc__SoapAlert
// Description	: Handle soap incoming packet. Write the appropriate event
//				  to Windows Event Log.
//*****************************************************************************
SOAP_FMAC5 int SOAP_FMAC6 __emc__SoapAlert(struct soap* soap, _emc__SoapAlertRequest *emc__SoapAlertRequest, _emc__SoapAlertResponse *emc__SoapAlertResponse)
{
	// TO-DO - parse PET and record event.
	emc__SoapAlertResponse->Status = PT_STATUS_SUCCESS;

	// Client used basic authentication:
	if (soap->userid && soap->passwd) // client used basic authentication
	{
#ifdef BASIC_AUTH
		// go ahead and compare info:
		if (!strcmp(soap->userid, UNSSoapServer::m_userid.c_str()) &&
			!strcmp(soap->passwd, UNSSoapServer::m_passwd.c_str()))
		{
			// Parse the PET message here:
			UNSSoapServer::HandlePetAlert(emc__SoapAlertRequest);
					return SOAP_OK;
		}
#endif
	}
	else if (soap->authrealm && soap->userid) // Digest authentication
	{
#ifdef DIGEST_AUTH
		if (!strcmp(soap->authrealm, UNS_AUTH_REALM) && !strcmp(soap->userid, UNSSoapServer::m_userid.c_str()))
		{
			if (!http_da_verify_post(soap, (char*)UNSSoapServer::m_passwd.c_str()))
			{
				UNSSoapServer::HandlePetAlert(emc__SoapAlertRequest);
				return SOAP_OK;
			}
			printf ("Digest authentication failed");
		}
#endif
	}

#ifdef NO_AUTH
	UNSSoapServer::HandlePetAlert(emc__SoapAlertRequest);
	return SOAP_OK;
#else
	printf ("401: Unauthorized\n");
	printf ("Info: Unauthorized connection while authorization is required.\n");
	printf ("      Code 401 returned to challenge basic/digest authentication.\n");
	soap->authrealm = UNS_AUTH_REALM; // set realm for challenge
	return 401; // Not authorized, challenge basic/digest authentication
#endif
}
