/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#include "EBOOKBitStream.h"
#include "EBOOKMemoryStream.h"

#include "EBOOKBitStreamTest.h"

using libebook::EBOOKBitStream;
using libebook::EBOOKMemoryStream;

namespace test
{

namespace
{

static const unsigned char TEST_DATA[] = {0xe3, 0x8e, 0x38, 0xe3};

}

void EBOOKBitStreamTest::setUp()
{
}

void EBOOKBitStreamTest::tearDown()
{
}

void EBOOKBitStreamTest::testRead1()
{
  testRead1(false);
  testRead1(true);
}

void EBOOKBitStreamTest::testRead3()
{
  testRead3(false);
  testRead3(true);
}

void EBOOKBitStreamTest::testRead4()
{
  testRead4(false);
  testRead4(true);
}

void EBOOKBitStreamTest::testRead8()
{
  testRead8(false);
  testRead8(true);
}

void EBOOKBitStreamTest::testRead16()
{
  EBOOKMemoryStream stream(TEST_DATA, sizeof TEST_DATA);

  {
    EBOOKBitStream bitStream(&stream);

    CPPUNIT_ASSERT(!bitStream.atEOS());

    CPPUNIT_ASSERT_EQUAL(0x8ee3u, bitStream.read(16));
    CPPUNIT_ASSERT_EQUAL(0xe338u, bitStream.read(16));

    CPPUNIT_ASSERT(bitStream.atEOS());
  }

  stream.seek(0, WPX_SEEK_SET);

  {
    EBOOKBitStream bitStream(&stream);

    CPPUNIT_ASSERT(!bitStream.atEOS());

    CPPUNIT_ASSERT_EQUAL(0xe38eu, bitStream.read(16, true));
    CPPUNIT_ASSERT_EQUAL(0x38e3u, bitStream.read(16, true));

    CPPUNIT_ASSERT(bitStream.atEOS());
  }
}

void EBOOKBitStreamTest::testRead32()
{
  EBOOKMemoryStream stream(TEST_DATA, sizeof TEST_DATA);
  {
    EBOOKBitStream bitStream(&stream);

    CPPUNIT_ASSERT(!bitStream.atEOS());

    CPPUNIT_ASSERT_EQUAL(0xe3388ee3u, bitStream.read(32));

    CPPUNIT_ASSERT(bitStream.atEOS());
  }

  stream.seek(0, WPX_SEEK_SET);

  {
    EBOOKBitStream bitStream(&stream);

    CPPUNIT_ASSERT(!bitStream.atEOS());

    CPPUNIT_ASSERT_EQUAL(0xe38e38e3u, bitStream.read(32, true));

    CPPUNIT_ASSERT(bitStream.atEOS());
  }
}

void EBOOKBitStreamTest::testReadVaryingSmall()
{
  testReadVaryingSmall(false);
  testReadVaryingSmall(true);
}

void EBOOKBitStreamTest::testReadVaryingBig()
{
  EBOOKMemoryStream stream(TEST_DATA, sizeof TEST_DATA);

  {
    EBOOKBitStream bitStream(&stream);

    CPPUNIT_ASSERT(!bitStream.atEOS());

    CPPUNIT_ASSERT_EQUAL(0x2e3u, bitStream.read(10));
    CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(2));
    CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1));
    CPPUNIT_ASSERT_EQUAL(0x1c7u, bitStream.read(12));
    CPPUNIT_ASSERT_EQUAL(0x31u, bitStream.read(6));
    CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1));

    CPPUNIT_ASSERT(bitStream.atEOS());
  }

  stream.seek(0, WPX_SEEK_SET);

  {
    EBOOKBitStream bitStream(&stream);

    CPPUNIT_ASSERT(!bitStream.atEOS());

    CPPUNIT_ASSERT_EQUAL(0x38eu, bitStream.read(10, true));
    CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(2, true));
    CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, true));
    CPPUNIT_ASSERT_EQUAL(0xc71u, bitStream.read(12, true));
    CPPUNIT_ASSERT_EQUAL(0x31u, bitStream.read(6, true));
    CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, true));

    CPPUNIT_ASSERT(bitStream.atEOS());
  }
}

void EBOOKBitStreamTest::testRead1(const bool bigEndian)
{
  EBOOKMemoryStream stream(TEST_DATA, sizeof TEST_DATA);
  EBOOKBitStream bitStream(&stream);

  CPPUNIT_ASSERT(!bitStream.atEOS());

  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));

  CPPUNIT_ASSERT(bitStream.atEOS());
}

void EBOOKBitStreamTest::testRead3(const bool bigEndian)
{
  EBOOKMemoryStream stream(TEST_DATA, sizeof TEST_DATA);
  EBOOKBitStream bitStream(&stream);

  CPPUNIT_ASSERT(!bitStream.atEOS());

  CPPUNIT_ASSERT_EQUAL(7u, bitStream.read(3, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(3, bigEndian));
  CPPUNIT_ASSERT_EQUAL(7u, bitStream.read(3, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(3, bigEndian));
  CPPUNIT_ASSERT_EQUAL(7u, bitStream.read(3, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(3, bigEndian));
  CPPUNIT_ASSERT_EQUAL(7u, bitStream.read(3, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(3, bigEndian));
  CPPUNIT_ASSERT_EQUAL(7u, bitStream.read(3, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(3, bigEndian));
  // read the remaining bits
  CPPUNIT_ASSERT_EQUAL(3u, bitStream.read(2, bigEndian));

  CPPUNIT_ASSERT(bitStream.atEOS());
}

void EBOOKBitStreamTest::testRead4(const bool bigEndian)
{
  EBOOKMemoryStream stream(TEST_DATA, sizeof TEST_DATA);
  EBOOKBitStream bitStream(&stream);

  CPPUNIT_ASSERT(!bitStream.atEOS());

  CPPUNIT_ASSERT_EQUAL(0xeu, bitStream.read(4, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0x3u, bitStream.read(4, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0x8u, bitStream.read(4, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0xeu, bitStream.read(4, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0x3u, bitStream.read(4, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0x8u, bitStream.read(4, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0xeu, bitStream.read(4, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0x3u, bitStream.read(4, bigEndian));

  CPPUNIT_ASSERT(bitStream.atEOS());
}

void EBOOKBitStreamTest::testRead8(const bool bigEndian)
{
  EBOOKMemoryStream stream(TEST_DATA, sizeof TEST_DATA);
  EBOOKBitStream bitStream(&stream);

  CPPUNIT_ASSERT(!bitStream.atEOS());

  CPPUNIT_ASSERT_EQUAL(0xe3u, bitStream.read(8, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0x8eu, bitStream.read(8, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0x38u, bitStream.read(8, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0xe3u, bitStream.read(8, bigEndian));

  CPPUNIT_ASSERT(bitStream.atEOS());
}

void EBOOKBitStreamTest::testReadVaryingSmall(const bool bigEndian)
{
  EBOOKMemoryStream stream(TEST_DATA, sizeof TEST_DATA);
  EBOOKBitStream bitStream(&stream);

  CPPUNIT_ASSERT(!bitStream.atEOS());

  CPPUNIT_ASSERT_EQUAL(1u, bitStream.read(1, bigEndian));
  CPPUNIT_ASSERT_EQUAL(3u, bitStream.read(2, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0u, bitStream.read(3, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0xeu, bitStream.read(4, bigEndian));
  CPPUNIT_ASSERT_EQUAL(7u, bitStream.read(5, bigEndian));
  CPPUNIT_ASSERT_EQUAL(7u, bitStream.read(6, bigEndian));
  CPPUNIT_ASSERT_EQUAL(0xeu, bitStream.read(7, bigEndian));
  // read the remaining bits
  CPPUNIT_ASSERT_EQUAL(3u, bitStream.read(4, bigEndian));

  CPPUNIT_ASSERT(bitStream.atEOS());
}


void EBOOKBitStreamTest::testAtLastByte()
{
  EBOOKMemoryStream stream(TEST_DATA, sizeof TEST_DATA);
  EBOOKBitStream bitStream(&stream);

  CPPUNIT_ASSERT(!bitStream.atLastByte());
  bitStream.read(23);
  CPPUNIT_ASSERT(!bitStream.atLastByte());
  bitStream.read(1);
  CPPUNIT_ASSERT(bitStream.atLastByte());
  CPPUNIT_ASSERT(!bitStream.atEOS());
  bitStream.read(1);
  CPPUNIT_ASSERT(bitStream.atLastByte());
  CPPUNIT_ASSERT(!bitStream.atEOS());
  bitStream.read(7);
  CPPUNIT_ASSERT(bitStream.atLastByte());
  CPPUNIT_ASSERT(bitStream.atEOS());
}

CPPUNIT_TEST_SUITE_REGISTRATION(EBOOKBitStreamTest);

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
