/*
 * Copyright (C) 2011-13 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.    See the GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program.    If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VanDine <ken.vandine@canonical.com>
 */

using Dee;
using Gee;
using Config;
using Unity;
using Friends;

namespace UnityFriends {

    /* DBus name for the place. Must match out .place file */
    const string BUS_NAME = "com.canonical.Unity.Scope.Friends";
    const string ICON_PATH = Config.DATADIR + "/icons/unity-icon-theme/places/svg/";
    const string STRIP_LINKS = """</?a[^>]*>""";

    /**
     * The Scope class implements all of the logic for the place.
     *
     */
    public class FriendsScope : Unity.AbstractScope
    {
        private Unity.FilterSet? filters;

        private Dee.Model? _model = null;
        private Dee.SharedModel? _streams_model = null;
        private Dee.Filter _sort_filter;
        /* Keep track of the previous search, so we can determine when to
         * filter down the result set instead of rebuilding it */
        private unowned Dee.ModelIter _stream_iter_first = null;
        private unowned Dee.ModelIter _stream_iter_last = null;

        private Dee.Analyzer _analyzer;
        private Dee.Index _index;
        private Dee.ICUTermFilter _ascii_filter;
        private Ag.Manager _account_manager;
        private bool _has_accounts = false;
        private HashMap<string,Variant> featureMap;

        public FriendsScope ()
        {
            featureMap = new HashMap<string,Variant> ();

            // Check for accounts
            _account_manager = new Ag.Manager.for_service_type ("microblogging");
            map_account_features ();

            _account_manager.enabled_event.connect ((id) =>
            {
                GLib.List<Ag.AccountService> accts = _account_manager.get_enabled_account_services ();
                if (accts.length () > 0 && !_has_accounts)
                {
                    _has_accounts = true;
                    setup_friends ();
                }
                else if (accts.length () == 0)
                {
                    _has_accounts = false;
                }
                else
                {
                    map_account_features ();
                }
            });
        }

        public override string get_group_name ()
        {
            return BUS_NAME;
        }

        public override string get_unique_name ()
        {
            return "/com/canonical/unity/scope/friends";
        }

        private void map_account_features ()
        {
            GLib.List<Ag.AccountService> accts = _account_manager.get_enabled_account_services ();
            foreach (Ag.AccountService account_service in accts) {
                Ag.Account account = account_service.get_account ();
                account.set_enabled (false);
                var _provider = account.get_provider_name ();
                if (!(featureMap.has_key(_provider)))
                {
                    var d = new Friends.Dispatcher ();
                    var builder = new VariantBuilder (VariantType.STRING_ARRAY);
                    foreach (var _feature in d.features (_provider))
                    {
                        builder.add ("s", _feature);
                    }
                    featureMap[_provider] = builder.end ();
                }
            }

            // We only want to trigger starting friends-dispatcher if there are accounts
            if (accts.length() > 0)
            {
                _has_accounts = true;
                setup_friends ();
            }
        }

        private void setup_friends ()
        {
            Intl.setlocale(LocaleCategory.COLLATE, "C");

            _streams_model = new Dee.SharedModel ("com.canonical.Friends.Streams");

            if (_streams_model is Dee.SharedModel)
            {
                    _streams_model.notify["synchronized"].connect (on_synchronized);
                    _streams_model.row_added.connect ((source) => {
                        results_invalidated (SearchType.DEFAULT);
                    });
                    _streams_model.row_removed.connect ((source) => {
                        results_invalidated (SearchType.DEFAULT);
                    });
                    _streams_model.row_changed.connect ((source) => {
                        results_invalidated (SearchType.DEFAULT);
                    });
            }
        }

        private void on_synchronized ()
        {
            debug ("on_synchronized");

            if (!_streams_model.is_synchronized ())
                return;

            debug ("%u ROWS", _streams_model.get_n_rows ());
            _sort_filter = Dee.Filter.new_collator_desc (StreamModelColumn.TIMESTAMP);
            _model = new Dee.FilterModel (_streams_model, _sort_filter);

            _ascii_filter = new Dee.ICUTermFilter.ascii_folder ();
            _analyzer = new Dee.TextAnalyzer ();
            _analyzer.add_term_filter ((terms_in, terms_out) =>
            {
                for (uint i = 0; i < terms_in.num_terms (); i++)
                {
                    unowned string term = terms_in.get_term (i);
                    var folded = _ascii_filter.apply (term);
                    terms_out.add_term (term);
                    if (folded != term) terms_out.add_term (folded);
                }
            });
            var reader = Dee.ModelReader.new ((model, iter) =>
            {
                var sender_col = StreamModelColumn.SENDER;
                var msg_col = StreamModelColumn.MESSAGE;
                return "%s\n%s".printf (model.get_string (iter, sender_col), 
                                        model.get_string (iter, msg_col));
            });
            _index = new Dee.TreeIndex (_model, _analyzer, reader);
        }

        public override Unity.FilterSet get_filters ()
        {
            if (filters != null)
                return filters;

            filters = new Unity.FilterSet ();
            /* Stream filter */
            {
                var filter = new CheckOptionFilter ("stream", _("Stream"));

                filter.add_option ("messages", _("Messages"));
                filter.add_option ("mentions", _("Mentions"));
                filter.add_option ("private", _("Private"));
                /* FIXME: Hide the filters for now, the streams don't exist
                filter.add_option ("images", _("Images"));
                filter.add_option ("videos", _("Videos"));
                filter.add_option ("links", _("Links"));
                filter.add_option ("public", _("Public"));
                */

                filters.add (filter);
            }

            /* Account filter */
            {
                var filter = create_account_filter ();
                filters.add (filter);
            }

            _account_manager.account_created.connect ((id) => {
                debug ("ACCOUNT ADDED");
                var _acct = _account_manager.get_account ((Ag.AccountId)id);
                var filter = filters.get_filter_by_id ("account_id") as CheckOptionFilter;
                create_account_filter_option (filter, _acct);
                map_account_features ();
            });

            /* FIXME: we need a way to remove an option or remove and re-add the
             * filter on account deletion */
            return filters;
        }

        private Unity.CheckOptionFilter create_account_filter ()
        {
            var filter = new CheckOptionFilter ("account_id", _("Account"));
            var accts = _account_manager.get_enabled_account_services ();
            if (accts.length () > 0)
            {
                foreach (var _acctservice in accts)
                {
                    // create a service icon, which isn't used yet as of libunity 4.24
                    // when it is used, we can drop _acct.service from the option name 
                    // and rely on the service icon and username to disambiguate accounts 
                    var _acct = _acctservice.get_account ();
                    create_account_filter_option (filter, _acct);
                }
            }
            return filter;
        }

        private void create_account_filter_option (CheckOptionFilter filter, Ag.Account _acct)
        {
            GLib.Icon icon;
            GLib.File icon_file;
            var _provider = _acct.get_provider_name ();
            var _name = _acct.get_display_name ();
            icon_file = GLib.File.new_for_path (GLib.Path.build_filename (Config.PKGDATADIR + "/plugins/" + _provider + "/ui/icons/16x16/" + _provider + ".png"));
            icon = new GLib.FileIcon (icon_file);

            filter.add_option (_acct.id.to_string(), _provider + "/" + _name, icon);
        }

        public override Unity.CategorySet get_categories ()
        {
            var categories = new Unity.CategorySet ();
            Icon icon;

            var icon_dir = File.new_for_path ("/usr/share/unity-lens-friends/ui/icons/hicolor/scalable/places/");

            icon = new FileIcon (icon_dir.get_child ("group-messages.svg"));
            var cat = new Unity.Category ("messages", _("Messages"), icon);
            cat.content_type = CategoryContentType.SOCIAL;
            categories.add (cat);

            icon = new FileIcon (icon_dir.get_child ("group-replies.svg"));
            cat =    new Unity.Category ("mentions", _("Mentions"), icon);
            cat.content_type = CategoryContentType.SOCIAL;
            categories.add (cat);

            icon = new FileIcon (icon_dir.get_child ("group-images.svg"));
            cat =    new Unity.Category ("images", _("Images"), icon);
            categories.add (cat);

            icon = new FileIcon (icon_dir.get_child ("group-videos.svg"));
            cat =    new Unity.Category ("videos", _("Videos"), icon);
            categories.add (cat);

            icon = new FileIcon (icon_dir.get_child ("group-links.svg"));
            cat =    new Unity.Category ("links", _("Links"), icon);
            cat.content_type = CategoryContentType.SOCIAL;
            categories.add (cat);

            icon = new FileIcon (icon_dir.get_child ("group-private.svg"));
            cat =    new Unity.Category ("private", _("Private"), icon);
            cat.content_type = CategoryContentType.SOCIAL;
            categories.add (cat);

            icon = new FileIcon (icon_dir.get_child ("group-public.svg"));
            cat =    new Unity.Category ("public", _("Public"), icon);
            cat.content_type = CategoryContentType.SOCIAL;
            categories.add (cat);

            return categories;
        }

        public override Unity.Schema get_schema ()
        {
            var schema = new Unity.Schema ();
            schema.add_field ("account_id", "t", Unity.Schema.FieldType.REQUIRED);
            schema.add_field ("status_id", "s", Unity.Schema.FieldType.REQUIRED);
            return schema;
        }

        public override string normalize_search_query (string search_query)
        {
            return search_query.strip ();
        }

        public override Unity.ScopeSearchBase create_search_for_query (Unity.SearchContext search_context)
        {
            return new FriendsSearch (this, search_context);
        }

        public void perform_search (Unity.SearchContext context)
        {
            unowned Dee.ModelIter iter, end;

            var stream_ids = new Gee.ArrayList<string> ();
            var filter = context.filter_state.get_filter_by_id ("stream") as CheckOptionFilter;
            if (filter.filtering)
            {
                foreach (Unity.FilterOption option in filter.options)
                {
                    if (option.active)
                    {
                        stream_ids.add (option.id);
                    }
                }
            }

            var account_ids = new Gee.ArrayList<string> ();
            filter = context.filter_state.get_filter_by_id ("account_id") as CheckOptionFilter;
            if (filter.filtering)
            {
                foreach (Unity.FilterOption option in filter.options)
                {
                    if (option.active)
                    {
                        account_ids.add (option.id);
                    }
                }
            }

            if (_model == null)
                setup_friends ();

            iter = _model.get_first_iter ();
            end = _model.get_last_iter ();

            _stream_iter_first = _model.get_first_iter ();
            _stream_iter_last = end;
         
            var term_list = Object.new (typeof (Dee.TermList)) as Dee.TermList;
            // search only the folded terms, FIXME: is that a good idea?
            _analyzer.tokenize (_ascii_filter.apply (context.search_query), term_list);

            var matches = new Sequence<Dee.ModelIter> ();
            for (uint i = 0; i < term_list.num_terms (); i++)
            {
                // FIXME: use PREFIX search only for the last term?
                var result_set = _index.lookup (term_list.get_term (i),
                                                Dee.TermMatchFlag.PREFIX);
                bool first_pass = i == 0;
                CompareDataFunc<Dee.ModelIter> cmp_func = (a, b) =>
                {
                    return a == b ? 0 : ((void*) a > (void*) b ? 1 : -1);
                };
                // intersect the results (cause we want to AND the terms)
                var remaining = new Sequence<Dee.ModelIter> ();
                foreach (var item in result_set)
                {
                    if (first_pass)
                        matches.insert_sorted (item, cmp_func);
                    else if (matches.lookup (item, cmp_func) != null)
                        remaining.insert_sorted (item, cmp_func);
                }
                if (!first_pass) matches = (owned) remaining;
                // final result set empty already?
                if (matches.get_begin_iter () == matches.get_end_iter ()) break;
            }

            matches.sort ((a, b) =>
            {
                var col = StreamModelColumn.TIMESTAMP;
                return _model.get_string (b, col).collate (_model.get_string (a, col));
            });

            var match_iter = matches.get_begin_iter ();
            var match_end_iter = matches.get_end_iter ();
            while (match_iter != match_end_iter)
            {
                iter = match_iter.get ();

                if (matches_filters (_model, iter, stream_ids, account_ids))
                {
                    add_result (_model, iter, context.result_set);
                }

                match_iter = match_iter.next ();
            }

            if (term_list.num_terms () > 0) return;

            /* Go over the whole model if we had empty search */
            while (iter != end)
            {
                if (matches_filters (_model, iter, stream_ids, account_ids))
                {
                    add_result (_model, iter, context.result_set);
                }
                iter = _model.next (iter);
            }

            //debug ("Results has %u rows", results_model.get_n_rows());
        }

        private bool matches_filters (Dee.Model model, Dee.ModelIter iter,
                                      Gee.List<string> stream_ids,
                                      Gee.List<string> account_ids)
        {
            bool stream_match = true;
            bool account_match = true;
            if (stream_ids.size > 0)
            {
                stream_match = model.get_string (iter, StreamModelColumn.STREAM) in stream_ids;
            }

            if (account_ids.size > 0)
            {
                var _account = (uint)model.get_uint64 (iter, StreamModelColumn.ACCOUNT_ID);
                if (!(_account.to_string() in account_ids))
                    account_match = false;
            }

            var _stream = model.get_string (iter, StreamModelColumn.STREAM);
            if ("reply_to/" in _stream)
                stream_match = false;

            return account_match && stream_match;
        }

        private void add_result (Dee.Model model, Dee.ModelIter iter, Unity.ResultSet result_set)
        {
            var result = Unity.ScopeResult ();

            result.uri = model.get_string (iter, StreamModelColumn.URL);
            unowned string stream_id =
                model.get_string (iter, StreamModelColumn.STREAM);
            string _icon_uri = model.get_string (iter, StreamModelColumn.ICON_URI);
            if (stream_id == "images")
                result.icon_hint = model.get_string (iter, StreamModelColumn.LINK_PICTURE);
            else if (stream_id == "videos")
            {
                result.icon_hint = model.get_string (iter, StreamModelColumn.LINK_PICTURE);
                if (result.icon_hint.length < 1)
                    result.icon_hint = get_avatar_path (_icon_uri);
            }
            else
                result.icon_hint = get_avatar_path (_icon_uri);

            result.category = Categories.MESSAGES;
            switch (stream_id)
            {
                case "messages": result.category = Categories.MESSAGES; break;
                case "mentions":
                    if (model.get_bool (iter, StreamModelColumn.FROM_ME))
                        result.category = Categories.MESSAGES;
                    else
                        result.category = Categories.REPLIES;
                    break;
                case "images": result.category = Categories.IMAGES; break;
                case "videos": result.category = Categories.VIDEOS; break;
                case "links": result.category = Categories.LINKS; break;
                case "private": result.category = Categories.PRIVATE; break;
                case "public": result.category = Categories.PUBLIC; break;
            }
            result.result_type = ResultType.PERSONAL;
            result.mimetype = "text/html";
            result.title = GLib.Markup.escape_text (_model.get_string(iter, StreamModelColumn.SENDER));
            result.comment = sanitize_message (_model.get_string(iter, StreamModelColumn.MESSAGE));
            result.dnd_uri = result.uri;
            result.metadata = new HashTable<string,Variant> (str_hash, str_equal);
            result.metadata["account_id"] = new Variant.uint64 (model.get_uint64(iter, StreamModelColumn.ACCOUNT_ID));
            result.metadata["status_id"] = new Variant.string (model.get_string (iter, StreamModelColumn.MESSAGE_ID));

            result_set.add_result (result);
        }

        private string get_avatar_path (string uri)
        {
            var _avatar_cache_image = uri;
            /* FIXME
            var _avatar_cache_image = avatar_path (uri);
            if (_avatar_cache_image == null)
            {
                try
                {
                    _avatar_cache_image = avatar_path (uri);
                } catch (GLib.Error e)
                {
                }
                if (_avatar_cache_image == null)
                    _avatar_cache_image = uri;
            }
            */

            return _avatar_cache_image;
        }

        public override Unity.ResultPreviewer create_previewer (Unity.ScopeResult result, Unity.SearchMetadata metadata)
        {
            return new FriendsResultPreviewer (this, result, metadata);
        }

        public Unity.Preview make_preview (string uri)
        {
            debug ("Previewing: %s", uri);
            Unity.SocialPreview preview = null;
            var utils = new Friends.Utils ();
            var icon_dir = File.new_for_path (ICON_PATH);
            Icon icon;
            string retweet_str, like_str, _img_uri = null;

            if (_streams_model == null)
                setup_friends ();

            var model = _streams_model;
            unowned Dee.ModelIter iter, end;
            iter = model.get_first_iter ();
            end = model.get_last_iter ();

            while (iter != end)
            {
                var url = model.get_string (iter, StreamModelColumn.URL);
                if (url == uri)
                {
                    debug ("Found %s", url);
                    var icon_uri = model.get_string (iter, StreamModelColumn.ICON_URI);
                    if ("_normal." in icon_uri)
                    {
                        //If it looks like a twitter avatar, try to get the original
                        var ss = icon_uri.split ("_normal.");
                        icon_uri = ss[0] + "." + ss[1];
                    }
                    var avatar = Icon.new_for_string (icon_uri);

                    //var avatar = new FileIcon (File.new_for_path (get_avatar_path (icon_uri)));
                    var likes = model.get_uint64 (iter, StreamModelColumn.LIKES);
                    var sender = model.get_string (iter, StreamModelColumn.SENDER);
                    var timestring = utils.create_time_string (model.get_string (iter, StreamModelColumn.TIMESTAMP));
                    var content = sanitize_message (_model.get_string(iter, StreamModelColumn.MESSAGE));
                    string title = timestring;
                    preview = new Unity.SocialPreview (sender, title, content, avatar);
                    // work around bug 1058198
                    preview.title = sender;
                    preview.subtitle = title;
                    // end work around
                    preview.add_info (new InfoHint ("likes", _("Favorites"), null, likes.to_string()));
                    var stream_id = model.get_string (iter, StreamModelColumn.STREAM);
                    if (stream_id == "images")
                        _img_uri = model.get_string (iter, StreamModelColumn.LINK_PICTURE);
                    else if (stream_id == "videos" && _img_uri != null && _img_uri.length < 1)
                        _img_uri = model.get_string (iter, StreamModelColumn.LINK_PICTURE);
                    if (_img_uri != null && _img_uri.length > 0)
                    {
                        Icon img = Icon.new_for_string (_img_uri);
                        preview.image = img;
                    }
                    break;
                }
                iter = model.next (iter);
            }

            uint _account_id = (uint)model.get_uint64(iter, StreamModelColumn.ACCOUNT_ID);
            debug ("account_id: %u", _account_id);
            var _account_service = model.get_string (iter, StreamModelColumn.PROTOCOL);
            var _status_id = model.get_string (iter, StreamModelColumn.MESSAGE_ID);
            var icon_str = icon_dir.get_child ("service-" + _account_service + ".svg");
            if (icon_str.query_exists ())
                icon = new FileIcon (icon_str);
            else
                icon = null;

            string[] _features = null;
            if (featureMap.has_key(_account_service))
            {
                _features = featureMap[_account_service].dup_strv ();
            }

            parse_comments (preview, _status_id);

            var view_action = new Unity.PreviewAction ("view", _("View"), icon);
            preview.add_action (view_action);

            if ("retweet" in _features)
            {
                if (_account_service == "twitter")
                    retweet_str = _("Retweet");
                else 
                    retweet_str = _("Share");
                var retweet_action = new Unity.PreviewAction ("retweet", retweet_str, icon);
                preview.add_action (retweet_action);
            }
            
            bool from_me = model.get_bool (iter, StreamModelColumn.FROM_ME);
            // REPLY
            //if ("reply" in _features && !from_me)
            //{
            //    var reply_action = new Unity.PreviewAction ("reply", _("Reply"), icon);
            //    preview.add_action (reply_action);
            //    reply_action.activated.connect ((source) => {
            //        return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);
            //    });
            //}

            bool liked = model.get_bool (iter, StreamModelColumn.LIKED);
            if ("like" in _features && "unlike" in _features && !from_me)
            {
                like_str = liked ? _("Unlike") : _("Like");
                var like_action = new Unity.PreviewAction ("like", like_str, icon);
                preview.add_action (like_action);
            }

            return preview;
        }

        private string sanitize_message (string msg)
        {
            var remove_links_regex = new Regex (STRIP_LINKS, RegexCompileFlags.CASELESS);
            string _msg_cleaned = remove_links_regex.replace (msg, msg.length, 0, "");
            return _msg_cleaned;
        }

        private void parse_comments (Unity.SocialPreview preview, string _message_id)
        {
            if (_message_id != null)
            {
                var utils = new Friends.Utils ();
                unowned Dee.ModelIter iter, end;
                iter = _model.get_first_iter ();
                end = _model.get_last_iter ();

                while (iter != end)
                {
                    var _stream = _model.get_string (iter, StreamModelColumn.STREAM).split ("/")[1];
                    if (_stream == _message_id)
                    {
                        debug ("stream: %s", _stream);
                        var text = sanitize_message (_model.get_string (iter, StreamModelColumn.MESSAGE));
                        var name = GLib.Markup.escape_text (_model.get_string (iter, StreamModelColumn.SENDER));
                        var time_string = utils.create_time_string (_model.get_string (iter, StreamModelColumn.TIMESTAMP));
                        preview.add_comment (new Unity.SocialPreview.Comment (name, name, text, time_string));
                    }
                    iter = _model.next (iter);
                }
            }
        }

        private string? avatar_path(string url)
        {
            string _avatar_cache_image = Path.build_path (Path.DIR_SEPARATOR_S, Environment.get_user_cache_dir(), "friends/avatars", GLib.Checksum.compute_for_string (GLib.ChecksumType.SHA1, url));
            var file = File.new_for_path (_avatar_cache_image);
            if (file.query_exists ())
                return _avatar_cache_image;
            else
                return null;
        }

        public override Unity.ActivationResponse? activate (Unity.ScopeResult result, Unity.SearchMetadata metadata, string? action_id)
        {
            uint _account_id = (uint)result.metadata["account_id"].get_uint64();
            var _status_id = result.metadata["status_id"].get_string();
            if (action_id == "retweet")
            {
                Idle.add (() => {
                    var dispatcher = new Friends.Dispatcher ();
                    dispatcher.retweet (_account_id, _status_id);
                    return false;
                });
                return new Unity.ActivationResponse (Unity.HandledType.SHOW_PREVIEW);
            }
            else if (action_id == "like")
            {
                var old_liked = toggle_liked (result.uri);
                Idle.add (() => {
                    var dispatcher = new Friends.Dispatcher ();
                    if (old_liked)
                    {
                        dispatcher.unlike (_account_id, _status_id);
                    }
                    else
                    {
                        dispatcher.like (_account_id, _status_id);
                    }
                    return false;
                });
                var new_preview = make_preview (result.uri);
                return new Unity.ActivationResponse.with_preview (new_preview);
            }
            else
            {
                // Default action, or "view"
                return new Unity.ActivationResponse (Unity.HandledType.NOT_HANDLED);
            }
        }

        public bool toggle_liked (string uri)
        {
            var model = _streams_model;
            var iter = model.get_first_iter ();
            var end = model.get_last_iter ();
            while (iter != end)
            {
                var row_uri = model.get_string (iter, StreamModelColumn.URL);
                if (row_uri == uri)
                {
                    var liked = model.get_bool (iter, StreamModelColumn.LIKED);
                    model.set_value (iter, StreamModelColumn.LIKED, !liked);
                    return liked;
                }
                iter = model.next (iter);
            }
            return false;
        }
    }

    public class FriendsSearch : Unity.ScopeSearchBase
    {
        private FriendsScope scope;

        public FriendsSearch(FriendsScope scope, Unity.SearchContext search_context)
        {
            this.scope = scope;
            this.search_context = search_context;
        }

        public override void run ()
        {
            // FIXME: no results for home screen of the dash?
            var context = this.search_context;
            if (context.search_type == SearchType.GLOBAL &&
                context.search_query.strip () == "")
            {
                return;
            }

            this.scope.perform_search (context);
        }

        public override void run_async (Unity.ScopeSearchBaseCallback callback)
        {
            // Run the search in the main thread, to avoid
            // synchronisation problems.
            run ();
            callback (this);
        }
    }

    public class FriendsResultPreviewer : Unity.ResultPreviewer
    {
        private FriendsScope scope;

        public FriendsResultPreviewer (FriendsScope scope, Unity.ScopeResult result, Unity.SearchMetadata metadata)
        {
            this.scope = scope;
            set_scope_result (result);
            set_search_metadata (metadata);
        }

        public override Unity.AbstractPreview? run ()
        {
            return scope.make_preview (result.uri);
        }

        public override void run_async (Unity.AbstractPreviewCallback callback)
        {
            // Build preview in the main thread, to avoid
            // synchronisation problems.
            var preview = run ();
            callback (this, preview);
        }
    }
} /* end Friends namespace */
