------------------------------------------------------------------------------
--                                                                          --
--                            GNATPP COMPONENTS                             --
--                                                                          --
--            G N A T P P . S O U R C E _ L I N E _ B U F F E R             --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                   Copyright (C) 2001-2009, AdaCore                       --
--                                                                          --
-- GNATPP is free software; you can redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNATPP is  distributed in the  hope that it will  be  useful, but --
-- WITHOUT ANY WARRANTY; without even the implied warranty of  MERCHANTABI- --
-- LITY or  FITNESS  FOR A  PARTICULAR  PURPOSE. See the GNU General Public --
-- License  for more details. You  should  have  received a copy of the GNU --
-- General Public License  distributed with GNAT; see file COPYING. If not, --
-- write  to  the Free  Software  Foundation,  59 Temple Place - Suite 330, --
-- Boston,                                                                  --
--                                                                          --
-- GNATPP is maintained by ACT Europe (http://www.act-europe.fr).           --
--                                                                          --
------------------------------------------------------------------------------

--  This package defines the buffer into which the string image of the source
--  line of the original text is placed during traversing the source. The main
--  idea of pretty-printing the Ada code is to traverse the abstract Ada
--  structure of the code in parallel with scanning the corresponding source
--  line.

with Asis;          use Asis;
with Asis.Text;     use Asis.Text;

with GNATPP.Common; use GNATPP.Common;

package GNATPP.Source_Line_Buffer is

   -----------------
   -- Line Buffer --
   -----------------

   --  The simple fixed-length buffer abstraction is used at the moment
   --  May be, we should implement this buffer on top of the GNAT Table
   --  abstraction???

   Max_Line_Buf_Len : constant Positive := 1024;
   --  Should be enough.

   Line_Buf : Program_Text (1 .. Max_Line_Buf_Len);

   Line_Len : Natural;
   --  The length of the line currently placed into the buffer (may be zero)

   Line_Pos : Natural;
   --  Pointer to the position where we are at the given stage of the source
   --  traversal process. Is set to zero in two (???) cases:
   --  (1) When the corresponding line in the source is empty;
   --  (2) If line traversal routine has reached the end of the source line

   HT_Passed : Boolean := False;
   --  This flag is set by Skip_Blank routines. It is set on if at least one
   --  HT character is passed when skipping the sequence of white spaces.

   ------------------------------
   -- End of the revised stuff --
   ------------------------------

   Word_End : Natural;
   --  Supposing that Line_Pos points to some non-space symbol, this variable
   --  may be set by Set_Word_End procedure to the end of the word pointed by
   --  Line_Pos. ("Words" are any character sequences separated by white
   --  spaces, the end of the buffer and beginning of the buffer. If Line_Pos
   --  is equal to 0, Set_Word_End also sets Word_End to 0.

   Keyword : Keyword_Kinds;
   --  The keyword in the source buffer pointer by Line_Pos. Set to Not_A_KW if
   --  the next word in the buffer is not a keyword

   Delimiter : Delimiter_Kinds;
   --  The delimiter in the source buffer pointer by Line_Pos. Set to
   --  Not_A_Dlm if the next word in the buffer is not a delimiter

   procedure In_Buffer (Str_Num : Line_Number_Positive);
   --  Places the argument line into the buffer, sets Line_Len to Str'Length
   --  and Line_Pos to 1. The first procedure also sets
   --  GNATPP.State.Current_Source_Line to Str_Num
   --
   --  Note, that this procedure is not supposed to be called for the empty
   --  arguments, so the buffer is always non-empty???

   procedure Next_Line_In_Buffer;
   --  Places the next line in the buffer. The caller is responsible for
   --  making sure that this procedure will not go beyond the last line of
   --  the source being processed.

   function End_Of_Line_Comment return Boolean;
   --  Checks if the substring between Line_Pos and Line_Len contains the
   --  end-of-line comment only. If it is, as a side-effect this function sets
   --  Line_Pos pointing to the first '-' of this comment. Otherwise it leaves
   --  Line_Pos unchanged

   procedure Skip_Blanks;
   --  If Line_Pos points to non-space character (HT is considered as space),
   --  does nothing. Else moves Line_Pos to the right till the first non-space
   --  character is found or the end of the line is reached. In the latter case
   --  Line_Pos is set to 0.

   procedure Skip_Word;
   --  Skips the next word in the line and sets Line_Pos to the first
   --  character of the next word. If there are no word left, sets Line_Pos
   --  to 0.

   function Get_End_Idx return Natural;
   --  Returns the index of the last non-blank character in of the line stored
   --  in the line buffer. Returns zero if the line in the line buffer is the
   --  blank line.

   function Is_Blank_Line return Boolean;
   --  Checks if the part of the line in the Buffer starting from Line_Pos is
   --  empty or contains only space and HT characters

   procedure Set_Word_End;
   --  Providing that Line_Pos points to the beginning of some word in the
   --  buffer, sets the Word_End variable to the end of this word. If Line_Pos
   --  is zero, Word_End is also set to zero. Here we consider that the word
   --  has the syntax of Ada identifier.

   procedure Set_Word_End_In_Comment;
   --  Similar to the previous procedure, but this procedure does not consider
   --  '--' as the end of the word. That is, only white spaces are considered
   --  as word separators.

   function Normalized_Word return Program_Text;
   --  This function supposes that Line_Pos and Word_End point to the beginning
   --  and to the end of some word in the buffer. If this word consists only of
   --  characters which belong to the first 256 positions of Wide_Character
   --  type, it returns the this word converted to lover case characters.
   --  Otherwise it returns this word as is.

   procedure Detect_Keyword;
   --  Set the value of the Keyword variable. This procedure is supposed to
   --  be called when the buffer contains some non-space symbols after Line_Pos

   procedure Detect_Delimiter;
   --  Set the value of the Delimiter variable. It is supposed that Line_Pos
   --  points to the (first character of the) delimiter

   procedure Save_Buf;
   procedure Restore_Buf;
   --  These routines save and restore the content of the line buffer and the
   --  settiing of Line_Pos

end GNATPP.Source_Line_Buffer;
