//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Beam/SourceItems.h
//! @brief     Defines BeamItem hierarchy
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_BEAM_SOURCEITEMS_H
#define BORNAGAIN_GUI_MODEL_BEAM_SOURCEITEMS_H

#include "GUI/Model/Beam/FootprintItemCatalog.h"
#include "GUI/Model/Descriptor/DoubleProperty.h"
#include "GUI/Model/Descriptor/SelectionProperty.h"
#include <functional>
#include <heinz/Vectors3D.h>

class BasicAxisItem;
class Beam;
class BeamAzimuthalAngleItem;
class BeamDistributionItem;
class BeamWavelengthItem;
class FootprintItem;
class GrazingScanItem;
class IBeamScan;
class ICoordSystem;
class IFootprint;
class Scale;

//! Base class for BeamItem and ScanItem. Name refers to radiation source.
class SourceItem {
public:
    DoubleProperty& intensity() { return m_intensity; }
    const DoubleProperty& intensity() const { return m_intensity; }
    void setIntensity(double v) { m_intensity.setValue(v); }

    double wavelength() const;
    void setWavelength(double value);
    BeamWavelengthItem* wavelengthItem() const;

    double azimuthalAngle() const;
    void setAzimuthalAngle(double value);
    BeamAzimuthalAngleItem* azimuthalAngleItem() const;

    bool isExpandBeamParameters() const { return m_expandBeamParameters; }
    void setExpandBeamParameters(bool b) { m_expandBeamParameters = b; }

    SelectionProperty<FootprintItemCatalog>& footprintSelection() { return m_footprint; }

    void setGaussianFootprint(double value);
    void setSquareFootprint(double value);

    void setFootprint(const IFootprint* footprint);

    bool isExpandFootprint() const { return m_expandFootprint; }
    void setExpandFootprint(bool b) { m_expandFootprint = b; }

protected:
    SourceItem();
    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

    std::unique_ptr<BeamWavelengthItem> m_wavelengthItem;
    std::unique_ptr<BeamAzimuthalAngleItem> m_azimuthalAngleItem;

    DoubleProperty m_intensity;
    bool m_expandBeamParameters = true;

    SelectionProperty<FootprintItemCatalog> m_footprint;
    bool m_expandFootprint = true;
};

class BeamItem : public SourceItem {
public:
    BeamItem();

    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

    BeamDistributionItem* beamDistributionItem() const;
    void setInclinationAngle(double value);
    double getInclinationAngle() const;

    std::unique_ptr<Beam> createBeam() const;

private:
    std::unique_ptr<BeamDistributionItem> m_inclinationAngleItem;
};

class ScanItem : public SourceItem {
public:
    explicit ScanItem();

    void setScan(const IBeamScan* scan);

    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

    double getInclinationAngle() const;

    GrazingScanItem* grazingScanItem() const;
    BasicAxisItem* inclinationAxisItem() const;

    void updateToData(const Scale& axis, QString units, const ICoordSystem& converter);

    std::unique_ptr<Beam> createBeam() const;

private:
    std::unique_ptr<GrazingScanItem> m_grazingScanItem;
};

#endif // BORNAGAIN_GUI_MODEL_BEAM_SOURCEITEMS_H
