# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-

"""Ubuntu Touch App autopilot tests."""

import os
import shutil
import subprocess

from autopilot.input import Pointer
from autopilot.matchers import Eventually
from testtools.matchers import Equals
from ubuntuuitoolkit import base, emulators


def _get_module_include_path():
    return os.path.join(_get_path_to_source_root(), 'modules')


def _get_path_to_source_root():
    return os.path.abspath(
        os.path.join(
            os.path.dirname(__file__), '..', '..', '..'))


def _get_local_desktop_file_directory():
    return os.path.join(os.environ['HOME'], '.local', 'share', 'applications')


class ClickUpdateManagerTestCase(base.UbuntuUIToolkitAppTestCase):

    qml_file_name = 'updatemanager.qml'
    path_to_installed_app = '/usr/share/click-update-manager/'
    desktop_file_name = 'click-update-manager.desktop'
    path_to_installed_desktop_file = '/usr/share/applications/'
    local_desktop_file_path = None

    def setUp(self):
        super(ClickUpdateManagerTestCase, self).setUp()
        self.pointing_device = Pointer(self.input_device_class.create())
        self.app_qml_source_path = os.path.join(
            _get_path_to_source_root(), self.qml_file_name)
        self.test_qml_file_path = self._get_test_qml_file_path()
        self.desktop_file_path = self._get_desktop_file_path()
        self.launch_application()

    def _get_test_qml_file_path(self):
        if self._application_source_exists():
            return self.app_qml_source_path
        else:
            return os.path.join(
                self.path_to_installed_app, self.qml_file_name)

    def _application_source_exists(self):
        return os.path.exists(self.app_qml_source_path)

    def _get_desktop_file_path(self):
        if self._application_source_exists():
            local_desktop_file_dir = _get_local_desktop_file_directory()
            if not os.path.exists(local_desktop_file_dir):
                os.makedirs(local_desktop_file_dir)
            source_desktop_file_path = os.path.join(
                _get_path_to_source_root(), self.desktop_file_name)
            local_desktop_file_path = os.path.join(
                local_desktop_file_dir, self.desktop_file_name)
            shutil.copy(source_desktop_file_path, local_desktop_file_path)
            # We can't delete the desktop file before we close the application,
            # so we save it on an attribute to be deleted on tear down.
            self.local_desktop_file_path = local_desktop_file_path
            return local_desktop_file_path
        else:
            return os.path.join(
                self.path_to_installed_desktop_file, self.desktop_file_name)

    def launch_application(self):
        arch = subprocess.check_output(
            ["dpkg-architecture", "-qDEB_HOST_MULTIARCH"]).strip()
        self.app = self.launch_test_application(
            os.path.join('/usr/lib', arch, 'qt5/bin/qmlscene'),
            '-I' + _get_module_include_path(),
            self.test_qml_file_path,
            '--desktop_file_hint={0}'.format(self.desktop_file_path),
            emulator_base=emulators.UbuntuUIToolkitEmulatorBase,
            app_type='qt')

        self.assertThat(
            self.main_view.visible, Eventually(Equals(True)))

    def tearDown(self):
        super(ClickUpdateManagerTestCase, self).tearDown()
        # We can't delete the desktop file before we close the application,
        # so we save it on an attribute to be deleted on tear down.
        if self.local_desktop_file_path is not None:
            os.remove(self.local_desktop_file_path)

    @property
    def main_view(self):
        return self.app.select_single(emulators.MainView)
