/*
    SPDX-FileCopyrightText: 2005 Joris Guisson <joris.guisson@gmail.com>
    SPDX-License-Identifier: GPL-2.0-or-later
*/

#include "logviewer.h"

#include <QApplication>
#include <QBoxLayout>
#include <QIcon>
#include <QMenu>

#include <KConfig>
#include <KLocalizedString>

#include "logflags.h"
#include "logviewerpluginsettings.h"

namespace kt
{
LogViewer::LogViewer(LogFlags *flags, QWidget *parent)
    : Activity(i18n("Log"), QStringLiteral("utilities-log-viewer"), 100, parent)
    , use_rich_text(true)
    , flags(flags)
    , suspended(false)
    , menu(nullptr)
    , max_block_count(200)
{
    setToolTip(i18n("View the logging output generated by KTorrent"));
    QVBoxLayout *layout = new QVBoxLayout(this);
    output = new QTextBrowser(this);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    layout->addWidget(output);
    output->document()->setMaximumBlockCount(max_block_count);
    output->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(output, &QTextBrowser::customContextMenuRequested, this, &LogViewer::showMenu);

    suspend_action = new QAction(QIcon::fromTheme(QStringLiteral("media-playback-pause")), i18n("Suspend Output"), this);
    suspend_action->setCheckable(true);
    connect(suspend_action, &QAction::toggled, this, &LogViewer::suspend);
}

LogViewer::~LogViewer()
{
}

void LogViewer::message(const QString &line, unsigned int arg)
{
    if (suspended)
        return;

    /*
        IMPORTANT: because QTextBrowser is not thread safe, we must use the Qt event mechanism
        to add strings to it, this will ensure that strings will only be added in the main application
        thread.
    */
    if (arg == 0x00 || flags->checkFlags(arg)) {
        if (!mutex.tryLock()) // Drop the message if we cannot acquire the lock, no deadlocks
            return;

        if (use_rich_text) {
            pending.append(flags->getFormattedMessage(arg, line));
        } else {
            pending.append(line);
        }

        while (pending.size() > max_block_count)
            pending.pop_front();
        mutex.unlock();
    }
}

void LogViewer::processPending()
{
    // Copy to tmp list so that we do not get a deadlock when Qt tries to print something when we add lines to the output
    QStringList tmp;
    {
        if (!mutex.tryLock()) // No deadlocks
            return;

        tmp = pending;
        pending.clear();
        mutex.unlock();
    }

    for (const QString &line : qAsConst(tmp)) {
        QTextCharFormat fm = output->currentCharFormat();
        output->append(line);
        output->setCurrentCharFormat(fm);
    }
}

void LogViewer::setRichText(bool val)
{
    use_rich_text = val;
}

void LogViewer::setMaxBlockCount(int max)
{
    max_block_count = max;
    output->document()->setMaximumBlockCount(max);
}

void LogViewer::showMenu(const QPoint &pos)
{
    if (!menu) {
        menu = output->createStandardContextMenu();
        QAction *first = menu->actions().at(0);
        QAction *sep = menu->insertSeparator(first);
        menu->insertAction(sep, suspend_action);
    }
    menu->popup(output->viewport()->mapToGlobal(pos));
}

void LogViewer::suspend(bool on)
{
    suspended = on;
    QTextCharFormat fm = output->currentCharFormat();
    if (on)
        output->append(i18n("<font color=\"#FF0000\">Logging output suspended</font>"));
    else
        output->append(i18n("<font color=\"#00FF00\">Logging output resumed</font>"));
    output->setCurrentCharFormat(fm);
}

}
