/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.cache.jmx;

import org.jboss.cache.config.Configuration;
import org.jboss.cache.interceptors.base.CommandInterceptor;

import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import java.util.List;

/**
 * Various JMX related utilities
 *
 * @author Jerry Gauthier
 * @author Manik Surtani
 * @version $Id: JmxUtil.java 6120 2008-06-30 11:58:59Z manik.surtani@jboss.com $
 */
public class JmxUtil
{
   public static final String JBOSS_SERVER_DOMAIN = "jboss";
   public static final String JBOSS_CACHE_DOMAIN = "jboss.cache";
   public static final String SERVICE_KEY_NAME = "service";
   public static final String BASE_PREFIX = JBOSS_CACHE_DOMAIN + ":" + SERVICE_KEY_NAME + "=JBossCache";
   public static final String CLUSTER_KEY = "cluster";
   public static final String PREFIX = BASE_PREFIX + "," + CLUSTER_KEY + "=";
   public static final String UNIQUE_ID_KEY = "uniqueId";
   public static final String NO_CLUSTER_PREFIX = BASE_PREFIX + "," + UNIQUE_ID_KEY + "=";
   public static final String CACHE_TYPE_KEY = "cacheType";
   public static final String PLAIN_CACHE_TYPE = "Cache";
   public static final String MBEAN_CLASS_SUFFIX = "MBean";
   public static final String INTERCEPTOR_KEY = ",cache-interceptor=";

   public static void registerCacheMBean(MBeanServer server, CacheJmxWrapperMBean cache, String cacheObjectName)
         throws JMException
   {
      ObjectName on = new ObjectName(cacheObjectName);
      if (!server.isRegistered(on))
      {
         server.registerMBean(cache, on);
      }
   }

   /*
    * Register the associated mbeans for cache interceptors
    *
    * @param server the mbean server with which the mbeans should be registered
    * @param cache the cache having the set of interceptors to be registered
    * @param registerCache whether the cache itself should be registered
    */
   public static void registerInterceptors(MBeanServer server, List<CommandInterceptor> interceptors, String cacheObjectName)
         throws JMException
   {
      if (server == null || interceptors == null || cacheObjectName == null)
      {
         return;
      }

      for (CommandInterceptor interceptor : interceptors)
      {
         if (!interceptor.getStatisticsEnabled())
            continue;

         boolean mbeanExists = true;
         try
         {
            // the mbean for interceptor AbcInterceptor will be named AbcInterceptorMBean
            Class.forName(interceptor.getClass().getName() + MBEAN_CLASS_SUFFIX);
         }
         catch (Throwable e)
         {
            // if the class can't be instantiated, no mbean is available
            mbeanExists = false;
         }

         // for JDK 1.4, must parse name and remove package prefix
         // for JDK 1.5, can use getSimpleName() to establish class name without package prefix
         String className = interceptor.getClass().getName();
         String serviceName = cacheObjectName + INTERCEPTOR_KEY + className.substring(className.lastIndexOf('.') + 1);

         ObjectName objName = new ObjectName(serviceName);
         if (mbeanExists && !server.isRegistered(objName))
         {
            // register associated interceptor mbean
            server.registerMBean(interceptor, objName);
         }
      }
   }

   public static String getDefaultCacheObjectName(org.jboss.cache.Cache cache)
   {
      // get the cache's registration name
      return getDefaultCacheObjectName(cache.getConfiguration(), cache.getClass().getName());
   }

   public static String getDefaultCacheObjectName(Configuration config, String cacheImplClass)
   {
      // get the cache's registration name
      String tmpName;
      if (config.getClusterName() == null)
      {
         tmpName = NO_CLUSTER_PREFIX + getUniqueId(cacheImplClass);
      }
      else
      {
         tmpName = PREFIX + config.getClusterName();
      }

      return tmpName;
   }

   public static String getUniqueId(String cacheImplClass)
   {
      return cacheImplClass + System.currentTimeMillis();
   }

   public static void unregisterCacheMBean(MBeanServer server, String cacheObjectName)
         throws Exception
   {
      server.unregisterMBean(new ObjectName(cacheObjectName));
   }

   /*
   * Unregister the associated mbeans for cache interceptors
   *
   * @param server the mbean server for which the mbeans should be unregistered
   * @param cache the cache having the set of interceptors to be unregistered
   * @param unregisterCache whether the cache itself should be unregistered
   */
   public static void unregisterInterceptors(MBeanServer server, List<CommandInterceptor> interceptors, String cacheObjectName)
         throws Exception
   {
      if (server == null || interceptors == null || cacheObjectName == null)
      {
         return;
      }

      for (CommandInterceptor interceptor : interceptors)
      {
         // for JDK 1.4, must parse name and remove package prefix
         // for JDK 1.5, can use getSimpleName() to establish class name without package prefix
         String className = interceptor.getClass().getName();
         String serviceName = cacheObjectName + INTERCEPTOR_KEY + className.substring(className.lastIndexOf('.') + 1);

         ObjectName objName = new ObjectName(serviceName);
         if (server.isRegistered(objName))
         {
            server.unregisterMBean(objName);
         }
      }
   }
}
