# Copyright 2013-2014 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Tests for `maastest.kvmfixture`."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

__metaclass__ = type
__all__ = []

import itertools
import os.path
import random
from random import randint
from tempfile import gettempdir
import textwrap

import fixtures
from lxml import etree
from maastest import kvmfixture
from maastest.testing.factory import make_file
from maastest.utils import read_file
import mock
import netaddr
from six import text_type
import testtools
from testtools.matchers import (
    FileContains,
    FileExists,
    HasPermissions,
    MatchesRegex,
    Not,
    StartsWith,
    )


class TestKVMFixture(testtools.TestCase):

    def setUp(self):
        super(TestKVMFixture, self).setUp()
        self.logger = self.useFixture(fixtures.FakeLogger())
        self.patch(kvmfixture, 'sleep', mock.MagicMock())
        # KVMFixture.generate_ssh_key() generates an ssh key in maas-test's
        # configuration directory, creating that directory if necessary.  Set
        # up a temporary directory so it can do what it needs to.
        self.state_dir = self.useFixture(fixtures.TempDir()).path

    def patch_run_command(self, return_value=0, stdout=b'', stderr=b''):
        """Patch out `kvmfixture` calls to `run_command`.

        Call this so your test doesn't accidentally make `KVMFixture` download
        machine images, ssh into it, etc.

        Returns the `MagicMock` fake it has installed.
        """
        self.assertIsInstance(stdout, bytes)
        self.assertIsInstance(stderr, bytes)
        fake = mock.MagicMock(return_value=(return_value, stdout, stderr))
        self.patch(kvmfixture, 'run_command', fake)
        return fake

    def make_KVMFixture(self, series=None, architecture=None, name=None,
                        direct_interface=None, direct_network=None,
                        archives=None, proxy_url=None, kvm_timeout=None,
                        simplestreams_filter=None):
        if series is None:
            series = self.getUniqueString()
        if architecture is None:
            architecture = self.getUniqueString()
        if proxy_url is None:
            proxy_url = "http://example.com:%s" % (
                random.randint(1, 65535))
        self.proxy_url = proxy_url
        # We don't bother dealing with name being None, since KVMFixture
        # deals with that itself.
        fixture = kvmfixture.KVMFixture(
            series, architecture, name=name,
            direct_interface=direct_interface,
            direct_network=direct_network, archives=archives,
            proxy_url=proxy_url, ssh_key_dir=self.state_dir,
            kvm_timeout=kvm_timeout, simplestreams_filter=simplestreams_filter)
        # Hack to get fixture.addDetail to work without actually calling
        # fixture.setUp().
        fixture._details = {}
        return fixture

    def find_matching_call(self, fake_run_command, command_prefix):
        """Find the first `run_command` call with a matching command line.

        This assumes that you have called `patch_run_command` first.

        :param fake_run_command: fake for `run_command` as returned by
            `patch_run_command`.
        :param command_prefix: A sequence of items on the command line.  A
            call to `run_commanad` will match this when this forms a prefix of
            the executed command line.  For instance, a `command_prefix` of
            `['ls']` will match a command line `['ls', '/tmp']` but not
            a command line `['sudo', 'ls', '/tmp']`.  On the other hand, a
            prefix `['sudo', 'ls']` will match `['sudo', 'ls', '/tmp']` but
            not `['ls', '/tmp']`.
        :rtype: `mock.call`
        """
        command_prefix = list(command_prefix)
        prefix_length = len(command_prefix)
        for invocation in fake_run_command.mock_calls:
            _, args, _ = invocation
            command_line = args[0]
            if command_line[:prefix_length] == command_prefix:
                return invocation
        return None

    def test_init_sets_properties(self):
        series = "test-series"
        architecture = "test-arch"
        fixture = self.make_KVMFixture(series, architecture)
        self.assertEqual(
            (series, architecture, 36),
            (fixture.series, fixture.architecture, len(fixture.name)))

    def test_ip_address_calls_out_for_ip(self):
        series = "test-series"
        architecture = "test-arch"
        ip_address = '192.168.2.1'
        self.patch_run_command(0, stdout=ip_address.encode("ascii"))
        with self.make_KVMFixture(series, architecture) as fixture:
            self.assertEqual(ip_address, fixture.ip_address())

    def test_ip_address_caches_ip_after_acquisition(self):
        series = "test-series"
        architecture = "test-arch"
        ip_address = '192.168.2.1'
        self.patch_run_command(0, stdout=ip_address.encode("ascii"))
        with self.make_KVMFixture(series, architecture) as fixture:
            fixture.ip_address()
            self.assertEqual(ip_address, fixture._ip_address)

    def test_ip_address_returns_cached_address(self):
        series = "test-series"
        architecture = "test-arch"
        ip_address = '192.168.2.1'
        self.patch_run_command()
        with self.make_KVMFixture(series, architecture) as fixture:
            fixture._ip_address = ip_address
            self.assertEqual(ip_address, fixture.ip_address())

    def test_identity(self):
        series = "test-series"
        architecture = "test-arch"
        ip_address = '192.168.2.1'
        self.patch_run_command(stdout=ip_address.encode("ascii"))
        with self.make_KVMFixture(series, architecture) as fixture:
            self.assertEqual("ubuntu@%s" % ip_address, fixture.identity())

    def test_setUp_calls_machine_creation_methods(self):
        self.patch_run_command()
        mock_import_image = mock.MagicMock()
        self.patch(kvmfixture.KVMFixture, 'import_image', mock_import_image)
        mock_start = mock.MagicMock()
        self.patch(kvmfixture.KVMFixture, 'start', mock_start)
        mock_install_base_packages = mock.MagicMock()
        self.patch(
            kvmfixture.KVMFixture, 'install_base_packages',
            mock_install_base_packages)
        mock_configure_network = mock.MagicMock()
        self.patch(
            kvmfixture.KVMFixture, 'configure_network',
            mock_configure_network)
        mock_configure_archives = mock.MagicMock()
        self.patch(
            kvmfixture.KVMFixture, 'configure_archives',
            mock_configure_archives)
        series = "test-series"
        architecture = "test-arch"
        with self.make_KVMFixture(series, architecture):
            self.assertEqual(
                [
                    mock.call(),
                    mock.call(),
                    mock.call(),
                    mock.call(),
                    mock.call(),
                ],
                [
                    mock_import_image.mock_calls,
                    mock_start.mock_calls,
                    mock_configure_network.mock_calls,
                    mock_configure_archives.mock_calls,
                    mock_install_base_packages.mock_calls,
                ])

    def test_setUp_awaits_boot_and_cloudinit_completion(self):
        fake_run_command = self.patch_run_command()
        self.patch(
            kvmfixture.KVMFixture, 'ip_address',
            mock.MagicMock(return_value='127.127.127.127'))

        with self.make_KVMFixture() as fixture:
            name = fixture.name

        fake_run_command.assert_has_calls([
            mock.call(['uvt-kvm', 'wait', name], check_call=True),
            mock.call(
                fixture._make_ssh_command(
                    ['test', '-f', kvmfixture.CLOUDINIT_COMPLETION_MARKER]),
                check_call=False, input=None),
            ])

    def test_running_is_set_while_running_only(self):
        self.patch_run_command()
        fixture = self.make_KVMFixture()

        running_before = fixture.running
        with fixture:
            running_during = fixture.running
        running_after = fixture.running

        self.assertEqual(
            (False, True, False),
            (running_before, running_during, running_after))

    def test_setUp_registers_cleanup_method(self):
        class InducedError(RuntimeError):
            """Deliberately induced error for testing."""

        self.patch_run_command()
        fixture = self.make_KVMFixture()
        self.patch(
            fixture, 'wait_for_vm', mock.MagicMock(side_effect=InducedError))

        self.assertRaises(InducedError, fixture.setUp)
        fixture.cleanUp()

        self.assertFalse(fixture.running)

    def test_get_simplestreams_filter_with_filters(self):
        filter1 = "%s=%s" % (self.getUniqueString(), self.getUniqueString())
        filter2 = "%s=%s" % (self.getUniqueString(), self.getUniqueString())
        fixture = self.make_KVMFixture(
            simplestreams_filter="%s %s" % (filter1, filter2))
        self.assertItemsEqual(
            [
                'arch=%s' % fixture.architecture,
                'release=%s' % fixture.series,
                filter1,
                filter2,
            ],
            fixture.get_simplestreams_filter())

    def test_get_simplestreams_filter_without_filters(self):
        fixture = self.make_KVMFixture(simplestreams_filter=None)
        self.assertItemsEqual(
            [
                'arch=%s' % fixture.architecture,
                'release=%s' % fixture.series,
            ],
            fixture.get_simplestreams_filter())

    def test_start_starts_vm(self):
        fake_run_command = self.patch_run_command()
        series = "test-series"
        architecture = "test-arch"
        self.patch(kvmfixture.KVMFixture, 'generate_ssh_key', mock.MagicMock())
        mock_make_kvm_template = mock.MagicMock()
        kvm_template = self.getUniqueString()
        mock_make_kvm_template.return_value = kvm_template
        self.patch(kvmfixture, 'make_kvm_template', mock_make_kvm_template)
        ssfilter = self.getUniqueString()

        kvm_fixture = self.make_KVMFixture(
            series, architecture, simplestreams_filter=ssfilter)
        with kvm_fixture as fixture:
            name = fixture.name
            public_key = fixture.get_ssh_public_key_file()

        fake_run_command.assert_has_calls(
            [
                mock.call([
                    'sudo', 'http_proxy=%s' % self.proxy_url,
                    'uvt-simplestreams-libvirt', 'sync',
                    'arch=%s' % architecture, 'release=%s' % series, ssfilter],
                    check_call=True),
                mock.call([
                    'sudo', 'uvt-kvm', 'create',
                    '--ssh-public-key-file=%s' % public_key,
                    '--unsafe-caching', '--memory', '2047', '--disk', '20',
                    name,
                    'arch=%s' % architecture, 'release=%s' % series, ssfilter,
                    '--template', '-'],
                    input=kvm_template,
                    check_call=True),
            ])
        self.assertEqual(
            [
                mock.call(None),
            ],
            mock_make_kvm_template.mock_calls)

    def test_import_image_logs_to_root_logger(self):
        self.patch_run_command()
        fixture = self.make_KVMFixture()
        fixture.import_image()
        self.assertEqual(
            [
                "Downloading KVM image for %s..." % ' '.join(
                    fixture.get_simplestreams_filter()),
                "Done downloading KVM image."
            ],
            self.logger.output.strip().split("\n"))

    def test_generate_ssh_key_sets_up_temporary_key_pair(self):
        fixture = self.make_KVMFixture()
        fixtures.Fixture.setUp(fixture)

        fixture.generate_ssh_key()

        self.assertThat(fixture.ssh_private_key_file, FileExists())
        self.assertThat(fixture.get_ssh_public_key_file(), FileExists())
        self.assertThat(
            fixture.ssh_private_key_file, StartsWith(self.state_dir + '/'))

        self.assertThat(
            fixture.ssh_private_key_file,
            FileContains(
                matcher=StartsWith('-----BEGIN RSA PRIVATE KEY-----')))
        self.assertThat(
            fixture.get_ssh_public_key_file(),
            FileContains(matcher=StartsWith('ssh-rsa')))

        # Other users are not allowed to read the private key.
        self.assertThat(fixture.ssh_private_key_file, HasPermissions('0600'))

        # The key pair is not protected by a passphrase, as indicated by
        # the absence of this line.
        self.assertThat(
            fixture.ssh_private_key_file,
            Not(FileContains(matcher=MatchesRegex("^Proc-Type: .*ENCRYPTED"))))

    def test_generate_ssh_key_reuses_existing_key_pair(self):
        earlier_fixture = self.make_KVMFixture()
        fixtures.Fixture.setUp(earlier_fixture)
        earlier_fixture.generate_ssh_key()

        # We decode the file contents here to work around bug #1250483
        # in FileContains later. This is for Python 3 compatibility.
        earlier_private_key_contents = read_file(
            earlier_fixture.ssh_private_key_file).decode("ascii")
        earlier_public_key_contents = read_file(
            earlier_fixture.get_ssh_public_key_file()).decode("ascii")

        later_fixture = self.make_KVMFixture()
        fixtures.Fixture.setUp(later_fixture)
        later_fixture.generate_ssh_key()

        self.assertEqual(
            earlier_fixture.ssh_private_key_file,
            later_fixture.ssh_private_key_file)
        self.assertEqual(
            earlier_fixture.get_ssh_public_key_file(),
            later_fixture.get_ssh_public_key_file())
        self.assertThat(
            later_fixture.ssh_private_key_file,
            FileContains(earlier_private_key_contents))
        self.assertThat(
            later_fixture.get_ssh_public_key_file(),
            FileContains(earlier_public_key_contents))

    def test_start_passes_public_ssh_key(self):
        fake_run_command = self.patch_run_command()

        with self.make_KVMFixture() as fixture:
            fixture.start()

        uvt_call = self.find_matching_call(
            fake_run_command, ['sudo', 'uvt-kvm'])
        self.assertIsNotNone(uvt_call)
        _, args, _ = uvt_call
        command_line = args[0]
        self.assertIn(
            '--ssh-public-key-file=%s' % fixture.get_ssh_public_key_file(),
            command_line)

    def test_wait_for_cloudinit_looks_for_completion_marker(self):
        ssh_run_command = mock.MagicMock(return_value=(0, '', ''))
        self.patch(kvmfixture.KVMFixture, 'run_command', ssh_run_command)
        fixture = self.make_KVMFixture()

        fixture.wait_for_cloudinit()

        self.assertEqual(
            [mock.call(
                ['test', '-f', kvmfixture.CLOUDINIT_COMPLETION_MARKER])],
            ssh_run_command.mock_calls)
        self.assertEqual([], kvmfixture.sleep.mock_calls)

    def test_wait_for_cloudinit_times_out_eventually(self):
        attempts = [(0, 15), (5, 10), (10, 5), (15, 0)]
        self.patch(
            kvmfixture, 'retries', mock.MagicMock(return_value=attempts))
        stderr = self.getUniqueString()
        ssh_run_command = mock.MagicMock(return_value=(1, '', stderr))
        self.patch(kvmfixture.KVMFixture, 'run_command', ssh_run_command)
        fixture = self.make_KVMFixture()

        exception = self.assertRaises(RuntimeError, fixture.wait_for_cloudinit)

        message = text_type(exception)
        self.assertIn(
            "Cloud-init initialization on virtual machine %s timed out"
            % fixture.name,
            message)
        self.assertIn(stderr, message)
        self.assertEqual(len(attempts), len(ssh_run_command.mock_calls))

    def test_wait_for_cloudinit_fails_immediately_on_unexpected_error(self):
        stderr = self.getUniqueString()
        ssh_run_command = mock.MagicMock(return_value=(99, '', stderr))
        self.patch(kvmfixture.KVMFixture, 'run_command', ssh_run_command)
        fixture = self.make_KVMFixture()

        exception = self.assertRaises(RuntimeError, fixture.wait_for_cloudinit)

        message = text_type(exception)
        self.assertIn(
            "Error contacting virtual machine %s" % fixture.name,
            message)
        self.assertIn(stderr, message)
        self.assertEqual(1, len(ssh_run_command.mock_calls))
        self.assertEqual([], kvmfixture.sleep.mock_calls)

    def test_get_ssh_public_key_file_returns_public_key(self):
        self.patch_run_command()

        with self.make_KVMFixture() as fixture:
            public_key = fixture.get_ssh_public_key_file()

        self.assertEqual(fixture.ssh_private_key_file + '.pub', public_key)

    def test_start_logs_to_root_logger(self):
        self.patch_run_command()
        fixture = self.make_KVMFixture()

        fixture.start()
        self.assertEqual(
            [
                "Creating virtual machine %s, arch=%s..."
                % (fixture.name, fixture.architecture),
                "Done creating virtual machine %s, arch=%s."
                % (fixture.name, fixture.architecture),
            ],
            self.logger.output.strip().split("\n"))

    def test_destroy_logs_to_root_logger(self):
        self.patch_run_command()
        fixture = self.make_KVMFixture()
        fixture.destroy()
        self.assertEqual(
            [
                "Destroying virtual machine %s..." % fixture.name,
                "Done destroying virtual machine %s." % fixture.name,
            ],
            self.logger.output.strip().split("\n"))

    def test_network_generated_if_not_specified(self):
        self.patch_run_command()
        series = "test-series"
        architecture = "test-arch"
        interface = self.getUniqueString()
        fixture = self.make_KVMFixture(
            series, architecture, direct_interface=interface)
        self.assertEqual(
            (fixture.direct_network, fixture.direct_ip),
            (netaddr.IPNetwork('192.168.2.0/24'), '192.168.2.1'))

    def test_direct_first_available_ip(self):
        self.patch_run_command()
        series = "test-series"
        architecture = "test-arch"
        interface = self.getUniqueString()
        network = netaddr.IPNetwork('192.168.12.0/24')
        fixture = self.make_KVMFixture(
            series, architecture, direct_interface=interface,
            direct_network=network)
        self.assertEqual(
            fixture.direct_first_available_ip(), "192.168.12.2")

    def test_configure_archives_adds_archives_and_updates(self):
        mock_identity = mock.MagicMock()
        mock_identity.return_value = "ubuntu@test"
        self.patch(kvmfixture.KVMFixture, 'identity', mock_identity)
        run_command_mock = mock.MagicMock(return_value=(0, '', ''))
        self.patch(kvmfixture.KVMFixture, 'run_command', run_command_mock)
        archives = [self.getUniqueString(), self.getUniqueString()]
        fixture = self.make_KVMFixture(
            'series', 'architecture', archives=archives)

        fixture.configure_archives()

        add_archive_cmds = [
            ["sudo", "add-apt-repository", "-y", archives[0]],
            ["sudo", "add-apt-repository", "-y", archives[1]],
            ["sudo", "apt-get", "update"],
        ]
        self.assertEqual(
            [mock.call(cmd, check_call=True) for cmd in add_archive_cmds],
            run_command_mock.mock_calls)

    def test_configure_archives_always_updates(self):
        mock_identity = mock.MagicMock()
        mock_identity.return_value = "ubuntu@test"
        self.patch(kvmfixture.KVMFixture, 'identity', mock_identity)
        run_command_mock = mock.MagicMock(return_value=(0, '', ''))
        self.patch(kvmfixture.KVMFixture, 'run_command', run_command_mock)
        fixture = self.make_KVMFixture(
            'series', 'architecture', archives=None)

        fixture.configure_archives()

        self.assertEqual(
            [mock.call(["sudo", "apt-get", "update"], check_call=True)],
            run_command_mock.mock_calls)

    def test_configure_network_configures_network(self):
        series = "test-series"
        architecture = "test-arch"
        mock_identity = mock.MagicMock()
        mock_identity.return_value = "ubuntu@test"
        self.patch(kvmfixture.KVMFixture, 'identity', mock_identity)
        network = netaddr.IPNetwork('192.168.12.0/24')
        ip = "192.168.12.1"
        interface = self.getUniqueString()
        fixture = self.make_KVMFixture(
            series, architecture, direct_interface=interface,
            direct_network=network)
        kvm_run_command = mock.MagicMock(return_value=(0, '', ''))
        self.patch(kvmfixture.KVMFixture, 'run_command', kvm_run_command)

        fixture.configure_network()

        netconfig_command = ["sudo", "tee", "-a", "/etc/network/interfaces"]
        input = kvmfixture.make_network_interface_config(
            'eth1', ip, network, 'eth0')
        netup_command = ["sudo", "ifup", "eth1"]
        self.assertEqual(
            [
                mock.call(netconfig_command, input=input, check_call=True),
                mock.call(netup_command, check_call=True),
            ],
            kvm_run_command.mock_calls)

    def test_machine_is_destroyed(self):
        fake_run_command = self.patch_run_command()
        series = "test-series"
        architecture = "test-arch"
        with self.make_KVMFixture(series, architecture) as fixture:
            pass
        self.assertEqual(
            mock.call([
                "sudo", "uvt-kvm", "destroy", fixture.name],
                check_call=True),
            fake_run_command.mock_calls[-1])

    def test_install_packages_runs_apt_get_noninteractively_with_check(self):
        self.patch_run_command()
        packages = ['foo', 'bar']
        fixture = self.make_KVMFixture('series', 'architecture')
        kvm_run_command = mock.MagicMock(return_value=(0, '', ''))
        self.patch(kvmfixture.KVMFixture, 'run_command', kvm_run_command)

        fixture.install_packages(packages)

        self.assertEqual(
            [mock.call(
                fixture._make_apt_get_install_command(packages),
                check_call=True)],
            kvm_run_command.mock_calls)

    def test_install_base_packages(self):
        self.patch_run_command()
        fixture = self.make_KVMFixture('series', 'architecture')
        kvm_run_command = mock.MagicMock(return_value=(0, '', ''))
        self.patch(kvmfixture.KVMFixture, 'run_command', kvm_run_command)

        fixture.install_base_packages()

        self.assertEqual(
            [mock.call(
                fixture._make_apt_get_install_command(['nmap']),
                check_call=True)],
            kvm_run_command.mock_calls)

    def test_run_command_runs_command_remotely(self):
        fake_run_command = self.patch_run_command()
        series = "test-series"
        architecture = "test-arch"
        input = self.getUniqueString()
        command = ["echo", "Hello"]
        with self.make_KVMFixture(series, architecture) as fixture:
            expected_command = fixture._make_ssh_command(command)
            fixture.run_command(command, input=input, check_call=False)
            ssh_command = fake_run_command.mock_calls[-1]

        self.assertEqual(
            mock.call(expected_command, input=input, check_call=False),
            ssh_command)

    def test_run_command_adds_details(self):
        series = "test-series"
        architecture = "test-arch"
        return_value = randint(0, 3)
        input = self.getUniqueString()
        stdout_content = self.getUniqueString().encode("ascii")
        stderr_content = self.getUniqueString().encode("ascii")
        self.patch_run_command(
            return_value, stdout=stdout_content, stderr=stderr_content)
        self.patch(
            kvmfixture.KVMFixture, 'wait_for_cloudinit', mock.MagicMock())

        with self.make_KVMFixture(series, architecture) as fixture:
            fixture.command_count = itertools.count(1)
            fixture.run_command(
                ["echo", "Hello"], input=input, check_call=False)
            details = fixture.getDetails()

        self.assertThat(
            details['cmd #0001'].as_text(),
            MatchesRegex('ssh .* echo Hello$'))
        self.assertEqual(
            (
                text_type(return_value),
                input,
                stdout_content,
                stderr_content,
            ),
            (
                details['cmd #0001 retcode'].as_text(),
                details['cmd #0001 input'].as_text(),
                details['cmd #0001 stdout'].as_text(),
                details['cmd #0001 stderr'].as_text(),
            ))

    def test_get_ip_from_network_scan(self):
        self.patch_run_command()
        mac = self.getUniqueString()
        ip = self.getUniqueString()
        network_str = '192.168.12.0/24'
        network = netaddr.IPNetwork(network_str)
        fixture = self.make_KVMFixture(
            'series', 'architecture', direct_network=network)
        mapping = {mac.upper(): ip}
        extract_mac_ip_mapping_mock = mock.MagicMock(return_value=mapping)
        xml_output = self.getUniqueString()
        self.patch(
            kvmfixture, 'extract_mac_ip_mapping',
            extract_mac_ip_mapping_mock)
        kvm_run_command = mock.MagicMock(return_value=(0, xml_output, ''))
        self.patch(kvmfixture.KVMFixture, 'run_command', kvm_run_command)

        returned_ip = fixture.get_ip_from_network_scan(mac)

        self.assertEqual(
            (
                ip,
                [
                    mock.call(
                        ['sudo', 'nmap', '-sP', network_str, '-oX', '-'],
                        check_call=True)
                ],
                [mock.call(xml_output)],
            ),
            (
                returned_ip,
                kvm_run_command.mock_calls,
                extract_mac_ip_mapping_mock.mock_calls,
            ))

    def test_wait_for_vm_uses_timeout(self):
        self.patch(kvmfixture, 'run_command', mock.MagicMock())
        timeout = randint(1, 1000)
        fixture = self.make_KVMFixture(kvm_timeout=timeout)
        fixture.wait_for_vm()
        self.assertEqual([
            mock.call([
                'uvt-kvm', 'wait', '--timeout', unicode(timeout),
                fixture.name], check_call=True)],
            kvmfixture.run_command.mock_calls)

    def test_wait_for_vm_ignores_timeout_if_not_set(self):
        self.patch(kvmfixture, 'run_command', mock.MagicMock())
        fixture = self.make_KVMFixture(kvm_timeout=None)
        fixture.wait_for_vm()
        self.assertEqual([
            mock.call([
                'uvt-kvm', 'wait', fixture.name], check_call=True)],
            kvmfixture.run_command.mock_calls)

    def test_upload_file_uploads_file(self):
        self.patch(kvmfixture, 'run_command', mock.MagicMock())
        fixture = self.make_KVMFixture(kvm_timeout=None)
        local_file = make_file(self)
        remote_file = os.path.join(gettempdir(), self.getUniqueString())
        fixture.upload_file(local_file, remote_file)
        self.assertEqual(
            [
                mock.call(
                    (
                        ['scp'] +
                        fixture._get_base_ssh_options() +
                        [local_file, remote_file]
                    ),
                    check_call=True)
            ],
            kvmfixture.run_command.mock_calls)

    def test_upload_file_defaults_to_base_name_in_default_dir(self):
        self.patch(kvmfixture, 'run_command', mock.MagicMock())
        fixture = self.make_KVMFixture(kvm_timeout=None)
        local_file = make_file(self)
        fixture.upload_file(local_file)
        [command] = kvmfixture.run_command.mock_calls
        name, args, kwargs = command
        (command_line,) = args
        self.assertEqual(os.path.basename(local_file), command_line[-1])


def xml_normalize(snippet):
    parser = etree.XMLParser(remove_blank_text=True)
    return etree.tostring(etree.fromstring(snippet, parser))


class TestKVMTemplateGeneration(testtools.TestCase):

    def test_kvm_template_creates_kvm_template(self):
        template = kvmfixture.make_kvm_template()
        expected_template = xml_normalize(kvmfixture.KVM_TEMPLATE)
        self.assertEqual(expected_template, template)

    def test_kvm_template_without_direct_interface_has_one_interface(self):
        parser = etree.XMLParser(remove_blank_text=True)
        template = kvmfixture.make_kvm_template()
        xml_template = etree.fromstring(template, parser)
        interface_tags = xml_template.xpath("/domain/devices/interface")
        expected_interfaces = ["""
            <interface type='network'>
              <source network='default'/>
              <model type='virtio'/>
            </interface>"""]
        self.assertEqual(
            [xml_normalize(iface) for iface in expected_interfaces],
            [etree.tostring(iface) for iface in interface_tags])

    def test_kvm_template_with_direct_interface_has_two_interface(self):
        interface = self.getUniqueString()
        parser = etree.XMLParser(remove_blank_text=True)
        template = kvmfixture.make_kvm_template(interface)
        xml_template = etree.fromstring(template, parser)
        interface_tags = xml_template.xpath("/domain/devices/interface")
        expected_interfaces = [
            """
            <interface type='network'>
              <source network='default'/>
              <model type='virtio'/>
            </interface>""",
            """
            <interface type='direct'>
              <source dev='%s' mode='vepa'/>
            </interface>""" % interface,
        ]
        self.assertEqual(
            [xml_normalize(iface) for iface in expected_interfaces],
            [etree.tostring(iface) for iface in interface_tags])


class TestNetworkConfigGeneration(testtools.TestCase):

    def test_make_network_interface_config(self):
        interface = 'eth0'
        direct_interface = 'eth1'
        ip = "192.168.12.22"
        network = netaddr.IPNetwork('192.168.12.3/24')
        expected_config = textwrap.dedent("""
            auto eth1
            iface eth1 inet static
            address 192.168.12.22
            network 192.168.12.0
            netmask 255.255.255.0
            broadcast 192.168.12.255
            post-up sysctl -w net.ipv4.ip_forward=1
            post-up iptables -t nat -A POSTROUTING -o eth0 -j MASQUERADE
            post-up iptables -A FORWARD -i eth0 -o eth1 -m state --state \
RELATED,ESTABLISHED -j ACCEPT
            post-up iptables -A FORWARD -i eth1 -o eth0 -j ACCEPT
            """)

        config = kvmfixture.make_network_interface_config(
            direct_interface, ip, network, interface)
        self.assertEqual(expected_config, config)
