/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/settings.h"

// Boost C++
#include <boost/algorithm/string/predicate.hpp>

// C++ Standard Library
#include <set>
#include <string>
#include <vector>
#include <utility>

// Media Scanner Library
#include "mediascanner/dbustypes.h"
#include "mediascanner/logging.h"
#include "mediascanner/utilities.h"

namespace mediascanner {

// Boost C++
using boost::algorithm::starts_with;

// Context specific logging domains
static const logging::Domain kWarning("warning/settings", logging::warning());

// Key constants
const Settings::Key<Settings::MediaFormatList>
Settings::kMandatoryContainers("mandatory-containers");
const Settings::Key<Settings::MediaFormatList>
Settings::kMandatoryDecoders("mandatory-decoders");
const Settings::Key<Settings::MetadataSourceList>
Settings::kMetadataSources("metadata-sources");
const Settings::Key<Settings::StringList>
Settings::kMediaRoots("media-roots");

// FIXME(M5): Apparently "dbus" is not the proper namespace for this things
namespace dbus {

template<> struct BoxedTypeTrait<Settings::MediaFormat> {
    typedef std::pair<std::string, std::string> boxed_type;
};

template<> struct BoxedTypeTrait<Settings::MetadataSource> {
    typedef boost::tuples::tuple
        <std::string, std::string, Settings::MetadataSource::Config>
         boxed_type;
};

template<> class Type<Settings::MediaFormat>
        : public BoxedType<Settings::MediaFormat> {
};

template<> class Type<Settings::MetadataSource>
        : public BoxedType<Settings::MetadataSource> {
};

template<> Settings::MediaFormat
BoxedType<Settings::MediaFormat>::make_unboxed(boxed_type value) {
    return Settings::MediaFormat(value.first, value.second);
}

template<> Settings::MetadataSource
BoxedType<Settings::MetadataSource>::make_unboxed(boxed_type value) {
    return Settings::MetadataSource(boost::get<0>(value),
                                    boost::get<1>(value),
                                    boost::get<2>(value));
}

} // namespace dbus

// TODO(M4): Handle change notifications
Settings::Settings()
    : settings_(take(g_settings_new(MEDIASCANNER_SETTINGS_ID))) {
}

template<typename T>
T Settings::lookup(const Key<T> &key) const {
    if (const Wrapper<GVariant> settings_value =
            take(g_settings_get_value(settings_.get(), key.name_.c_str())))
        return dbus::Type<T>::make_value(settings_value.get());

    return T();
}

static void on_change(void *data) {
    (*static_cast<Settings::ChangeListener *>(data))();
}

unsigned Settings::connect(const KeyName &key,
                           const ChangeListener &listener) const {
    return g_signal_connect_data
            (settings_.get(), ("changed::" + key.name_).c_str(),
             G_CALLBACK(&on_change), new ChangeListener(listener),
             ClosureNotify<ChangeListener>, static_cast<GConnectFlags>(0));
}

void Settings::disconnect(unsigned handler_id) const {
    g_signal_handler_disconnect(settings_.get(), handler_id);
}

Settings::MediaFormatList Settings::mandatory_containers() const {
    return lookup(kMandatoryContainers);
}

Settings::MediaFormatList Settings::mandatory_decoders() const {
    return lookup(kMandatoryDecoders);
}

Settings::MetadataSourceList Settings::metadata_sources() const {
    return lookup(kMetadataSources);
}

Settings::StringList Settings::media_root_urls() const {
    return lookup(kMediaRoots);
}

static std::string user_dir_url(GUserDirectory dir) {
    switch (dir) {
    case G_USER_DIRECTORY_DESKTOP:
        return "user:desktop";

    case G_USER_DIRECTORY_DOCUMENTS:
        return "user:documents";

    case G_USER_DIRECTORY_DOWNLOAD:
        return "user:download";

    case G_USER_DIRECTORY_MUSIC:
        return "user:music";

    case G_USER_DIRECTORY_PICTURES:
        return "user:pictures";

    case G_USER_DIRECTORY_PUBLIC_SHARE:
        return "user:public";

    case G_USER_DIRECTORY_TEMPLATES:
        return "user:templates";

    case G_USER_DIRECTORY_VIDEOS:
        return "user:videos";

    case G_USER_N_DIRECTORIES:
        break;
    }

    return std::string();
}

static std::string resolve_url(const std::string &url) {
    if (starts_with(url, "user:")) {
        for (unsigned i = 0; i < G_USER_N_DIRECTORIES; ++i) {
            const GUserDirectory dir = static_cast<GUserDirectory>(i);

            if (url == user_dir_url(dir)) {
                const char *const path = g_get_user_special_dir(dir);

                if (not path || not g_file_test(path, G_FILE_TEST_IS_DIR))
                    break;

                return path;
            }
        }

        return std::string();
    }

    Wrapper<GFile> file = take(g_file_new_for_uri(url.c_str()));

    if (not g_file_is_native(file.get()))
        return std::string();

    if (G_FILE_TYPE_DIRECTORY != g_file_query_file_type
            (file.get(), G_FILE_QUERY_INFO_NOFOLLOW_SYMLINKS, nullptr))
        return std::string();

    return safe_string(g_file_get_path(file.get()));
}

Settings::StringList Settings::media_root_paths() const {
    const StringList urls = media_root_urls();

    StringList paths;
    paths.reserve(urls.size());

    for (const std::string &root_url: urls) {
        std::string root_path = resolve_url(root_url);

        if (root_path.empty()) {
            kWarning("Ignoring unsupported media root <{1}>") % root_url;
            continue;
        }

        paths.push_back(root_path);
    }

    return paths;
}

std::vector<std::string> Settings::LoadMetadataSources() const {
    return LoadMetadataSources(metadata_sources());
}

std::vector<std::string> Settings::LoadMetadataSources
                                        (const MetadataSourceList &sources) {
    const Wrapper<GrlRegistry> registry = wrap(grl_registry_get_default());
    std::vector<std::string> available_sources;
    std::set<std::string> loaded_plugins;

    for (const Settings::MetadataSource &s: sources) {
        if (s.config.empty())
            continue;

        Wrapper<GrlConfig> config =
                take(grl_config_new(s.plugin_id.c_str(),
                                    s.source_id.c_str()));

        typedef Settings::MetadataSource::Config::value_type ConfigParam;

        for (const ConfigParam &p: s.config) {
            grl_config_set_string(config.get(), p.first.c_str(),
                                  p.second.c_str());
        }

        Wrapper<GError> error;

        if (not grl_registry_add_config(registry.get(), config.release(),
                                        error.out_param())) {
            const std::string error_message = to_string(error);
            kWarning("Cannot configure the {1} plugin's {2} source: {3}")
                    % s.plugin_id % s.source_id % error_message;
        }
    }

    for (const Settings::MetadataSource &s: sources) {
        Wrapper<GError> error;

        if (loaded_plugins.find(s.plugin_id) == loaded_plugins.end()) {
            if (not grl_registry_load_plugin_by_id(registry.get(),
                                                   s.plugin_id.c_str(),
                                                   error.out_param())) {
                const std::string error_message = to_string(error);
                kWarning("Cannot load metadata plugin {1}: {2}")
                        % s.plugin_id % error_message;
                continue;
            }

            loaded_plugins.insert(s.plugin_id);
        }

        if (not grl_registry_lookup_source(registry.get(),
                                           s.source_id.c_str())) {
            kWarning("No such metadata source: {1}") % s.source_id;
            continue;
        }

        available_sources.push_back(s.source_id);
    }

    return available_sources;
}

} // namespace mediascanner
