/* 
  Copyright (C) 2008 Kai Hertel

	This file is part of mmpong.

	mmpong is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	mmpong is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <math.h>
#include "game.h"
#include "message.h"
#include "linear.h"
#include "badminton.h"
#include "padflat.h"
#include "padround.h"


#define INIT_RANGE_X 		0.2
#define INIT_RANGE_Y 		0.6
#define INIT_RANGE_ANGLE 	0.5
#define INIT_PADDLE_SIZE 	0.25
#define INIT_VELOCITY 		0.6
#define STALL_TIMEOUT 		5.0


static int gameplay_newgame(struct gameplay *);
//static int set_geometry(const enum gamemode, struct gamegeometry *);

static const struct {
	enum gamemode mode;
	const char *spell;
	int (*fct)(struct gameplay *, float,
		int (*)(const float[2], struct gameball *));
} gamemode_handler[]= {
	{ gamemode_linear, "linear", linear_model },
	{ gamemode_badminton, "badminton", badminton_model },
	{ (-1), NULL, NULL }
};

static const struct {
	enum gamepadprofile mode;
	const char *spell;
	int (*fct)(const float[2], struct gameball *);
} gamepadprofile_handler[]= {
	{ gamepadprofile_flat, "flat", flat_padprofile },
	{ gamepadprofile_round, "round", round_padprofile },
	{ (-1), NULL, NULL }
};



EXPORT short gameplay_getversion(void)
{
	return sizeof(struct gameplay);
}



EXPORT struct gameplay *gameplay_create(void)
{
	struct gameplay *instance= calloc(1, sizeof(struct gameplay));
	if (instance) instance->version= gameplay_getversion();
	return instance;
}



EXPORT int gameplay_init(igame, size, model, profile)
struct gameplay *igame;
int size;
const enum gamemode model;
const enum gamepadprofile profile;
{
	if (!igame) return PONG_ARGINVALID;
	if (size < sizeof(struct gameplay)) return PONG_UNSUPPORTED;

	// check for valid model implementations
	int mode;
	for (mode= 0; gamemode_handler[mode].fct; mode++)
		if (gamemode_handler[mode].mode == model) break;
	if (!gamemode_handler[mode].fct) return PONG_UNSUPPORTED;
	for (mode= 0; gamepadprofile_handler[mode].fct; mode++)
		if (gamepadprofile_handler[mode].mode == profile) break;
	if (!gamepadprofile_handler[mode].fct) return PONG_UNSUPPORTED;

	memset(igame, 0, sizeof(struct gameplay));
	// set headers
	igame->version= sizeof(struct gameplay);
	igame->mode= model;
	// set paddles
	for (int idx= 0; idx <= 1; idx++) {
		igame->pad[idx].mean= 0.5f;
		igame->pad[idx].var= 0;
		igame->pad[idx].size= INIT_PADDLE_SIZE;
		igame->pad_attr[idx].profile= profile;
		// we have to set this value somewhere later in the game. but i have  no idea where.
//		igame->pad[idx].dir= 0.f;  
		// score has been nulled by memset
	}
  
//	if ((mode= set_geometry(model, &igame->geo)) != PONG_SUCCESS)
//		return mode;
	return gameplay_newgame(igame);
}



static int gameplay_newgame(igame)
struct gameplay *igame;
{
	if (gettimeofday(&igame->stamp, NULL) == (-1))
		return PONG_INTERROR;
	memcpy(&igame->lasthit, &igame->stamp, sizeof(igame->lasthit));
	igame->status= gamestatus_running;

	// set ball pos
	igame->ball.pos[0]= ((float)rand())/RAND_MAX * INIT_RANGE_X + 0.5 - INIT_RANGE_X*0.5;
	igame->ball.pos[1]= ((float)rand())/RAND_MAX * INIT_RANGE_Y + 0.5 - INIT_RANGE_Y*0.5;

	float angle = ( ((float)rand())/RAND_MAX * 0.5 - 0.25 + (rand()%2) ) * M_PI;
	igame->ball.dir[0]= cosf(angle) * INIT_VELOCITY;
	igame->ball.dir[1]= sinf(angle) * INIT_VELOCITY;
	return PONG_SUCCESS;
}



EXPORT int gameplay_update(igame, newtime)
struct gameplay *igame;
const struct timeval *newtime;
{
	if ((!igame) || (!newtime)) return PONG_ARGINVALID;
	if (igame->status != gamestatus_running)
		return gameplay_newgame(igame);

	float timediff=
		((float)( newtime->tv_sec - igame->stamp.tv_sec))
		+ ((float)( newtime->tv_usec - igame->stamp.tv_usec)) /1000.0f /1000.0f;
	if (timediff < 0) return PONG_ARGINVALID; 	// we only move forward in time (especially in non-linear models)
	float stalldiff=
		((float)( newtime->tv_sec - igame->lasthit.tv_sec))
		+ ((float)( newtime->tv_usec - igame->lasthit.tv_usec)) /1000.0f /1000.0f;
	if (stalldiff >= STALL_TIMEOUT) {
		igame->status= gamestatus_stall;
		return PONG_SUCCESS;
	}

	// find model implementation
	int model;
	for (model= 0; gamemode_handler[model].fct; model++)
		if (igame->mode == gamemode_handler[model].mode) break;
	if (gamemode_handler[model].fct == NULL) return PONG_UNSUPPORTED;
	memcpy(&igame->stamp, newtime, sizeof(igame->stamp));

	enum gamepadprofile reqprofile= igame->pad_attr[0].profile;
	int padprofile;
	for (padprofile= 0; padprofile<= 1; padprofile++) 	// require all paddles to use the same profile (for now)
		if (igame->pad_attr[padprofile].profile != reqprofile) return PONG_UNSUPPORTED; 	// ... so far
	for (padprofile= 0; gamepadprofile_handler[padprofile].fct; padprofile++)
		if (gamepadprofile_handler[padprofile].mode == reqprofile) break;
	if (gamepadprofile_handler[padprofile].fct == NULL) return PONG_UNSUPPORTED;

//	struct gameplay gameintern;
	int retcode;
//	if ((retcode= gameplay_public_to_internal(game, newtime, &gameintern)) != PONG_SUCCESS) return retcode;
	retcode= gamemode_handler[model].fct(igame, timediff, gamepadprofile_handler[padprofile].fct);
	switch (retcode) {
	case PONG_PADHIT:
		memcpy(&igame->lasthit, newtime, sizeof(igame->lasthit));
		break;
	case PONG_SCORE:
		igame->status= gamestatus_onhold; 	// leave the score counting to the model (not necessarily linear)
		break;
	case PONG_SUCCESS:
		break;
	default:
		if (retcode == PONG_INTERROR) 	// reset the game (which will hopefully help the situation)
			igame->status= gamestatus_onhold;
		return retcode;
	}
//	if ((retcode= gameplay_internal_to_public(&gameintern, newtime, game)) != PONG_SUCCESS) return retcode;
	return PONG_SUCCESS; 	// mask internal state
}



EXPORT int gameplay_public_to_internal(pgame, igame)
const struct gameplay_public *pgame;
struct gameplay *igame;
{
	if ((!pgame) || (!igame)) return PONG_ARGINVALID;

	igame->stamp.tv_sec = pgame->stamp.tv_sec;
	igame->stamp.tv_usec = pgame->stamp.tv_usec;

	igame->mode= pgame->mode;
	igame->status= pgame->status;
	// ball
	for (int idx= 0; idx <= 1; idx++) {
		igame->ball.pos[idx]= ((float)(pgame->ball.pos[idx] % PONG_RANGE_SPREAD)) /PONG_RANGE_SPREAD;
		igame->ball.dir[idx]= (((float)(pgame->ball.dir[idx] % PONG_RANGE_SPREAD)) /PONG_RANGE_SPREAD *2.f -1.f) * GAMEBALL_DIR_MAX;
	}
	// paddles
	for (int idx= 0; idx <= 1; idx++) {
		igame->pad[idx].mean= ((float)(pgame->pad[idx].mean % PONG_RANGE_SPREAD)) /PONG_RANGE_SPREAD;
		igame->pad[idx].var=  ((float)(pgame->pad[idx].var  % PONG_RANGE_SPREAD)) /PONG_RANGE_SPREAD;
		igame->pad[idx].size= ((float)(pgame->pad[idx].size % PONG_RANGE_SPREAD)) /PONG_RANGE_SPREAD;
		igame->pad_attr[idx].score= pgame->pad_attr[idx].score;
		igame->pad_attr[idx].peers= pgame->pad_attr[idx].peers;
		igame->pad_attr[idx].profile= pgame->pad_attr[idx].profile;
	}
	return PONG_SUCCESS;
}



EXPORT int gameplay_internal_to_public(igame, pgame)
const struct gameplay *igame;
struct gameplay_public *pgame;
{
	if ((!igame) || (!pgame)) return PONG_ARGINVALID;

	pgame->stamp.tv_sec = igame->stamp.tv_sec;
	pgame->stamp.tv_usec = igame->stamp.tv_usec;

	pgame->mode= igame->mode;
	pgame->status= igame->status;
	// ball
	for (int idx= 0; idx <= 1; idx++) {
		// cut off values and transform
		float crop= igame->ball.pos[idx];
		if (crop > 1.f) crop= 1.f;
		if (crop < 0.f) crop= 0.f;
		pgame->ball.pos[idx]= (uint16_t)(crop * PONG_RANGE_SPREAD);

		crop= igame->ball.dir[idx];
		if (crop > GAMEBALL_DIR_MAX) crop= GAMEBALL_DIR_MAX;
		if (crop < -GAMEBALL_DIR_MAX) crop= -GAMEBALL_DIR_MAX;
		pgame->ball.dir[idx]= (uint16_t)((crop + GAMEBALL_DIR_MAX) /2.f /GAMEBALL_DIR_MAX * PONG_RANGE_SPREAD);
	}
	// paddles
	for (int idx= 0; idx <= 1; idx++) {
		pgame->pad[idx].mean= (uint16_t)(igame->pad[idx].mean * PONG_RANGE_SPREAD);
		pgame->pad[idx].var=  (uint16_t)(igame->pad[idx].var  * PONG_RANGE_SPREAD);
		pgame->pad[idx].size= (uint16_t)(igame->pad[idx].size * PONG_RANGE_SPREAD);
//		pgame->pad[idx].dir= (short)(igame->pad[idx].dir  * PONG_RANGE_SPREAD);
		pgame->pad_attr[idx].score= igame->pad_attr[idx].score;
		pgame->pad_attr[idx].peers= igame->pad_attr[idx].peers;
		pgame->pad_attr[idx].profile= igame->pad_attr[idx].profile;
	}
	return PONG_SUCCESS;
}



EXPORT const char *gameplay_spell(model, padprofile)
const enum gamemode model;
const enum gamepadprofile padprofile;
{
	int idx;
	if (model >= 0) {
		for (idx= 0; gamemode_handler[idx].fct; idx++)
			if (gamemode_handler[idx].mode == model)
				return gamemode_handler[idx].spell;
		return NULL;
	}
	if (padprofile >= 0) {
		for (idx= 0; gamepadprofile_handler[idx].fct; idx++)
			if (gamepadprofile_handler[idx].mode == padprofile)
				return gamepadprofile_handler[idx].spell;
		return NULL;
	}
	return NULL;
}



EXPORT int gameplay_parse(spell, model, padprofile)
const char *spell;
enum gamemode *model;
enum gamepadprofile *padprofile;
{
	int idx;
	if (model) {
		for (idx= 0; gamemode_handler[idx].fct; idx++)
			if (!strcmp(gamemode_handler[idx].spell, spell)) {
				*model= gamemode_handler[idx].mode;
				return PONG_SUCCESS;
			}
		return PONG_UNSUPPORTED;
	}
	if (padprofile) {
		for (idx= 0; gamepadprofile_handler[idx].fct; idx++)
			if (!strcmp(gamepadprofile_handler[idx].spell, spell)) {
				*padprofile= gamepadprofile_handler[idx].mode;
				return PONG_SUCCESS;
			}
		return PONG_UNSUPPORTED;
	}
	return PONG_UNSUPPORTED;
}



EXPORT int gameplay_apply_state(msg, igame, team)
const struct netmessage *msg;
struct gameplay *igame;
uint16_t *team;
{
	if (!msg || !igame || !team) return PONG_ARGINVALID;

	struct gameplay_public const *pub= NULL;
	switch (msg->hdr.id) {

	case NETMSG_STAT:
		if (msg->hdr.len < (sizeof(struct netmessage_header) + sizeof(struct game_state_full))) 
			return PONG_UNSUPPORTED;

		*team = msg->payload.full.team;
//		if (*team <0) *team= 0;
		if (*team >GAME_MAX_TEAMS) *team= GAME_MAX_TEAMS -1;
		pub= &msg->payload.full.game;
		break;

	case NETMSG_UPDT:
		if (msg->hdr.len < (sizeof(struct netmessage_header) + sizeof(struct game_state_part)))
			return PONG_UNSUPPORTED;

		struct gameplay_public game_full;
		gameplay_internal_to_public(igame, &game_full);
		memcpy (&game_full.ball, &msg->payload.part.ball, sizeof(game_full.ball));
		memcpy (&game_full.pad, msg->payload.part.pad, 2*sizeof(game_full.pad));
		pub= &game_full;
		break;
	}

	if (!pub)
		return PONG_ARGINVALID;
	gameplay_public_to_internal(pub, igame);
	memcpy (&igame->lasthit, &igame->stamp, sizeof(igame->lasthit)); 	// take a guess

	return PONG_SUCCESS;
}



/*
static int set_geometry(mode, igeo)
const enum gamemode mode;
struct gamegeometry *igeo;
{
	switch(mode) {
	case gamemode_linear:
		igeo->num_lines= 2;
		if( igeo->num_lines >= GAME_MAX_OBJECTS-2)
			return PONG_INTERROR;
		// bottom wall;
		igeo->start[0][0]= 0.;
		igeo->start[0][1]= 0.;
		igeo->end[0][0]= 1.;
		igeo->end[0][1]= 0.;
		// top wall;
		igeo->start[1][0]= 1.;
		igeo->start[1][1]= 1.;
		igeo->end[1][0]= 0.;
		igeo->end[1][1]= 1.;
		break;

	case gamemode_badminton:
		igeo->num_lines= 3;
		if( igeo->num_lines >= GAME_MAX_OBJECTS-2)
			return PONG_INTERROR;
		// bottom wall;
		igeo->start[0][0]= 0.;
		igeo->start[0][1]= 0.;
		igeo->end[0][0]= 1.;
		igeo->end[0][1]= 0.;
		// top wall;
		igeo->start[1][0]= 1.;
		igeo->start[1][1]= 1.;
		igeo->end[1][0]= 0.;
		igeo->end[1][1]= 1.;
		// net
		igeo->start[2][0]= 0.5;
		igeo->start[2][1]= 0.0; 	// maybe, we need to switch coordinate systems for an accurate physics mapping
		igeo->end[2][0]= 0.5;
		igeo->end[2][1]= 0.3;
		break;

	default:
		return PONG_UNSUPPORTED;
	}

	// Compute normals by the right hand rule:
	// Daumen schaut aus dem Monitor raus
	// Zeigefinger zeigt von startpunkt zu endpunkt
	// Mittelfinger schaut in Normalenrichtung
	for(int l= 0;l< igeo->num_lines; ++l) {
		igeo->normal[l][0]= -igeo->end[l][1] + igeo->start[l][1];
		igeo->normal[l][1]=  igeo->end[l][0] - igeo->start[l][0];
	}

	return PONG_SUCCESS;
}
*/

