/*
 * Copyright (C) 2016 Igalia S.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "config.h"
#include "MathMLScriptsElement.h"

#if ENABLE(MATHML)

#include "RenderMathMLScripts.h"

namespace WebCore {

using namespace MathMLNames;

static ScriptType scriptTypeOf(const QualifiedName& tagName)
{
    if (tagName == msubTag)
        return ScriptType::Sub;
    if (tagName == msupTag)
        return ScriptType::Super;
    if (tagName == msubsupTag)
        return ScriptType::SubSup;
    if (tagName == munderTag)
        return ScriptType::Under;
    if (tagName == moverTag)
        return ScriptType::Over;
    if (tagName == munderoverTag)
        return ScriptType::UnderOver;
    ASSERT(tagName == mmultiscriptsTag);
    return ScriptType::Multiscripts;
}

MathMLScriptsElement::MathMLScriptsElement(const QualifiedName& tagName, Document& document)
    : MathMLPresentationElement(tagName, document)
    , m_scriptType(scriptTypeOf(tagName))
{
}

Ref<MathMLScriptsElement> MathMLScriptsElement::create(const QualifiedName& tagName, Document& document)
{
    return adoptRef(*new MathMLScriptsElement(tagName, document));
}

const MathMLElement::Length& MathMLScriptsElement::subscriptShift()
{
    return cachedMathMLLength(subscriptshiftAttr, m_subscriptShift);
}

const MathMLElement::Length& MathMLScriptsElement::superscriptShift()
{
    return cachedMathMLLength(superscriptshiftAttr, m_superscriptShift);
}

void MathMLScriptsElement::parseAttribute(const QualifiedName& name, const AtomicString& value)
{
    if (name == subscriptshiftAttr)
        m_subscriptShift = std::nullopt;
    else if (name == superscriptshiftAttr)
        m_superscriptShift = std::nullopt;

    MathMLElement::parseAttribute(name, value);
}

RenderPtr<RenderElement> MathMLScriptsElement::createElementRenderer(RenderStyle&& style, const RenderTreePosition&)
{
    ASSERT(hasTagName(msubTag) || hasTagName(msupTag) || hasTagName(msubsupTag) || hasTagName(mmultiscriptsTag));
    return createRenderer<RenderMathMLScripts>(*this, WTFMove(style));
}

}

#endif // ENABLE(MATHML)
