///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ATOMS_IMPORT_OBJECT_H
#define __ATOMS_IMPORT_OBJECT_H

#include <core/Core.h>
#include <core/scene/objects/SceneObject.h>
#include <core/gui/properties/PropertiesEditor.h>
#include <core/gui/properties/PropertiesPanel.h>
#include <core/gui/ElidedTextLabel.h>
#include <core/scene/animation/controller/Controller.h>

#include <atomviz/AtomViz.h>
#include <atomviz/atoms/AtomsObject.h>
#include "AtomsFileParser.h"

namespace AtomViz {

/******************************************************************************
* A placeholder object that reads in an atoms file and creates an
* AtomsObject from it.
******************************************************************************/
class ATOMVIZ_DLLEXPORT AtomsImportObject : public SceneObject
{
public:

	/// \brief Constructs a new instance of this class.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	AtomsImportObject(bool isLoading = false);

	/// \brief Returns the AtomsObject that stores the imported atoms.
	///
	/// This method returns a pointer to the internal AtomsObject that stores the
	/// current atoms dataset. If the input file contains more than movie frame
	/// then the last frame returned by evalObject() is stored in the returned AtomsObject.
	AtomsObject* atomsObject() const { return _atoms; }

	/// \brief Returns the movie frame that is currently loaded.
	///
	/// Usually this is the animation frame that was requested from evalObject() the last time.
	/// The atomic configuration for the currently loaded movie frame can be accessed via atomsObject().
	int loadedMovieFrame() const { return _loadedMovieFrame; }

	/// \brief Sets the parser used by this object.
	///
	/// After the parser has been set, reloadInputFile() must be called.
	void setParser(AtomsFileParser* parser);

	/// \brief Returns the format specific parser that loads the input file.
	AtomsFileParser* parser() const { return _parser; }

	/// \brief Returns the path to the input file.
	/// \return The path to the input file. This is the same as AtomsFileParser::inputFile() returns.
	/// \sa AtomsFileParser::inputFile()
	QString inputFile() const { return parser() ? parser()->inputFile() : QString(); }

	/// \brief Returns the name of the current source file.
	/// \return The name/path of the file from which the current snapshot was loaded. This is the same as AtomsFileParser::sourceFile() returns.
	/// \sa AtomsFileParser::sourceFile()
	QString sourceFile() const { return parser() ? parser()->sourceFile() : QString(); }

	/// \brief This reloads the input file.
	/// \return \a false when the operation has been canceled by the user; \a true on success.
	/// \throws Exception on error.
	bool reloadInputFile();

	/// \brief Return the status returned by the parser during its last call.
	const EvaluationStatus& status() const { return _loadStatus; }

	/// \brief Returns whether atomic coordinates are saved along with the scene.
	/// \return \c true if data is stored in the scene file; \c false if the data resides in an external file.
	bool storeAtomsWithScene() const { return atomsObject() ? atomsObject()->serializeAtoms() : false; }

	/// \brief Returns whether atomic coordinates are saved along with the scene.
	/// \param on \c true if data should be stored in the scene file; \c false if the data resides in an external file.
	/// \undoable
	void setStoreAtomsWithScene(bool on) { if(atomsObject()) atomsObject()->setSerializeAtoms(on); }

	/// \brief Returns the number of animation frames per simulation snapshot.
	/// \return The number of animation frames per simulation snapshot. This is always equal or greater than 1.
	int framesPerSnapshot() const { return max((int)_framesPerSnapshot, 1); }

	/// \brief Sets the number of animation frames per simulation snapshot to control the playback speed.
	/// \param fps The number of animation frames per simulation snapshot. This must be equal or greater than 1.
	void setFramesPerSnapshot(int fps) { _framesPerSnapshot = fps; }

	// From SceneObject:

	/// Asks the object for its validity interval at the given time.
	virtual TimeInterval objectValidity(TimeTicks time);
	/// Makes the object render itself into the viewport.
	virtual void renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp) {}
	/// Returns the bounding box of the object in local object coordinates.
	virtual Box3 boundingBox(TimeTicks time, ObjectNode* contextNode) { return Box3(); }

	/// Asks the object for the result of the geometry pipeline at the given time.
	virtual PipelineFlowState evalObject(TimeTicks time);

	// From RefTarget:

	/// Returns the title of this object.
	virtual QString schematicTitle();

public Q_SLOTS:

	/// \brief Displays the file selection dialog and lets the user select a new input file.
	void showSelectionDialog(QWidget* parent = NULL);

public:

	Q_PROPERTY(QString inputFile READ inputFile)
	Q_PROPERTY(bool storeAtomsWithScene READ storeAtomsWithScene WRITE setStoreAtomsWithScene)

protected:

	/// \brief Stores the parser status and sends a notification message.
	void setStatus(const EvaluationStatus& status);

	/// \brief Saves the class' contents to the given stream.
	/// \sa RefTarget::saveToStream()
	virtual void saveToStream(ObjectSaveStream& stream);

	/// \brief Loads the class' contents from the given stream.
	/// \sa RefTarget::loadFromStream()
	virtual void loadFromStream(ObjectLoadStream& stream);

	/// \brief This method is called once for this object after it has been loaded from the input stream
	virtual void loadFromStreamComplete();

	/// \brief Creates a copy of this object.
	/// \sa RefTarget::Clone()
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

private:

	/// The movie frame currently loaded from the input file.
	int _loadedMovieFrame;

	/// The AtomsObject containing the imported data.
	ReferenceField<AtomsObject> _atoms;

	/// The format specific parser that loads the input file.
	ReferenceField<AtomsFileParser> _parser;

	/// The status returned by the parser during its last call.
	EvaluationStatus _loadStatus;

	/// Controls the playback speed of simulation snapshots.
	PropertyField<int> _framesPerSnapshot;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(AtomsImportObject)
	DECLARE_REFERENCE_FIELD(_atoms)
	DECLARE_REFERENCE_FIELD(_parser)
	DECLARE_PROPERTY_FIELD(_framesPerSnapshot)
};

/******************************************************************************
* A properties editor for the AtomsImportObject class.
******************************************************************************/
class ATOMVIZ_DLLEXPORT AtomsImportObjectEditor : public PropertiesEditor
{
public:

	virtual void setEditObject(RefTarget* newObject);

protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

	/// This method is called when a reference target changes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

protected Q_SLOTS:

	/// Is called when the user presses the Reload button.
	void onReload();

	/// Is called when the user presses the Parser Settings button.
	void onParserSettings();

	/// Updates the contents of the status label.
	void updateInformationLabel();

	/// Is called when the user presses the Animation Settings button.
	void onAnimationSettings();

private:

	QAction* parserSettingsAction;
	QAction* animationSettingsAction;

	ElidedTextLabel* filenameLabel;
	ElidedTextLabel* filepathLabel;
	QLabel* _statusTextLabel;
	QLabel* _statusIconLabel;

	QPixmap statusWarningIcon;
	QPixmap statusErrorIcon;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(AtomsImportObjectEditor)
};


/******************************************************************************
* This dialog lets the user adjust the animation settings of an AtomsImportObject.
******************************************************************************/
class AtomsImportObjectAnimationSettingsDialog : public QDialog
{
	Q_OBJECT

public:

	/// Constructor.
	AtomsImportObjectAnimationSettingsDialog(AtomsImportObject* importObject, QWidget* parent = 0);

protected Q_SLOTS:

	/// This is called when the user has pressed the OK button.
	void onOk();

protected:

	/// The object whose settings are being edited.
	intrusive_ptr<AtomsImportObject> importObject;

	SpinnerWidget* framePerSnapshotSpinner;
	QCheckBox* adjustAnimationIntervalBox;
};

};	// End of namespace AtomViz

#endif // __ATOMS_IMPORT_OBJECT_H
