/*
 * Copyright (C) 2017 Valve Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/** \file
 *
 * Test cases which exercise illegal operations when a texture/sampler object
 * has been referenced by one or more texture handles.
 */

#include "common.h"

static struct piglit_gl_test_config *piglit_config;

PIGLIT_GL_TEST_CONFIG_BEGIN

	piglit_config = &config;
	config.supports_gl_compat_version = 33;
	config.supports_gl_core_version = 33;
	config.khr_no_error_support = PIGLIT_HAS_ERRORS;

PIGLIT_GL_TEST_CONFIG_END

static enum piglit_result
call_TextureParameter_when_texture_is_referenced(void *data)
{
	GLuint tex;

	tex = piglit_integer_texture(GL_RGBA32I, 16, 16, 0, 0);

	glTextureParameteri(tex, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glGetTextureHandleARB(tex);
	if (!piglit_check_gl_error(GL_NO_ERROR))
		return PIGLIT_FAIL;

	/* The ARB_bindless_texture_spec says:
	 *
	 * "The error INVALID_OPERATION is generated by TexImage*,
	 *  CopyTexImage*, CompressedTexImage*, TexBuffer*, TexParameter*, as
	 *  well as other functions defined in terms of these, if the texture
	 *  object to be modified is referenced by one or more texture or image
	 *  handles."
	 */
	glTextureParameteri(tex, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	if (!piglit_check_gl_error(GL_INVALID_OPERATION))
		return PIGLIT_FAIL;

	return PIGLIT_PASS;
}

static enum piglit_result
call_TexImage_when_texture_is_referenced(void *data)
{
	GLuint tex;

	tex = piglit_rgbw_texture(GL_RGBA32F, 16, 16, GL_FALSE, GL_FALSE,
				  GL_UNSIGNED_NORMALIZED);

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32I, 16, 16, 0, GL_RGBA_INTEGER,
		     GL_INT, NULL);
	glGetTextureHandleARB(tex);
	if (!piglit_check_gl_error(GL_NO_ERROR))
		return PIGLIT_FAIL;

	/* The ARB_bindless_texture_spec says:
	 *
	 * "The error INVALID_OPERATION is generated by TexImage*,
	 *  CopyTexImage*, CompressedTexImage*, TexBuffer*, TexParameter*, as
	 *  well as other functions defined in terms of these, if the texture
	 *  object to be modified is referenced by one or more texture or image
	 *  handles."
	 */
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32I, 16, 16, 0, GL_RGBA_INTEGER,
		     GL_INT, NULL);
	if (!piglit_check_gl_error(GL_INVALID_OPERATION))
		return PIGLIT_FAIL;

	return PIGLIT_PASS;
}

static enum piglit_result
call_CopyTexImage_when_texture_is_referenced(void *data)
{
	GLuint tex;

	tex = piglit_rgbw_texture(GL_RGBA32F, 16, 16, GL_FALSE, GL_FALSE,
				  GL_UNSIGNED_NORMALIZED);

	glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, 0, 0, 16, 16, 0);
	glGetTextureHandleARB(tex);
	if (!piglit_check_gl_error(GL_NO_ERROR))
		return PIGLIT_FAIL;

	/* The ARB_bindless_texture_spec says:
	 *
	 * "The error INVALID_OPERATION is generated by TexImage*,
	 *  CopyTexImage*, CompressedTexImage*, TexBuffer*, TexParameter*, as
	 *  well as other functions defined in terms of these, if the texture
	 *  object to be modified is referenced by one or more texture or image
	 *  handles."
	 */
	glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, 0, 0, 16, 16, 0);
	if (!piglit_check_gl_error(GL_INVALID_OPERATION))
		return PIGLIT_FAIL;

	return PIGLIT_PASS;
}

static enum piglit_result
call_CompressedTexImage_when_texture_is_referenced(void *data)
{
	void *compressed;
	GLuint tex;
	GLint size;

	tex = piglit_rgbw_texture(GL_COMPRESSED_RGBA_BPTC_UNORM, 16, 16,
				  GL_FALSE, GL_FALSE, GL_UNSIGNED_NORMALIZED);

	glGetTexLevelParameteriv(GL_TEXTURE_2D, 0,
				 GL_TEXTURE_COMPRESSED_IMAGE_SIZE, &size);

	compressed = malloc(size);
	glGetCompressedTexImage(GL_TEXTURE_2D, 0, compressed);

	glCompressedTexImage2D(GL_TEXTURE_2D, 0, GL_COMPRESSED_RGBA_BPTC_UNORM,
			       16, 16, 0, size, compressed);
	glGetTextureHandleARB(tex);
	if (!piglit_check_gl_error(GL_NO_ERROR))
		return PIGLIT_FAIL;

	/* The ARB_bindless_texture_spec says:
	 *
	 * "The error INVALID_OPERATION is generated by TexImage*,
	 *  CopyTexImage*, CompressedTexImage*, TexBuffer*, TexParameter*, as
	 *  well as other functions defined in terms of these, if the texture
	 *  object to be modified is referenced by one or more texture or
	 *  image handles."
	 */
	glCompressedTexImage2D(GL_TEXTURE_2D, 0, GL_COMPRESSED_RGBA_BPTC_UNORM,
			       16, 16, 0, size, compressed);
	if (!piglit_check_gl_error(GL_INVALID_OPERATION))
		return PIGLIT_FAIL;

	free(compressed);
	return PIGLIT_PASS;
}

static enum piglit_result
call_TexBuffer_when_texture_is_referenced(void *data)
{
	static const float red[4] = {1, 0, 0, 0};
	GLuint tex, tbo;

	glGenBuffers(1, &tbo);
	glBindBuffer(GL_TEXTURE_BUFFER, tbo);
	glBufferData(GL_TEXTURE_BUFFER, sizeof(red), red, GL_STATIC_DRAW);

	glGenTextures(1, &tex);
	glBindTexture(GL_TEXTURE_BUFFER, tex);
	glTexBuffer(GL_TEXTURE_BUFFER, GL_RGBA32F, tbo);

	glTexBuffer(GL_TEXTURE_BUFFER, GL_RGBA32F, tbo);
	glGetTextureHandleARB(tex);
	if (!piglit_check_gl_error(GL_NO_ERROR))
		return PIGLIT_FAIL;

	/* The ARB_bindless_texture_spec says:
	 *
	 * "The error INVALID_OPERATION is generated by TexImage*,
	 *  CopyTexImage*, CompressedTexImage*, TexBuffer*, TexParameter*, as
	 *  well as other functions defined in terms of these, if the texture
	 *  object to be modified is referenced by one or more texture or
	 *  image handles."
	 */
	glTexBuffer(GL_TEXTURE_BUFFER, GL_RGBA32F, tbo);
	if (!piglit_check_gl_error(GL_INVALID_OPERATION))
		return PIGLIT_FAIL;

	return PIGLIT_PASS;
}

static enum piglit_result
call_BufferData_when_texture_is_referenced(void *data)
{
	static const float red[4] = {1, 0, 0, 0};
	GLuint tex, tbo;

	glGenBuffers(1, &tbo);
	glBindBuffer(GL_TEXTURE_BUFFER, tbo);
	glBufferData(GL_TEXTURE_BUFFER, sizeof(red), red, GL_STATIC_DRAW);

	glGenTextures(1, &tex);
	glBindTexture(GL_TEXTURE_BUFFER, tex);
	glTexBuffer(GL_TEXTURE_BUFFER, GL_RGBA32F, tbo);

	glBufferData(GL_TEXTURE_BUFFER, sizeof(red), red, GL_STATIC_DRAW);
	glGetTextureHandleARB(tex);
	if (!piglit_check_gl_error(GL_NO_ERROR))
		return PIGLIT_FAIL;

	/* The ARB_bindless_texture spec says:
	 *
	 * "The error INVALID_OPERATION is generated by BufferData if it is
	 *  called to modify a buffer object bound to a buffer texture while
	 *  that texture object is referenced by one or more texture handles."
	 */
	glBufferData(GL_TEXTURE_BUFFER, sizeof(red), red, GL_STATIC_DRAW);
	if (!piglit_check_gl_error(GL_INVALID_OPERATION))
		return PIGLIT_FAIL;

	return PIGLIT_PASS;
}

static enum piglit_result
call_SamplerParameter_when_texture_is_referenced(void *data)
{
	GLuint sampler, texture;

	texture = piglit_integer_texture(GL_RGBA32I, 16, 16, 0, 0);
	sampler = new_sampler();

	glSamplerParameteri(sampler, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glGetTextureSamplerHandleARB(texture, sampler);
	if (!piglit_check_gl_error(GL_NO_ERROR))
		return PIGLIT_FAIL;

	/* The ARB_bindless_texture spec says:
	 *
	 * "When a sampler object is referenced by one or more texture
	 *  handles, the sampler parameters of the object may not be changed.
	 *  The error INVALID_OPERATION is generated when calling
	 *  SamplerParameter* functions to modify a sampler object referenced
	 *  by one or more texture handles."
	 */
	glSamplerParameteri(sampler, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	if (!piglit_check_gl_error(GL_INVALID_OPERATION))
		return PIGLIT_FAIL;

	return PIGLIT_PASS;
}

static const struct piglit_subtest subtests[] = {
	{
		"Call glTextureParameter* when a texture handle is referenced",
		"call_TextureParameter_when_texture_referenced",
		call_TextureParameter_when_texture_is_referenced,
		NULL
	},
	{
		"Call glTexImage* when a texture handle is referenced",
		"call_TexImage_when_texture_referenced",
		call_TexImage_when_texture_is_referenced,
		NULL
	},
	{
		"Call glCopyTexImage* when a texture handle is referenced",
		"call_CopyTexImage_when_texture_referenced",
		call_CopyTexImage_when_texture_is_referenced,
		NULL
	},
	{
		"Call glCompressedTexImage* when a texture handle is referenced",
		"call_CompressedTexImage_when_texture_referenced",
		call_CompressedTexImage_when_texture_is_referenced,
		NULL
	},
	{
		"Call glTexBuffer* when a texture handle is referenced",
		"call_TexBuffer_when_texture_referenced",
		call_TexBuffer_when_texture_is_referenced,
		NULL
	},
	{
		"Call glBufferData when a texture handle is referenced",
		"call_BufferData_when_texture_referenced",
		call_BufferData_when_texture_is_referenced,
		NULL
	},
	{
		"Call glSamplerParameter* when a texture handle is referenced",
		"call_SamplerParameter_when_texture_referenced",
		call_SamplerParameter_when_texture_is_referenced,
		NULL
	},
	{
		NULL,
		NULL,
		NULL,
		NULL
	}
};

enum piglit_result
piglit_display(void)
{
	return PIGLIT_FAIL;
}

void
piglit_init(int argc, char **argv)
{
	enum piglit_result result;

	piglit_require_extension("GL_ARB_bindless_texture");
	result = piglit_run_selected_subtests(subtests,
					      piglit_config->selected_subtests,
					      piglit_config->num_selected_subtests,
					      PIGLIT_SKIP);
	piglit_report_result(result);
}
