/*	Reference_Resolver

PIRL CVS ID: Reference_Resolver.java,v 1.47 2012/04/16 06:04:10 castalia Exp

Copyright (C) 2003-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Conductor;

import java.util.Vector;
import java.util.List;
import java.util.Iterator;
import java.util.Enumeration;
import java.io.*;
import java.text.ParseException;
import java.lang.IllegalArgumentException;

//	JCM package for expression parser/evaluator.
import edu.hws.jcm.data.*;

import PIRL.Database.Database;
import PIRL.Database.Database_Exception;
import PIRL.Configuration.Configuration;
import PIRL.PVL.Parameter;
import PIRL.PVL.Value;
import PIRL.PVL.Selector;
import PIRL.PVL.Selection;
import PIRL.PVL.PVL_Exception;

/**	A <I>Reference_Resolver</I> resolves embedded references in
	text strings.
<P>
	A Reference_Resolver recognizes two kinds of references:
	<I>database field references</I> and <I>parameter source
	references</I>. All references are enclosed in curly brace ('{' and
	'}') characters. A leading dollar sign ('$') character indicates a
	parameter source reference; otherwise it's a database field
	reference. All field references are resolved against a specified
	Database, and all parameter references are resolved against a
	specified list of Parmerters; if no Parameter list is provided the
	Configuration Parameters from the Database will be used.
<P>
	The resolved value of a reference replaces the reference in the
	string being resolved. Replacement occurs depth-wise: the most
	deeply nested references are resolved first. The substitution of
	a resolved reference value in a nested reference modifies the
	reference accordingly. Thus a reference may be constructed from
	the values of other references.
<P>
	A field reference is a token that produces a database query: the
	query is submitted to the Reference_Resolver's Database and then
	replaced by the first result. Field references have the following
	form:
<P>
	<BLOCKQUOTE>
	<B>{</B><I>Catalog</I><B>.</B><I>Table</I><B>.</B><I>Field</I><B>:</B><I>Key</I><B>}</B>
	</BLOCKQUOTE>
<P>
	The components preceding the colon form a fully qualified field
	name, and the component following the colon is a match (SQL WHERE)
	criteria. Data are retrieved from the field only when they match
	the criteria. If more than one result is returned, only the first
	is used.
<P>
	A parameter reference is resolved against the Parameter source
	list. The parameter reference is replaced with the value of the
	Parameter having the reference name. Parameter references have the
	following form:
<P>
	<BLOCKQUOTE>
	<B>${</B><I>Pathname</I><B>@</B><I>Sequence</I><B>[</B><I>Index</I><B>]...}</B>
	</BLOCKQUOTE>
<P>
	The <I>Pathname</I> is a simple, relative or absolute Parameter
	pathname used to {@link PIRL.PVL.Parameter#Find(String) Find} the
	parameter in the Database Configuration. The <I>Sequence</I>, with
	its preceeding "at" ('@') delimiter , is optional; if present it
	indicates which of a multiple sequence parameters that match the
	Pathname to select (the first parameter is Sequence 0). The
	<I>Index</I>, with its enclosing square bracket ('[' and ']')
	characters, is also optional; if present it indicates which element
	of an Array Value to select (the first element is Index 0).
	Multiple Array Value element specifiers may be provided to select
	from multidimensional Array Values. Both the Sequence and Index
	specifiers may contain nested references. They may also be
	mathematical expressions that evaluate to any numeric value, which
	is truncated to an integer for selecting Array Value elements.
<P>
	References that can not be resolved are replaced with the {@link
	#Default_Value Default_Value}. However, if the default is null,
	then an Unresolved_Reference will be thrown.
<P>
	Any escaped character - a character preceeded by a backslash ('\')
	character - is not interpreted with any special meaning. Both
	the backslash and the special character are left in place.
<P>
	@author		Bradford Castalia, Christian Schaller - UA/PIRL
	@version	1.47
	@see		PIRL.Database
	@see		PIRL.Configuration
*/
public class Reference_Resolver
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Conductor.Reference_Resolver (1.47 2012/04/16 06:04:10)";

/**	The delimiter of database field reference components.
<p>
	<b>N.B.</b>: This is not necessarily the same as the
	{@link Database#Table_Reference_Component_Delimiter() Database table
	reference component delimiter}.
*/
public static final char
	COMPONENTS_DELIMITER			= '.';

/**	The marker preceeding a key specification of a database field reference.
*/
public static final char
	KEY_MARKER						= ':';

/**	The marker preceeding an enclosed parameter reference.
*/
public static final char
	PARAMETER_REFERENCE_MARKER		= '$';

/**	The marker that starts an enclosed reference.
*/
public static final char
	REFERENCE_START_MARKER			= '{';

/**	The marker that ends an enclosed reference.
*/
public static final char
	REFERENCE_END_MARKER			= '}';

/**	The marker that preceeds a parameter reference sequence number.
*/
public static final char
	PARAMETER_SEQUENCE_MARKER		= '@';

/**	The marker that starts a parameter reference array value index number.
*/
public static final char
	ARRAY_ELEMENT_START_MARKER		= '[';

/**	The marker that ends a parameter reference array value index number.
*/
public static final char
	ARRAY_ELEMENT_END_MARKER		= ']';

/**	The marker that escapes any special interpretation of the following
	character.
*/
public static final char
	ESCAPE_MARKER					= '\\';

/**	Alternative source parameter reference token character.
*/
public static final char
	ALTERNATIVE_MARKER				= '|';

/*	Programmer's note:

	Other kinds of references could be recognized. For example, perhaps
	a URL reference of the form "#{<URL>}" might be useful. It is
	helpful to think of references as placeholders for information,
	which is retrieved from another source in the process of resolving
	the reference. This definition is similar to that of a URL.
*/

//	Pattern string being resolved.
private String			Pattern = null;
private int				Pattern_Index;

//	Resolved string with references being replaced by resolved values.
private StringBuffer	Resolved = null;
private int				Resolved_Index;

//	Database to use for resolving field references.
private Database		The_Database = null;

//	Aggregate to use for resolving parameter references.
private Parameter		The_Parameters = null;

//	Case sensitive parameter name matching.
public static final int	CASE_SENSITIVE		= Selector.SPECIFIC;
//	Case insensitive parameter name matching.
public static final int	CASE_INSENSITIVE	= Selector.ANY;
//	Pattern (regex) parameter name matching.
public static final int	PATTERN_MATCH		= Selector.PATTERN_MATCH;
//	The comparison mode to use when finding parameter names.
private	int				Match_Mode			= CASE_SENSITIVE;

//	The default when a reference can not be resolved (null := throw).
private String			Default_Value = null;

// Counter to keep track of depth in a nested reference.
private int				Level_Counter;

//	Debug control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONSTRUCTOR			= 1 << 0,
	DEBUG_RESOLVE				= 1 << 1,
	DEBUG_PARSE					= 1 << 2,
	DEBUG_FIELD_REFERENCE		= 1 << 3,
	DEBUG_DATABASE				= 1 << 4,
	DEBUG_PARAMETER_REFERENCE	= 1 << 5,
	DEBUG_ARRAY_REFERENCE		= 1 << 6,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a Reference_Resolver to use a Database.
<P>
	@param	database	The Database to use when resolving
		embedded references.
*/
public Reference_Resolver
	(
	Database	database
	)
{
if((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">-< Reference_Resolver");
Database (database);
}

/**	Constructs an empty Reference_Resolver.

	Until a {@link #Database(Database) Database} is assigned this
	Reference_Resolver will not be able to {@link #Resolve Resolve}
	references to their values.
*/
public Reference_Resolver ()
{}

/*==============================================================================
	Accessors
*/
/**	Sets the Database to use when resolving database references.
<P>
	If no Parameters have been specified, then the Database
	Configuration parameters will be used to supply them.
<P>
	@param	database	The Database to use.
	@return	This Reference_Resolver.
	@see	Database#Configuration()
*/
public Reference_Resolver Database
	(
	Database	database
	)
{
The_Database = database;
if (The_Parameters == null)
	Parameters (The_Database.Configuration ());
return this;
}

/**	Gets the Database that will be used when resolving references.
<P>
	@return	The current Database.
*/
public Database Database ()
{return The_Database;}

/**	Sets the Parameters to use when resolving parameter references.
<P>
	Typically an Aggregate is provided for use. Otherwise an
	Aggregate is created to contain the Parameter that is provided.
<P>
	@param	parameters	The Parameters to use. If null an empty
		Aggregate will be used.
	@return	This Reference_Resolver.
*/
public Reference_Resolver Parameters
	(
	Parameter	parameters
	)
{
if (parameters == null)
	The_Parameters = new Parameter ()
		.Classification (Parameter.GROUP);
else if (! parameters.Is_Aggregate ())
	{
	The_Parameters = new Parameter ()
		.Classification (Parameter.GROUP);
	try {The_Parameters.Add (parameters);}
	catch (PVL_Exception exception) {/* Shouldn't happen */}
	}
else
	The_Parameters = parameters;
return this;
}

/**	Gets the Parameters that will be used when resolving parameter
	references.
<P>
	@return	The current Parameter Aggregate. This may be null if
		
*/
public Parameter Parameters ()
{return The_Parameters;}

/**	Sets the parameter match mode.
<p.
	When searching for a parameter to resolve a parameter reference
	the parameter name will be matched using one of these modes:
<p>
<dl>
	<dt>{@link #CASE_SENSITIVE}
		<dd>A case sensitive comparsion of the parameter name with
		the reference name will be made.
	<dt>{@link #CASE_INSENSITIVE}
		<dd>A case insensitive comparison of the parameter name with
		the reference name will be made.
	<dt>{@link #PATTERN_MATCH}
		<dd>The reference name will be treated as a regular expression
		(regex) pattern for a match with the parameter name.
</dl>
<p>
	The default match mode is CASE_SENSITIVE.
<p>
	@param	match_mode	One of the CASE_SENSITIVE, CASE_INSENSITIVE, or
		PATTERN_MATCH values.
	@return	This Reference_Resolver.
	@throws	IllegalArgumentException	If the match mode is not a
		valid value.
	@see	Selection
*/
public Reference_Resolver Match_Mode
	(
	int		match_mode
	)
	throws IllegalArgumentException
{
if (match_mode == CASE_SENSITIVE ||
	match_mode == CASE_INSENSITIVE ||
	match_mode == PATTERN_MATCH)
	Match_Mode = match_mode;
else
	throw new IllegalArgumentException (ID + '\n'
		+ "Not a valid Match Mode value - " + match_mode);
return this;
}

/**	Gets the parameter match mode.
<p>
	@return the current parameter match mode code.
	@see	#Match_Mode(int)
*/
public int Match_Mode ()
{return Match_Mode;}

/**	Sets the default to use when a reference can not be resolved.

	@param	default_value	The String to use as the default.
		If this is null an Unresolved_Reference will be thrown when
		a reference can not be resolved.
	@return	This Reference_Resolver.
*/
public Reference_Resolver Default_Value
	(
	String		default_value
	)
{
Default_Value = default_value;
return this;
}

/**	Gets the default that will be used when a reference can not be resolved.
<P>
	@return	The String to use as the default. This is null if
		Unresolved_Reference will be thrown when a reference can not be
		resolved.
*/
public String Default_Value ()
{return Default_Value;}

/**	Gets the last pattern String that was resolved.
<P>
	@return	The last pattern String. This will be null if no previous
		pattern String was {@link #Resolve(String) Resolve}d.
*/
public String Pattern ()
{return Pattern;}

/**	Gets the last resolved String.
<P>
	<B>Note</B>: If the last {@link #Pattern} did not fully resolve
	due to some problem, then the String returned will be the
	partially resolved result.
<P>
	@return	The last resolved String. This will be null if no previous
		pattern String was {@link #Resolve(String) Resolve}d.
*/
public String Resolved ()
{
if (Resolved != null)
	return Resolved.toString ();
return null;
}

/*==============================================================================
	Resolvers
*/
/**	Resolve references in a pattern String.
<P>
	@param	pattern	The pattern String potentially containing embedded
		references.
	@return The pattern String with its references resolved.
	@throws ParseException if a reference is not correctly formed.
	@throws	Database_Exception	if there is a problem accessing the
		Database.
	@throws	Unresolved_Reference	if the reference is unresolved and
		the Default_Value is null.
*/
public String Resolve
	(
	String	pattern
	)
	throws ParseException, Database_Exception, Unresolved_Reference
{
//	Ensures that references will be completely resolved to a value.
Level_Counter = 1;
return Resolution (pattern);
}

/**	Resolve references in a pattern String down to their end references.
<P>
	@param	pattern	The pattern String potentially containing embedded
		references.
	@return The pattern String with its references resolved to end
		references.
	@throws ParseException if a reference is not correctly formed.
	@throws	Database_Exception	if there is a problem accessing the
		Database.
	@throws	Unresolved_Reference	if the reference is unresolved and
		the Default_Value is null.
*/
public String Resolve_to_End_Reference
	(
	String	pattern
	)
	throws ParseException, Database_Exception, Unresolved_Reference
{
//	Prevents the final resolution of a reference to its value.
Level_Counter = 0;
return Resolution (pattern);
}

/**	Provides the resolution of a pattern.
<P>
	The pattern String is copied to a Resolved StringBuffer where
	references will be substituted with their resolved values.
<P>
	@param	pattern	The Pattern to be subject to resolution.
	@throws	ParseException	if the pattern contains a syntax error.
	@throws	Database_Exception	if there is a problem accessing the
		Database.
	@throws	Unresolved_Reference	if the reference is unresolved and
		the Default_Value is null.
*/
private String Resolution
	(
	String	pattern
	)
	throws ParseException, Database_Exception, Unresolved_Reference
{
if (pattern == null)
	pattern = "";
Pattern	= pattern;
Resolved = new StringBuffer (Pattern);
Resolved_Index = Pattern_Index = 0;
if ((DEBUG & DEBUG_RESOLVE) != 0)
	System.out.println
		(">>> Resolve_References:\n"
		+"    " + pattern.length () + " characters in " + pattern);

//	Parse the pattern.
while (Resolved_Index < Resolved.length ())
	{
	Parse ();
	if ((DEBUG & DEBUG_RESOLVE) != 0)
		System.out.println
			("    Resolve_References: Parse completed up to "
			+ Pattern_Index + '/' + Resolved_Index + '\n'
			+ "    Resolved: " + Resolved);
	}
if ((DEBUG & DEBUG_RESOLVE) != 0)
	System.out.println
	("<<< Resolve_References\n"
	+"    Resolved: " + Resolved);
return Resolved.toString ();
}

/**	The pattern parser.
<P>
	Parsing is applied recursively on each reference that is
	encountered. A reference that does not contain any nested
	references is sent to the {@link #Resolve_Field_Reference(String)
	Resolve_Field_Reference} or {@link
	#Resolve_Parameter_Reference(String) Resolve_Parameter_Reference}
	resolver method.
<P>
	References are substituted into the Resolved StringBuffer with
	their resolved values when tbey contain no nested references.
<P>
	The parser keeps the Pattern_Index location and the Resolved_Index
	location synchronized to the same logical position. The parser
	also keeps track of the nesting level.
<P>
	@throws	ParseException	if the pattern contains a syntax error.
	@throws	Database_Exception	if there is a problem accessing the
		Database.
	@throws	Unresolved_Reference	if the reference is unresolved and
		the Default_Value is null.
*/
private void Parse ()
	throws ParseException, Database_Exception, Unresolved_Reference
{
if((DEBUG & DEBUG_PARSE) != 0)
	System.out.println
		(">>> Parse: from " + Pattern_Index + '/' + Resolved_Index + '\n' +
		Resolved.length () + "> " + Resolved);
boolean
	is_parameter_reference = false;
int
	pattern_start = -1,	//	No reference.
	resolved_start = -1;
if (Resolved.charAt (Resolved_Index) == REFERENCE_START_MARKER)
	{
	if ((DEBUG & DEBUG_PARSE) != 0)
		System.out.println
			("*** Field reference start at "
			+ Pattern_Index + '/' + Resolved_Index
			+" - " + Resolved.substring (Resolved_Index) + '\n'
			+"    Drop '" + REFERENCE_START_MARKER + "' at "
			+ Pattern_Index + '/' + Resolved_Index);
	//	Increment Level_Counter.
	Level_Counter++;
	//	Drop the opening marker.
	Resolved.deleteCharAt (Resolved_Index);
	resolved_start = Resolved_Index;
	pattern_start = Pattern_Index++;
	}
else if ((Resolved.charAt (Resolved_Index) == PARAMETER_REFERENCE_MARKER) &&
		 (Resolved.charAt (Resolved_Index + 1) == REFERENCE_START_MARKER))
	{
	if ((DEBUG & DEBUG_PARSE) != 0)
		System.out.println
			("*** Parameter reference start at "
			+ Pattern_Index + '/' + Resolved_Index
			+" - " + Resolved.substring (Resolved_Index) + '\n'
			+"    Drop '" + PARAMETER_REFERENCE_MARKER + "' and '"
			+ REFERENCE_START_MARKER + "' at "
			+ Pattern_Index + '/' + Resolved_Index);
	is_parameter_reference = true;
	// Increment Level_Counter.
	Level_Counter++;
	/*	Drop the opening markers.

		The first deleteCharAt removes the PARAMETER_REFERENCE_MARKER.
		The second removes the REFERENCE_MARKER_START,
		which is bumped into the PARAMETER_REFERENCE_MARKER's position
		by the latter's deletion.
	*/
	Resolved.deleteCharAt (Resolved_Index);
	Resolved.deleteCharAt (Resolved_Index);
	resolved_start = Resolved_Index;
	pattern_start = Pattern_Index;
	Pattern_Index += 2;
	}

while (Resolved_Index < Resolved.length ())
	{
	switch (Resolved.charAt (Resolved_Index))
		{
		case ESCAPE_MARKER:
			//	Skip the ESCAPE_MARKER.
			if ((DEBUG & DEBUG_PARSE) != 0)
				System.out.println
					("    Skip '" + ESCAPE_MARKER + "' at "
					+ Pattern_Index + '/' + Resolved_Index);
			Resolved_Index++;
			Pattern_Index++;
			if (Resolved_Index < Resolved.length ())
				{
				//	Skip the character that was escaped.
				if ((DEBUG & DEBUG_PARSE) != 0)
					System.out.println
						("    Skip escaped '"
							+ Resolved.charAt (Resolved_Index)
							+ "' at " + Pattern_Index + '/' + Resolved_Index);
				Resolved_Index++;
				Pattern_Index++;
				}
			break;
		case REFERENCE_START_MARKER:
			Parse ();
			break;
		case PARAMETER_REFERENCE_MARKER:
			/*
				May have a parameter reference,
				but only if the character immediately following the
				PARAMETER_REFERENCE_MARKER is a REFERENCE_START_MARKER.
			*/
			if (Resolved.charAt (Resolved_Index + 1) == REFERENCE_START_MARKER)
				Parse ();
			else
				{
				//	Not a parameter reference.
				Resolved_Index++;
				Pattern_Index++;
				}
			break;
		case REFERENCE_END_MARKER:
			if (resolved_start >= 0)
				{
				if ((DEBUG & DEBUG_PARSE) != 0)
					System.out.println
						("*** Reference end at "
						+ Pattern_Index + '/' + Resolved_Index);
				//	Remove the closing marker.
				if ((DEBUG & DEBUG_PARSE) != 0)
					System.out.println
						("    Drop '" + REFERENCE_END_MARKER
						+"' at " + Pattern_Index + '/' + Resolved_Index);
				Resolved.deleteCharAt (Resolved_Index);

				//	Decrement the Level_Counter.
				if (--Level_Counter != 0)
					{
					//	Resolve the reference.
					String
						reference = Resolved.substring
							(resolved_start, Resolved_Index);
					if((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
						System.out.println
							("    " + (is_parameter_reference ?
								"Parameter" : "Field")
							+" reference: from "
							+ pattern_start + '/' + resolved_start
							+" to " + Pattern_Index + '/' + Resolved_Index
							+" - " + reference + '\n'
							+ Resolved.length () + "> " + Resolved);
					try
						{
						Resolved.replace (resolved_start, Resolved_Index,
							(is_parameter_reference ?
								Resolve_Parameter_Reference (reference) :
								Resolve_Field_Reference (reference)));
						}
					catch (ParseException exception)
						{
						throw new ParseException
							(exception.getMessage () + '\n'
							+ "At parse index " + pattern_start
							+ " in \"" + Pattern + "\".",
							pattern_start + exception.getErrorOffset ());
						}
					catch (Database_Exception exception)
						{
						throw new Database_Exception
							(exception.getMessage () + '\n'
							+ "At parse index " + pattern_start
							+ " in \"" + Pattern + "\".");
						}
					catch (Unresolved_Reference exception)
						{
						throw new Unresolved_Reference
							(exception.getMessage () + '\n'
							+ "At parse index " + pattern_start
							+ " in \"" + Pattern + "\".",
							exception.Reference ());
						}
					}

				//	Reset the indices to Parse the resolved value.
				Pattern_Index = pattern_start;
				Resolved_Index = resolved_start;
				if ((DEBUG & DEBUG_PARSE) != 0)
					System.out.println
						("<<< Parse: Next index at: "
						+ Pattern_Index + '/' + Resolved_Index + '\n'
						+ Resolved.length () + "> " + Resolved);
				return;
				}
			else
				//	Throw up: Unexpected reference close.
				throw Parse_Error ("", -Pattern_Index);
		default:
			//	Everything else; ignore.
			Resolved_Index++;
			Pattern_Index++;
	}
}
if (resolved_start >= 0)
	//	Throw up: Unbalanced opening reference.
	throw Parse_Error ("",
		(pattern_start + (is_parameter_reference ? 1 : 0)));
if ((DEBUG & DEBUG_PARSE) != 0)
	System.out.println
		("<<< Parse: No reference found.\n"
		+ Resolved.length () + "> " + Resolved);
return;
}

/**	Resolves a database field reference.
<P>
	A field reference has the following form:
<P>
	<BLOCKQUOTE>
	<I>Catalog</I><B>.</B><I>Table</I><B>.</B><I>Field</I><B>:</B><I>Key</I>
	</BLOCKQUOTE>
<P>
	The components are used to {@link
	PIRL.Database.Database#Select(String, Vector, String) Select} a
	Database Field value from a Catalog.Table where the table record
	matches the Key criteria.
<P>
	<B>N.B.</B>: The  field reference is presumed to have been resolved
	of any nested references.
<P>
	@param	field_reference	The field reference String to be resolved by the
		database.
	@return	The resolved String value.
	@throws	ParseException	if the field reference does not have all of
		the components of the field reference format.
	@throws	Database_Exception	if there is a problem accessing the
		Database.
	@throws	Unresolved_Reference	if the reference is unresolved and
		the Default_Value is null.
*/
public String Resolve_Field_Reference
	(
	String	field_reference
	)
	throws ParseException, Database_Exception, Unresolved_Reference
{
if((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
	System.out.println
		(">>> Reference_Resolver.Resolve_Field_Reference: \""
		+ field_reference + '"');
Vector
	result_vector = new Vector ();
if (The_Database != null)
	{
	String
		catalog	= Catalog_Name (field_reference),
		table	= Table_Name (field_reference),
		field	= Field_Name (field_reference),
		key		= Key (field_reference);
	if((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.println
			("    catalog: " + catalog + '\n'
			+"      table: " + table + '\n'
			+"      field: " + field + '\n'
			+"        key: " + key);
	if (catalog == null ||
		table == null ||
		field == null ||
		key == null)
		throw new ParseException (Error_Message
			("Missing "
			+	((catalog == null) ? "catalog name" :
				((table == null) ? "table name" :
				((field == null) ? "field name" : "key specification")))
			+" in database field reference: \"" + field_reference + "\"."),
			0);
	Vector<String>
		field_vector = new Vector<String> ();
	field_vector.add (field);
	/*
		The Select method returns a Vector of data record Vectors. The
		first Vector always contains the names of each field returned,
		with subsequent Vectors containing the returned data.

		There may be cases in which we receive more than one result
		from the query; we should consider only the first. Further,
		there may be cases in which the query yields no results, in
		which case the results Vector will hold only one element: the
		Vector of the field names. We throw an exception when this
		happens. (We should not throw an exception when more than one
		matching record is found, however.)

		The field reference structure guarantees that the Select method
		will query for only one field, even though we use a form that
		allows for multiple fields. Thus, we can be pretty sure that we
		will receive the following from a query:

		Vector {
			Vector { <Field> },
			Vector { First Matching Result }, // If at least one match!
			Vector { Second Match }, // If more than one match!
			...
		}

		Thus, as long as the Vector has two or more records,
		result_vector.get(1).get(0) will be the result of the query
		that interests us.
	*/
	if((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.println
			("    The_Database.Table_Reference (" + catalog + ", " + table
				+ ") = " + The_Database.Table_Reference (catalog, table));
	result_vector = The_Database.Select
		(The_Database.Table_Reference (catalog, table), field_vector, key);
	if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.println ("    Query results: " + result_vector);
	}
String
	value = null;
if (result_vector.size () > 1)
	{
	value = (String)(((Vector)(result_vector.get (1))).get (0));
	if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.println
			("*** Field reference resolved: " + value);
	}
else
	{
	if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.println
			("*** Field reference unresolved. Using " + Default_Value);
	if ((value = Default_Value) == null)
		throw new Unresolved_Reference (Error_Message
			("Unresolved database field reference \""
				+ field_reference + "\".\n"
			+"Using " + Database_Reference ()),
			field_reference);
	}
if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
	System.out.println
		("<<< Reference_Resolver.Resolve_Field_Reference: \"" + value + '"');
return value;
}

/**	Resolves a parameter source reference.
<P>
	A parameter reference has the following form:
<P>
	<BLOCKQUOTE>
	<I>Pathname</I><B>@</B><I>Sequence</I><B>[</B><I>Index</I><B>]...</B>
	</BLOCKQUOTE>
<P>
	The Pathname is used to retrieve a Parameter value from the source
	Parameters list.
<P>
	The optional Sequence refers to which of a multiple sequence of
	Parameters that all match the Pathname is to be selected; the first
	parameter is Sequence 0.
<P>
	The Index specifies which element of a Parameter with an Array
	Value to use; the first element is Index 0. Mulitple Index values
	access multidimensional Arrays with the left-most specifier
	accessing an element of the top-most Array, the next specifier
	accessing an element of the Array Value selected by the previous
	specifier, etc. If there are less specifiers than the depth of
	Array Values, then the first non-Array Value of the last Array
	selected will satisfy the reference (as if [0] had been specified);
	thus when a parameter reference has no Index specifiers but the
	referenced Parameter has an Array Value, then the first non-Array
	Value will be selected regardless of its nesting depth.
<P>
	<B>N.B.</B>: As a special case, a parameter reference index that is
	negative results in the Value being returned regardless of whether
	it is an Array or not. A nested Array value may be selected by
	providing leading array indices before the negative index in the
	parameter reference, but any additional array indices provided after
	a negative index are ignored.
<P>
	Both the Sequence and Index are subject to mathematical expression
	evaluation. Any expression that can be evaluated to a numerical
	value is allowed; the result will be truncated to an integer.
<P>
	<B>N.B.</B>: The parameter reference is presumed to have been
	resolved of any nested references.
<P>
	@param	parameter_reference	The reference String to be resolved by the
		source Parameters list.
	@return	The resolved String value.
	@throws	ParseException	if the reference contains a syntax error such
		as an unclosed Array Value element specifier, or a Sequence or Index
		specifier that does not evaluate to a numerical value.
	@throws	Unresolved_Reference	if the reference is unresolved and
		the Default_Value is null.
	@see	#Parameter_Reference_Value(String)
*/
public String Resolve_Parameter_Reference
	(
	String	parameter_reference
	)
	throws ParseException, Unresolved_Reference
{
if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
	System.out.println
		(">>> Reference_Resolver.Resolve_Parameter_Reference: \""
		+ parameter_reference + '"');
Value
	parameter_Value = Parameter_Reference_Value (parameter_reference);
String
	value = null;
if (parameter_Value != null)
	{
	//	Get the Value as a String representation.
	if (parameter_Value.Is_Array ())
		{
		if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
			System.out.println ("    Parameter Value is an Array -");
		value = "";
		Value
			array_Value;
		Enumeration
			enumerator = parameter_Value.depthFirstEnumeration ();
		while (enumerator.hasMoreElements ())
			{
			array_Value = (Value)enumerator.nextElement ();
			if (! array_Value.Is_Array ())
				{
				if (value.length () != 0)
					value += ',';
				try {value += array_Value.String_Data ();}
				catch (PVL_Exception exception) {/* Shouldn't happen. */}
				if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
					System.out.println ("    " + value);
				}
			}
		}
	else
		{
		try {value = parameter_Value.String_Data ();}
		catch (PVL_Exception exception) {/* Shouldn't happen. */}
		}
	}
if (value == null)
	{
	if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
		System.out.println
			("*** Parameter reference unresolved. Using " + Default_Value);
	if ((value = Default_Value) == null)
		throw new Unresolved_Reference (Error_Message
			("Unresolved parameter reference \""
			+ parameter_reference + "\"."),
			parameter_reference);
	}
if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
	System.out.println
		("<<< Reference_Resolver.Resolve_Parameter_Reference: \"" + value + '"');
return value;
}

/**	Gets the Parameter that resolves a parameter reference.
<P>
	A parameter reference has the following form:
<P>
	<BLOCKQUOTE>
	<I>Pathname</I><B>@</B><I>Sequence</I><B>[</B><I>Index</I><B>]...</B>
	</BLOCKQUOTE>
<P>
	The Pathname is used to retrieve a Parameter value from the source
	Parameters list.
<P>
	The optional Sequence refers to which of a multiple sequence of
	Parameters that all match the Pathname is to be selected; the first
	parameter is Sequence 0. The Sequence may be a mathematical
	expression which is cast to an integer value.
<P>
	The Array Value Index specifiers are not used here.
<P>
	<B>N.B.</B>: The parameter_reference String argument may contain
	nested parameter source references
	(<B>${</B><I>parameter_reference</I><B>}</B>) or database field
	references (<B>{</B><I>field_reference</I><B>}</B>). However, the
	resolved result is expected to be an unenclosed parameter reference
	which is used to find the resolving Parameter from this
	Reference_Resolver's {@link #Parameters() Parameters}.
<P>
	@param	parameter_reference	The reference String to be resolved by the
		source Parameters list.
	@return	The Parameter that resolves the reference, or null if
		the reference can not be resolved to a Parameter.
	@throws	ParseException	if the reference contains a syntax error.
	@throws	Database_Exception	if there is a problem accessing the
		Database.
*/
public Parameter Resolve_to_Parameter
	(
	String	parameter_reference
	)
	throws ParseException, Database_Exception, Unresolved_Reference
{
Parameter
	parameter = null;
if (parameter_reference != null)
	{
	String
		resolved = Resolve (parameter_reference);
	try {parameter = Parameter_Reference_Parameter (resolved);}
	catch (ParseException exception)
		{
		throw new ParseException (exception.getMessage () + '\n'
			+ "While resolving to Parameter: \"" + parameter_reference + "\".",
			exception.getErrorOffset ());
		}
	}
return parameter;
}

/**	Gets the Parameter that resolves a parameter reference.
<P>
	<B>N.B.</B>: The parameter reference is presumed to have been
	resolved of any nested references.
<P>
	@param	parameter_reference	The reference String to be resolved by the
		source Parameters list.
	@return	The Parameter that resolves the reference, or null if
		the reference can not be resolved to a Parameter.
	@throws	ParseException	if the reference contains a syntax error.
	@see	#Resolve_to_Parameter(String)
*/
public Parameter Parameter_Reference_Parameter
	(
	String	parameter_reference
	)
	throws ParseException
{
if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
	System.out.println
		(">>> Reference_Resolver.Parameter_Reference_Parameter: \""
		+ parameter_reference + '"');
Parameter
	parameter = null;
String
	pathname = Pathname (parameter_reference);
if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
	System.out.println ("    Pathname - " + pathname);
if (pathname != null)
	{
	String
		sequence = Sequence (parameter_reference);
	if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
		System.out.println
			("    parameter sequence specifier: \""
				+ Sequence (parameter_reference) + '"');
	int
		count = 0;
	if (sequence != null)
		{
		try {count = (int)Evaluate_to_double (sequence);}
		catch (ParseException exception)
			{
			throw new ParseException
				(exception.getMessage () + '\n'
				+ "For Parameter sequence of reference \""
				+ parameter_reference + "\".",
				parameter_reference.indexOf (PARAMETER_SEQUENCE_MARKER)
				+ exception.getErrorOffset ());
			}
		}
	//	Find the parameter.
	parameter = Find_Parameter (pathname, count);
	}
if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
	{
	System.out.print
		("<<< Reference_Resolver.Parameter_Reference_Parameter:");
	if (parameter == null)
		System.out.println (" null");
	else
		{
		System.out.println ("");
		try {parameter.Write ();}
		catch (Exception exception)
			{System.out.println (exception.getMessage ());}
		}
	}
return parameter;
}

/**	Gets the Value that resolves a parameter reference.
<P>
	A parameter reference has the following form:
<P>
	<BLOCKQUOTE>
	<I>Pathname</I><B>@</B><I>Sequence</I><B>[</B><I>Index</I><B>]...</B>
	</BLOCKQUOTE>
<P>
	The Pathname is used to retrieve a Parameter Value from the source
	Parameters list.
<P>
	The optional Sequence refers to which of a multiple sequence of
	Parameters, that all match the Pathname, is to be selected; the
	first is Sequence 0. The Sequence may be a mathematical expression
	which is cast to an integer value.
<P>
	The optional Array Value Index specifiers are used to select which
	Value to use from an Array of Values. When no Index specifiers are
	present, the implicit "[0]" specifier is used. If the specifiers
	select an Array Value, then the first non-Array Value of that Array
	Value is selected.
<P>
	<B>N.B.</B>: As a special case, a parameter reference index that is
	negative results in the Value being returned regardless of whether
	it is an Array or not. A nested Array value may be selected by
	providing leading array indices before the negative index in the
	parameter reference, but any additional array indices provided after
	a negative index are ignored.
<P>
	<B>N.B.</B>: The parameter_reference String argument may contain
	nested parameter source references
	(<B>${</B><I>parameter_reference</I><B>}</B>) or database field
	references (<B>{</B><I>field_reference</I><B>}</B>). However, the
	resolved result is expected to be an unenclosed parameter reference
	which is used to find the resolving Parameter from this
	Reference_Resolver's {@link #Parameters() Parameters}.
<P>
	@param	parameter_reference	The reference String to be resolved by the
		source Parameters list.
	@return	The Value that resolves the reference, or null if
		the reference can not be resolved to a Value.
	@throws	ParseException	if the reference contains a syntax error.
	@throws	Database_Exception	if there is a problem accessing the
		Database.
	@throws	Unresolved_Reference	if the reference is unresolved and
		the Default_Value is null.
	@see	#Parameter_Reference_Value(String)
*/
public Value Resolve_to_Value
	(
	String	parameter_reference
	)
	throws ParseException, Database_Exception, Unresolved_Reference
{
Value
	value = null;
if (parameter_reference != null)
	{
	String
		resolved = Resolve (parameter_reference);
	try {value = Parameter_Reference_Value (resolved);}
	catch (ParseException exception)
		{
		throw new ParseException (exception.getMessage () + '\n'
			+ "While resolving to Value: \"" + parameter_reference + "\".",
			exception.getErrorOffset ());
		}
	}
return value;
}

/**	Gets the Value that resolves a parameter reference.
<P>
	When the parameter reference resolves to an Array Value, the
	reference is expected to include array indices that will select a
	non-Array Value from within the Array. However, if no indices are
	provided the first non-Array Value is selected, regardless of its
	Array nesting depth.

	<B>N.B.</B>: As a special case, a parameter reference index that is
	negative results in the Value being returned regardless of whether
	it is an Array or not. A nested Array value may be selected by
	providing leading array indices before the negative index in the
	parameter reference, but any additional array indices provided after
	a negative index are ignored.

	<B>N.B.</B>: The parameter reference (as described for {@link
	#Resolve_to_Value(String)}) is presumed to have been
	resolved of any nested references.
<P>
	@param	parameter_reference	The parameter reference String.
	@return	A Parameter Value selected by the reference, or null if
		no Value was selected.
	@throws	ParseException	if the reference contains a syntax error.
	@see	#Parameter_Reference_Parameter(String)
	@see	#Array_Indices(String)
*/
public Value Parameter_Reference_Value
	(
	String	parameter_reference
	)
	throws ParseException
{
if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
	System.out.println
		(">>> Reference_Resolver.Parameter_Reference_Value: \""
		+ parameter_reference + '"');
Parameter
	parameter = Parameter_Reference_Parameter (parameter_reference);
Value
	value = null;
Get_Value:
if (parameter != null)
	{
	//	Get the value.
	Vector
		indices = Array_Indices (parameter_reference);
	int
		dimension = 0,
		index;
	try {value = parameter.Value ();}
	catch (PVL_Exception exception) {}
	while (value != null &&
		   value.Is_Array () &&
		   dimension < indices.size ())
		{
		if ((index = ((Integer)(indices.get (dimension))).intValue ()) < 0)
			{
			//	Return the entire Array Value.
			dimension = indices.size ();	//	Ignore any remaining dimensions.
			break Get_Value;
			}
		if ((value = value.Get (index)) == null)
			break;
		++dimension;
		}
	if (value != null)
		{
		if (dimension < indices.size () &&
			(index = ((Integer)(indices.get (dimension))).intValue ()) >= 0)
			{
			//	Still have indices; ran out of arrays.
			if (value.Is_Array () ||
				index != 0 ||
				dimension != 0)
				//	Unresolved array reference.
				value = null;
			}
		else
			{
			//	Use the first non-Array Value.
			while (value.Is_Array ())
				value = value.Get (0);
			}
		}
	}
if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
	{
	System.out.print ("<<< Reference_Resolver.Parameter_Reference_Value:");
	if (parameter == null)
		System.out.println (" null");
	else
		{
		System.out.print ("\n"
			+"    parameter - " + parameter.Path_Name () + '\n'
			+"        value - ");
		if (value == null)
			System.out.println ("null");
		else
			{
			try {value.Write ();}
			catch (Exception exception)
				{System.out.println (exception.getMessage ());}
			}
		}
	}
return value;
}

/**	Gets the Vector of Array Value dimension indices from a parameter
	reference.
<P>
	The parameter reference String may contain zero or more Arrary Value
	element specifiers enclosed in square brackets ('[' and ']'). The
	reference String has any nested references {@link #Resolve(String)
	resolved} to their values. Then the substring within each set of
	square brackets is evaluated as a mathematical {@link
	#Resolve_to_double(String) expression}.
<P>
	Array Value elements are accessed outer-to-inner in left-to-right
	order: The left-most array element reference applies to the
	top-most, outer, Value Array; the next array element reference
	applies to the Value Array selected by the previous reference.
<P>
	If no Array Value specifier is found an empty indices Vector is
	returned.
<P>
	@param	parameter_reference	The String containing zero or more Array Value
		element specifiers.
	@return	A Vector of zero or more element indices.
	@throws	ParseException	if the references contain a syntax error.
	@throws	Database_Exception	if there is a problem accessing the
		Database.
	@throws	Unresolved_Reference	if the reference is unresolved and
		the Default_Value is null.
	@see	#Resolve_to_double(String)
*/
public Vector Resolve_Array_Indices
	(
	String	parameter_reference
	)
	throws Database_Exception, ParseException, Unresolved_Reference
{
Vector
	indices = new Vector ();
if (parameter_reference != null)
	{
	String
		resolved = Resolve (parameter_reference);
	try {indices = Array_Indices (resolved);}
	catch (ParseException exception)
		{
		throw new ParseException (exception.getMessage () + '\n'
			+ "While resolving Array Indices: \"" + parameter_reference + "\".",
			exception.getErrorOffset ());
		}
	}
return indices;
}

/**	Resolves a reference to a double value.
<P>
	The reference is fully resolved by an independent
	Reference_Resolver that shares the Database, source Parameters and
	Default_Value with this Reference_Resolver. This ensures that the
	reference will be fully resolved even when the method is not being
	used by this Reference_Resolver object to resolve its own Pattern;
	i.e. the Pattern of this Reference_Resolver will not be disturbed.
<P>
	The resolved reference is parsed by a mathematical expression
	evaluator to produce a double value.
<P>
	@param	reference	The reference String to be resolved. A null
	@return	The resolved double value. If the reference is null, 0.0
		will be returned.
	@throws	ParseException	if the reference contains a syntax error.
	@throws	Database_Exception	if there is a problem accessing the
		Database.
	@throws	Unresolved_Reference	if the reference is unresolved and
		the Default_Value is null.
	@see	#Resolve(String)
	@see	edu.hws.jcm.data.Parser#Parser()
*/
public double Resolve_to_double
	(
	String	reference
	)
	throws Database_Exception, ParseException, Unresolved_Reference
{
double
	value = 0.0;
if (reference != null)
	{
	String
		expression = Resolve (reference);
	try {value = Evaluate_to_double (expression);}
	catch (ParseException exception)
		{
		throw new ParseException (exception.getMessage () + '\n'
			+ "While resolving to double: \"" + reference + "\".",
			exception.getErrorOffset ());
		}
	}
return value;
}

/*==============================================================================
	Utilities
*/
/**	Gets a catlog name from a field reference.
<P>
	A field reference has catalog, table, field and optional key
	components:
<P>
	<BLOCKQUOTE>
	<I>Catalog</I><B>.</B><I>Table</I><B>.</B><I>Field</I><B>:</B><I>Key</I>
	</BLOCKQUOTE>
<P>
	Other than the Key component, if only one component is present, it
	is the Field; if only two components are present, they are the
	Catalog and Table.
<P>
	@param	field_reference		The field reference to parse.
	@return	The catlog name or null if one is not present.
*/
public static String Catalog_Name
	(
	String	field_reference
	)
{
String
	catalog_name = null;
if (field_reference != null)
	{
	int
		catalog_table_index =
			field_reference.indexOf (COMPONENTS_DELIMITER),
		table_field_index =
			field_reference.indexOf (COMPONENTS_DELIMITER,
				catalog_table_index + 1),
		field_key_index =
			field_reference.indexOf (KEY_MARKER);
	if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.print
			("Catalog_Name: \"" + field_reference + "\" "
			+ catalog_table_index + ", "
			+ table_field_index + ", "
			+ field_key_index + " ");
	if (catalog_table_index > 0 &&	//	Must have the first delimiter.
		(field_key_index < 0 ||
		 field_key_index > catalog_table_index))
		{
		catalog_name =
			field_reference.substring (0, catalog_table_index).trim ();
		if (catalog_name.length () == 0)
			catalog_name = null;
		if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
			System.out.print
				("(0, " + catalog_table_index + ") ");
		}
	if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.println (catalog_name);
	}
return catalog_name;
}

/**	Gets a table name from a field reference.
<P>
	A field reference has catalog, table, field and optional key
	components:
<P>
	<BLOCKQUOTE>
	<I>Catalog</I><B>.</B><I>Table</I><B>.</B><I>Field</I><B>:</B><I>Key</I>
	</BLOCKQUOTE>
<P>
	Other than the Key component, if only one component is present,
	it is the Field; if only two components are present, they are
	the Catalog and Table.
<P>
	@param	field_reference		The field reference to parse.
	@return	The table name or null if one is not present.
*/
public static String Table_Name
	(
	String	field_reference
	)
{
String
	table_name = null;
if (field_reference != null)
	{
	int
		catalog_table_index =
			field_reference.indexOf (COMPONENTS_DELIMITER),
		table_field_index =
			field_reference.indexOf (COMPONENTS_DELIMITER,
				catalog_table_index + 1),
		field_key_index =
			field_reference.indexOf (KEY_MARKER);
	if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.print
			("  Table_Name: \"" + field_reference + "\" "
			+ catalog_table_index + ", "
			+ table_field_index + ", "
			+ field_key_index + " ");
	if (field_key_index >= 0 &&
		catalog_table_index > field_key_index)		//	In the Key.
		catalog_table_index = -1;
	if (field_key_index >= 0 &&
		table_field_index > field_key_index)		//	In the Key.
		table_field_index = -1;
	if (catalog_table_index > 0 &&	//	Must have the first delimiter.
		(field_key_index < 0 ||
		(field_key_index > catalog_table_index &&
		 field_key_index > table_field_index)))
		{
		if (table_field_index < 0)
			{
			if (field_key_index > 0)
				table_field_index = field_key_index;
			else
				table_field_index = field_reference.length ();
			}
		if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
			System.out.print
				("(" + (catalog_table_index + 1)
				+", " + table_field_index + ") ");
		table_name = field_reference.substring
			(catalog_table_index + 1, table_field_index).trim ();
		if (table_name.length () == 0)
			table_name = null;
		}
	if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.println (table_name);
	}
return table_name;
}

/**	Gets a field name from a field reference.
<P>
	A field reference has catalog, table, field and optional key
	components:
<P>
	<BLOCKQUOTE>
	<I>Catalog</I><B>.</B><I>Table</I><B>.</B><I>Field</I><B>:</B><I>Key</I>
	</BLOCKQUOTE>
<P>
	Other than the Key component, if only one component is present,
	it is the Field; if only two components are present, they are
	the Catalog and Table.
<P>
	@param	field_reference		The field reference to parse.
	@return	The field name or null if one is not present.
*/
public static String Field_Name
	(
	String	field_reference
	)
{
String
	field_name = null;
if (field_reference != null)
	{
	int
		catalog_table_index =
			field_reference.indexOf (COMPONENTS_DELIMITER),
		table_field_index =
			field_reference.indexOf (COMPONENTS_DELIMITER,
				catalog_table_index + 1),
		field_key_index =
			field_reference.indexOf (KEY_MARKER);
	if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.print
			("  Field_Name: \"" + field_reference + "\" "
			+ catalog_table_index + ", "
			+ table_field_index + ", "
			+ field_key_index + " ");
	if (field_key_index >= 0 &&
		catalog_table_index > field_key_index)		//	In the Key.
		catalog_table_index = -1;
	if (field_key_index >= 0 &&
		table_field_index > field_key_index)		//	In the Key.
		table_field_index = -1;
	if (catalog_table_index <= table_field_index)	//	One or three components.
		{
		if (field_key_index < 0)					//	No Key.
			field_key_index = field_reference.length ();
		if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
			System.out.print
				("(" + (table_field_index + 1)
				+", " + field_key_index + ") ");
		field_name = field_reference.substring
			(table_field_index + 1, field_key_index).trim ();
		if (field_name.length () == 0)
			field_name = null;
		}
	if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.println (field_name);
	}
return field_name;
}

/**	Gets a key from a field reference.
<P>
	A field reference has catalog, table, field and optional key
	components:
<P>
	<BLOCKQUOTE>
	<I>Catalog</I><B>.</B><I>Table</I><B>.</B><I>Field</I><B>:</B><I>Key</I>
	</BLOCKQUOTE>
<P>
	Other than the Key component, if only one component is present,
	it is the Field; if only two components are present, they are
	the Catalog and Table.
<P>
	@param	field_reference		The field reference to parse.
	@return	The key or null if one is not present.
*/
public static String Key
	(
	String	field_reference
	)
{
String
	key = null;
if (field_reference != null)
	{
	int
		field_key_index =
			field_reference.indexOf (KEY_MARKER);
	if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.print
			("         Key: \"" + field_reference + "\" "
			+ field_key_index + " ");
	if (field_key_index >= 0)
		{
		if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
			System.out.print ("(" + (field_key_index + 1) + ") ");
		key = field_reference.substring (field_key_index + 1).trim ();
		if (key.length () == 0)
			key = null;
		}
	if ((DEBUG & DEBUG_FIELD_REFERENCE) != 0)
		System.out.println (key);
	}
return key;
}

/**	Gets a pathname from a parameter reference.
<P>
	A parameter reference has a pathname and optional sequence and
	Array Value element index specifiers:
<P>
	<BLOCKQUOTE>
	<I>Pathname</I><B>@</B><I>Sequence</I><B>[</B><I>Index</I><B>]</B>...
	</BLOCKQUOTE>
<P>
	@param	parameter_reference	The parameter reference to parse.
	@return	The parameter pathname or null if one is not present.
*/
public static String Pathname
	(
	String	parameter_reference
	)
{
String
	pathname = null;
if (parameter_reference != null)
	{
	int
		index = parameter_reference.indexOf (PARAMETER_SEQUENCE_MARKER);
	if (index < 0)
		index = parameter_reference.indexOf (ARRAY_ELEMENT_START_MARKER);
	if (index > 0)
		pathname = parameter_reference.substring (0, index).trim ();
	else
		pathname = parameter_reference;
	if (pathname.length () == 0)
		pathname = null;
	}
return pathname;
}

/**	Gets a parameter sequence from a parameter reference.
<P>
	A parameter reference has a pathname and optional sequence and
	Array Value element index specifiers:
<P>
	<BLOCKQUOTE>
	<I>Pathname</I><B>@</B><I>Sequence</I><B>[</B><I>Index</I><B>]</B>...
	</BLOCKQUOTE>
<P>
	@param	parameter_reference	The parameter reference to parse.
	@return	The parameter sequence or null if one is not present.
*/
public static String Sequence
	(
	String	parameter_reference
	)
{
String
	sequence = null;
if (parameter_reference != null)
	{
	int
		sequence_index =
			parameter_reference.indexOf (PARAMETER_SEQUENCE_MARKER) + 1;
	if (sequence_index > 0)
		{
		int
			index = parameter_reference.indexOf (ARRAY_ELEMENT_START_MARKER);
		if (index < 0)
			index = parameter_reference.length ();
		sequence = parameter_reference.substring (sequence_index, index)
			.trim ();
		if (sequence.length () == 0)
			sequence = null;
		}
	}
return sequence;
}

/**	Gets the parameter Array Value element index specifiers from a
	parameter reference.
<P>
	A parameter reference has a pathname and optional sequence and
	Array Value element index specifiers:
<P>
	<BLOCKQUOTE>
	<I>Pathname</I><B>@</B><I>Sequence</I><B>[</B><I>Index</I><B>]</B>...
	</BLOCKQUOTE>
<P>
	@param	parameter_reference	The parameter reference to parse.
	@return	The element index specifiers or null if none are present. The
		element index enclosures are included.
*/
public static String Index_Specifiers
	(
	String	parameter_reference
	)
{
String
	indices = null;
if (parameter_reference != null)
	{
	int
		index =
			parameter_reference.indexOf (ARRAY_ELEMENT_START_MARKER);
	if (index >= 0)
		{
		indices = parameter_reference.substring (index).trim ();
		if (indices.length () == 0)
			indices = null;
		}
	}
return indices;
}

/**	Gets the Vector of Array Value element indices from a parameter
	reference.
<P>
	<B>N.B.</B>: The parameter reference is presumed to have been
	resolved of any nested references.
<P>
	@param	parameter_reference	The parameter reference String
		containing zero or more Array Value element specifiers.
	@return	A Vector of zero or more element indices.
	@throws	ParseException	if the references contain a syntax error.
	@throws	Database_Exception	if there is a problem accessing the
		Database, or a reference is unresolved and the Default_Value
		is null.
	@see	#Evaluate_to_double(String)
*/
public static Vector<Integer> Array_Indices
	(
	String	parameter_reference
	)
	throws ParseException
{
if ((DEBUG & (DEBUG_ARRAY_REFERENCE | DEBUG_PARAMETER_REFERENCE)) != 0)
	System.out.println (">>> Reference_Resolver.Array_Indices: \""
		+ parameter_reference + '"');
Vector<Integer>
	indices = new Vector<Integer> ();
if (parameter_reference == null)
	parameter_reference = "";
int
	dimension = 0,
	start_index = 0,
	end_index = 0,
	length = parameter_reference.length ();

for (start_index = parameter_reference.indexOf
		(ARRAY_ELEMENT_START_MARKER);
	 start_index >= 0;
	 start_index = parameter_reference.indexOf
	 	(ARRAY_ELEMENT_START_MARKER, end_index++))
	{
	//	Find the end of the array element specifier.
	int
		nested = 0;
	for (end_index = start_index + 1;
		 end_index < length;
		 end_index++)
		{
		if (parameter_reference.charAt(end_index) == ESCAPE_MARKER)
			{
			//	Ignore the next character.
			++end_index;
			continue;
			}
		if (parameter_reference.charAt(end_index) == ARRAY_ELEMENT_END_MARKER)
			{
			if (nested-- == 0)
				//	Final closing marker.
				break;
			//	Nested closing marker.
			continue;
			}
		if (parameter_reference.charAt(end_index) == ARRAY_ELEMENT_START_MARKER)
			//	Nested opening marker.
			++nested;
		}

	if (end_index < length)
		{
		//	Extract, resolve and evaluate the array element specifier.
		String
			specifier = parameter_reference.substring
				(start_index + 1, end_index);
		if ((DEBUG & (DEBUG_ARRAY_REFERENCE | DEBUG_PARAMETER_REFERENCE)) != 0)
			System.out.println
				("    Reference_Resolver.Array_Indices: \""
					+ parameter_reference + "\"\n"
				+"    dimension " + dimension + ": \"" + specifier + '"');
		try {indices.add (new Integer ((int)Evaluate_to_double (specifier)));}
		catch (ParseException exception)
			{
			throw new ParseException
				(exception.getMessage () + '\n'
				+ "In Array Value element specifier " + dimension
					+ ": \"" + specifier + "\".",
				start_index + exception.getErrorOffset ());
			}
		++dimension;
		}
	else
		throw new ParseException (Error_Message
			("Unclosed Array Value element specifier " + dimension
				+ ": \"" + parameter_reference.substring (start_index) + "\"."),
			start_index);
	}
if ((DEBUG & (DEBUG_ARRAY_REFERENCE | DEBUG_PARAMETER_REFERENCE)) != 0)
	System.out.println ("<<< Reference_Resolver.Array_Indices: " + indices);
return indices;
}

/**	Resolves a reference to a double value.
<P>
	<B>N.B.</B>: The expression is presumed to have been resolved of
	any nested references.
<P>
	@param	expression	The expression String to be evaluated.
	@see	#Resolve_to_double(String)
*/
public static double Evaluate_to_double
	(
	String	expression
	)
	throws ParseException
{
if ((DEBUG & (DEBUG_PARAMETER_REFERENCE | DEBUG_ARRAY_REFERENCE)) != 0)
	System.out.println
		(">>> Reference_Resolver.Evaluate_to_double: \"" + expression + '"');
double
	value = 0.0;
if (expression != null)
	{
	try {value = new Parser ().parse (expression).getVal ();}
	catch (ParseError exception)
		{
		throw new ParseException (Error_Message
			(exception.getMessage () + '\n'
			+"Unable to evaluate the expression \"" + expression + "\"\n"
			+"  while parsing the sequence \"" + exception.context.data + "\"\n"
			+"  near sequence position " + exception.context.pos + '.'),
			0);
		}
	}
if ((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
	System.out.println
		("<<< Reference_Resolver.Evaluate_to_double: " + value);
return value;
}

/**	Tests if a String is an enclosed reference.
<p>
	The reference may be a parameter reference or a database reference, but
	it must be completely enclosed in reference markers.
<p>
	<b>N.B.</b>: No check is made to determine if the reference enclosure,
	if it exits, is properly balanced.
<p>
	@param	string	A String to be tested.
	@return	true if the string is an enclosed reference; false otherwise.
*/
public static boolean Enclosed_Reference
	(
	String	string
	)
{
int
	level = -1;
if (string != null &&
	string.length () > 1 &&
	(string.charAt (0) == Reference_Resolver.REFERENCE_START_MARKER ||
	(string.charAt (0) == Reference_Resolver.PARAMETER_REFERENCE_MARKER &&
	 string.charAt (1) == Reference_Resolver.REFERENCE_START_MARKER)))
	{
	level = 0;
	char
		character;
	for (int index = 0;
			 index < string.length ();
			 index++)
		{
		character = string.charAt (index);
		if (character == Reference_Resolver.ESCAPE_MARKER)
			index++;
		else if (character == Reference_Resolver.REFERENCE_START_MARKER)
			level++;
		else if (character == Reference_Resolver.REFERENCE_END_MARKER)
			level--;
		}
	}
return level == 0;
}

/**	Gets the index of an {@link #ALTERNATIVE_MARKER} in a String.
<p>
	The first occurance is found of the {@link #ALTERNATIVE_MARKER}
	character that is not escaped by a preceeding {@link #ESCAPE_MARKER},
	not immediately followed by the same character when this occurs
	the second character is also skipped), and not contained in
	a quoted (single or double) sequence.
<p>
	@param	string	The String to search.
	@return	The index of the first {@link #ALTERNATIVE_MARKER} in the
		string, or -1 if the marker is not found.
*/
public static int Alternative_Marker_Index
	(
	String	string
	)
{
int
	index = -1;
if (string != null)
	{
	char
		character;
	int
		length = string.length ();
	while (++index < length)
		{
		character = string.charAt (index);
		if (character == ALTERNATIVE_MARKER)
			{
			if ((index + 1) == length ||
				string.charAt (index + 1) != ALTERNATIVE_MARKER)
				break;
			++index;
			continue;
			}
		if (character == '\\')
			//	Skip the escaped character.
			++index;
		else
		if (character == '"' ||
			character == '\'')
			{
			//	Skip the quoted sequence.
			char
				quote = character;
			while (++index < length)
				{
				character = string.charAt (index);
				if (character == quote)
					break;
				else
				if (character == '\\')
					//	Skip the escaped character.
					++index;
				}
			}
		}
	if (index >= length)
		index = -1;
	}
return index;
}

/*==============================================================================
	Helpers
*/
/**	Searches for a Parameter from The_Parameters list.
<p>
	The Match_Mode is used to find an Assignment Parameter with a
	matching name. When PATTERN_MATCH is being used, the pathname is
	treated as a regex.
<p>
	@param	pathname	The pathname for the parameter to find.
	@param	skip		The number of matching parameters to skip over.
		If skip is negative, then the last matching parameter, if there
		is one, is selected.
	@return	The matching Parameter, or null if one is not found.
*/
private Parameter Find_Parameter
	(
	String	pathname,
	int		skip
	)
{
if (pathname == null)
	return null;
if((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
	System.out.println (">>> Reference_Resolver.Find_Parameter: "
		+ pathname + ' ' + skip + '\n'
		+ The_Parameters.Description ());

Parameter
	pattern_parameter =
		new Parameter (pathname).Classification (Parameter.ASSIGNMENT),
	last_parameter = null,
	parameter = null;
Selector
	selection = new Selection (Match_Mode)
		.Name (true)
		.And (true)
		.Classification (true);
while ((parameter = The_Parameters.Find
			(pattern_parameter, selection, last_parameter))
		!= null)
	{
	if (skip == 0)
		break;
	if (skip > 0)
		skip--;
	last_parameter = parameter;
	}
if (skip < 0)
	parameter = last_parameter;
if((DEBUG & DEBUG_PARAMETER_REFERENCE) != 0)
	System.out.println ("<<< Reference_Resolver.Find_Parameter: "
		+ ((parameter == null) ? "(not found)" : parameter.Name ()));
return parameter;
}

private static String Array_Elements
	(
	List	list
	)
{
String
	description = "";
Iterator
	entries = list.iterator ();
while (entries.hasNext ())
	description += "[" + entries.next () + "]";
return description;
}


private static String Error_Message
	(
	String	message
	)
{
if (message == null || message.length () == 0)
	message = ID;
else
	message = ID + "\n" + message;
return message;
}


private Database_Exception Database_Error
	(
	String	message
	)
{
return new Database_Exception (Error_Message
	("Problem accessing " + Database_Reference () + ":\n"
	+ message));
}

/**	Produces a one line description of the Database.
<p>
	The description is of the form:
<blockquote><p>
	<i>TYPE</i> database on host <i>HOST</i>
</blockquote>
	The Configuration is used to obtain the {@link Database#TYPE TYPE}
	and {@link Database#HOST HOST} parameters.
<p>
	@return	A database reference string.
*/
private String Database_Reference ()
{
if (The_Database == null)
	return "No Database";
return
	The_Database.Configuration ().Get_One (Database.TYPE) +
	" database on host " +
	The_Database.Configuration ().Get_One (Configuration.HOST);
}


private ParseException Parse_Error
	(
	String	message,
	int		index
	)
{
/*	The message will be empty if there was a problem with reference
	enclosure balance. In this case a negative index for the error
	indicates an unbalanced closing marker, while a positive index
	indicates an unbalanced opening marker.

	Any other message is from the method that is used to resolve the
	specific type of reference.
*/
if (message.length () == 0)
	{
	if (index < 0)
		{
		index = -index;
		message = "Unexpected closing";
		}
	else
		message = "Unbalanced opening";
	message += " reference marker.";
	}
message += "\nAt parse index " + index + " in \"" + Pattern + "\".";
return new ParseException (Error_Message (message), index);
}


}
