/***********************************************************************************
* Smooth Tasks
* Copyright (C) 2009-2010 Toni Dietze <smooth-tasks@derflupp.e4ward.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "TaskbuttonFrameSvg.h"

#include "Global.h"

#include <QRectF>

namespace SmoothTasks {
	TaskbuttonFrameSvg::TaskbuttonFrameSvg(QObject *parent)
	 :	Plasma::FrameSvg(parent),
		m_customMargins(false),
		m_customMarginLeft(0.0),
		m_customMarginTop(0.0),
		m_customMarginRight(0.0),
		m_customMarginBottom(0.0)
	{
	}
	
	void TaskbuttonFrameSvg::getMaximalMargins(qreal &left, qreal &top, qreal &right, qreal &bottom) {
		QString elementPrefixes[] = {ACTIVE, MINIMIZED, FOCUS, ATTENTION, HOVER, NORMAL};

		setElementPrefix(elementPrefixes[0]);
		getMargins(left, top, right, bottom);
		
		for (unsigned i = 1; i < sizeof(elementPrefixes) / sizeof(QString); ++i) {
			setElementPrefix(elementPrefixes[i]);
			qreal l, t, r, b;
			getMargins(l, t, r, b);
			if (l > left  ) left = l;
			if (t > top   ) top = t;
			if (r > right ) right = r;
			if (b > bottom) bottom = b;
		}
	}
	
	QRectF TaskbuttonFrameSvg::minimalContentsRect() {
		QString elementPrefixes[] = {ACTIVE, MINIMIZED, FOCUS, ATTENTION, HOVER, NORMAL};

		setElementPrefix(elementPrefixes[0]);
		QRectF contentsRect(this->contentsRect());
		
		for (unsigned i = 1; i < sizeof(elementPrefixes) / sizeof(QString); ++i) {
			setElementPrefix(elementPrefixes[i]);
			contentsRect &= this->contentsRect();
		}
		return contentsRect;
	}

	void TaskbuttonFrameSvg::getMaximalMarginsCustom(qreal &left, qreal &top, qreal &right, qreal &bottom) {
		if (m_customMargins) {
			getMarginsCustom(left, top, right, bottom);
		} else {
			getMaximalMargins(left, top, right, bottom);
		}
	}
	
	QRectF TaskbuttonFrameSvg::minimalContentsRectCustom() {
		if (m_customMargins) {
			return contentsRectCustom();
		} else {
			return minimalContentsRect();
		}
	}

	void TaskbuttonFrameSvg::setCustomMargins(const qreal &left, const qreal &top, const qreal &right, const qreal &bottom) {
		m_customMarginLeft = left;
		m_customMarginTop = top;
		m_customMarginRight = right;
		m_customMarginBottom = bottom;
	}
	
	qreal TaskbuttonFrameSvg::customMarginSize(const Plasma::MarginEdge edge) const {
		switch(edge) {
			case Plasma::LeftMargin  : return m_customMarginLeft  ; break;
			case Plasma::TopMargin   : return m_customMarginTop   ; break;
			case Plasma::RightMargin : return m_customMarginRight ; break;
			case Plasma::BottomMargin: return m_customMarginBottom; break;
		}
		return 0.0;
	}
	
	void TaskbuttonFrameSvg::enableCustomMargins(const bool &enable) {
		m_customMargins = enable;
	}
	
	QRectF TaskbuttonFrameSvg::contentsRectCustom() const {
		if (m_customMargins) {
			QSizeF size = ::Plasma::FrameSvg::frameSize();
			return QRectF(
					m_customMarginLeft,
					m_customMarginTop,
					size.width()  - m_customMarginLeft - m_customMarginRight,
					size.height() - m_customMarginTop  - m_customMarginBottom
				);
		} else {
			return ::Plasma::FrameSvg::contentsRect();
		}
	}
	
	void TaskbuttonFrameSvg::getMarginsCustom(qreal &left, qreal &top, qreal &right, qreal &bottom) const {
		if (m_customMargins) {
			left   = m_customMarginLeft;
			top    = m_customMarginTop;
			right  = m_customMarginRight;
			bottom = m_customMarginBottom;
		} else {
			::Plasma::FrameSvg::getMargins(left, top, right, bottom);
		}
	}
	
	qreal TaskbuttonFrameSvg::marginSizeCustom(const Plasma::MarginEdge edge) const {
		if (m_customMargins) {
			return customMarginSize(edge);
		} else {
			return ::Plasma::FrameSvg::marginSize(edge);
		}
	}
}