/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 * 
 *    Project              : SagCAD
 *    Description          : CAD/CAM
 *    Source               : culcfunc.c
 * 
 *    ----------------------------------
 * 
 *    License              : GNU General Public License (GPL)
 *    Copyright            : (C) 1998-2007 by Yutaka Sagiya
 *    email                : kappa@a6s.highway.ne.jp
 *                         : yutaka@sagiya.com
 *    Begin                : 1998/10/13
 *    Last                 : 2007/11/08
 * ====================================================================
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <math.h>

#include "intl.h"
#define _CULCFUNC_
#include "culcfunc.h"



/* -------------------------------------------------------------------
 * 説明 : 二つの値が同じかを調べる
 * 関数 : isEqual
 * 引数 : double a
 *		: double b
 * 戻値 : int 1 : Equal
 *		: 
 * 外部 : 
 * 内部 : 
 */
int isEqual(double a, double b, int isEqual_digits)
{
	if (a + (1/pow(10, isEqual_digits)) > b 
		&& 
		a - (1/pow(10, isEqual_digits)) < b) 
	{
		return 1;
	}
	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : 円弧の最大値と最小値を求める
 * 関数 : arc_max_and_min
 * 引数 : rtndat
 *		:	円弧  (cx[1] , cy[1]) , r[1] (sx[1] , st[1]) - (ex[1] , ey[1])
 * 戻値 : rtndat
 *		:	max x  sx[2]
 *      :   max y  sy[2]
 *		:	min x  ex[2]
 *      :   min y  ey[2]
 * 外部 : 無し
 * 内部 : RtnDat lah
 */
int arc_max_and_min(struct RtnDat *rtn)
{
	double right, hi, left, low;
	double sa, ea;
	int sa_frag, ea_frag;
	double max_x, max_y, min_x, min_y;
	struct RtnDat lah;

	right = rtn->cx[1] + rtn->r[1];
	hi    = rtn->cy[1] + rtn->r[1];
	left  = rtn->cx[1] - rtn->r[1];
	low   = rtn->cy[1] - rtn->r[1];
	
	/* SA を求める */
	lah.sx[1] = rtn->cx[1];
	lah.sy[1] = rtn->cy[1];
	lah.ex[1] = rtn->sx[1];
	lah.ey[1] = rtn->sy[1];
	la(&lah);
	sa = lah.angle;
	/* EA を求める */
	lah.sx[1] = rtn->cx[1];
	lah.sy[1] = rtn->cy[1];
	lah.ex[1] = rtn->ex[1];
	lah.ey[1] = rtn->ey[1];
	la(&lah);
	ea = lah.angle;
	
	/* ---------------------------------
	 * 象限
	 *     |   
	 *   2 | 1 
	 *     |   
	 *  -------
	 *     |   
	 *   3 | 4 
	 *     |   
	 * ---------------------------------
	 */
	/* SA の象限 */
	if (sa >= 0 && sa < 90) {
		sa_frag = 1;
	}
	else if (sa >= 90 && sa < 180) {
		sa_frag = 2;
	}
	else if (sa >= 180 && sa < 270) {
		sa_frag = 3;
	}
	else if (sa >= 270 && sa < 360) {
		sa_frag = 4;
	}
	else {
		return -1;
	}
	
	/* EA の象限 */
	if (ea >= 0 && ea < 90) {
		ea_frag = 1;
	}
	else if (ea >= 90 && ea < 180) {
		ea_frag = 2;
	}
	else if (ea >= 180 && ea < 270) {
		ea_frag = 3;
	}
	else if (ea >= 270 && ea < 360) {
		ea_frag = 4;
	}
	else {
		return -1;
	}


	/* SA = 1 */
	if (sa_frag == 1 && ea_frag == 1) {
		if (sa < ea) {
			max_x = rtn->sx[1];
			max_y = rtn->ey[1];
			min_x = rtn->ex[1];
			min_y = rtn->sy[1];
		}
		else if (sa > ea) {
			max_x = right;
			max_y = hi;
			min_x = left;
			min_y = low;
		}
	}
	else if (sa_frag == 1 && ea_frag == 2) {
		max_x = rtn->sx[1];
		max_y = hi;
		min_x = rtn->ex[1];
		if (rtn->sy[1] < rtn->ey[1]) {
			min_y = rtn->sy[1];
		}
		else {
			min_y = rtn->ey[1];
		}
	}
	else if (sa_frag == 1 && ea_frag == 3) {
		max_x = rtn->sx[1];
		max_y = hi;
		min_x = left;
		min_y = rtn->ey[1];
	}
	else if (sa_frag == 1 && ea_frag == 4) {
		if (rtn->sx[1] > rtn->ex[1]) {
			max_x = rtn->sx[1];
		}
		else {
			max_x = rtn->ex[1];
		}
		max_y = hi;
		min_x = left;
		min_y = low;
	}


	/* SA = 2 */
	else if (sa_frag == 2 && ea_frag == 2) {
		if (sa < ea) {
			max_x = rtn->sx[1];
			max_y = rtn->sy[1];
			min_x = rtn->ex[1];
			min_y = rtn->ey[1];
		}
		else if (sa > ea) {
			max_x = right;
			max_y = hi;
			min_x = left;
			min_y = low;
		}
	}
	else if (sa_frag == 2 && ea_frag == 3) {
		if (rtn->sx[1] > rtn->ex[1]) {
			max_x = rtn->sx[1];
		}
		else {
			max_x = rtn->ex[1];
		}
		max_y = rtn->sy[1];
		min_x = left;
		min_y = rtn->ey[1];
	}
	else if (sa_frag == 2 && ea_frag == 4) {
		max_x = rtn->ex[1];
		max_y = rtn->sy[1];
		min_x = left;
		min_y = low;
	}
	else if (sa_frag == 2 && ea_frag == 1) {
		max_x = right;
		if (rtn->sy[1] > rtn->ey[1]) {
			max_y = rtn->sy[1];
		}
		else {
			max_y = rtn->ey[1];
		}
		min_x = left;
		min_y = low;
	}


	/* SA = 3 */
	else if (sa_frag == 3 && ea_frag == 3) {
		if (sa < ea) {
			max_x = rtn->ex[1];
			max_y = rtn->sy[1];
			min_x = rtn->sx[1];
			min_y = rtn->ey[1];
		}
		else if (sa > ea) {
			max_x = right;
			max_y = hi;
			min_x = left;
			min_y = low;
		}
	}
	else if (sa_frag == 3 && ea_frag == 4) {
		max_x = rtn->ex[1];
		if (rtn->sy[1] > rtn->ey[1]) {
			max_y = rtn->sy[1];
		}
		else {
			max_y = rtn->ey[1];
		}
		min_x = rtn->sx[1];
		min_y = low;
	}
	else if (sa_frag == 3 && ea_frag == 1) {
		max_x = right;
		max_y = rtn->ey[1];
		min_x = rtn->sx[1];
		min_y = low;
	}
	else if (sa_frag == 3 && ea_frag == 2) {
		max_x = right;
		max_y = hi;
		if (rtn->sx[1] < rtn->ex[1]) {
			min_x = rtn->sx[1];
		}
		else {
			min_x = rtn->ex[1];
		}
		min_y = low;
	}


	/* SA = 4 */
	else if (sa_frag == 4 && ea_frag == 4) {
		if (sa < ea) {
			max_x = rtn->ex[1];
			max_y = rtn->ey[1];
			min_x = rtn->sx[1];
			min_y = rtn->sy[1];
		}
		else if (sa > ea) {
			max_x = right;
			max_y = hi;
			min_x = left;
			min_y = low;
		}
	}
	else if (sa_frag == 4 && ea_frag == 1) {
		max_x = right;
		max_y = rtn->ey[1];
		if (rtn->sx[1] < rtn->ex[1]) {
			min_x = rtn->sx[1];
		}
		else {
			min_x = rtn->ex[1];
		}
		min_y = rtn->sy[1];
	}
	else if (sa_frag == 4 && ea_frag == 2) {
		max_x = right;
		max_y = hi;
		min_x = rtn->ex[1];
		min_y = rtn->sy[1];
	}
	else if (sa_frag == 4 && ea_frag == 3) {
		max_x = right;
		max_y = hi;
		min_x = left;
		if (rtn->sy[1] < rtn->ey[1]) {
			min_y = rtn->sy[1];
		}
		else {
			min_y = rtn->ey[1];
		}
	}
	else {
		return -1;
	}

	rtn->sx[2] = max_x;
	rtn->sy[2] = max_y;
	rtn->ex[2] = min_x;
	rtn->ey[2] = min_y;
	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [1] 角度の単位を、度 (DEG) からラジアン (RAD) に変換する。
 *		: 度 → RAD
 * 関数 : degrad
 * 引数 : double
 * 戻値 : double
 * 外部 : 無し
 * 内部 : 無し
 */
double degrad(double a)
{	return (a * (PI / 180));	}



/* -------------------------------------------------------------------
 * 説明 : [2] 角度の単位を、ラジアン (RAD) 度 (DEG) から度 (DEG) に変換する。
 *		: RAD → 度
 * 関数 : raddeg
 * 引数 : double
 * 戻値 : double
 * 外部 : 無し
 * 内部 : 無し
 */
double raddeg(double a)
{	return ((180 / PI) * a);	}



/* -------------------------------------------------------------------
 * 説明 : [3] 始点・終点からＳＡ・ＥＡを求める
 * 関数 : pa
 * 引数 : rtndat
 *			 円 中心		  (cx[1] , cy[1])
 *			 円上の点または点 (sx[1] , sy[1])
 * 戻値 : rtndat
 *			 角度  angle
 * 外部 : 無し
 * 内部 : RtnDat lah
 */
int pa(struct RtnDat *dumy)
{
	struct RtnDat lah;
	int msg;
	
	lah.sx[1] = dumy->cx[1];
	lah.sy[1] = dumy->cy[1];
	lah.ex[1] = dumy->sx[1];
	lah.ey[1] = dumy->sy[1];
	msg = la(&lah);
	dumy->angle = lah.angle;
	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [4] 始点と終点を結ぶ線分を sx[2] : ex[2] に
 *		: 内分する点求める
 * 関数 : ppp
 * 引数 : rtndat
 *		:	始点	(sx[1] , sy[1])
 *		:	終点	(ex[1] , ey[1])
 *		:	比		(sx[2] , ex[2])
 * 戻値 : rtndat
 *		:	点		(sx[3] , sy[3])
 * 外部 : 無し
 * 内部 : 
 */
int ppp(struct RtnDat *dumy)
{
	dumy->sx[3] = ((dumy->sx[2]) * (dumy->ex[1]) + (dumy->ex[2]) * (dumy->sx[1])) / ((dumy->sx[2]) + (dumy->ex[2]));
	dumy->sy[3] = ((dumy->sx[2]) * (dumy->ey[1]) + (dumy->ex[2]) * (dumy->sy[1])) / ((dumy->sx[2]) + (dumy->ex[2]));
	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [5] 直線に直交する直線の角度
 * 関数 : lla
 * 引数 : rtndat
 *		:	線	  (sx[1] , sy[1]) - 終点 (ex[1] , ey[1])
 * 戻値 : rtndat
 *		:	角度   angle
 * 外部 : 無し
 * 内部 : RtnDat lah
 */
int lla(struct RtnDat *dumy)
{
	struct RtnDat lah;
	int msg;

	lah.sx[1] = dumy->sx[1];
	lah.sy[1] = dumy->sy[1];
	lah.ex[1] = dumy->ex[1];
	lah.ey[1] = dumy->ey[1];
	msg = la(&lah);
	dumy->angle = lah.angle;
	dumy->angle = (dumy->angle) + 90;

	if ((dumy->angle) < 0)
		dumy->angle = (dumy->angle) + 360.;
	else if ((dumy->angle) >= 360)
		dumy->angle = (dumy->angle) - 360.;
	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [6] 始点と角度と距離で直線の終点を求める
 * 関数 : pap
 * 引数 : rtndat
 *		:	始点   (sx[1] , sy[1])
 *		:	角度   angle
 *		:	長さ   l
 * 戻値 : rtndat
 *		:	点	   (ex[1] , ey[1])
 * 外部 : 無し
 * 内部 : 
 */
int pap(struct RtnDat *dumy)
{
	if ((dumy->angle) < 0)
		dumy->angle = 360 + (dumy->angle);

	dumy->angle = degrad(dumy->angle);
	dumy->ex[1] = (dumy->sx[1]) + (dumy->l) * cos(dumy->angle);
	dumy->ey[1] = (dumy->sy[1]) + (dumy->l) * sin(dumy->angle);

	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [7] 始点と終点を与えた直線のオフセット
 * 関数 : lo
 * 引数 : rtndat
 *		:	線	(sx[1] , sy[1]) - (ex[1] , ey[1])
 *		:	長さ	 l
 * 戻値 : rtndat
 *		:	線	(sx[2] , sy[2]) - (ex[2] , ey[2]) 進行方向	左
 *		:	線	(sx[3] , sy[3]) - (ex[3] , ey[3]) 進行方向	右
 * 外部 : 無し
 * 内部 : RtnDat llah,paph
 */
int lo(struct RtnDat *dumy)
{
	struct RtnDat llah,paph;
	int msg;

	/* 直線に直交する直線の角度 */
	llah.sx[1] = dumy->sx[1]; 
	llah.sy[1] = dumy->sy[1];
	llah.ex[1] = dumy->ex[1]; 
	llah.ey[1] = dumy->ey[1];
	msg = lla(&llah);

	/* 始点終点のそれぞれを始点とするLLAで求めた角度に
	 * オフセットの距離離れた点を求める。
	 * 進行方向左にオフセット
	 */
	paph.sx[1] = dumy->sx[1]; 
	paph.sy[1] = dumy->sy[1]; 
	paph.angle = llah.angle; 
	paph.l = dumy->l;
	msg = pap(&paph);
	dumy->sx[2] = paph.ex[1]; 
	dumy->sy[2] = paph.ey[1];

	paph.sx[1] = dumy->ex[1]; 
	paph.sy[1] = dumy->ey[1]; 
	paph.angle = llah.angle; 
	paph.l = dumy->l;
	msg = pap(&paph);
	dumy->ex[2] = paph.ex[1]; 
	dumy->ey[2] = paph.ey[1];

	/* １８０度反転方向のオフセット
	 * 進行方向右にオフセット
	 * 直線に直交する直線の角度
	 */
	llah.sx[1] = dumy->ex[1];
	llah.sy[1] = dumy->ey[1];
	llah.ex[1] = dumy->sx[1];
	llah.ey[1] = dumy->sy[1];
	msg = lla(&llah);

	paph.sx[1] = dumy->sx[1];
	paph.sy[1] = dumy->sy[1];
	paph.angle = llah.angle; 
	paph.l = dumy->l;
	msg = pap(&paph);
	dumy->sx[3] = paph.ex[1];
	dumy->sy[3] = paph.ey[1];

	paph.sx[1] = dumy->ex[1];
	paph.sy[1] = dumy->ey[1];
	paph.angle = llah.angle; 
	paph.l = dumy->l;
	msg = pap(&paph);
	dumy->ex[3] = paph.ex[1];
	dumy->ey[3] = paph.ey[1];

	dumy->type = 2;
	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [8] 直線の角度
 * 関数 : la
 * 引数 : *rtndat
 *		:	線		(sx[1] , sy[1]) - (ex[1] , ey[1])
 * 戻値 : int
 *		:	角度	 angle
 * 外部 : 無し
 * 内部 : 
 */
int la(struct RtnDat *dumy)
{
	double dx;
	double dy;


	dx = dumy->ex[1] - dumy->sx[1];
	dy = dumy->ey[1] - dumy->sy[1];
	dx = sg(dx, calcu_digits);
	dy = sg(dy, calcu_digits);

	/* ---------------------------------
	 * X not move
	 * Y + move  
	 * Deg90 
	 */
	if (dx == 0 && dy > 0) {
		dumy->angle = 90;
		dumy->type = 1;
	}

	/* ---------------------------------
	 * X not move
	 * Y - move
	 * Deg 270
	 */
	else if (dx == 0 && dy < 0) {
		dumy->angle = 270;
		dumy->type = 1;
	}

	/* ---------------------------------
	 * X + move
	 * Y not move 
	 * Deg 0
	 */
	else if (dx > 0 && dy == 0) {
		dumy->angle = 0;
		dumy->type = 1;
	}

	/* ---------------------------------
	 * X - move
	 * Y not move
	 * Deg 180 
	 */
	else if (dx < 0 && dy == 0) {
		dumy->angle = 180;
		dumy->type = 1;
	}

	/* ---------------------------------
	 * X not move
	 * Y not move
	 */
	else if (dx == 0 && dy == 0) {
		/* msg "移動が有りません" */
		dumy->type = 0;
	}

	else {
		dumy->angle = atan(fabs(dy) / fabs(dx));
		dumy->angle = raddeg(dumy->angle);
		
		if (dx < 0 && dy > 0)
			dumy->angle = 180 - dumy->angle;
		if (dx < 0 && dy < 0)
			dumy->angle = 180 + dumy->angle;
		if (dx > 0 && dy < 0)
			dumy->angle = 360 - dumy->angle;
		if (dumy->angle < 0)
			dumy->angle = dumy->angle + 360;
		if (dumy->angle >= 360)
			dumy->angle = dumy->angle - 360;
		dumy->type = 1;
	}
	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [9] ２点間の距離
 * 関数 : pp
 * 引数 : RtnDat
 *		:	点 1	(sx[1] , sy[1])
 *		:	点 2	(ex[1] , ey[1])
 * 戻値 : int
 *		:	l
 * 外部 : 無し
 * 内部 : 無し
 */
int pp(struct RtnDat *dumy)
{
	dumy->l = sqrt(((dumy->ex[1] - dumy->sx[1]) * (dumy->ex[1] - dumy->sx[1])) + ((dumy->ey[1] - dumy->sy[1]) * (dumy->ey[1] - dumy->sy[1])));
	dumy->type = 1;

	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [10] ２直線の交点  初めてのグラフィックス P120
 * 関数 : llp
 * 引数 : RtnDat
 *		:	(sx[1] , sy[1]) (ex[1] , ey[1])
 *		:	(sx[2] , sy[2]) (ex[2] , ey[2])
 * 戻値 : int
 *		:	(sx[3] , sy[3]) , (type = 1:正常終了   0:エラー )
 * 外部 : 無し
 * 内部 : 
 */
int llp(struct RtnDat *dumy)
{
//#define LLP_TEST
#ifdef LLP_TEST
	char str[256];
#endif

	double a1, a2, b1, b2;


#ifdef LLP_TEST
	sprintf(str, "llp 1 : LINE 1 (%f , %f)-(%f , %f)\n", dumy->sx[1], dumy->sy[1], dumy->ex[1], dumy->ey[1]);
	OneShotLog(str);
	sprintf(str, "llp 1 : LINE 2 (%f , %f)-(%f , %f)\n", dumy->sx[2], dumy->sy[2], dumy->ex[2], dumy->ey[2]);
	OneShotLog(str);
#endif

	/* 線１が垂直線で、線２が水平線のとき */
	if (
		sg(dumy->sx[1], compa_digits) == sg(dumy->ex[1], compa_digits) 
		&& 
		sg(dumy->sy[2], compa_digits) == sg(dumy->ey[2], compa_digits)
		) 
	{
		dumy->sx[3] = sg(dumy->sx[1], calcu_digits);
		dumy->sy[3] = sg(dumy->sy[2], calcu_digits);
		dumy->type = 1;
#ifdef LLP_TEST
	sprintf(str, "llp 2 : 線１が垂直線で、線２が水平線\n");
	OneShotLog(str);
#endif
		return 1;
	}

	/* 線２が垂直線で、線１が水平線のとき */
	if (
		sg(dumy->sx[2], compa_digits) == sg(dumy->ex[2], compa_digits) 
		&& 
		sg(dumy->sy[1], compa_digits) == sg(dumy->ey[1], compa_digits)
		) 
	{
		dumy->sx[3] = sg(dumy->sx[2], calcu_digits);
		dumy->sy[3] = sg(dumy->sy[1], calcu_digits);
		dumy->type = 1;
#ifdef LLP_TEST
	sprintf(str, "llp 2 : 線２が垂直線で、線１が水平線\n");
	OneShotLog(str);
#endif
		return 1;
	}

	/* 線１が、９０度線で線２が斜め線のとき */
	if (
		sg(dumy->ex[1] - dumy->sx[1], compa_digits) == 0 
		&& 
		sg(dumy->ex[2] - dumy->sx[2], compa_digits) != 0
		) 
	{
		a2 = (dumy->ey[2] - dumy->sy[2]) / (dumy->ex[2] - dumy->sx[2]);
		b2 = dumy->sy[2] - a2 * dumy->sx[2];
		dumy->sx[3] = dumy->sx[1];
		dumy->sy[3] = a2 * dumy->sx[3] + b2;
		dumy->type = 1;
#ifdef LLP_TEST
	sprintf(str, "llp 2 : 線１が、９０度線で線２が斜め線\n");
	OneShotLog(str);
#endif
	}

	/* ２直線ともＹ軸に平行のとき */
	if (
		sg(dumy->ex[1] - dumy->sx[1], compa_digits) == 0 
		&& 
		sg(dumy->ex[2] - dumy->sx[2], compa_digits) == 0
		) 
	{
		dumy->type = 0;
#ifdef LLP_TEST
	sprintf(str, "llp 3 : ２直線ともＹ軸に平行\n");
	OneShotLog(str);
#endif
		return 0;
	}

	/* 線２が、９０度線で線１が斜め線のとき */
	if (
		sg(dumy->ex[2] - dumy->sx[2], compa_digits) == 0 
		&& 
		sg(dumy->ex[1] - dumy->sx[1], compa_digits) != 0
		) 
	{
		a1 = (dumy->ey[1] - dumy->sy[1]) / (dumy->ex[1] - dumy->sx[1]);
		b1 = dumy->sy[1] - a1 * dumy->sx[1];
		dumy->sx[3] = dumy->sx[2];
		dumy->sy[3] = a1 * dumy->sx[3] + b1;
		dumy->type = 1;
#ifdef LLP_TEST
	sprintf(str, "llp 4 : 線２が、９０度線で線１が斜め線\n");
	OneShotLog(str);
#endif
	}

	/* 線１＆線２が、斜め線または、水平線のとき */
	if (
		sg(dumy->ex[1] - dumy->sx[1], compa_digits) != 0 
		&& 
		sg(dumy->ex[2] - dumy->sx[2], compa_digits) != 0
		) 
	{
		a1 = (dumy->ey[1] - dumy->sy[1]) / (dumy->ex[1] - dumy->sx[1]);
		a2 = (dumy->ey[2] - dumy->sy[2]) / (dumy->ex[2] - dumy->sx[2]);
		b1 = dumy->sy[1] - a1 * dumy->sx[1];
		b2 = dumy->sy[2] - a2 * dumy->sx[2];
		if (a1 - a2 != 0) {
			dumy->sx[3] = (b2 - b1) / (a1 - a2);
			dumy->sy[3] = a1 * dumy->sx[3] + b1;
			dumy->type = 1;
		}
		else if (a1 - a2 == 0) {
			/* ２直線が平行 */
			dumy->type = 0;
			return 0;
		}
#ifdef LLP_TEST
	sprintf(str, "llp 5 : 線１＆線２が、斜め線または、水平線\n");
	OneShotLog(str);
	sprintf(str, "llp 5 : sx[3] = %f   sy[3] = %f\n", dumy->sx[3], dumy->sy[3]);
	OneShotLog(str);
#endif
	}
	return 1;
}



/* -------------------------------------------------------------------
 * 説明 : [11] 直線と円の交点  初めてのグラフィックス P126
 * 関数 : lcp
 * 引数 : RtnDat
 *		:	線 1	(sx[1],sy[1]) - (ex[1],ey[1])
 *		:	円 1	(cx[1],cy[1]) ,r[1]
 * 戻値 : RtnDat
 *		:	type	0 , 1 , 2
 *		:	点 1	(sx[2],sy[2])
 *		:	点 2	(sx[3],sy[3])
 * 外部 : 無し
 * 内部 : RtnDat plph,pph
 */
int lcp(struct RtnDat *dumy)
{
//#define LCP
	struct RtnDat plph, pph;
	int lcptype, msg;
	double aa1, bb, cc, dd, mm, nn;
	double f_dumy, disp, r1;
	int debug = 0;


#ifdef LCP
	debug = 1;
#endif


	lcptype = 0;

	/* 直線と円の中心との距離を求める */
	plph.sx[2] = dumy->cx[1];
	plph.sy[2] = dumy->cy[1];
	plph.sx[1] = dumy->sx[1];
	plph.sy[1] = dumy->sy[1];
	plph.ex[1] = dumy->ex[1];
	plph.ey[1] = dumy->ey[1];
	msg = plp(&plph);

	/* ２点間の距離 */
	pph.sx[1] = dumy->cx[1];
	pph.sy[1] = dumy->cy[1];
	pph.ex[1] = plph.ex[2];
	pph.ey[1] = plph.ey[2];
	msg = pp(&pph);

	disp = pph.l;
	r1 = dumy->r[1];



	/* 接している */
	if (debug > 0) g_print("接している : (disp == r1) = %d\n", 
			(disp + (1/pow(10, compa_digits)) > r1 
			&& 
			disp - (1/pow(10, compa_digits)) < r1) );
	/* 重なっている */
	if (debug > 0) g_print("重なっている : sg(disp, compa_digits) < sg(r1, compa_digits) = %d\n", 
			sg(disp, compa_digits) < sg(r1, compa_digits));
	/* 離れている */
	if (debug > 0) g_print("離れている : sg(disp, compa_digits) > sg(r1, compa_digits) = %d\n", 
			sg(disp, compa_digits) > sg(r1, compa_digits));

	if (debug > 0) g_print("           : sg(disp, compa_digits) = %f\n", sg(disp, compa_digits));
	if (debug > 0) g_print("           : sg(r1, compa_digits) = %f\n", sg(r1, compa_digits));


	/* -----------------------------------------------------
	 * 円と直線の位置関係と交点
	 *
	 */

// (1/pow(10, compa_digits))

	/* -----------------------------------------------------
	 * 接している 
	 */
	if (disp + (1/pow(10, compa_digits)) > r1 
		&& 
		disp - (1/pow(10, compa_digits)) < r1) 
	{
//	if (sg(disp, compa_digits) == sg(r1, compa_digits)) {
		dumy->type = 1;
		dumy->sx[2] = plph.ex[2];
		dumy->sy[2] = plph.ey[2];
	}



	/* -----------------------------------------------------
	 * 重なっている
	 */
	else if (sg(disp, compa_digits) < sg(r1, compa_digits)) {
		dumy->type = 2;
		/* 分母が0になるとエラーとなるためX座標とY座標を計算時に入れ換える。 */
		if (sg((dumy->ex[1] - dumy->sx[1]), compa_digits) == 0) {
			f_dumy = dumy->sx[1];
			dumy->sx[1] = dumy->sy[1];
			dumy->sy[1] = f_dumy;
			f_dumy = dumy->ex[1];
			dumy->ex[1] = dumy->ey[1];
			dumy->ey[1] = f_dumy;
			f_dumy = dumy->cx[1];
			dumy->cx[1] = dumy->cy[1];
			dumy->cy[1] = f_dumy;
			lcptype = 1;
		}

		mm = (dumy->ey[1] - dumy->sy[1]) / (dumy->ex[1] - dumy->sx[1]);
		nn = dumy->sy[1] - mm * dumy->sx[1];
		aa1 = (mm * mm + 1);
		bb = 2 * (mm * nn - dumy->cx[1] - dumy->cy[1] * mm);
		cc = dumy->cx[1] * dumy->cx[1] - dumy->r[1] * dumy->r[1] + (nn - dumy->cy[1]) * (nn - dumy->cy[1]);
		dd = bb * bb - aa1 * aa1 * cc;

		dumy->sx[2] = (-bb - sqrt(bb * bb - 4 * aa1 * cc)) / (2 * aa1);
		dumy->sy[2] = mm * dumy->sx[2] + nn;
		dumy->sx[3] = (-bb + sqrt(bb * bb - 4 * aa1 * cc)) / (2 * aa1);
		dumy->sy[3] = mm * dumy->sx[3] + nn;

		/* X 座標とY 座標を入れ換えたものを元に戻す。 */
		if (lcptype == 1) {
			f_dumy = dumy->sx[1];
			dumy->sx[1] = dumy->sy[1];
			dumy->sy[1] = f_dumy;
			f_dumy = dumy->ex[1];
			dumy->ex[1] = dumy->ey[1];
			dumy->ey[1] = f_dumy;
			f_dumy = dumy->cx[1];
			dumy->cx[1] = dumy->cy[1];
			dumy->cy[1] = f_dumy;
			f_dumy = dumy->sx[2];
			dumy->sx[2] = dumy->sy[2];
			dumy->sy[2] = f_dumy;
			f_dumy = dumy->sx[3];
			dumy->sx[3] = dumy->sy[3];
			dumy->sy[3] = f_dumy;
			lcptype = 0;
		}
	}



	/* -----------------------------------------------------
	 * 離れている
	 */
	else if (sg(disp, compa_digits) > sg(r1, compa_digits)) {
		dumy->type = 0;
		return 0;
	}

	return 1;
}



/* -------------------------------------------------------------------
 * 説明 : [12] ２円の交点
 * 関数 : ccp
 * 引数 : rtndat
 *		:	円 1	  (cx[1] , cy[1]) , r[1]
 *		:	円 2	  (cx[2] , cy[2]) , r[2]
 * 戻値 : rtndat
 *		:	type	   0 , 1 , 2
 *		:	交点 1	  (sx[1] , sy[1])
 *		:	交点 2	  (sx[2] , sy[2])
 * 外部 : 無し
 * 内部 : RtnDat pph,lah,paph,lcph,llah
 */
int ccp(struct RtnDat *dumy)
{
	struct RtnDat pph,lah,paph,lcph,llah, DumyRtn;
	double p;
	int msg;
	double dist,r1pr2,r1mr2;


	/* 円1の方が大きくなるようにデータを入れかえる */
	if (dumy->r[1] < dumy->r[2]) {
		DumyRtn.cx[1] = dumy->cx[2];
		DumyRtn.cy[1] = dumy->cy[2];
		DumyRtn.r[1] = dumy->r[2];

		DumyRtn.cx[2] = dumy->cx[1];
		DumyRtn.cy[2] = dumy->cy[1];
		DumyRtn.r[2] = dumy->r[1];

		dumy->cx[1] = DumyRtn.cx[1];
		dumy->cy[1] = DumyRtn.cy[1];
		dumy->r[1] = DumyRtn.r[1];

		dumy->cx[2] = DumyRtn.cx[2];
		dumy->cy[2] = DumyRtn.cy[2];
		dumy->r[2] = DumyRtn.r[2];
	}

	/* ２中心の距離 */
	pph.sx[1] = dumy->cx[1];
	pph.sy[1] = dumy->cy[1];
	pph.ex[1] = dumy->cx[2];
	pph.ey[1] = dumy->cy[2];
	msg = pp(&pph);

	/* ２円の位置関係と交点 */
	dist = pph.l;
	/* R1 + R2 */
	r1pr2 = dumy->r[1] + dumy->r[2];
	/* R1 - R2 */
	r1mr2 = dumy->r[1] - dumy->r[2];
	
	
	/* -----------------------------------------------------
	 * 交点が 1 個 
	 */
	if (
		(dist + (1/pow(10, compa_digits)) > r1pr2 
		&& 
		dist - (1/pow(10, compa_digits)) < r1pr2) 
		
		|| 
		
		(dist + (1/pow(10, compa_digits)) > r1mr2 
		&& 
		dist - (1/pow(10, compa_digits)) < r1mr2) 
		) 
	{
		dumy->type = 1;
		lah.sx[1] = dumy->cx[1];
		lah.sy[1] = dumy->cy[1];
		lah.ex[1] = dumy->cx[2];
		lah.ey[1] = dumy->cy[2];
		msg = la(&lah);

		paph.sx[1] = dumy->cx[1];
		paph.sy[1] = dumy->cy[1];
		paph.angle = lah.angle;
		paph.l = dumy->r[1];
		msg = pap(&paph);

		dumy->sx[1] = paph.ex[1];
		dumy->sy[1] = paph.ey[1];
		dumy->sx[2] = 0;
		dumy->sy[2] = 0;
	}

	/* -----------------------------------------------------
	 * 交点が2個
	 */
	else if (
				sg(dist, compa_digits) < sg(r1pr2, compa_digits) 
				&& 
				sg(dist, compa_digits) > sg(r1mr2, compa_digits)
			) 
	{
		dumy->type = 2;
		p = ((dumy->r[1] * dumy->r[1]) - (dumy->r[2] * dumy->r[2]) + (pph.l * pph.l)) / (2 * pph.l);
		/* -------------------------------------------------
		 * LA 直線の角度
		 */
		lah.sx[1] = dumy->cx[1];
		lah.sy[1] = dumy->cy[1];
		lah.ex[1] = dumy->cx[2];
		lah.ey[1] = dumy->cy[2];
		msg = la(&lah);

		/* -------------------------------------------------
		 * PAP 始点と角度と距離で直線の終点を求める
		 */
		paph.sx[1] = dumy->cx[1];
		paph.sy[1] = dumy->cy[1];
		paph.angle = lah.angle;
		paph.l = p;
		msg = pap(&paph);

		/* -------------------------------------------------
		 * LLA 直線に直交する直線の角度
		 */
		llah.sx[1] = dumy->cx[1];
		llah.sy[1] = dumy->cy[1];
		llah.ex[1] = dumy->cx[2];
		llah.ey[1] = dumy->cy[2];
		msg = lla(&llah);

		/* -------------------------------------------------
		 * PAP 始点と角度と距離で直線の終点を求める
		 */
		paph.sx[1] = paph.ex[1];
		paph.sy[1] = paph.ey[1];
		paph.angle = llah.angle;
		paph.l = 10;
		msg = pap(&paph);

		/* -------------------------------------------------
		 * LCP 直線と円の交点
		 */
		lcph.sx[1] = paph.sx[1];
		lcph.sy[1] = paph.sy[1];
		lcph.ex[1] = paph.ex[1];
		lcph.ey[1] = paph.ey[1];
		lcph.cx[1] = dumy->cx[1];
		lcph.cy[1] = dumy->cy[1];
		lcph.r[1] = dumy->r[1];
		msg = lcp(&lcph);

		dumy->sx[1] = lcph.sx[2];
		dumy->sy[1] = lcph.sy[2];
		dumy->sx[2] = lcph.sx[3];
		dumy->sy[2] = lcph.sy[3];
	}

	/* -----------------------------------------------------
	 * 交点が 0 個 
	 * (dist > r1pr2) || (dist < r1mr2 && dist > 0)
	 */
	else {
		dumy->type = 0;
		return 0;
	}
	return 1;
}



/* -------------------------------------------------------------------
 * 説明 : [13] PLP	 点と直線の最近点（１点を通る垂線）
 * 関数 : plp
 * 引数 : RtnDat
 *		:	線 1	(sx[1],sy[1]) - (ex[1],ey[1])
 *		:	点 1	(sx[2],sy[2])
 * 戻値 : RtnDat
 *		:	type	 0 , 1
 *		:	点 1	(ex[2],ey[2])
 *		:	点 2	(ex[3],ey[3]) 交点と関係ない点
 * 外部 : 無し
 * 内部 : RtnDat llah,paph,llph
 */
int plp(struct RtnDat *dumy)
{
	struct RtnDat llah,paph,llph;
	int msg;


	/* 直線に直交する直線の角度を求める。 */
	llah.sx[1] = dumy->sx[1];
	llah.sy[1] = dumy->sy[1];
	llah.ex[1] = dumy->ex[1];
	llah.ey[1] = dumy->ey[1];
	msg = lla(&llah);
	/* dumy->ANGLE = llah.ANGLE;  */
	
	/* 始点（１点）と角度と距離[1]で直線の終点を求める。 */
	paph.sx[1] = dumy->sx[2];
	paph.sy[1] = dumy->sy[2];
	paph.angle = llah.angle;
	paph.l = 100;
	msg = pap(&paph);
	
	/* ２直線の交点 */
	llph.sx[1] = dumy->sx[1];
	llph.sy[1] = dumy->sy[1];
	llph.ex[1] = dumy->ex[1];
	llph.ey[1] = dumy->ey[1];
	llph.ex[2] = dumy->sx[2];
	llph.ey[2] = dumy->sy[2];
	llph.sx[2] = paph.ex[1];
	llph.sy[2] = paph.ey[1];
	msg = llp(&llph);
	
	dumy->ex[2] = llph.sx[3];
	dumy->ey[2] = llph.sy[3];
	dumy->ex[3] = paph.ex[1];
	dumy->ey[3] = paph.ey[1];
	dumy->type = 1;
	return 1;
}



/* -------------------------------------------------------------------
 * 説明 : [14] １点を通り円に接する直線の終点  初めてのグラフィックス P129
 * 関数 : pcl
 * 引数 : rtndat
 *		:	点		(sx[1] , sy[1])
 *		:	円		(cx[1] , cy[1]) , r[1]
 * 戻値 : rtndat
 *		:	type	 0 , 1 , 21])
 *		:	点 1	(ex[1] , ey[1])
 *		:	点 2	(ex[2] , ey[2])
 * 外部 : 無し
 * 内部 : RtnDat pph,lah,llah,paph,ccph
 */
int pcl(struct RtnDat *dumy)
{
	struct RtnDat pph,lah,llah,paph,ccph;
	int msg;


	/* ２点間の距離 */
	pph.sx[1] = dumy->sx[1];
	pph.sy[1] = dumy->sy[1];
	pph.ex[1] = dumy->cx[1];
	pph.ey[1] = dumy->cy[1];
	msg = pp(&pph);

	if (sg(pph.l, compa_digits) < sg(dumy->r[1], compa_digits)) {
		dumy->type = 0;
		return 1;
	}

	if (sg(pph.l, compa_digits) == sg(dumy->r[1], compa_digits)) {
		/* 点が円上にあるから、点と円の中心を結ぶ線に垂直で点を通る線 */
		/* LLA	 直線に直交する直線の角度 */
		llah.sx[1] = dumy->sx[1];
		llah.sy[1] = dumy->sy[1];
		llah.ex[1] = dumy->cx[1];
		llah.ey[1] = dumy->cy[1];
		msg = lla(&llah);
		
		/* PAP	始点と角度と距離で直線の終点を求める */
		paph.sx[1] = dumy->sx[1];
		paph.sy[1] = dumy->sy[1];
		paph.angle = llah.angle;
		paph.l = 100;
		msg = pap(&paph);
		
		dumy->type = 1;
		dumy->sx[1] = paph.sx[1];
		dumy->sy[1] = paph.sy[1];
		dumy->ex[1] = paph.ex[1];
		dumy->ey[1] = paph.ey[1];
		
		return 0;
	}

	if (sg(pph.l, compa_digits) > sg(dumy->r[1], compa_digits)) {
		/* 中心から点を結ぶ線１の距離　pph.l */
		/* LA	直線の角度 */
		lah.sx[1] = dumy->sx[1];
		lah.sy[1] = dumy->sy[1];
		lah.ex[1] = dumy->cx[1];
		lah.ey[1] = dumy->cy[1];
		msg = la(&lah);
		
		/* PAP	始点と角度と距離で直線の終点を求める */
		paph.sx[1] = dumy->sx[1];
		paph.sy[1] = dumy->sy[1];
		paph.angle = lah.angle;
		paph.l = pph.l / 2;
		msg = pap(&paph);
		
		/* CCP	２円の交点 */
		ccph.cx[1] = dumy->cx[1];
		ccph.cy[1] = dumy->cy[1];
		ccph.r[1] = dumy->r[1];
		ccph.cx[2] = paph.ex[1];
		ccph.cy[2] = paph.ey[1];
		ccph.r[2] = pph.l / 2;
		msg = ccp(&ccph);
		
		dumy->type = 2;
		dumy->ex[1] = ccph.sx[1];
		dumy->ey[1] = ccph.sy[1];
		dumy->ex[2] = ccph.sx[2];
		dumy->ey[2] = ccph.sy[2];
		
		return 0;
	}
	return 1;
}



/* -------------------------------------------------------------------
 * 説明 : [15] ２円に接する直線   初めてのグラフィックス P131
 * 関数 : ccl
 * 引数 : rtndat
 *		:	円 1	(cx[1] , cy[1]) ,r[1]
 *		:	円 2	(cx[2] , cy[2]) ,r[2]
 * 戻値 : rtndat
 *		:	type	 0, 1 , 2 , 3 , 4
 *		:	線 1	(sx[1] , sy[1]) - (ex[1] , ey[1])
 *		:	線 2	(sx[2] , sy[2]) - (ex[2] , ey[2])
 *		:	線 3	(sx[3] , sy[3]) - (ex[3] , ey[3])
 *		:	線 4	(sx[4] , sy[4]) - (ex[4] , ey[4])
 * 外部 : 無し
 * 内部 : RtnDat pph,pp2h,ppph,pclh,lah,la2h,paph,plph,lcph,pap2h
 */
int ccl(struct RtnDat *dumy)
{
//#define CCL
	struct RtnDat pph,pp2h,ppph,lah,la2h,paph,plph,lcph,pap2h;
	int msg,ccltype;
	double f_dumy,kaku_b1,kaku_b2,kaku_o,kaku_a;
	double x,x1,x2;
	double dist,r1pr2,r1mr2;
	int debug = 0;


#ifdef CCL
	debug = 1;
#endif

	ccltype = 0;
	pph.sx[1] = dumy->cx[1];
	pph.sy[1] = dumy->cy[1];
	pph.ex[1] = dumy->cx[2];
	pph.ey[1] = dumy->cy[2];
	msg = pp(&pph);

	dist = sg(pph.l, compa_digits);
	r1pr2 = sg((dumy->r[1] + dumy->r[2]), compa_digits);
	r1mr2 = sg((dumy->r[1] - dumy->r[2]), compa_digits);


//#ifdef TEAT
	/* ２円が離れている。接線４本 */
	if (debug > 0) g_print("type 4 : dist > r1pr2 = %d\n", dist > r1pr2);
	/* ２円が接している。接線３本 */
	if (debug > 0) g_print("type 3 : dist == r1pr2 = %d\n", dist == r1pr2);
	/* ２円が重なっている。接線２本 */
	if (debug > 0) g_print("type 2 : dist < r1pr2 && dist > r1mr2 = %d\n", (dist < r1pr2 && dist > r1mr2));
	if (debug > 0) g_print("  type 2-1 : dist < r1pr2 = %d\n", (dist < r1pr2));
	if (debug > 0) g_print("  type 2-2 : dist > r1mr2 = %d\n", (dist > r1mr2));
	/* ２円が重なって接している。接線１本 */
	if (debug > 0) g_print("type 1 : dist == r1mr2 = %d\n", (dist == r1mr2));
	/* ２円が重なって離れている。接線 0 本 */
	if (debug > 0) g_print("type 0 : (dist < r1mr2 && sg(dumy->l, compa_digits) > 0) = %d\n", 
			(dist < r1mr2 && sg(dumy->l, compa_digits) > 0));
	if (debug > 0) g_print("  type 0-1 : (dist < r1mr2) = %d\n", (dist < r1mr2));
	if (debug > 0) g_print("  type 0-2 : (sg(dumy->l, compa_digits) > 0) = %d\n", (sg(dumy->l, compa_digits) > 0));
//#endif


	/* ２円が離れている。接線４本 */
	if (dist > r1pr2) {
		if (debug > 0) g_print("type 4 : in\n");
		dumy->type = 4;
		
		/* 内接する直線 */

		/* O1(dumy.cx[1],dumy.cy[1]) - O2(dumy.cx[2],dumy.cy[2]) を */
		/* dumy.r[1] : dumy.r[2] に内分する点 P を求める。 */
		ppph.sx[1] = dumy->cx[1];
		ppph.sy[1] = dumy->cy[1];
		ppph.ex[1] = dumy->cx[2];
		ppph.ey[1] = dumy->cy[2];
		ppph.sx[2] = dumy->r[1];
		ppph.ex[2] = dumy->r[2];
		msg = ppp(&ppph);

		/* O1(dumy.cx[1] , dumy.cy[1]) から点 P (pph.sx[3] , pph.sy[3]) の角度を求める。 */
		la2h.sx[1] = dumy->cx[1];
		la2h.sy[1] = dumy->cy[1];
		la2h.ex[1] = ppph.sx[3];
		la2h.ey[1] = ppph.sy[3];
		msg = la(&la2h);

		/* O1(dumy.cx[1] , dumy.cy[1]) から点 P (pph.sx[3] , pph.sy[3]) の距離を求める。 */
		pp2h.sx[1] = dumy->cx[1];
		pp2h.sy[1] = dumy->cy[1];
		pp2h.ex[1] = ppph.sx[3];
		pp2h.ey[1] = ppph.sy[3];
		msg = pp(&pp2h);
		
		x = dumy->r[1] / pp2h.l;
		x = atan(x / sqrt(-x * x + 1));
		x = (180 / PI) * x;
		
		x1 = la2h.angle + x;
		x2 = la2h.angle - x;

		/* 点 P から角度 x1 の線 */
		pap2h.sx[1] = ppph.sx[3];
		pap2h.sy[1] = ppph.sy[3];
		pap2h.angle = x1;
		pap2h.l = 100;
		msg = pap(&pap2h);

		lcph.sx[1] = ppph.sx[3];
		lcph.sy[1] = ppph.sy[3];
		lcph.ex[1] = pap2h.ex[1];
		lcph.ey[1] = pap2h.ey[1];
		lcph.cx[1] = dumy->cx[1];
		lcph.cy[1] = dumy->cy[1];
		lcph.r[1] = dumy->r[1];
		msg = lcp(&lcph);

		if (lcph.type != 1) {
			if (debug > 0) g_print("type 4-1 : lcp.type != 1\n");
			dumy->type = 0;
			return 1;
		}
		dumy->sx[3] = lcph.sx[2];
		dumy->sy[3] = lcph.sy[2];

		/* 円２ */
		lcph.sx[1] = ppph.sx[3];
		lcph.sy[1] = ppph.sy[3];
		lcph.ex[1] = pap2h.ex[1];
		lcph.ey[1] = pap2h.ey[1];
		lcph.cx[1] = dumy->cx[2];
		lcph.cy[1] = dumy->cy[2];
		lcph.r[1] = dumy->r[2];
		msg = lcp(&lcph);

		if (lcph.type != 1) {
			if (debug > 0) g_print("type 4-2 : lcp.type != 1\n");
			dumy->type = 0;
			return 1;
		}
		dumy->ex[3] = lcph.sx[2];
		dumy->ey[3] = lcph.sy[2];


		/* 点 P から角度 x2 の線 */
		pap2h.sx[1] = ppph.sx[3];
		pap2h.sy[1] = ppph.sy[3];
		pap2h.angle = x2;
		pap2h.l = 100;
		msg = pap(&pap2h);

		lcph.sx[1] = ppph.sx[3];
		lcph.sy[1] = ppph.sy[3];
		lcph.ex[1] = pap2h.ex[1];
		lcph.ey[1] = pap2h.ey[1];
		lcph.cx[1] = dumy->cx[1];
		lcph.cy[1] = dumy->cy[1];
		lcph.r[1] = dumy->r[1];
		msg = lcp(&lcph);

		if (lcph.type != 1) {
			if (debug > 0) g_print("type 4-3 : lcp.type != 1\n");
			dumy->type = 0;
			return 1;
		}
		dumy->sx[4] = lcph.sx[2];
		dumy->sy[4] = lcph.sy[2];

		/* 円２ */
		lcph.sx[1] = ppph.sx[3];
		lcph.sy[1] = ppph.sy[3];
		lcph.ex[1] = pap2h.ex[1];
		lcph.ey[1] = pap2h.ey[1];
		lcph.cx[1] = dumy->cx[2];
		lcph.cy[1] = dumy->cy[2];
		lcph.r[1] = dumy->r[2];
		msg = lcp(&lcph);

		if (lcph.type != 1) {
			if (debug > 0) g_print("type 4-4 : lcp.type != 1\n");
			dumy->type = 0;
			return 1;
		}
		dumy->ex[4] = lcph.sx[2];
		dumy->ey[4] = lcph.sy[2];
	}



	/* ２円が接している。接線３本 */
	else if (dist == r1pr2) {
		if (debug > 0) g_print("type 3 : in\n");
		dumy->type = 3;
	}



	/* ２円が重なっている。接線２本 */
	else if (dist < r1pr2 && dist > r1mr2) {
		if (debug > 0) g_print("type 2 : in\n");
		dumy->type = 2;
	}



	/* ２円が重なって接している。接線１本 */
	else if (dist == r1mr2) {
		/* 重なって接する */
		dumy->type = 1;

		/* dumy->r[1] と dumy->r[2] の大小を比べ、大きい方を円１にする。 */
		if (dumy->r[1] < dumy->r[2]) {
			f_dumy = dumy->cx[1];
			dumy->cx[1] = dumy->cx[2];
			dumy->cx[2] = f_dumy;
			f_dumy = dumy->cy[1];
			dumy->cy[1] = dumy->cy[2];
			dumy->cy[2] = f_dumy;
			f_dumy = dumy->r[1];
			dumy->r[1] = dumy->r[2];
			dumy->r[2] = f_dumy;
			ccltype = 2;
		}

		/* 円１の中心から円２の中心への角度Ａを求める。 */
		lah.sx[1] = dumy->cx[1];
		lah.sy[1] = dumy->cy[1];
		lah.ex[1] = dumy->cx[2];
		lah.ey[1] = dumy->cy[2];
		msg = la(&lah);

		/* 円１の中心から角度ＡにＲ１だけ行った点Ｐを求める。 */
		paph.sx[1] = dumy->cx[1];
		paph.sy[1] = dumy->cy[1];
		paph.angle = lah.angle;
		paph.l = dumy->r[1];
		msg = pap(&paph);

		/* 点Ｐを通り、円１の中心から円２の中心への直線に垂直な線を求める。 */
		plph.sx[2] = paph.ex[1];
		plph.sy[2] = paph.ey[1];
		plph.sx[1] = dumy->cx[1];
		plph.sy[1] = dumy->cy[1];
		plph.ex[1] = dumy->cx[2];
		plph.ey[1] = dumy->cy[2];
		msg = plp(&plph);
	
		dumy->sx[1] = plph.sx[2];
		dumy->sy[1] = plph.sy[2];
		dumy->ex[1] = plph.ex[3];
		dumy->ey[1] = plph.ey[3];
	
		if (ccltype == 2) {
			f_dumy = dumy->cx[1];
			dumy->cx[1] = dumy->cx[2];
			dumy->cx[2] = f_dumy;
			f_dumy = dumy->cy[1];
			dumy->cy[1] = dumy->cy[2];
			dumy->cy[2] = f_dumy;
			f_dumy = dumy->r[1];
			dumy->r[1] = dumy->r[2];
			dumy->r[2] = f_dumy;
			ccltype = 0;
		}
	}



	/* ２円が重なって離れている。接線 0 本 */
	else if (dist < r1mr2 && sg(dumy->l, compa_digits) > 0) {
		if (debug > 0) g_print("type 1 : in\n");
		dumy->type = 0;
		return 1;
	}



	/* 外接する直線 */
	if (dumy->type >= 2) {
		/* Ｒ１の小さい方がＯ１にする。 */
		if (dumy->r[1] > dumy->r[2]) {
			f_dumy = dumy->cx[1];
			dumy->cx[1] = dumy->cx[2];
			dumy->cx[2] = f_dumy;
			f_dumy = dumy->cy[1];
			dumy->cy[1] = dumy->cy[2];
			dumy->cy[2] = f_dumy;
			f_dumy = dumy->r[1];
			dumy->r[1] = dumy->r[2];
			dumy->r[2] = f_dumy;
			ccltype = 1;
		}

		/* Ｏ１→Ｏ２のベクトルの角度を求める。(RAD) */
		lah.sx[1] = dumy->cx[1];
		lah.sy[1] = dumy->cy[1];
		lah.ex[1] = dumy->cx[2];
		lah.ey[1] = dumy->cy[2];
		msg = la(&lah);
		lah.angle = degrad(lah.angle);
		kaku_o = lah.angle;

		/* Ｏ１→Ｏ２の距離を求める。 */
		pph.sx[1] = dumy->cx[1];
		pph.sy[1] = dumy->cy[1];
		pph.ex[1] = dumy->cx[2];
		pph.ey[1] = dumy->cy[2];
		msg = pp(&pph);

		/* 角Ａを求める。 */
		kaku_a = (dumy->r[2] - dumy->r[1]) / pph.l;
		kaku_a = atan(kaku_a / sqrt(-kaku_a * kaku_a + 1));

		/* Ｏ１→Ｏ２の角度±（角Ａ＋９０）＝角Ｂを求める。 */

		kaku_a = kaku_a + (PI / 2);
		kaku_b1 = kaku_o + kaku_a;
		kaku_b2 = kaku_o - kaku_a;

		kaku_b1 = raddeg(kaku_b1);
		kaku_b2 = raddeg(kaku_b2);

		/* 各円の中心から角Ｂの方向に各Ｒ行った点を求める。(4コ) */
		paph.sx[1] = dumy->cx[1];
		paph.sy[1] = dumy->cy[1];
		paph.angle = kaku_b1;
		paph.l = dumy->r[1];
		msg = pap(&paph);
		dumy->sx[1] = paph.ex[1];
		dumy->sy[1] = paph.ey[1];

		paph.sx[1] = dumy->cx[1];
		paph.sy[1] = dumy->cy[1];
		paph.angle = kaku_b2;
		paph.l = dumy->r[1];
		msg = pap(&paph);
		dumy->sx[2] = paph.ex[1];
		dumy->sy[2] = paph.ey[1];

		paph.sx[1] = dumy->cx[2];
		paph.sy[1] = dumy->cy[2];
		paph.angle = kaku_b1;
		paph.l = dumy->r[2];
		msg = pap(&paph);
		dumy->ex[1] = paph.ex[1];
		dumy->ey[1] = paph.ey[1];
		paph.sx[1] = dumy->cx[2];
		paph.sy[1] = dumy->cy[2];
		paph.angle = kaku_b2;
		paph.l = dumy->r[2];
		msg = pap(&paph);
		dumy->ex[2] = paph.ex[1];
		dumy->ey[2] = paph.ey[1];

		if (ccltype == 1) {
			f_dumy = dumy->cx[1];
			dumy->cx[1] = dumy->cx[2];
			dumy->cx[2] = f_dumy;
			f_dumy = dumy->cy[1];
			dumy->cy[1] = dumy->cy[2];
			dumy->cy[2] = f_dumy;
			f_dumy = dumy->r[1];
			dumy->r[1] = dumy->r[2];
			dumy->r[2] = f_dumy;
			ccltype = 0;
		}
	}



	if (dumy->type == 3) {
		/* 並んで接する */
		/* 円 1 の中心から円 2 の中心までの線分を  */
		/* dumy->r[1] : dumy->r[2] に内分する点Ｐを求める。 */
		ppph.sx[1] = dumy->cx[1];
		ppph.sy[1] = dumy->cy[1];
		ppph.ex[1] = dumy->cx[2];
		ppph.ey[1] = dumy->cy[2];
		ppph.sx[2] = dumy->r[1];
		ppph.ex[2] = dumy->r[2];
		msg = ppp(&ppph);

		/* １点Ｐを通って、円 1 の中心から円 2 の中心までの線分に垂直な線を求める。 */
		plph.sx[2] = ppph.sx[3];
		plph.sy[2] = ppph.sy[3];
		plph.sx[1] = dumy->cx[1];
		plph.sy[1] = dumy->cy[1];
		plph.ex[1] = dumy->cx[2];
		plph.ey[1] = dumy->cy[2];
		msg = plp(&plph);

		dumy->sx[3] = ppph.sx[3];
		dumy->sy[3] = ppph.sy[3];
		dumy->ex[3] = plph.ex[3];
		dumy->ey[3] = plph.ey[3];
	}
	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [16] １点を通り円に接する円
 * 関数 : pcc
 * 引数 : rtndat
 *		:	求める円の半径	  r[2]
 *		:	円の中心		 (cx[1] , cy[1]) ,r[1]
 *		:	点				 (sx[1] , sy[1])
 * 戻値 : rtndat
 *		:	type			  0 , 1 , 2 , 3 , 4
 *		:	円 1			 (cx[2] , cy[2]) , r[2]
 *		:	円 2			 (cx[3] , cy[3]) , r[3]
 *		:	円 3			 (cx[4] , cy[4]) , r[4]
 *		:	円 4			 (cx[5] , cy[5]) , r[5]
 * 外部 : 無し
 * 内部 : RtnDat pph,ccph,pp2h,lah,paph
 */
int pcc(struct RtnDat *dumy)
{
	struct RtnDat pph,ccph,lah,paph;
	int msg;


	dumy->type = 0;
	/* 1点と円の中心との距離 */
	pph.sx[1] = dumy->sx[1];
	pph.sy[1] = dumy->sy[1];
	pph.ex[1] = dumy->cx[1];
	pph.ey[1] = dumy->cy[1];
	msg = pp(&pph);
	
	
	
	/* 点が円の外にあるとき ------------------------------------------ */
	if (sg(pph.l, compa_digits) > sg(dumy->r[1], compa_digits))
	{
		/* 1.2 */
		if (sg(pph.l - dumy->r[1], compa_digits) <= sg((2 * dumy->r[2]), compa_digits))
		{
			/* 点から半径Ｒ２の円と円の中心から半径Ｒ１＋Ｒ２の円の交点が求める円の中心。 */
			ccph.cx[1] = dumy->sx[1];
			ccph.cy[1] = dumy->sy[1];
			ccph.r[1] = dumy->r[2];
			ccph.cx[2] = dumy->cx[1];
			ccph.cy[2] = dumy->cy[1];
			ccph.r[2] = dumy->r[1] + dumy->r[2];
			msg = ccp(&ccph);
			dumy->type = ccph.type;
			if (ccph.type == 1)
			{
				dumy->cx[2] = ccph.sx[1];
				dumy->cy[2] = ccph.sy[1];
				dumy->r[2] = dumy->r[2];
			}
			if (ccph.type == 2)
			{
				dumy->cx[2] = ccph.sx[1];
				dumy->cy[2] = ccph.sy[1];
				dumy->r[2] = dumy->r[2];
				dumy->cx[3] = ccph.sx[2];
				dumy->cy[3] = ccph.sy[2];
				dumy->r[3] = dumy->r[2];
			}
		}

		if (sg((pph.l + dumy->r[1]), compa_digits) <= sg((2 * dumy->r[2]), compa_digits))
		{
			/* 点から半径Ｒ２の円と円の中心から半径Ｒ２－Ｒ１の円の交点が求める円の中心。 */
			ccph.cx[1] = dumy->sx[1];
			ccph.cy[1] = dumy->sy[1];	
			ccph.r[1] = dumy->r[2];
			ccph.cx[2] = dumy->cx[1];	
			ccph.cy[2] = dumy->cy[1];	
			ccph.r[2] = dumy->r[2] - dumy->r[1];
			msg = ccp(&ccph);
			if (dumy->type == 0)
			{
				dumy->cx[2] = ccph.sx[1];	
				dumy->cy[2] = ccph.sy[1];	
				dumy->r[2] = dumy->r[2];
				if (ccph.type == 2)
				{
					dumy->cx[3] = ccph.sx[2];	
					dumy->cy[3] = ccph.sy[2];	
					dumy->r[3] = dumy->r[2];
				}
			}
			else if (dumy->type == 1)
			{
				dumy->cx[3] = ccph.sx[1];	
				dumy->cy[3] = ccph.sy[1];	
				dumy->r[3] = dumy->r[2];
				if (ccph.type == 2)
				{
					dumy->cx[4] = ccph.sx[2];	
					dumy->cy[4] = ccph.sy[2];	
					dumy->r[4] = dumy->r[2];
				}
			}
			else if (dumy->type == 2)
			{
				dumy->cx[4] = ccph.sx[1];	
				dumy->cy[4] = ccph.sy[1];	
				dumy->r[4] = dumy->r[2];
				if (ccph.type == 2)
				{
					dumy->cx[5] = ccph.sx[2];	
					dumy->cy[5] = ccph.sy[2];	
					dumy->r[5] = dumy->r[2];
				}
			}
			dumy->type = dumy->type + ccph.type;
		}
		else
		{
			dumy->type = dumy->type + 0;
		}
	}





	/* 点が円の中にあるとき ------------------------------------------ */
	if (sg(pph.l, compa_digits) < sg(dumy->r[1], compa_digits))
	{
		if ((sg((dumy->r[1] + pph.l), compa_digits) >= sg((2 * dumy->r[2]), compa_digits))	
			&& (sg((dumy->r[1] - pph.l), compa_digits) <= sg((2 * dumy->r[2]), compa_digits)))
		{
			/* 中心から半径(R1-R2)の円と点から半径(R2)の円の交点  */
			ccph.cx[1] = dumy->cx[1];	
			ccph.cy[1] = dumy->cy[1];	
			ccph.r[1] = (dumy->r[1] - dumy->r[2]);
			ccph.cx[2] = dumy->sx[1];	
			ccph.cy[2] = dumy->sy[1];	
			ccph.r[2] = dumy->r[2];
			msg = ccp(&ccph);
			
			if (ccph.type != 0)
			{
				dumy->type = ccph.type;
				dumy->cx[2] = ccph.sx[1];	
				dumy->cy[2] = ccph.sy[1];
				if (dumy->type == 2)
				{
					dumy->cx[3] = ccph.sx[2];	
					dumy->cy[3] = ccph.sy[2];	
					dumy->r[3] = dumy->r[2];
				}
			}
			else
			{
				dumy->type = 0;
			}
		}
	}





	/* 点が円の上にあるとき ------------------------------------------ */
	if (sg(pph.l, compa_digits) == sg(dumy->r[1], compa_digits))
	{
		dumy->type = 2;
		/* 中心から点までの線の角度Ａを求める。 */
		lah.sx[1] = dumy->cx[1];	
		lah.sy[1] = dumy->cy[1];	
		lah.ex[1] = dumy->sx[1];	
		lah.ey[1] = dumy->sy[1];
		msg = la(&lah);

		/* 点から角度Ａで半径Ａ.r[2]の点  */
		paph.sx[1] = dumy->sx[1];	
		paph.sy[1] = dumy->sy[1];
		paph.angle = lah.angle; 
		paph.l = dumy->r[2];
		msg = pap(&paph);
				
		dumy->cx[2] = paph.ex[1];	
		dumy->cy[2] = paph.ey[1];

		/* 点から角度Ａ＋１８０で半径Ａ.r[2]の点 */
		paph.sx[1] = dumy->sx[1];	
		paph.sy[1] = dumy->sy[1];
		paph.angle = (lah.angle + 180); 
		paph.l = dumy->r[2];
		msg = pap(&paph);
				
		dumy->cx[3] = paph.ex[1];	
		dumy->cy[3] = paph.ey[1];	
		dumy->r[3] = dumy->r[2];
	}


	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [17] 直線と円に接する円
 * 関数 : lcc
 * 引数 : rtndat
 *		:	半径		r[2]
 *		:	線		   (sx[1] , sy[1]) - (ex[1] , ey[1])
 *		:	円		   (cx[1] , cy[1]) , r[1]
 * 戻値 : rtndat
 *		:	type
 *		:	円 1	   (cx[2] , cy[2]) , r[2]
 *		:	円 2	   (cx[3] , cy[3]) , r[3]
 *		:	円 3	   (cx[4] , cy[4]) , r[4]
 *		:	円 4	   (cx[5] , cy[5]) , r[5]
 *		:	円 5	   (cx[6] , cy[6]) , r[6]
 *		:	円 6	   (cx[7] , cy[7]) , r[7]
 *		:	円 7	   (cx[8] , cy[8]) , r[8]
 *		:	円 8	   (cx[9] , cy[9]) , r[9]
 * 外部 : 無し
 * 内部 : RtnDat plph,pph,lah,paph,lcph,loh
 */
int lcc(struct RtnDat *dumy)
{
	struct RtnDat plph,pph,lah,paph,lcph,loh;
	int msg;
	double dist,r1,lmr1,r2x2,lpr1,r2,r1ml;


	dumy->type = 0;
	/* 1 ==============================================================  */
	/* ========== まず円の中心と線の距離を求める。			 ==========  */
	/* === start ======================================================  */


	/* ------------------------------------------------  */
	/* PLP	点と直線の最近点（１点を通る垂線）			 */
	/* ------------------------------------------------  */
	plph.sx[1] = dumy->sx[1];
	plph.sy[1] = dumy->sy[1];
	plph.ex[1] = dumy->ex[1];
	plph.ey[1] = dumy->ey[1];
	plph.sx[2] = dumy->cx[1];
	plph.sy[2] = dumy->cy[1];
	msg = plp(&plph);
	
	/* 最近点(plph.ex[2],plph.ey[2])  */
	/* ------------------------------------------------  */
	/* PP	２点間の距離								 */
	/* ------------------------------------------------  */
	pph.sx[1] = dumy->cx[1];	
	pph.sy[1] = dumy->cy[1];	
	pph.ex[1] = plph.ex[2]; 
	pph.ey[1] = plph.ey[2];
	msg = pp(&pph);
	/* === end ========================================================  */
	/* ========== まず円の中心と線の距離を求める。			 ==========  */
	/* ================================================================  */


	dist = sg(pph.l, compa_digits);
	r1 = sg(dumy->r[1], compa_digits);
	lmr1 = sg(pph.l - dumy->r[1], compa_digits);
	r2x2 = sg(2 * dumy->r[2], compa_digits);
	lpr1 = sg(pph.l + dumy->r[1], compa_digits);
	r2 = sg(dumy->r[2], calcu_digits);
	r1ml = sg(dumy->r[1] - pph.l, compa_digits);



	/* 2 = start ======================================================  */
	/* ========== <   線と円が、離れている。 >				 ==========  */
	/* ================================================================  */
	if (dist > r1)
	{
		/* -< 1 & 2 >-	*/
		/* 中心から、直線の最近点までの、線の角度Ａを求める。 */
		/* ------------------------------------------------  */
		/* LA	直線の角度 */
		/* ------------------------------------------------  */
		lah.sx[1] = dumy->cx[1];	
		lah.sy[1] = dumy->cy[1];	
		lah.ex[1] = plph.ex[2]; 
		lah.ey[1] = plph.ey[2];
		msg = la(&lah);
		
		if (lmr1 <= r2x2)
		{
			/* 線を円の中心方向に dumy.r[2] オフセットした線Ａを求める。 */
			/* 最近点から円の中心方向の角度を求める。						*/
			/* lah.ANGLE+180  */
			/* ------------------------------------------------  */
			/* PAP	始点と角度と距離で直線の終点を求める		 */
			/* ------------------------------------------------  */
			/* 始点を計算  */
			paph.sx[1] = dumy->sx[1];	
			paph.sy[1] = dumy->sy[1];
			paph.angle = lah.angle + 180;	
			paph.l = dumy->r[2];
			msg = pap(&paph);
		
			dumy->sx[2] = paph.ex[1];	
			dumy->sy[2] = paph.ey[1];
		
			/* 終点を計算  */
			paph.sx[1] = dumy->ex[1];	
			paph.sy[1] = dumy->ey[1];
			paph.angle = lah.angle + 180;	
			paph.l = dumy->r[2];
			msg = pap(&paph);
		
			dumy->ex[2] = paph.ex[1];	
			dumy->ey[2] = paph.ey[1];
		
			/* 円の中心から半径 (dumy.r[1] + dumy.r[2]) の円と線Ａの交点を求める。 */
			/* ------------------------------------------------  */
			/* LCP 直線と円の交点	初めてのグラフィックス P126  */
			/* ------------------------------------------------  */
			lcph.sx[1] = dumy->sx[2];	
			lcph.sy[1] = dumy->sy[2];	
			lcph.ex[1] = dumy->ex[2];	
			lcph.ey[1] = dumy->ey[2];
			lcph.cx[1] = dumy->cx[1];	
			lcph.cy[1] = dumy->cy[1];	
			lcph.r[1] = (dumy->r[1] + dumy->r[2]);
			msg = lcp(&lcph);
			
			dumy->type = lcph.type;
			
			dumy->cx[2] = lcph.sx[2];	
			dumy->cy[2] = lcph.sy[2];	
			dumy->r[2] = dumy->r[2];
			
			if (lcph.type == 2)
			{
				dumy->cx[3] = lcph.sx[3];	
				dumy->cy[3] = lcph.sy[3];	
				dumy->r[3] = dumy->r[2];
			}
		}



		/* -< 1 & 2 >- OR -< 3 & 4 >-  */
		if (lmr1 <= r2x2)
		{
			/* 線を円の中心方向に dumy.r[2] オフセットした線Ａを求める。  */
			/* 最近点から円の中心方向の角度を求める。 */
			/* lah.ANGLE+180  */
			/* ------------------------------------------------  */
			/* PAP	始点と角度と距離で直線の終点を求める */
			/* ------------------------------------------------  */
			/* 始点を計算  */
			paph.sx[1] = dumy->sx[1];	
			paph.sy[1] = dumy->sy[1];
			paph.angle = lah.angle + 180;	
			paph.l = dumy->r[2];
			msg = pap(&paph);
		
			dumy->sx[2] = paph.ex[1];	
			dumy->sy[2] = paph.ey[1];
		
			/* 終点を計算  */
			paph.sx[1] = dumy->ex[1];	
			paph.sy[1] = dumy->ey[1];
			paph.angle = lah.angle + 180;	
			paph.l = dumy->r[2];
			msg = pap(&paph);
		
			dumy->ex[2] = paph.ex[1];	
			dumy->ey[2] = paph.ey[1];
		
			/* 円の中心から半径 (dumy.r[2] - dumy.r[1]) の円と線Ａの交点を求める。 */
			/* ------------------------------------------------  */
			/* LCP 直線と円の交点	初めてのグラフィックス P126  */
			/* ------------------------------------------------  */
			lcph.sx[1] = dumy->sx[2];	
			lcph.sy[1] = dumy->sy[2];	
			lcph.ex[1] = dumy->ex[2];	
			lcph.ey[1] = dumy->ey[2];
			lcph.cx[1] = dumy->cx[1];	
			lcph.cy[1] = dumy->cy[1];	
			lcph.r[1] = (dumy->r[2] - dumy->r[1]);
			msg = lcp(&lcph);
			
			if (dumy->type == 0)
			{
				if (lcph.type == 1)
				{
					dumy->cx[2] = lcph.sx[2];	
					dumy->cy[2] = lcph.sy[2];	
					dumy->r[2] = dumy->r[2];
				}
				if (lcph.type == 2)
				{
					dumy->cx[2] = lcph.sx[2];	
					dumy->cy[2] = lcph.sy[2];	
					dumy->r[2] = dumy->r[2];
					dumy->cx[3] = lcph.sx[3];	
					dumy->cy[3] = lcph.sy[3];	
					dumy->r[3] = dumy->r[2];
				}
			}
			
			if (dumy->type == 1)
			{
				if (lcph.type == 1)
				{
					dumy->cx[3] = lcph.sx[2];	
					dumy->cy[3] = lcph.sy[2];	
					dumy->r[3] = dumy->r[2];
				}
				if (lcph.type == 2)
				{
					dumy->cx[3] = lcph.sx[2];	
					dumy->cy[3] = lcph.sy[2];	
					dumy->r[3] = dumy->r[2];
					dumy->cx[4] = lcph.sx[3];	
					dumy->cy[4] = lcph.sy[3];	
					dumy->r[4] = dumy->r[2];
				}
			}
			
			if (dumy->type == 2)
			{
				if (lcph.type == 1)
				{
					dumy->cx[4] = lcph.sx[2];	
					dumy->cy[4] = lcph.sy[2];	
					dumy->r[4] = dumy->r[2];
				}
				if (lcph.type == 2)
				{
					dumy->cx[4] = lcph.sx[2];	
					dumy->cy[4] = lcph.sy[2];	
					dumy->r[4] = dumy->r[2];
					dumy->cx[5] = lcph.sx[3];	
					dumy->cy[5] = lcph.sy[3];	
					dumy->r[5] = dumy->r[2];
				}
			}
			dumy->type = dumy->type + lcph.type;
		}
	}
	/* 2 = end ========================================================  */
	/* ========== <   線と円が、離れている。 >				 ==========  */
	/* ================================================================  */
	
	
	
	
	
	/* 3 = start ======================================================  */
	/* ========== < 線と円が、接している。 >				 ==========  */
	/* ================================================================  */
	if (dist == r1)
	{
		/* -< 1 & 2 >-	*/
		/* -----	<	最近点(plph.ex[2],plph.ey[2])で接する円、２個。  >------  */
		dumy->type = 4;
		/* 中心から、直線の最近点までの、線の角度Ａを求める。 */
		/* ------------------------------------------------  */
		/* LA	直線の角度									 */
		/* ------------------------------------------------  */
		lah.sx[1] = dumy->cx[1];	
		lah.sy[1] = dumy->cy[1];	
		lah.ex[1] = plph.ex[2]; 
		lah.ey[1] = plph.ey[2];
		msg = la(&lah);
		
		/* 最近点から角度Ａで長さ半径dumy.r[2]の点	*/
		/* ------------------------------------------------  */
		/* PAP	始点と角度と距離で直線の終点を求める		 */
		/* ------------------------------------------------  */
		paph.sx[1] = plph.ex[2];	
		paph.sy[1] = plph.ey[2];
		paph.angle = lah.angle; 
		paph.l = dumy->r[2];
		msg = pap(&paph);
		
		dumy->cx[2] = paph.ex[1];	
		dumy->cy[2] = paph.ey[1];	
		dumy->r[2] = dumy->r[2];
		
		/* 最近点から角度Ａ＋１８０で長さ半径dumy.r[2]の点	*/
		/* paph.sx[1] = plph.ex[2]; paph.sy[1] = plph.ey[2]  */
		paph.angle = (lah.angle + 180);
		paph.l = dumy->r[2];
		msg = pap(&paph);
		
		dumy->cx[3] = paph.ex[1];
		dumy->cy[3] = paph.ey[1];
		dumy->r[3] = dumy->r[2];
		
		
		
		/* -< 3 & 4 >-	*/
		/* 線を円の中心方向に dumy.r[2] オフセットした線Ａを求める。 */
		/* 最近点から円の中心方向の角度を求める。 */
		/* lah.ANGLE+180  */
		/* ------------------------------------------------  */
		/* PAP	始点と角度と距離で直線の終点を求める		 */
		/* ------------------------------------------------  */
		/* 始点を計算  */
		paph.sx[1] = dumy->sx[1];	
		paph.sy[1] = dumy->sy[1];
		paph.angle = lah.angle + 180;	
		paph.l = dumy->r[2];
		msg = pap(&paph);
		
		dumy->sx[2] = paph.ex[1];	
		dumy->sy[2] = paph.ey[1];
		
		/* 終点を計算  */
		paph.sx[1] = dumy->ex[1];	
		paph.sy[1] = dumy->ey[1];
		paph.angle = lah.angle + 180;	
		paph.l = dumy->r[2];
		msg = pap(&paph);
		
		dumy->ex[2] = paph.ex[1];	
		dumy->ey[2] = paph.ey[1];
		
		/* 円の中心から半径 (dumy.r[1] + dumy.r[2]) の円と線Ａの交点を求める。 */
		/* ------------------------------------------------  */
		/* LCP 直線と円の交点	初めてのグラフィックス P126  */
		/* ------------------------------------------------  */
		lcph.sx[1] = dumy->sx[2]; 
		lcph.sy[1] = dumy->sy[2];	
		lcph.ex[1] = dumy->ex[2];	
		lcph.ey[1] = dumy->ey[2];
		lcph.cx[1] = dumy->cx[1];	
		lcph.cy[1] = dumy->cy[1];	
		lcph.r[1] = (dumy->r[1] + dumy->r[2]);
		msg = lcp(&lcph);
		
		dumy->cx[4] = lcph.sx[2];	
		dumy->cy[4] = lcph.sy[2];	
		dumy->r[4] = dumy->r[2];
		dumy->cx[5] = lcph.sx[3];	
		dumy->cy[5] = lcph.sy[3];	
		dumy->r[5] = dumy->r[2];
	}
	/* 3 = end ========================================================  */
	/* ========== < 線と円が、接している。 >				 ==========  */
	/* ================================================================  */
	
	
	
	
	
	/* 4 = start ======================================================  */
	/* ========== < 線と円が、重なっている。 >				 ==========  */
	/* ================================================================  */
	if (dist < r1)
	{
		/* -< 1 & 2 & 3 & 4 >-	*/
		/* 線を両方に dumy.r[2] オフセットした線Ａ、線Ｂを求める。 */
		/* ------------------------------------------------  */
		/* LO	始点と終点を与えた直線のオフセット			 */
		/* ------------------------------------------------  */
		loh.sx[1] = dumy->sx[1];	
		loh.sy[1] = dumy->sy[1];	
		loh.ex[1] = dumy->ex[1];	
		loh.ey[1] = dumy->ey[1];	
		loh.l = dumy->r[2];
		msg = lo(&loh);
		
		/* 線Ａと中心から半径 (dumy.r[1] + dumy.r[2]) の円の交点を求める。 */
		/* ------------------------------------------------  */
		/* LCP 直線と円の交点	初めてのグラフィックス P126  */
		/* ------------------------------------------------  */
		lcph.sx[1] = loh.sx[2]; 
		lcph.sy[1] = loh.sy[2]; 
		lcph.ex[1] = loh.ex[2]; 
		lcph.ey[1] = loh.ey[2];
		lcph.cx[1] = dumy->cx[1];	
		lcph.cy[1] = dumy->cy[1];	
		lcph.r[1] = (dumy->r[1] + dumy->r[2]);
		msg = lcp(&lcph);
		
		dumy->cx[2] = lcph.sx[2];	
		dumy->cy[2] = lcph.sy[2];	
		dumy->r[2] = dumy->r[2];
		dumy->cx[3] = lcph.sx[3];	
		dumy->cy[3] = lcph.sy[3];	
		dumy->r[3] = dumy->r[2];
		
		/* 線Ｂと中心から半径 (dumy.r[1] + dumy.r[2]) の円の交点を求める。 */
		lcph.sx[1] = loh.sx[3]; 
		lcph.sy[1] = loh.sy[3]; 
		lcph.ex[1] = loh.ex[3]; 
		lcph.ey[1] = loh.ey[3];
		lcph.cx[1] = dumy->cx[1];	
		lcph.cy[1] = dumy->cy[1];	
		lcph.r[1] = (dumy->r[1] + dumy->r[2]);
		msg = lcp(&lcph);
		
		dumy->cx[4] = lcph.sx[2];	
		dumy->cy[4] = lcph.sy[2];	
		dumy->r[4] = dumy->r[2];
		dumy->cx[5] = lcph.sx[3];	
		dumy->cy[5] = lcph.sy[3];	
		dumy->r[5] = dumy->r[2];
		
		dumy->type = 4;
		
		
		
		
		/* 広い方 ---  */
		if ((lpr1 > r2) && dist != 0)
		{
			/* 線を中心の方向にdumy.r[2]だけオフセットした線Ａを求める。 */
			/* 直線の最近点から、中心までの、線の角度Ａを求める。 */
			/* ------------------------------------------------  */
			/* LA	直線の角度									 */
			/* ------------------------------------------------  */
			lah.sx[1] = plph.ex[2]; 
			lah.sy[1] = plph.ey[2]; 
			lah.ex[1] = dumy->cx[1];	
			lah.ey[1] = dumy->cy[1];
			msg = la(&lah);
		
			/* 最近点から角度Ａで長さ半径dumy.r[2]の点	*/
			/* ------------------------------------------------  */
			/* PAP	始点と角度と距離で直線の終点を求める		 */
			/* ------------------------------------------------  */
			paph.sx[1] = dumy->sx[1];	
			paph.sy[1] = dumy->sy[1];
			paph.angle = lah.angle; 
			paph.l = dumy->r[2];
			msg = pap(&paph);
			dumy->sx[2] = paph.ex[1];	
			dumy->sy[2] = paph.ey[1];
		
			paph.sx[1] = dumy->ex[1];	
			paph.sy[1] = dumy->ey[1];
			paph.angle = lah.angle; 
			paph.l = dumy->r[2];
			msg = pap(&paph);
			dumy->ex[2] = paph.ex[1];	
			dumy->ey[2] = paph.ey[1];
		
			/* 中心から半径(dumy.r[1] - dumy.r[2])の円と線Ａの交点を求める。 */
		
			/* ------------------------------------------------  */
			/* LCP 直線と円の交点	初めてのグラフィックス P126  */
			/* ------------------------------------------------  */
			lcph.sx[1] = dumy->sx[2];	
			lcph.sy[1] = dumy->sy[2];	
			lcph.ex[1] = dumy->ex[2];	
			lcph.ey[1] = dumy->ey[2];
			lcph.cx[1] = dumy->cx[1];	
			lcph.cy[1] = dumy->cy[1];	
			lcph.r[1] = (dumy->r[1] - dumy->r[2]);
			msg = lcp(&lcph);
		
			if (lcph.type == 1)
			{
				dumy->cx[6] = lcph.sx[2];	
				dumy->cy[6] = lcph.sy[2];	
				dumy->r[6] = dumy->r[2];
			}
			if (lcph.type == 2)
			{
				dumy->cx[6] = lcph.sx[2];	
				dumy->cy[6] = lcph.sy[2];	
				dumy->r[6] = dumy->r[2];
				dumy->cx[7] = lcph.sx[3];	
				dumy->cy[7] = lcph.sy[3];	
				dumy->r[7] = dumy->r[2];
			}
			dumy->type = dumy->type + lcph.type;
		}
		
		
		
		
		/* 狭い方 ---  */
		if ((r1ml > r2) && dist != 0)
		{
			/* 線を中心と逆の方向にdumy.r[2]だけオフセットした線Ｂを求める。 */
			/* 直線の最近点から、中心までの、線の角度Ａを求める。 */
			/* ------------------------------------------------  */
			/* LA	直線の角度									 */
			/* ------------------------------------------------  */
			lah.sx[1] = plph.ex[2]; 
			lah.sy[1] = plph.ey[2]; 
			lah.ex[1] = dumy->cx[1];	
			lah.ey[1] = dumy->cy[1];
			msg = la(&lah);
		
			/* 最近点から角度Ａで長さ半径dumy.r[2]の点	*/
			/* ------------------------------------------------  */
			/* PAP	始点と角度と距離で直線の終点を求める		 */
			/* ------------------------------------------------  */
			paph.sx[1] = dumy->sx[1];	
			paph.sy[1] = dumy->sy[1];
			paph.angle = (lah.angle + 180); 
			paph.l = dumy->r[2];
			msg = pap(&paph);
			dumy->sx[2] = paph.ex[1];	
			dumy->sy[2] = paph.ey[1];
		
			paph.sx[1] = dumy->ex[1];	
			paph.sy[1] = dumy->ey[1];
			paph.angle = (lah.angle + 180); 
			paph.l = dumy->r[2];
			msg = pap(&paph);
			dumy->ex[2] = paph.ex[1];	
			dumy->ey[2] = paph.ey[1];
		
			/* 中心から半径(dumy->r[1]-dumy.r[2])の円と線Ｂの交点を求める。 */
		
			/* ------------------------------------------------  */
			/* LCP 直線と円の交点	初めてのグラフィックス P126  */
			/* ------------------------------------------------  */
			lcph.sx[1] = dumy->sx[2];	
			lcph.sy[1] = dumy->sy[2];	
			lcph.ex[1] = dumy->ex[2];	
			lcph.ey[1] = dumy->ey[2];
			lcph.cx[1] = dumy->cx[1];	
			lcph.cy[1] = dumy->cy[1];	
			lcph.r[1] = (dumy->r[1] - dumy->r[2]);
			msg = lcp(&lcph);
			
			
			
			if (dumy->type == 4)
			{
				if (lcph.type == 1)
				{
					dumy->cx[6] = lcph.sx[2];	
					dumy->cy[6] = lcph.sy[2];	
					dumy->r[6] = dumy->r[2];
				}
			
				if (lcph.type == 2)
				{
					dumy->cx[6] = lcph.sx[2];	
					dumy->cy[6] = lcph.sy[2];	
					dumy->r[6] = dumy->r[2];
					dumy->cx[7] = lcph.sx[3];	
					dumy->cy[7] = lcph.sy[3];	
					dumy->r[7] = dumy->r[2];
				}
			}
			
			if (dumy->type == 5)
			{
				if (lcph.type == 1)
				{
					dumy->cx[7] = lcph.sx[2];	
					dumy->cy[7] = lcph.sy[2]; 
					dumy->r[7] = dumy->r[2];
				}
			
				if (lcph.type == 2)
				{
					dumy->cx[7] = lcph.sx[2];	
					dumy->cy[7] = lcph.sy[2];	
					dumy->r[7] = dumy->r[2];
					dumy->cx[8] = lcph.sx[3];	
					dumy->cy[8] = lcph.sy[3];	
					dumy->r[8] = dumy->r[2];
				}
			}
			
			if (dumy->type == 6)
			{
				if (lcph.type == 1)
				{
					dumy->cx[8] = lcph.sx[2];	
					dumy->cy[8] = lcph.sy[2];	
					dumy->r[8] = dumy->r[2];
				}
			
				if (lcph.type == 2)
				{
					dumy->cx[8] = lcph.sx[2];	
					dumy->cy[8] = lcph.sy[2];	
					dumy->r[8] = dumy->r[2];
					dumy->cx[9] = lcph.sx[3];	
					dumy->cy[9] = lcph.sy[3];	
					dumy->r[9] = dumy->r[2];
				}
			}
			dumy->type = dumy->type + lcph.type;
		}
		
		
		
		/* 線が円の中心を通るとき。 ---  */
		if ((dist == 0) && r1 >= r2x2)
		{
			/* 線を両方に dumy->r[2] オフセットした線Ａ、線Ｂを求める。 */
			/* ------------------------------------------------  */
			/* LO	始点と終点を与えた直線のオフセット			 */
			/* ------------------------------------------------  */
			loh.sx[1] = dumy->sx[1];	
			loh.sy[1] = dumy->sy[1];	
			loh.ex[1] = dumy->ex[1];	
			loh.ey[1] = dumy->ey[1];	
			loh.l = dumy->r[2];
			msg = lo(&loh);
		
			/* 線Ａと中心から半径 (dumy->r[1]-dumy->r[2]) の円の交点を求める。 */
			/* ------------------------------------------------  */
			/* LCP 直線と円の交点	初めてのグラフィックス P126  */
			/* ------------------------------------------------  */
			lcph.sx[1] = loh.sx[2]; 
			lcph.sy[1] = loh.sy[2]; 
			lcph.ex[1] = loh.ex[2]; 
			lcph.ey[1] = loh.ey[2];
			lcph.cx[1] = dumy->cx[1];	
			lcph.cy[1] = dumy->cy[1];	
			lcph.r[1] = (dumy->r[1] - dumy->r[2]);
			msg = lcp(&lcph);
		
			if (lcph.type == 1)
			{
				dumy->cx[6] = lcph.sx[2];	
				dumy->cy[6] = lcph.sy[2];	
				dumy->r[6] = dumy->r[2];
			}
			
			if (lcph.type == 2)
			{
				dumy->cx[6] = lcph.sx[2];	
				dumy->cy[6] = lcph.sy[2];	
				dumy->r[6] = dumy->r[2];
				dumy->cx[7] = lcph.sx[3];	
				dumy->cy[7] = lcph.sy[3];	
				dumy->r[7] = dumy->r[2];
			}
						
			dumy->type = dumy->type + lcph.type;
			
			/* 線Ｂと中心から半径 (dumy->r[1]-dumy->r[2]) の円の交点を求める。 */
			lcph.sx[1] = loh.sx[3]; 
			lcph.sy[1] = loh.sy[3]; 
			lcph.ex[1] = loh.ex[3]; 
			lcph.ey[1] = loh.ey[3];
			lcph.cx[1] = dumy->cx[1];	
			lcph.cy[1] = dumy->cy[1];	
			lcph.r[1] = (dumy->r[1] - dumy->r[2]);
			msg = lcp(&lcph);
		
			if (dumy->type == 4)
			{
				if (lcph.type == 1)
				{
					dumy->cx[6] = lcph.sx[2];	
					dumy->cy[6] = lcph.sy[2];	
					dumy->r[6] = dumy->r[2];
				}
				if (lcph.type == 2)
				{
					dumy->cx[6] = lcph.sx[2];	
					dumy->cy[6] = lcph.sy[2];	
					dumy->r[6] = dumy->r[2];
					dumy->cx[7] = lcph.sx[3];	
					dumy->cy[7] = lcph.sy[3];	
					dumy->r[7] = dumy->r[2];
				}
			}
				
			if (dumy->type == 5)
			{
				if (lcph.type == 1)
				{
					dumy->cx[7] = lcph.sx[2];	
					dumy->cy[7] = lcph.sy[2];	
					dumy->r[7] = dumy->r[2];
				}
				if (lcph.type == 2)
				{
					dumy->cx[7] = lcph.sx[2];	
					dumy->cy[7] = lcph.sy[2];	
					dumy->r[7] = dumy->r[2];
					dumy->cx[8] = lcph.sx[3];	
					dumy->cy[8] = lcph.sy[3];	
					dumy->r[8] = dumy->r[2];
				}
			}
				
			if (dumy->type == 6)
			{
				if (lcph.type == 1)
				{
					dumy->cx[8] = lcph.sx[2];	
					dumy->cy[8] = lcph.sy[2];	
					dumy->r[8] = dumy->r[2];
				}
				if (lcph.type == 2)
				{
					dumy->cx[8] = lcph.sx[2];
					dumy->cy[8] = lcph.sy[2];
					dumy->r[8] = dumy->r[2];
					dumy->cx[9] = lcph.sx[3];
					dumy->cy[9] = lcph.sy[3];
					dumy->r[9] = dumy->r[2];
				}
			}
			dumy->type = dumy->type + lcph.type;
		}
	}
	/* 4 = end ========================================================  */
	/* ========== < 線と円が、重なっている。 >				 ==========  */
	/* ================================================================  */

	return 0;
} 



/* -------------------------------------------------------------------
 * 説明 : [18] ２点を通る円
 * 関数 : ppc
 * 引数 : rtndat
 *		:	点 1	   (sx[1] , sy[1])
 *		:	点 2	   (sx[2] , sy[2])
 *		:	円の半径	r[1]
 * 戻値 : rtndat
 *		:	type		0 , 1 , 2
 *		:	円 1	   (cx[1] , cy[1]) , r[1]
 *		:	円 2	   (cx[2] , cy[2]) , r[2]
 * 外部 : 無し
 * 内部 : RtnDat pph,ccph
 */
int ppc(struct RtnDat *dumy)
{
	struct RtnDat pph,ccph;
	int msg;


	/* ---	２点間の距離から円を判別  ---  */
	pph.sx[1] = dumy->sx[1];	
	pph.sy[1] = dumy->sy[1];	
	pph.ex[1] = dumy->sx[2];	
	pph.ey[1] = dumy->sy[2];
	msg = pp(&pph);

	if ( sg(pph.l, compa_digits) < sg(2 * dumy->r[1], compa_digits) )
		dumy->type = 2;
	else if ( sg(pph.l, compa_digits) == sg(2 * dumy->r[1], compa_digits) )
		dumy->type = 1;
	else if ( sg(pph.l, compa_digits) > sg(2 * dumy->r[1], compa_digits) )
		dumy->type = 0;


	if (dumy->type != 0)
	{
		/* 点１と点２を中心とする半径Ｒの２円の交点が求める円の交点  */
		ccph.cx[1] = dumy->sx[1];	
		ccph.cy[1] = dumy->sy[1];	
		ccph.r[1] = dumy->r[1];
		ccph.cx[2] = dumy->sx[2];	
		ccph.cy[2] = dumy->sy[2];	
		ccph.r[2] = dumy->r[1];
		msg = ccp(&ccph);

		dumy->cx[1] = ccph.sx[1];	
		dumy->cy[1] = ccph.sy[1];
		dumy->cx[2] = ccph.sx[2];
		dumy->cy[2] = ccph.sy[2];
		dumy->r[2] = dumy->r[1];
	}

	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [20] １点を通り直線に接する円
 * 関数 : plc
 * 引数 : rtndat
 *		:	点			 (sx[1] , sy[1])
 *		:	線			 (sx[2] , sy[2]) - (ex[2] , ey[2])
 *		:	円の半径	  r[3]
 * 戻値 : rtndat
 *		:	type
 *		:	円 1		 (cx[3]  , cy[3]) , r[3]
 *		:	円 2		 (cx[4]  , cy[4]) , r[4]
 * 外部 : 無し
 * 内部 : RtnDat plph,pph,lah,paph,lcph,loh
 */
int plc(struct RtnDat *dumy)
{
	struct RtnDat plph,pph,lah,paph,lcph,loh;
	int msg;
	double dist;


	/* 点と線の距離を求める。 */
	/* 線を半径Ｒ３オフセットした線Ａを求める。 */
	plph.sx[1] = dumy->sx[2];	
	plph.sy[1] = dumy->sy[2];	
	plph.ex[1] = dumy->ex[2];	
	plph.ey[1] = dumy->ey[2];
	plph.sx[2] = dumy->sx[1];	
	plph.sy[2] = dumy->sy[1];
	msg = plp(&plph);
	/* 交点(plph.ex[2],plph.ey[2])	*/

	pph.sx[1] = dumy->sx[1];	
	pph.sy[1] = dumy->sy[1];	
	pph.ex[1] = plph.ex[2]; 
	pph.ey[1] = plph.ey[2];
	msg = pp(&pph);

	dist = sg(pph.l, compa_digits);



	if (dist > 0)
	{
		lah.sx[1] = plph.ex[2]; 
		lah.sy[1] = plph.ey[2];
		lah.ex[1] = dumy->sx[1];	
		lah.ey[1] = dumy->sy[1];
		msg = la(&lah);

		paph.sx[1] = dumy->sx[2];	
		paph.sy[1] = dumy->sy[2];	
		paph.angle = lah.angle; 
		paph.l = dumy->r[3];
		msg = pap(&paph);

		lcph.sx[1] = paph.ex[1];	
		lcph.sy[1] = paph.ey[1];

		paph.sx[1] = dumy->ex[2];	
		paph.sy[1] = dumy->ey[2];	
		paph.angle = lah.angle; 
		paph.l = dumy->r[3];
		msg = pap(&paph);

		lcph.ex[1] = paph.ex[1];	
		lcph.ey[1] = paph.ey[1];

		/* 点を中心とした半径Ｒ３の円と、線Ａの交点を求める。 */
		lcph.cx[1] = dumy->sx[1];	
		lcph.cy[1] = dumy->sy[1];	
		lcph.r[1] = dumy->r[3];
		msg = lcp(&lcph);

		if (lcph.type == 0)
		{
			dumy->type = 0;
		}

		if (lcph.type == 1)
		{
			dumy->type = 1;
			dumy->cx[3] = lcph.sx[2];	
			dumy->cy[3] = lcph.sy[2];
		}

		if (lcph.type == 2)
		{
			dumy->type = 2;
			dumy->cx[3] = lcph.sx[2];	
			dumy->cy[3] = lcph.sy[2];
			dumy->cx[4] = lcph.sx[3];	
			dumy->cy[4] = lcph.sy[3];	
			dumy->r[4] = dumy->r[3];
		}
	}


	if (dist == 0)
	{
		loh.sx[1] = dumy->sx[2];	
		loh.sy[1] = dumy->sy[2];	
		loh.ex[1] = dumy->ex[2];	
		loh.ey[1] = dumy->ey[2];	
		loh.l = dumy->r[3];
		msg = lo(&loh);

		lcph.sx[1] = loh.sx[2]; 
		lcph.sy[1] = loh.sy[2]; 
		lcph.ex[1] = loh.ex[2]; 
		lcph.ey[1] = loh.ey[2];
		lcph.cx[1] = dumy->sx[1];	
		lcph.cy[1] = dumy->sy[1];	
		lcph.r[1] = dumy->r[3];
		msg = lcp(&lcph);

		dumy->cx[3] = lcph.sx[2];	
		dumy->cy[3] = lcph.sy[2];	
		dumy->r[3] = dumy->r[3];

		lcph.sx[1] = loh.sx[3]; 
		lcph.sy[1] = loh.sy[3]; 
		lcph.ex[1] = loh.ex[3]; 
		lcph.ey[1] = loh.ey[3];
		lcph.cx[1] = dumy->sx[1];	
		lcph.cy[1] = dumy->sy[1];	
		lcph.r[1] = dumy->r[3];
		msg = lcp(&lcph);

		dumy->cx[4] = lcph.sx[2];	
		dumy->cy[4] = lcph.sy[2];	
		dumy->r[4] = dumy->r[3];

		dumy->type = 2;
	}

	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [21] ２直線に接する円
 * 関数 : llc
 * 引数 : rtndat
 *		:	線 1	(sx[1] , sy[1]) -  (ex[1] , ey[1])
 *		:	線 2	(sx[2] , sy[2]) -  (ex[2] , ey[2])
 *		:	半径	 r[1]
 * 戻値 : rtndat
 *		:	円 1	(cx[1] , cy[1]) , r[1]
 *		:	円 2	(cx[2] , cy[2]) , r[2]
 *		:	円 3	(cx[3] , cy[3]) , r[3]
 *		:	円 4	(cx[4] , cy[4]) , r[4]
 * 外部 : 無し
 * 内部 : RtnDat loh,llph
 */
int llc(struct RtnDat *dumy)
{
	struct RtnDat loh,llph;
	double sx11,sy11,ex11,ey11;
	double sx12,sy12,ex12,ey12;
	double sx21,sy21,ex21,ey21;
	double sx22,sy22,ex22,ey22;
	int msg;


	/* 線１を半径Ｒ分オフセットして線１１と線１２を求める。 */
	loh.sx[1] = dumy ->sx[1];	
	loh.sy[1] = dumy ->sy[1];	
	loh.ex[1] = dumy ->ex[1];	
	loh.ey[1] = dumy ->ey[1];	
	loh.l = dumy ->r[1];
	msg = lo(&loh);
	
	sx11 = loh.sx[2];	
	sy11 = loh.sy[2];	
	ex11 = loh.ex[2];	
	ey11 = loh.ey[2];
	sx12 = loh.sx[3];	
	sy12 = loh.sy[3];	
	ex12 = loh.ex[3];	
	ey12 = loh.ey[3];
	
	/* 線２を半径Ｒ分オフセットして線２１と線２２を求める。 */
	loh.sx[1] = dumy ->sx[2];	
	loh.sy[1] = dumy ->sy[2];	
	loh.ex[1] = dumy ->ex[2];	
	loh.ey[1] = dumy ->ey[2];	
	loh.l = dumy ->r[1];
	msg = lo(&loh);
	
	sx21 = loh.sx[2];	
	sy21 = loh.sy[2];	
	ex21 = loh.ex[2];	
	ey21 = loh.ey[2];
	sx22 = loh.sx[3];	
	sy22 = loh.sy[3];	
	ex22 = loh.ex[3];	
	ey22 = loh.ey[3];
	
	/* 線１１と線２１との交点が求める円１の中心 */
	llph.sx[1] = sx11;	
	llph.sy[1] = sy11;	
	llph.ex[1] = ex11;	
	llph.ey[1] = ey11;
	llph.sx[2] = sx21;	
	llph.sy[2] = sy21;	
	llph.ex[2] = ex21;	
	llph.ey[2] = ey21;
	msg = llp(&llph);
	
	if (llph.type == 0)
	{
		dumy ->type = 0;
		return 1;
	}

	dumy ->type = 4;

	dumy ->cx[1] = llph.sx[3];	
	dumy ->cy[1] = llph.sy[3];

	/* 線１１と線２２との交点が求める円２の中心 */
	llph.sx[1] = sx11;	
	llph.sy[1] = sy11;	
	llph.ex[1] = ex11;	
	llph.ey[1] = ey11;
	llph.sx[2] = sx22;	
	llph.sy[2] = sy22;	
	llph.ex[2] = ex22;	
	llph.ey[2] = ey22;
	msg = llp(&llph);

	dumy ->cx[2] = llph.sx[3];	
	dumy ->cy[2] = llph.sy[3];	
	dumy ->r[2] = dumy ->r[1];

	/* 線１２と線２１との交点が求める円３の中心 */
	llph.sx[1] = sx12;	
	llph.sy[1] = sy12;	
	llph.ex[1] = ex12;	
	llph.ey[1] = ey12;
	llph.sx[2] = sx21;	
	llph.sy[2] = sy21;	
	llph.ex[2] = ex21;	
	llph.ey[2] = ey21;
	msg = llp(&llph);

	dumy ->cx[3] = llph.sx[3];	
	dumy ->cy[3] = llph.sy[3];	
	dumy ->r[3] = dumy ->r[1];

	/* 線１２と線２２との交点が求める円４の中心 */
	llph.sx[1] = sx12;	
	llph.sy[1] = sy12;	
	llph.ex[1] = ex12;	
	llph.ey[1] = ey12;
	llph.sx[2] = sx22;	
	llph.sy[2] = sy22;	
	llph.ex[2] = ex22;	
	llph.ey[2] = ey22;
	msg = llp(&llph);

	dumy ->cx[4] = llph.sx[3];	
	dumy ->cy[4] = llph.sy[3];	
	dumy ->r[4] = dumy ->r[1];

	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [22] ２等分線
 * 関数 : lll
 * 引数 : rtndat
 *		:	線 1	(sx[1] , sy[1]) -  (ex[1] , ey[1])
 *		:	線 2	(sx[2] , sy[2]) -  (ex[2] , ey[2])
 * 戻値 : rtndat
 *		:	type	 0:２直線が平行
 *		:	線		(sx[3] , sy[3]) - (ex[3] , ey[3])  ベクトルで考えれる。
 * 外部 : 無し
 * 内部 : RtnDat lah,llph,paph
 */
int lll(struct RtnDat *dumy)
{
	struct RtnDat lah,llph,paph;
	double a0 = 0,a1 = 0,a2 = 0;
	int msg;


	/* 直線１の角度 */
	lah.sx[1] = dumy->sx[1];	
	lah.sy[1] = dumy->sy[1];	
	lah.ex[1] = dumy->ex[1];	
	lah.ey[1] = dumy->ey[1];
	msg = la(&lah);
	a1 = lah.angle;

	/* 直線２の角度 */
	lah.sx[1] = dumy->sx[2];	
	lah.sy[1] = dumy->sy[2];	
	lah.ex[1] = dumy->ex[2];	
	lah.ey[1] = dumy->ey[2];
	msg = la(&lah);
	a2 = lah.angle;

	/* 直線１と直線２の差が角度Ａ */
	if (sg(a1, compa_digits) > sg(a2, compa_digits)){
		a0 = a1 - a2;	
		a0 = a2 + (a0 / 2);
	}
	if (sg(a1, compa_digits) == sg(a2, compa_digits)){
		dumy->type = 0;
		return 1;
	}
	if (sg(a1, compa_digits) < sg(a2, compa_digits)){
		a0 = a2 - a1;	
		a0 = a1 + (a0 / 2);
	}

	/* 直線１と直線２の交点が始点 */
	llph.sx[1] = dumy->sx[1];
	llph.sy[1] = dumy->sy[1];
	llph.ex[1] = dumy->ex[1];
	llph.ey[1] = dumy->ey[1];
	llph.sx[2] = dumy->sx[2];
	llph.sy[2] = dumy->sy[2];
	llph.ex[2] = dumy->ex[2];
	llph.ey[2] = dumy->ey[2];
	msg = llp(&llph);

	dumy->sx[3] = llph.sx[3];
	dumy->sy[3] = llph.sy[3];

	/* 始点から角度Ａに距離 100 の直線 */
	paph.sx[1] = dumy->sx[3];	
	paph.sy[1] = dumy->sy[3];	
	paph.angle = a0;	
	paph.l = 100;
	msg = pap(&paph);
	
	dumy->ex[3] = paph.ex[1];	
	dumy->ey[3] = paph.ey[1];
	dumy->type = 1;

	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [23] 点を通り、直線に平行な直線
 * 関数 : pll
 * 引数 : rtndat
 *		:	点		 (sx[1] , sy[1])
 *		:	線		 (sx[2] , sy[2]) -	(ex[2] , ey[2])
 * 戻値 : rtndat
 *		:	type
 *		:	線		 (sx[3] , sy[3]) - (ex[3] , ey[3])
 * 外部 : 無し
 * 内部 : RtnDat plph
 */
int pll(struct RtnDat *dumy)
{
	struct RtnDat plph;
	double dx,dy;
	int msg;


	/* PLP	点と直線の最近点（１点を通る垂線） */
	plph.sx[2] = dumy->sx[1];	
	plph.sy[2] = dumy->sy[1];
	plph.sx[1] = dumy->sx[2];	
	plph.sy[1] = dumy->sy[2];	
	plph.ex[1] = dumy->ex[2];	
	plph.ey[1] = dumy->ey[2];
	msg = plp(&plph);
	/* 最近点 (plph.ex[2],plph.ey[2])  */

	dx = dumy->sx[1] - plph.ex[2];
	dy = dumy->sy[1] - plph.ey[2];

	dumy->sx[3] = dumy->sx[2] + dx;
	dumy->sy[3] = dumy->sy[2] + dy;
	dumy->ex[3] = dumy->ex[2] + dx;
	dumy->ey[3] = dumy->ey[2] + dy;

	dumy->type = 1;
	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : [24] 円に接する角度Ａの接線
 * 関数 : cal
 * 引数 : rtndat
 *		:	円		(cx[1] , cy[1]) , r[1]
 *		:	角度	 angle
 *		:	長さ	 l (l = 0 のときは l=100)
 * 戻値 : rtndat
 *		:	type	 0 , 2
 *		:	線 1	(sx[1] , sy[1]) - (ex[1] , ey[1])
 *		:	線 2	(sx[2] , sy[2]) - (ex[2] , ey[2])
 * 外部 : 無し
 * 内部 : RtnDat paph,loh
 */
int cal(struct RtnDat *dumy)
{
	struct RtnDat paph,loh;
	int msg;


	if (dumy->l == 0)
		dumy->l = 100.;

	/* 中心から角度Ａの線を引く */
	paph.sx[1] = dumy->cx[1];	
	paph.sy[1] = dumy->cy[1];
	paph.angle = dumy->angle;	
	paph.l = dumy->l;
	msg = pap(&paph);
	/* (paph.sx[1],paph.sy[1])-(paph.ex[1],paph.ey[1])	*/

	/* その線を半径オフセットした線  */
	loh.sx[1] = paph.sx[1]; 
	loh.sy[1] = paph.sy[1];
	loh.ex[1] = paph.ex[1]; 
	loh.ey[1] = paph.ey[1];
	loh.l = dumy->r[1];
	msg = lo(&loh);
	/* (loh.sx[2],loh.sy[2])-(loh.ex[2],loh.ey[2]),,(loh.sx[3],loh.sy[3])-(loh.ex[3],loh.ey[3])  */

	dumy->type = 2;
	dumy->sx[1] = loh.sx[2];
	dumy->sy[1] = loh.sy[2];
	dumy->ex[1] = loh.ex[2];
	dumy->ey[1] = loh.ey[2];
	dumy->sx[2] = loh.sx[3];
	dumy->sy[2] = loh.sy[3];
	dumy->ex[2] = loh.ex[3];
	dumy->ey[2] = loh.ey[3];
	return 0;
}



/* -------------------------------------------------------------------
 * 四捨五入 小数点以下桁指定 (20)
 *	
 * double : 小数点データ 
 * int	  : 丸める桁数
 */
double sg(double value, int keta)
{
	int i = 1;

	if (value < 0)
		i = -1;
	value = value * (double) i;
	value = (double) i * (floor( ( pow(10, keta) * value) + 0.5) ) / pow(10, keta);
	return value;
}



/* -------------------------------------------------------------------
 * 説明 : [19] ２円に接する円
 * 関数 : ccc
 * 引数 : rtndat
 *		:	円 1	 (cx[1] , cy[1]) ,r[1]
 *		:	円 2	 (cx[2] , cy[2]) ,r[2]
 *		:	円の半径  r[3]
 * 戻値 : rtndat
 *		:	type
 *		:	円 1	 (cx[ 3]  , cy[ 3]) , r[ 3]
 *		:	円 2	 (cx[ 4]  , cy[ 4]) , r[ 4]
 *		:	円 3	 (cx[ 5]  , cy[ 5]) , r[ 5]
 *		:	円 4	 (cx[ 6]  , cy[ 6]) , r[ 6]
 *		:	円 5	 (cx[ 7]  , cy[ 7]) , r[ 7]
 *		:	円 6	 (cx[ 8]  , cy[ 8]) , r[ 8]
 *		:	円 7	 (cx[ 9]  , cy[ 9]) , r[ 9]
 *		:	円 8	 (cx[10]  , cy[10]) , r[10]
 *		:	円 9	 (cx[11]  , cy[11]) , r[11]
 *		:	円 10	 (cx[12]  , cy[12]) , r[12]
 * 外部 : 無し
 * 内部 : RtnDat pph,ccph,,lah,paph
 */
int ccc(struct RtnDat *dumy)
{
	struct RtnDat pph, ccph, lah, paph;
	double DumyD;
	int i;


	i = 2;

	/* 円２の方を大きくなるようにする。 */
	if (dumy->r[1] > dumy->r[2]) {
		DumyD = dumy->cx[1];
		dumy->cx[1] = dumy->cx[2];
		dumy->cx[2] = DumyD;

		DumyD = dumy->cy[1];
		dumy->cy[1] = dumy->cy[2];
		dumy->cy[2] = DumyD;

		DumyD = dumy->r[1];
		dumy->r[1] = dumy->r[2];
		dumy->r[2] = DumyD;
	}

	/* 2円の中心の距離を計算する */
	pph.sx[1] = dumy->cx[1];
	pph.sy[1] = dumy->cy[1];
	pph.ex[1] = dumy->cx[2];
	pph.ey[1] = dumy->cy[2];
	pp(&pph);



	/* ① 円１と円２が離れている。 ----------------------------------- */
	if (sg(pph.l, calcu_digits) > sg(dumy->r[1] + dumy->r[2], calcu_digits)) {
		/* < 1 & 2 > */
		if (sg(2 * dumy->r[3], calcu_digits) >= sg(pph.l - dumy->r[1] - dumy->r[2], calcu_digits)) {
			/* 円１の中心から半径 dumy->r[1]+dumy->R[3] の円と */
			/* 円２の中心から半径 dumy->r[2]+dumy->R[3] の円の交点 */
			/* ------------------------------------------------ */
			/* CCP	２円の交点 */
			/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX[2],dumy->cy[2]) dumy->R[2] */
			/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX[2],dumy->SY[2]) */
			/*	   dumy->TYPE */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[1] + dumy->r[3];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[2] + dumy->r[3];
			ccp(&ccph);
						
			if (ccph.type == 1) {
				i = i + 1;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}

			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
			
		}



		/* < 3 & 4 > */
		if (sg(2 * dumy->r[3], calcu_digits) >= sg(pph.l + dumy->r[1] - dumy->r[2], calcu_digits)) {
			/* 円１の中心から半径 dumy->r[3]-dumy->R[1] の円と */
			/* 円２の中心から半径 dumy->r[3]+dumy->R[2] の円の交点 */
			/* ------------------------------------------------ */
			/* CCP	２円の交点 */
			/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX[2],dumy->cy[2]) dumy->R[2] */
			/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX[2],dumy->SY[2]) */
			/*	   dumy->TYPE */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[3] - dumy->r[1];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[3] + dumy->r[2];
			ccp(&ccph);
						
			if (ccph.type == 1) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}
			
			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
		}



		/* < 5 & 6 > */
		if (sg(2 * dumy->r[3], calcu_digits) >= sg(pph.l + dumy->r[2] - dumy->r[1], calcu_digits)) {
			/* 円１の中心から半径 dumy->r[3]+dumy->R[1] の円と */
			/* 円２の中心から半径 dumy->r[3]-dumy->R[2] の円の交点 */
			/* ------------------------------------------------ */
			/* CCP	２円の交点 */
			/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX[2],dumy->cy[2]) dumy->R[2] */
			/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX[2],dumy->SY[2]) */
			/*	   dumy->TYPE */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[3] + dumy->r[1];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[3] - dumy->r[2];
			ccp(&ccph);
						
			if (ccph.type == 1) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}

			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
		}



		/* < 7 & 8 > */
		if (sg(2 * dumy->r[3], calcu_digits) >= sg(pph.l + dumy->r[1] + dumy->r[2], calcu_digits)) {
			/* 円１の中心から半径 dumy->r[3]-dumy->R[1] の円と */
			/* 円２の中心から半径 dumy->r[3]-dumy->R[2] の円の交点 */
			/* ------------------------------------------------ */
			/* CCP	２円の交点 */
			/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX[2],dumy->cy[2]) dumy->R[2] */
			/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX[2],dumy->SY[2]) */
			/*	   dumy->TYPE */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[3] - dumy->r[1];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[3] - dumy->r[2];
			ccp(&ccph);

			if (ccph.type == 1) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}

			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
		}
	}





	/* ② 円１と円２が隣同士接している。 ----------------------------- */
	if (sg(pph.l, calcu_digits) == sg(dumy->r[1] + dumy->r[2], calcu_digits)) {
		/* < 1 & 2 > */
		/* 円１の中心から半径 dumy->r[3]+dumy->R[1] の円と */
		/* 円２の中心から半径 dumy->r[3]+dumy->R[2] の円の交点 */
		/* ------------------------------------------------ */
		/* CCP	２円の交点 */
		/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX[2],dumy->cy[2]) dumy->R[2] */
		/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX[2],dumy->SY[2]) */
		/*	   dumy->TYPE */
		/* ------------------------------------------------ */
		ccph.cx[1] = dumy->cx[1];
		ccph.cy[1] = dumy->cy[1];
		ccph.r[1] = dumy->r[3] + dumy->r[1];
		ccph.cx[2] = dumy->cx[2];
		ccph.cy[2] = dumy->cy[2];
		ccph.r[2] = dumy->r[3] + dumy->r[2];
		ccp(&ccph);

		if (ccph.type == 1) {
			i++;
			dumy->cx[i] = ccph.sx[1];
			dumy->cy[i] = ccph.sy[1];
			dumy->r[i] = dumy->r[3];
		}

		if (ccph.type == 2) {
			i++;
			dumy->cx[i] = ccph.sx[1];
			dumy->cy[i] = ccph.sy[1];
			dumy->r[i] = dumy->r[3];
			i++;
			dumy->cx[i] = ccph.sx[2];
			dumy->cy[i] = ccph.sy[2];
			dumy->r[i] = dumy->r[3];
		}


		/* < 3 & 4 > */
		/* ------------------------------------------------ */
		/* CCP	２円の交点 */
		/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX(2),dumy->CY(2)) dumy->R(2) */
		/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX(2),dumy->SY(2)) */
		/*	   dumy->TYPE */
		/* ------------------------------------------------ */
		ccph.cx[1] = dumy->cx[1];
		ccph.cy[1] = dumy->cy[1];
		ccph.r[1] = dumy->r[1];
		ccph.cx[2] = dumy->cx[2];
		ccph.cy[2] = dumy->cy[2];
		ccph.r[2] = dumy->r[2];
		ccp(&ccph);
						
		if (ccph.type == 1) {
			/* 交点(ccph.sx[1],ccph.sy[1]) */
			/* ------------------------------------------------ */
			/* la	直線の角度 */
			/*	  (dumy->sx[1],dumy->sy[1]),(dumy->ex[1],dumy->ey[1]) */
			/*		   ans dumy->angle */
			/* ------------------------------------------------ */
			lah.sx[1] = ccph.sx[1];
			lah.sy[1] = ccph.sy[1];
			lah.ex[1] = dumy->cx[1];
			lah.ey[1] = dumy->cy[1];
			la(&lah);

			/* 3 */
			/* ------------------------------------------------ */
			/* pap	始点と角度と距離で直線の終点を求める */
			/*	  (dumy->sx[1] , dumy->sy[1]) , dumy->angle , dumy->l */
			/*		   ans	(dumy->ex[1] , dumy->ey[1]) */
			/* ------------------------------------------------ */
			paph.sx[1] = ccph.sx[1];
			paph.sy[1] = ccph.sy[1];
			paph.angle = lah.angle;
			paph.l = dumy->r[3];
			pap(&paph);

			i++;
			dumy->cx[i] = paph.ex[1];
			dumy->cy[i] = paph.ey[1];
			dumy->r[i] = dumy->r[3];

			/* 4 */
			/* ------------------------------------------------ */
			/* pap	始点と角度と距離で直線の終点を求める */
			/*	  (dumy->sx[1] , dumy->sy[1]) , dumy->angle , dumy->l */
			/*		   ans	(dumy->ex[1] , dumy->ey[1]) */
			/* ------------------------------------------------ */
			paph.sx[1] = ccph.sx[1];
			paph.sy[1] = ccph.sy[1];
			paph.angle = (lah.angle + 180);
			paph.l = dumy->r[3];
			pap(&paph);

			i++;
			dumy->cx[i] = paph.ex[1];
			dumy->cy[i] = paph.ey[1];
			dumy->r[i] = dumy->r[3];
		}



		/* < 5 & 6 > */
		if (sg(2 * dumy->r[3], calcu_digits) >= sg(pph.l + dumy->r[1] + dumy->r[2], calcu_digits)) {
			/* 円１の中心から半径 dumy->r[3]-dumy->R[1] の円と */
			/* 円２の中心から半径 dumy->r[3]-dumy->R[2] の円の交点 */
			/* ------------------------------------------------ */
			/* CCP	２円の交点 */
			/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX[2],dumy->cy[2]) dumy->R[2] */
			/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX[2],dumy->SY[2]) */
			/*	   dumy->TYPE */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[3] - dumy->r[1];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[3] - dumy->r[2];
			ccp(&ccph);

			if (ccph.type == 1) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}
			
			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
		}
	}





	/* ③ 円１と円２が重なっている。---------------------------------- */
	if ((sg(pph.l, calcu_digits) < sg(dumy->r[1] + dumy->r[2], calcu_digits))
		&&
			(sg(pph.l, calcu_digits) > sg(dumy->r[2] - dumy->r[1], calcu_digits)))
	{
		/* < 1 & 2 >  絶対ある */
		/* 円１の中心から半径 dumy->r[3]+dumy->R[1] の円と */
		/* 円２の中心から半径 dumy->r[3]+dumy->R[2] の円の交点 */
		/* ------------------------------------------------ */
		/* CCP	２円の交点 */
		/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX[2],dumy->cy[2]) dumy->R[2] */
		/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX[2],dumy->SY[2]) */
		/*	   dumy->TYPE */
		/* ------------------------------------------------ */
		ccph.cx[1] = dumy->cx[1];
		ccph.cy[1] = dumy->cy[1];
		ccph.r[1] = dumy->r[3] + dumy->r[1];
		ccph.cx[2] = dumy->cx[2];
		ccph.cy[2] = dumy->cy[2];
		ccph.r[2] = dumy->r[3] + dumy->r[2];
		ccp(&ccph);
						
		if (ccph.type == 1) {
			i++;
			dumy->cx[i] = ccph.sx[1];
			dumy->cy[i] = ccph.sy[1];
			dumy->r[i] = dumy->r[3];
		}
			
		if (ccph.type == 2) {
			i++;
			dumy->cx[i] = ccph.sx[1];
			dumy->cy[i] = ccph.sy[1];
			dumy->r[i] = dumy->r[3];
			i++;
			dumy->cx[i] = ccph.sx[2];
			dumy->cy[i] = ccph.sy[2];
			dumy->r[i] = dumy->r[3];
		}



		/* < 3 & 4 > */
		if (sg(2 * dumy->r[3], calcu_digits) <= sg(pph.l - dumy->r[2] + dumy->r[1], calcu_digits)) {
			/* 円１の中心から半径 dumy->r[1]-dumy->R[3] の円と */
			/* 円２の中心から半径 dumy->r[3]+dumy->R[2] の円の交点 */
			/* ------------------------------------------------ */
			/* CCP	２円の交点 */
			/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX[2],dumy->cy[2]) dumy->R[2] */
			/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX[2],dumy->SY[2]) */
			/*	   dumy->TYPE */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[1] - dumy->r[3];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[3] + dumy->r[2];
			ccp(&ccph);

			if (ccph.type == 1) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}
			
			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
		}



		/* < 5 & 6 > */
		if (sg(2 * dumy->r[3], calcu_digits) <= sg(pph.l - dumy->r[1] + dumy->r[2], calcu_digits)) {
			/* 円１の中心から半径 dumy->r[1]+dumy->R[3] の円と */
			/* 円２の中心から半径 dumy->r[2]-dumy->R[3] の円の交点 */
			/* ------------------------------------------------ */
			/* CCP	２円の交点 */
			/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX[2],dumy->cy[2]) dumy->R[2] */
			/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX[2],dumy->SY[2]) */
			/*	   dumy->TYPE */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[1] + dumy->r[3];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[2] - dumy->r[3];
			ccp(&ccph);
						
			if (ccph.type == 1) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}
			
			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
		}



		/* < 7 & 8 > */
		if (sg(2 * dumy->r[3], calcu_digits) <= sg(dumy->r[1] + dumy->r[2] - pph.l, calcu_digits)) {
			/* 円１の中心から半径 dumy->r[1]-dumy->R[3] の円と */
			/* 円２の中心から半径 dumy->r[2]-dumy->R[3] の円の交点 */
			/* ------------------------------------------------ */
			/* CCP	２円の交点 */
			/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX[2],dumy->cy[2]) dumy->R[2] */
			/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX[2],dumy->SY[2]) */
			/*	   dumy->TYPE */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[1] - dumy->r[3];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[2] - dumy->r[3];
			ccp(&ccph);
						
			if (ccph.type == 1) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}
			
			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
		}



		/* < 9 & 10 > */
		if (sg(2 * dumy->r[3], calcu_digits) >= sg(pph.l + dumy->r[1] + dumy->r[2], calcu_digits)) {
			/* 円１の中心から半径 dumy->r[3]-dumy->R[1] の円と */
			/* 円２の中心から半径 dumy->r[3]-dumy->R[2] の円の交点 */
			/* ------------------------------------------------ */
			/* CCP	２円の交点 */
			/* (dumy->cx[1],dumy->CY[1]) dumy->r[1] (dumy->CX[2],dumy->cy[2]) dumy->R[2] */
			/*	 Ans (dumy->SX[1],dumy->SY[1]) (dumy->SX[2],dumy->SY[2]) */
			/*	   dumy->TYPE */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[3] - dumy->r[1];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[3] - dumy->r[2];
			ccp(&ccph);
						
			if (ccph.type == 1) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}
			
			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
		}
	}





	/* ④ 円２の中に円１が入って接している。-------------------------- */
	if (sg(pph.l, calcu_digits) == sg(dumy->r[2] - dumy->r[1], calcu_digits)
		&& sg(dumy->r[2], calcu_digits) != sg(dumy->r[1], calcu_digits))
	{
		/* < 1 & 2 > */
		/* ------------------------------------------------ */
		/* ccp	２円の交点 */
		/* (dumy->cx[1],dumy->cy[1]) dumy->r[1] (dumy->cx[2],dumy->cy[2]) dumy->r[2] */
		/*	 ans (dumy->sx[1],dumy->sy[1]) (dumy->sx[2],dumy->sy[2]) */
		/*	   dumy->type */
		/* ------------------------------------------------ */
		ccph.cx[1] = dumy->cx[1];
		ccph.cy[1] = dumy->cy[1];
		ccph.r[1] = dumy->r[1];
		ccph.cx[2] = dumy->cx[2];
		ccph.cy[2] = dumy->cy[2];
		ccph.r[2] = dumy->r[2];
		ccp(&ccph);
						
		/* 交点(ccph.sx[1],ccph.sy[1]) */
		/* ------------------------------------------------ */
		/* la	直線の角度 */
		/*	  (dumy->sx[1],dumy->sy[1]),(dumy->ex[1],dumy->ey[1]) */
		/*		   ans dumy->angle */
		/* ------------------------------------------------ */
		lah.sx[1] = ccph.sx[1];
		lah.sy[1] = ccph.sy[1];
		lah.ex[1] = dumy->cx[1];
		lah.ey[1] = dumy->cy[1];
		la(&lah);
			
		/* 3 */
		/* ------------------------------------------------ */
		/* pap	始点と角度と距離で直線の終点を求める */
		/*	  (dumy->sx[1] , dumy->sy[1]) , dumy->angle , dumy->l */
		/*		   ans	(dumy->ex[1] , dumy->ey[1]) */
		/* ------------------------------------------------ */
		paph.sx[1] = ccph.sx[1];
		paph.sy[1] = ccph.sy[1];
		paph.angle = lah.angle;
		paph.l = dumy->r[3];
		pap(&paph);
			
		i++;
		dumy->cx[i] = paph.ex[1];
		dumy->cy[i] = paph.ey[1];
		dumy->r[i] = dumy->r[3];
		
		/* 4 */
		/* ------------------------------------------------ */
		/* pap	始点と角度と距離で直線の終点を求める */
		/*	  (dumy->sx[1] , dumy->sy[1]) , dumy->angle , dumy->l */
		/*		   ans	(dumy->ex[1] , dumy->ey[1]) */
		/* ------------------------------------------------ */
		paph.sx[1] = ccph.sx[1];
		paph.sy[1] = ccph.sy[1];
		paph.angle = (lah.angle + 180);
		paph.l = dumy->r[3];
		pap(&paph);
			
		i++;
		dumy->cx[i] = paph.ex[1];
		dumy->cy[i] = paph.ey[1];
		dumy->r[i] = dumy->r[3];



		/* < 3 & 4 > */
		if (sg(2 * dumy->r[3], calcu_digits) <= sg(2 * dumy->r[2] - 2 * dumy->r[1], calcu_digits)) {
			/* 円１の中心から半径 dumy->r[1]+dumy->r[3] の円と */
			/* 円２の中心から半径 dumy->r[2]-dumy->r[3] の円の交点 */
			/* ------------------------------------------------ */
			/* ccp	２円の交点 */
			/* (dumy->cx[1],dumy->cy[1]) dumy->r[1] (dumy->cx[2],dumy->cy[2]) dumy->r[2] */
			/*	 ans (dumy->sx[1],dumy->sy[1]) (dumy->sx[2],dumy->sy[2]) */
			/*	   dumy->type */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[1] + dumy->r[3];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[2] - dumy->r[3];
			ccp(&ccph);
						
			if (ccph.type == 1) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}
			
			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
		}
	}



	/* ⑤ 円２の中に円１が入って離れている。-------------------------- */
	if (sg(pph.l, calcu_digits) < sg(dumy->r[2] - dumy->r[1], calcu_digits)
		&& sg(dumy->r[2], calcu_digits) != sg(dumy->r[1], calcu_digits))
	{
		/* < 1 & 2 > */
		if ((sg(2 * dumy->r[3], calcu_digits) >= sg(dumy->r[2] - pph.l - dumy->r[1], calcu_digits))
			&&
				(sg(2 * dumy->r[3], calcu_digits) <= sg(dumy->r[2] + pph.l - dumy->r[1], calcu_digits)))
		{
			/* 円１の中心から半径 dumy->r[1]+dumy->r[3] の円と */
			/* 円２の中心から半径 dumy->r[2]-dumy->r[3] の円の交点 */
			/* ------------------------------------------------ */
			/* ccp	２円の交点 */
			/* (dumy->cx[1],dumy->cy[1]) dumy->r[1] (dumy->cx[2],dumy->cy[2]) dumy->r[2] */
			/*	 ans (dumy->sx[1],dumy->sy[1]) (dumy->sx[2],dumy->sy[2]) */
			/*	   dumy->type */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[1] + dumy->r[3];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[2] - dumy->r[3];
			ccp(&ccph);
						
			if (ccph.type == 1) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}
			
			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
		}



		/* < 3 & 4 > */
		if ((sg(2 * dumy->r[3], calcu_digits) >= sg(dumy->r[2] - pph.l + dumy->r[1], calcu_digits))
			&& (sg(2 * dumy->r[3], calcu_digits) <= sg(dumy->r[2] + pph.l + dumy->r[1], calcu_digits)))
		{
			/* 円１の中心から半径 dumy->r[3]-dumy->r[1] の円と */
			/* 円２の中心から半径 dumy->r[2]-dumy->r[3] の円の交点 */
			/* ------------------------------------------------ */
			/* ccp	２円の交点 */
			/* (dumy->cx[1],dumy->cy[1]) dumy->r[1] (dumy->cx[2],dumy->cy[2]) dumy->r[2] */
			/*	 ans (dumy->sx[1],dumy->sy[1]) (dumy->sx[2],dumy->sy[2]) */
			/*	   dumy->type */
			/* ------------------------------------------------ */
			ccph.cx[1] = dumy->cx[1];
			ccph.cy[1] = dumy->cy[1];
			ccph.r[1] = dumy->r[3] - dumy->r[1];
			ccph.cx[2] = dumy->cx[2];
			ccph.cy[2] = dumy->cy[2];
			ccph.r[2] = dumy->r[2] - dumy->r[3];
			ccp(&ccph);
						
			if (ccph.type == 1) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
			}
			
			if (ccph.type == 2) {
				i++;
				dumy->cx[i] = ccph.sx[1];
				dumy->cy[i] = ccph.sy[1];
				dumy->r[i] = dumy->r[3];
				i++;
				dumy->cx[i] = ccph.sx[2];
				dumy->cy[i] = ccph.sy[2];
				dumy->r[i] = dumy->r[3];
			}
		}
	}



	i = i - 2;
	dumy->type = i;
	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : ３点を通る円
 * 関数 : pppc()
 * 引数 : struct RtnDat *a
 *		:	( a->sx[1] , a->sy[1] )
 *		:	( a->sx[2] , a->sy[2] )
 *		:	( a->sx[3] , a->sy[3] )
 * 戻値 : int
 *		:	(a->cx[1],a->cy[1]),a->r[1]
 * 外部 : 無し
 * 内部 : 無し
 * 備考 : 参考文献	[Visual Basic による CG] 光成豊明著  産業図書
 */
int pppc(struct RtnDat *dumy)
{
	double x1, y1, x2, y2, x3, y3, dev, eps = 0.000001;
	double f1, f2, f3, g1, g2, g3, fg1, fg2, fg3;
	double a1, a2, b1, b2, c1, c2;
	double a11, a22, b11, b22, c11, c22;
	double xa, ya, xb, yb, xc, yc, r;


	x1 = dumy->sx[1];
	y1 = dumy->sy[1];

	x2 = dumy->sx[2];
	y2 = dumy->sy[2];

	f1 = x2 - x1;
	g1 = y2 - y1;
	fg1 = f1*f1 + g1*g1;
	if (fg1 < eps) {
		g_print( _("Circle from three points : Error : Identical Point") );
		return 0;
	}

	/* 中点の計算 */
	xa = x1 + f1/2;
	ya = y1 + g1/2;
	/* 線分L1の係数 */
	a1 = -g1;
	b1 = f1;
	c1 = x1*g1 - y1*f1;
	/* 線分L11の係数 */
	a11 = b1;
	b11 = -a1;
	c11 = a1*ya - b1*xa;

	x3 = dumy->sx[3];
	y3 = dumy->sy[3];

	f2 = x3 - x2;
	g2 = y3 - y2;
	fg2 = f2*f2 + g2*g2;
	if (fg2 < eps) {
		g_print( _("Circle from three points : Error : Identical Point") );
		return 0;
	}


	f3 = x3 - x1;
	g3 = y3 - y1;
	fg3 = f3*f3 + g3*g3;
	if (fg3 < eps) {
		g_print( _("Circle from three points : Error : Identical Point") );
		return 0;
	}

	/* 中点の計算 */
	xb = x2 + f2/2;
	yb = y2 + g2/2;
	/* 線分L2の係数 */
	a2 = -g2;
	b2 = f2;
	c2 = x2*g2 - y2*f2;
	/* 線分L22の係数 */
	a22 = b2;
	b22 = -a2;
	c22 = a2*yb - b2*xb;

	dev = a11*b22 - a22*b11;
	/* 中心の計算 */
	xc = (b11*c22 - b22*c11) / dev;
	yc = (a22*c11 - a11*c22) / dev;
	/* 半径の計算 */
	r = sqrt( (xc-x1)*(xc-x1) + (yc-y1)*(yc-y1) );


	dumy->cx[1] = xc;
	dumy->cy[1] = yc;
	dumy->r[1]	= r;
	return 1;
}



/* -------------------------------------------------------------------
 * 説明 : ３辺に接する円
 * 関数 : lllc()
 * 引数 : struct RtnDat *a
 *		:	  (a->sx[1],a->sy[1]) - (a->ex[1],a->ey[1])
 *		:	  (a->sx[2],a->sy[2]) - (a->ex[2],a->ey[2])
 *		:	  (a->sx[3],a->sy[3]) - (a->ex[3],a->ey[3])
 * 戻値 : int Type = 0,2,4	
 *		:	  (a->cx[1],a->cy[1]),a->r[1]
 *		:	  (a->cx[2],a->cy[2]),a->r[2]
 *		:	  (a->cx[3],a->cy[3]),a->r[3]
 *		:	  (a->cx[4],a->cy[4]),a->r[4]
 * 外部 : 無し
 * 内部 : 無し
 * 備考 : 参考文献	[Visual Basic による CG] 光成豊明著  産業図書
 */
int lllc(struct RtnDat *a)
{
/* 1：４個	普通は */
/* 2：２個	２直線が平行 */
/* 0：０個	３直線が平行 */
/* ------------------------------------------------ */
/*	もし３直線の２直線が平行なら、その２直線から半径を求めて、 */
/* ２直線に平行で真ん中の直線Ｍ上に円の中心が来るから、その直線 */
/* Ｍと３本目の直線の交点を求めて、その交点から直線Ｍの±の２方 */
/* 向に半径分離れた点が円の中心。 */
	double dumy;
	struct RtnDat PPPH, PAPH, LOH;
	struct RtnDat LLLH, LLPH, PLPH, PPH;
	struct RtnDat lah;
	double La1, La2 , La3;
	double AX = 0, AY = 0, BX = 0, BY = 0, FX = 0, FY = 0;
	double SXA, SYA, EXA, EYA,	SXB, SYB, EXB, EYB;


	/* --- < 前処理 > ---------------------------------------------------------- */



	/* ----------------------------------------------------- */
	/* --- < 各直線の角度を求める。 > ---------------------- */
	/* --- */
	/* --- */

	/* 線１の角度を求める。 */
	lah.sx[1] = a->sx[1]; lah.sy[1] = a->sy[1]; lah.ex[1] = a->ex[1]; lah.ey[1] = a->ey[1];
	la(&lah);
	La1 = lah.angle;
	if (La1 > 180) La1 = La1 - 180;

	/* 線２の角度を求める。 */
	lah.sx[1] = a->sx[2]; lah.sy[1] = a->sy[2]; lah.ex[1] = a->ex[2]; lah.ey[1] = a->ey[2];
	la(&lah);
	La2 = lah.angle;
	if (La2 > 180) La2 = La2 - 180;

	/* 線３の角度を求める。 */
	lah.sx[1] = a->sx[3]; lah.sy[1] = a->sy[3]; lah.ex[1] = a->ex[3]; lah.ey[1] = a->ey[3];
	la(&lah);
	La3 = lah.angle;
	if (La3 > 180) La3 = La3 - 180;



	/* ----------------------------------------------------- */
	/* --- < 角度で次の処理を判定する。 > ------------------ */
	/* --- */
	/* --- */

	/* ３直線とも平行 */
	if (sg(La1, compa_digits) == sg(La2, compa_digits) && sg(La1, compa_digits) == sg(La3, compa_digits)) {
		a->type = 0;
		return 0;
	}

	/* ２直線が平行 */
	if (sg(La1, compa_digits) == sg(La2, compa_digits) || sg(La1, compa_digits) == sg(La3, compa_digits) || sg(La2, compa_digits) == sg(La3, compa_digits)) {
		a->type = 2;
	}
	else a->type = 4;
	/* --- < 前処理 > ---------------------------------------------------------- */



	/* --- < ２直線が平行。> --------------------------------------------------- */
	if (a->type == 2) {
		/* 直線1と直線2が平行になるようにデータを入れ替える。 */
		if (sg(La1, compa_digits) == sg(La2, compa_digits)) {
			/* OK */
		}
		else if (sg(La1, compa_digits) == sg(La3, compa_digits)) {
			/* 2と3を入れ替える。 */
			dumy = La3; La3 = La2; La2 = dumy;	
			dumy = a->sx[3]; a->sx[3] = a->sx[2]; a->sx[2] = dumy;
			dumy = a->sy[3]; a->sy[3] = a->sy[2]; a->sy[2] = dumy;
			dumy = a->ex[3]; a->ex[3] = a->ex[2]; a->ex[2] = dumy;
			dumy = a->ey[3]; a->ey[3] = a->ey[2]; a->ey[2] = dumy;
		}
		else if (sg(La2, compa_digits) == sg(La3, compa_digits)) {
			/* 1と3を入れ替える。 */
			dumy = La3; La3 = La1; La1 = dumy;
			dumy = a->sx[3]; a->sx[3] = a->sx[1]; a->sx[1] = dumy;
			dumy = a->sy[3]; a->sy[3] = a->sy[1]; a->sy[1] = dumy;
			dumy = a->ex[3]; a->ex[3] = a->ex[1]; a->ex[1] = dumy;
			dumy = a->ey[3]; a->ey[3] = a->ey[1]; a->ey[1] = dumy;
		}


		/* --- < 直線 1 と直線 2 の距離を求める。> ----------------------------- */
		/* 直線 1 の始点と直線 2 の最近点を求める。 */
		PLPH.sx[2] = a->sx[1]; PLPH.sy[2] = a->sy[1];	
		PLPH.sx[1] = a->sx[2]; PLPH.sy[1] = a->sy[2]; PLPH.ex[1] = a->ex[2]; PLPH.ey[1] = a->ey[2];
		plp(&PLPH); /* ( PLPH.ex[2] , PLPH.ey[2] ) */

		/* 直線 1 の始点と直線 2 の最近点の距離を求め、半径 R を求める。 */
		PPH.sx[1] = a->sx[1]; PPH.sy[1] = a->sy[1]; PPH.ex[1] = PLPH.ex[2]; PPH.ey[1] = PLPH.ey[2];
		pp(&PPH);
		a->r[1] = PPH.l / 2;
		a->r[2] = PPH.l / 2;

		/* 直線 1 の始点と直線 2 の最近点の中点を求め、中点から直線 1と平行な直線 4を求める。 */
		PPPH.sx[1] = a->sx[1]; PPPH.sy[1] = a->sy[1]; PPPH.ex[1] = PLPH.ex[2]; PPPH.ey[1] = PLPH.ey[2];
		PPPH.sx[2] = 1; PPPH.ex[2] = 1;
		ppp(&PPPH);
		/* 中点 ＆ 直線 4 の始点( PPPH.sx[3] , PPPH.sy[3]) */

		PAPH.sx[1] = PPPH.sx[3]; PAPH.sy[1] = PPPH.sy[3]; PAPH.angle = La1; PAPH.l = 50;
		pap(&PAPH);
		/* 直線 4 の終点( PAPH.ex[1] , PAPH.ey[1]) */

		/* 直線 4 と直線 3 の交点 5 を求める。 */
		LLPH.sx[1] = PPPH.sx[3]; LLPH.sy[1] = PPPH.sy[3]; LLPH.ex[1] = PAPH.ex[1]; LLPH.ey[1] = PAPH.ey[1];
		LLPH.sx[2] = a->sx[3]; LLPH.sy[2] = a->sy[3]; LLPH.ex[2] = a->ex[3]; LLPH.ey[2] = a->ey[3];
		llp(&LLPH);
		/* 交点 5( LLPH.sx[3] , LLPH.sy[3] ) */



		/* 直線 3 をオフセットした直線 5,6 を求める。 */
		/* ------------------------------------------------------------------- */
		/* 説明 : 始点と終点を与えた直線のオフセット */
		/* ------------------------------------------------------------------- */
		LOH.sx[1] = a->sx[3];
		LOH.sy[1] = a->sy[3];
		LOH.ex[1] = a->ex[3];
		LOH.ey[1] = a->ey[3];
		LOH.l = a->r[1];
		lo(&LOH);
		/* (LOH.sx[2] , LOH.sy[2]) - (LOH.ex[2] , LOH.ey[2]) */
		/* (LOH.sx[3] , LOH.sy[3]) - (LOH.ex[3] , LOH.ey[3]) */

		/* 直線 5,6 と直線 4 の交点 */
		/* ------------------------------------------------------------------- */
		/* 説明 : ２直線の交点	初めてのグラフィックス P120 */
		/* ------------------------------------------------------------------- */
		LLPH.sx[1] = LOH.sx[2];
		LLPH.sy[1] = LOH.sy[2];
		LLPH.ex[1] = LOH.ex[2];
		LLPH.ey[1] = LOH.ey[2];

		LLPH.sx[2] = PPPH.sx[3];
		LLPH.sy[2] = PPPH.sy[3];
		LLPH.ex[2] = PAPH.ex[1];
		LLPH.ey[2] = PAPH.ey[1];
		llp(&LLPH);
		a->cx[1] = LLPH.sx[3];
		a->cy[1] = LLPH.sy[3];


		LLPH.sx[1] = LOH.sx[3];
		LLPH.sy[1] = LOH.sy[3];
		LLPH.ex[1] = LOH.ex[3];
		LLPH.ey[1] = LOH.ey[3];

		LLPH.sx[2] = PPPH.sx[3];
		LLPH.sy[2] = PPPH.sy[3];
		LLPH.ex[2] = PAPH.ex[1];
		LLPH.ey[2] = PAPH.ey[1];
		llp(&LLPH);
		a->cx[2] = LLPH.sx[3];
		a->cy[2] = LLPH.sy[3];
		return 1;
	}
	/* --- < ２直線が平行。> --------------------------------------------------- */





	/* --- < ３辺が平行でない。> ----------------------------------------------- */
	if (a->type == 4) {
		/* ----------------------------------------------------- */
		/* --- < ３辺の交点を求める。 > ------------------------ */
		/* --- */
		/* --- */

		/* ***	直線１と直線２の交点Ａを求める。 */
		LLPH.sx[1] = a->sx[1]; LLPH.sy[1] = a->sy[1]; LLPH.ex[1] = a->ex[1]; LLPH.ey[1] = a->ey[1];
		LLPH.sx[2] = a->sx[2]; LLPH.sy[2] = a->sy[2]; LLPH.ex[2] = a->ex[2]; LLPH.ey[2] = a->ey[2];
		llp(&LLPH);
		/* =0 なら平行 */
		if (LLPH.type != 0) {
			AX = LLPH.sx[3];
			AY = LLPH.sy[3];
		}

		/* ***	直線２と直線３の交点Ｂを求める。 */
		LLPH.sx[1] = a->sx[2]; LLPH.sy[1] = a->sy[2]; LLPH.ex[1] = a->ex[2]; LLPH.ey[1] = a->ey[2];
		LLPH.sx[2] = a->sx[3]; LLPH.sy[2] = a->sy[3]; LLPH.ex[2] = a->ex[3]; LLPH.ey[2] = a->ey[3];
		llp(&LLPH);
		if (LLPH.type != 0) {
			BX = LLPH.sx[3];
			BY = LLPH.sy[3];
		}

		/* ***	直線３と直線１の交点Ｆを求める。 */
		LLPH.sx[1] = a->sx[3]; LLPH.sy[1] = a->sy[3]; LLPH.ex[1] = a->ex[3]; LLPH.ey[1] = a->ey[3];
		LLPH.sx[2] = a->sx[1]; LLPH.sy[2] = a->sy[1]; LLPH.ex[2] = a->ex[1]; LLPH.ey[2] = a->ey[1];
		llp(&LLPH);
		if (LLPH.type != 0) {
			FX = LLPH.sx[3];
			FY = LLPH.sy[3];
		}


		if (sg(AX, compa_digits) == sg(BX, compa_digits) && sg(BX, compa_digits) == sg(FX, compa_digits) && sg(AY, compa_digits) == sg(BY, compa_digits) && sg(BY, compa_digits) == sg(FY, compa_digits)) {
			a->type = 0;
			return -1;
		}



		/* ----------------------------------------------------- */
		/* --- < 1個目 > --------------------------------------- */
		/* --- */
		/* --- */

		/* Ａ点を始点とする２直線の２等分線Ａ(SXA,SYA)-(EXA,EYA)を求める  --- */
		LLLH.sx[1] = AX; LLLH.sy[1] = AY; LLLH.ex[1] = BX; LLLH.ey[1] = BY; 
		LLLH.sx[2] = AX; LLLH.sy[2] = AY; LLLH.ex[2] = FX; LLLH.ey[2] = FY;
		lll(&LLLH);
		SXA = LLLH.sx[3]; SYA = LLLH.sy[3]; EXA = LLLH.ex[3]; EYA = LLLH.ey[3];

		/* Ｂ点を始点とする２直線の２等分線Ｂ(SXB,SYB)-(EXB,EYB)を求める  --- */
		LLLH.sx[1] = BX; LLLH.sy[1] = BY; LLLH.ex[1] = AX; LLLH.ey[1] = AY;
		LLLH.sx[2] = BX; LLLH.sy[2] = BY; LLLH.ex[2] = FX; LLLH.ey[2] = FY;
		lll(&LLLH);
		SXB = LLLH.sx[3]; SYB = LLLH.sy[3]; EXB = LLLH.ex[3]; EYB = LLLH.ey[3];


		/* ２等分線Ａと２等分線Ｂの交点を求める。求める円の中心(CX,CY)	--- */
		LLPH.sx[1] = SXA; LLPH.sy[1] = SYA; LLPH.ex[1] = EXA; LLPH.ey[1] = EYA;
		LLPH.sx[2] = SXB; LLPH.sy[2] = SYB; LLPH.ex[2] = EXB; LLPH.ey[2] = EYB;
		llp(&LLPH);
		a->cx[1] = LLPH.sx[3];
		a->cy[1] = LLPH.sy[3];

		/* 中心から直線１までの距離を求める。求める円の半径 R	--- */
		PLPH.sx[2] = a->cx[1]; PLPH.sy[2] = a->cy[1];	
		PLPH.sx[1] = a->sx[1]; PLPH.sy[1] = a->sy[1]; PLPH.ex[1] = a->ex[1]; PLPH.ey[1] = a->ey[1];
		plp(&PLPH);
		/*	 --- */
		PPH.sx[1] = a->cx[1]; PPH.sy[1] = a->cy[1]; PPH.ex[1] = PLPH.ex[2]; PPH.ey[1] = PLPH.ey[2];
		pp(&PPH);
		a->r[1] = PPH.l;



		/* ----------------------------------------------------- */
		/* --- < 2個目 > --------------------------------------- */
		/* --- */
		/* --- */

		/* Ａ点を始点と終点とする２直線の２等分線Ａ(SXA,SYA)-(EXA,EYA)を求める	--- */
		LLLH.sx[1] = FX; LLLH.sy[1] = FY; LLLH.ex[1] = AX; LLLH.ey[1] = AY; 
		LLLH.sx[2] = AX; LLLH.sy[2] = AY; LLLH.ex[2] = BX; LLLH.ey[2] = BY;
		lll(&LLLH);
		SXA = LLLH.sx[3]; SYA = LLLH.sy[3]; EXA = LLLH.ex[3]; EYA = LLLH.ey[3];

		/* Ｂ点を始点と終点とする２直線の２等分線Ｂ(SXB,SYB)-(EXB,EYB)を求める	--- */
		LLLH.sx[1] = FX; LLLH.sy[1] = FY; LLLH.ex[1] = BX; LLLH.ey[1] = BY;
		LLLH.sx[2] = BX; LLLH.sy[2] = BY; LLLH.ex[2] = AX; LLLH.ey[2] = AY;
		lll(&LLLH);
		SXB = LLLH.sx[3]; SYB = LLLH.sy[3]; EXB = LLLH.ex[3]; EYB = LLLH.ey[3];


		/* ２等分線Ａと２等分線Ｂの交点を求める。求める円の中心(CX,CY)	--- */
		LLPH.sx[1] = SXA; LLPH.sy[1] = SYA; LLPH.ex[1] = EXA; LLPH.ey[1] = EYA;
		LLPH.sx[2] = SXB; LLPH.sy[2] = SYB; LLPH.ex[2] = EXB; LLPH.ey[2] = EYB;
		llp(&LLPH);
		a->cx[2] = LLPH.sx[3];
		a->cy[2] = LLPH.sy[3];

		/* 中心から直線１までの距離を求める。求める円の半径 R	--- */
		PLPH.sx[2] = a->cx[2]; PLPH.sy[2] = a->cy[2];	
		PLPH.sx[1] = a->sx[1]; PLPH.sy[1] = a->sy[1]; PLPH.ex[1] = a->ex[1]; PLPH.ey[1] = a->ey[1];
		plp(&PLPH);
		/*	 --- */
		PPH.sx[1] = a->cx[2]; PPH.sy[1] = a->cy[2]; PPH.ex[1] = PLPH.ex[2]; PPH.ey[1] = PLPH.ey[2];
		pp(&PPH);
		a->r[2] = PPH.l;



		/* ----------------------------------------------------- */
		/* --- < 3個目 > --------------------------------------- */
		/* --- */
		/* --- */

		/* B 点を始点と終点とする２直線の２等分線Ａ(SXA,SYA)-(EXA,EYA)を求める	--- */
		LLLH.sx[1] = AX; LLLH.sy[1] = AY; LLLH.ex[1] = BX; LLLH.ey[1] = BY; 
		LLLH.sx[2] = BX; LLLH.sy[2] = BY; LLLH.ex[2] = FX; LLLH.ey[2] = FY;
		lll(&LLLH);
		SXA = LLLH.sx[3]; SYA = LLLH.sy[3]; EXA = LLLH.ex[3]; EYA = LLLH.ey[3];

		/* F 点を始点と終点とする２直線の２等分線Ｂ(SXB,SYB)-(EXB,EYB)を求める	--- */
		LLLH.sx[1] = AX; LLLH.sy[1] = AY; LLLH.ex[1] = FX; LLLH.ey[1] = FY;
		LLLH.sx[2] = FX; LLLH.sy[2] = FY; LLLH.ex[2] = BX; LLLH.ey[2] = BY;
		lll(&LLLH);
		SXB = LLLH.sx[3]; SYB = LLLH.sy[3]; EXB = LLLH.ex[3]; EYB = LLLH.ey[3];


		/* ２等分線Ａと２等分線Ｂの交点を求める。求める円の中心(CX,CY)	--- */
		LLPH.sx[1] = SXA; LLPH.sy[1] = SYA; LLPH.ex[1] = EXA; LLPH.ey[1] = EYA;
		LLPH.sx[2] = SXB; LLPH.sy[2] = SYB; LLPH.ex[2] = EXB; LLPH.ey[2] = EYB;
		llp(&LLPH);
		a->cx[3] = LLPH.sx[3];
		a->cy[3] = LLPH.sy[3];

		/* 中心から直線１までの距離を求める。求める円の半径 R	--- */
		PLPH.sx[2] = a->cx[3]; PLPH.sy[2] = a->cy[3];	
		PLPH.sx[1] = a->sx[1]; PLPH.sy[1] = a->sy[1]; PLPH.ex[1] = a->ex[1]; PLPH.ey[1] = a->ey[1];
		plp(&PLPH);
		/*	 --- */
		PPH.sx[1] = a->cx[3]; PPH.sy[1] = a->cy[3]; PPH.ex[1] = PLPH.ex[2]; PPH.ey[1] = PLPH.ey[2];
		pp(&PPH);
		a->r[3] = PPH.l;



		/* ----------------------------------------------------- */
		/* --- < 4個目 > --------------------------------------- */
		/* --- */
		/* --- */

		/* F 点を始点と終点とする２直線の２等分線Ａ(SXA,SYA)-(EXA,EYA)を求める	--- */
		LLLH.sx[1] = BX; LLLH.sy[1] = BY; LLLH.ex[1] = FX; LLLH.ey[1] = FY; 
		LLLH.sx[2] = FX; LLLH.sy[2] = FY; LLLH.ex[2] = AX; LLLH.ey[2] = AY;
		lll(&LLLH);
		SXA = LLLH.sx[3]; SYA = LLLH.sy[3]; EXA = LLLH.ex[3]; EYA = LLLH.ey[3];

		/* A 点を始点と終点とする２直線の２等分線Ｂ(SXB,SYB)-(EXB,EYB)を求める	--- */
		LLLH.sx[1] = BX; LLLH.sy[1] = BY; LLLH.ex[1] = AX; LLLH.ey[1] = AY;
		LLLH.sx[2] = AX; LLLH.sy[2] = AY; LLLH.ex[2] = FX; LLLH.ey[2] = FY;
		lll(&LLLH);
		SXB = LLLH.sx[3]; SYB = LLLH.sy[3]; EXB = LLLH.ex[3]; EYB = LLLH.ey[3];

		/* ２等分線Ａと２等分線Ｂの交点を求める。求める円の中心(CX,CY)	--- */
		LLPH.sx[1] = SXA; LLPH.sy[1] = SYA; LLPH.ex[1] = EXA; LLPH.ey[1] = EYA;
		LLPH.sx[2] = SXB; LLPH.sy[2] = SYB; LLPH.ex[2] = EXB; LLPH.ey[2] = EYB;
		llp(&LLPH);
		a->cx[4] = LLPH.sx[3];
		a->cy[4] = LLPH.sy[3];

		/* 中心から直線１までの距離を求める。求める円の半径 R	--- */
		PLPH.sx[2] = a->cx[4]; PLPH.sy[2] = a->cy[4];	
		PLPH.sx[1] = a->sx[1]; PLPH.sy[1] = a->sy[1]; PLPH.ex[1] = a->ex[1]; PLPH.ey[1] = a->ey[1];
		plp(&PLPH);
		/*	 --- */
		PPH.sx[1] = a->cx[4]; PPH.sy[1] = a->cy[4]; PPH.ex[1] = PLPH.ex[2]; PPH.ey[1] = PLPH.ey[2];
		pp(&PPH);
		a->r[4] = PPH.l;
		return 1;
	}
	return 0;
}



/* -------------------------------------------------------------------
 * 説明 : 点が円の中にあるか外にあるか。
 * 関数 : ChkPointCircleSide
 * 引数 : struct RtnDat *
 *		:	(a->sx[1] , a->sy[1]) 
 *		:	(a->cx[1] , a->cy[1]) , a->r[1]
 * 戻値 : int	 a->type = 0：ERR	1：中	2：外	3：線上
 * 外部 : 無し
 * 内部 :
 */
int ChkPointCircleSide(struct RtnDat *a)
{
	struct RtnDat PPH;


	PPH.sx[1] = a->sx[1];
	PPH.sy[1] = a->sy[1];
	PPH.ex[1] = a->cx[1];
	PPH.ey[1] = a->cy[1];
	pp(&PPH);

	if (sg(PPH.l, calcu_digits) == sg(a->r[1], calcu_digits))	  a->type = 0;
	else if (sg(PPH.l, calcu_digits) < sg(a->r[1], calcu_digits)) a->type = 1;
	else if (sg(PPH.l, calcu_digits) > sg(a->r[1], calcu_digits)) a->type = 2;

	return 1;
}



/* -------------------------------------------------------------------
 * 説明 : 点が直線の進行方向の右(1)にあるか、左(2)にあるか？
 * 関数 : ChkPointLineSide
 * 引数 : struct RtnDat *
 *		:	( a->sx[1] , a->sy[1] )-( a->ex[1] , a->ey[1] )
 *		:	( a->sx[2] , a->sy[2] )
 * 戻値 : int
 *		:	a->type = 0：ERR	1：右	 2：左	  3：線上
 * 外部 : 無し
 * 内部 : 
 */
int ChkPointLineSide(struct RtnDat *a)
{
	struct RtnDat PLPH, LAH;
	double Ang1, Ang2, Dumy;


	a->type = 0;

	/* --- < 最近点を求める。 > --- */
	PLPH = *a;
	plp(&PLPH);
	/* 最近点	(PLPH.ex[2],PLPH.ey[2]) */

	if (a->sx[2] == PLPH.ex[2] && a->sy[2] == PLPH.ey[2]) {
		a->type = 3;
		return 1;
	}

	/* --- < 直線の角度 > --- */
	LAH.sx[1] = a->sx[1];
	LAH.sy[1] = a->sy[1];
	LAH.ex[1] = a->ex[1];
	LAH.ey[1] = a->ey[1];
	la(&LAH);
	Ang1 = LAH.angle;
		
	/* --- < 最近点から点までの角度 > --- */
	LAH.sx[1] = PLPH.ex[2];
	LAH.sy[1] = PLPH.ey[2];
	LAH.ex[1] = a->sx[2];
	LAH.ey[1] = a->sy[2];
	la(&LAH);
	Ang2 = LAH.angle;
	
	/* 進行方向左の判定 */
	Dumy = Ang1 + 90;
	if (Dumy >= 360)	Dumy = Dumy - 360;
	if (sg(Dumy, calcu_digits) <= sg(Ang2, calcu_digits)+1	&& sg(Dumy, calcu_digits) >= sg(Ang2, calcu_digits)-1)  a->type = 2;
	else a->type = 1;

	return 1;
}



/* -------------------------------------------------------------------
 * 説明 : 直線と円の交点(クリック点に近い１点)
 * 関数 : LCP1
 * 引数 : struct RtnDat *
 *		:	(a->sx[1] , a->sy[1])-(a->ex[1] , a->ey[1])
 *		:	(a->cx[1] , a->CY[1]) , a->R[1]
 *		:	(a->sx[4] , a->SY[4]) , (a->ex[4] , a->EY[4])  クリック点
 * 戻値 : int
 *		:	(a->sx[2] , a->SY[2])
 * 外部 : 無し
 * 内部 : 
 */
int LCP1(struct RtnDat *a)
{
	struct RtnDat PPH;
	double P1X, P1Y, P2X, P2Y, L1, L2;
	double Ck1X, Ck1Y, Ck2X, Ck2Y;


	/* クリック点 */
	Ck1X = a->sx[4];
	Ck1Y = a->sy[4];
	Ck2X = a->ex[4];
	Ck2Y = a->ey[4];

	/* 交点を求める。 */
	lcp(a);

	if (a->type == 0)	return 0;
	else if (a->type == 1)	return 1;


	else if (a->type == 2) {
		/* 交点２個 P1(a->sx[2],a->sy[2]) P2(a->sx[3],a->sy[3]) */
		P1X = a->sx[2];
		P1Y = a->sy[2];
		P2X = a->sx[3];
		P2Y = a->sy[3];
		
		/* クリックした点(a->sx[4],a->sy[4])と交点１(P1X,P1Y)の距離を求める。 */
		PPH.sx[1] = Ck1X;
		PPH.sy[1] = Ck1Y;
		PPH.ex[1] = P1X;
		PPH.ey[1] = P1Y;
		pp(&PPH);
		L1 = PPH.l;
		/* クリックした点(a->sx[4],a->sy[4])と交点２(P2X,P2Y)の距離を求める。 */
		PPH.sx[1] = Ck1X;
		PPH.sy[1] = Ck1Y;
		PPH.ex[1] = P2X;
		PPH.ey[1] = P2Y;
		pp(&PPH);
		L2 = PPH.l;
		/* どっちの交点がクリック点に近いか？ */
		if (L1 < L2) {
			a->sx[2] = P1X;
			a->sy[2] = P1Y;
			a->sx[3] = 0;
			a->sy[3] = 0;
		}
		else if (L1 > L2) {
			a->sx[2] = P2X;
			a->sy[2] = P2Y;
			a->sx[3] = 0;
			a->sy[3] = 0;
		}
	}

	a->type = 1;
	return 1;
}



/* -------------------------------------------------------------------
 * 説明 : ２円の交点(クリック点に近い１点)
 * 関数 : CCP1
 * 引数 : struct RtnDat *
 *		:	(a->cx[1] , a->cy[1]) a->r[1] (a->cx[2] , a->cy[2]) a->r[2]
 *		:	(a->sx[1] , a->sy[1]) (a->sx[2] , a->sy[2])    クリック点
 * 戻値 : int 0,1
 *		:	(a->sx[1] , a->sy[1])
 * 外部 : 無し
 * 内部 : 
 */
int CCP1(struct RtnDat *a)
{
	struct RtnDat PPH;
	double P1X, P1Y, P2X, P2Y, L1, L2, Ck1X, Ck1Y, Ck2X, Ck2Y;


	/* クリック点 */
	Ck1X = a->sx[1];
	Ck1Y = a->sy[1];
	Ck2X = a->sx[2];
	Ck2Y = a->sy[2];
	/* 交点を求める。 */
	ccp(a);    /* 交点(a->sx[1],a->sy[1])(a->sx[2],a->sy[2]) */

	/* 交点が 0 のとき。------------------------------ */
	if (a->type == 0) {
		a->sx[2] = 0;
		a->sy[2] = 0;
		return 0;
	}
	/* 交点が 1個のとき。---------------------------- */
	else if (a->type == 1) return 1;
	/* 交点が 2個のとき。---------------------------- */
	else if (a->type == 2) {
		/* 交点２個 P1(a->sx[2],a->sy[2]) P2(a->sx[3],a->sy[3]) */
		P1X = a->sx[1];
		P1Y = a->sy[1];
		P2X = a->sx[2];
		P2Y = a->sy[2];
		
		/* クリックした点(a->sx[4],a->sy[4])と交点１(P1X,P1Y)の距離を求める。 */
		PPH.sx[1] = Ck1X;
		PPH.sy[1] = Ck1Y;
		PPH.ex[1] = P1X;
		PPH.ey[1] = P1Y;
		pp(&PPH);
		L1 = PPH.l;
		/* クリックした点(a->sx[4],a->sy[4])と交点２(P2X,P2Y)の距離を求める。 */
		PPH.sx[1] = Ck2X;
		PPH.sy[1] = Ck2Y;
		PPH.ex[1] = P2X;
		PPH.ey[1] = P2Y;
		pp(&PPH);
		L2 = PPH.l;
		/* どっちの交点がクリック点に近いか？ */
		if (L1 < L2) {
			a->sx[1] = P1X;
			a->sy[1] = P1Y;
		}
		else if (L1 > L2) {
			a->sx[1] = P2X;
			a->sy[1] = P2Y;
		}
	}
	a->sx[2] = 0;
	a->sy[2] = 0;
	a->type = 1;
	return 1;
}





/* --------------------------------------------------------------------------
 * NAME: 
 *     near_point_on_line
 * 
 * DESCRIPTION:
 *     (x,y)点と線の最近点を求める
 * 
 * PARAMS: 
 *     double *x                       
 *     double *y                       
 *     double sx                       
 *     double sy                       
 *     double ex                       
 *     double ey                       
 * 
 * RETURN VALUE:
 *     int                             0 : 正常終了
 * 
 * USAGE: 
 * 
 * --------------------------------------------------------------------------
 */
int near_point_on_line(double *x, double *y, double sx, double sy, double ex, double ey)
{
	struct RtnDat rtn_plp;
	
	/* 点と線の最近点を求める */
	rtn_plp.sx[2] = *x;
	rtn_plp.sy[2] = *y;
	rtn_plp.sx[1] = sx;
	rtn_plp.sy[1] = sy;
	rtn_plp.ex[1] = ex;
	rtn_plp.ey[1] = ey;
	plp(&rtn_plp);
	/* 最近点	 rtn_plp.ex[2] , rtn_plp.ey[2] */
	*x = rtn_plp.ex[2];
	*y = rtn_plp.ey[2];
	
	return 0;
}


/* --------------------------------------------------------------------------
 * NAME: 
 *     near_point_on_circle
 * 
 * DESCRIPTION:
 *     (x,y)点と円の最近点を求める
 * 
 * PARAMS: 
 *     double *x                       
 *     double *y                       
 *     double cx                       
 *     double cy                       
 *     double r                        
 *     double sx                       
 *     double sy                       
 *     double ex                       
 *     double ey                       
 * 
 * RETURN VALUE:
 *     int                             0  : 正常終了
 *                                     -1 : 異常終了
 * 
 * USAGE: 
 * 
 * --------------------------------------------------------------------------
 */
int near_point_on_circle(double *x, double *y, double cx, double cy, double r)
{
	struct RtnDat rtn_lcp, rtn_la;
	double angle, p1_angle, p2_angle;


	/* -------------------------------------------------------------------
	 * 説明 : [11] 直線と円の交点  初めてのグラフィックス P126
	 */
	rtn_lcp.sx[1] = cx;
	rtn_lcp.sy[1] = cy;
	rtn_lcp.ex[1] = *x;
	rtn_lcp.ey[1] = *y;
	rtn_lcp.cx[1] = cx;
	rtn_lcp.cy[1] = cy;
	rtn_lcp.r[1]  = r;
	lcp(&rtn_lcp);
	if (rtn_lcp.type < 2) {
		//g_print("culcfunc.c : near_point_on_circle() : type = %d error !\n", rtn_lcp.type);
		return -1;
	}
	
	
	/* -------------------------------------------------------------------
	 * 説明 : [8] 直線の角度
	 */
	rtn_la.sx[1] = cx;
	rtn_la.sy[1] = cy;
	rtn_la.ex[1] = *x;
	rtn_la.ey[1] = *y;
	la(&rtn_la);
	angle = rtn_la.angle;
	
	rtn_la.sx[1] = cx;
	rtn_la.sy[1] = cy;
	rtn_la.ex[1] = rtn_lcp.sx[2];
	rtn_la.ey[1] = rtn_lcp.sy[2];
	la(&rtn_la);
	p1_angle = rtn_la.angle;
	
	rtn_la.sx[1] = cx;
	rtn_la.sy[1] = cy;
	rtn_la.ex[1] = rtn_lcp.sx[3];
	rtn_la.ey[1] = rtn_lcp.sy[3];
	la(&rtn_la);
	p2_angle = rtn_la.angle;
	
//	g_print("culcfunc.c : near_point_on_circle() : mouse(%f,%f)  p1(%f,%f)  p2(%f,%f)\n", 
//			*x, *y, rtn_lcp.sx[2], rtn_lcp.sy[2], rtn_lcp.sx[3], rtn_lcp.sy[3]);
	
	if ( sg(angle, compa_digits) == sg(p1_angle, compa_digits) ) {
		*x = rtn_lcp.sx[2];
		*y = rtn_lcp.sy[2];
		return 0;
	}
	else if ( sg(angle, compa_digits) == sg(p2_angle, compa_digits) ) {
		*x = rtn_lcp.sx[3];
		*y = rtn_lcp.sy[3];
		return 0;
	}
	
	return -1;
}


/* --------------------------------------------------------------------------
 * NAME: 
 *     point_on_line
 * 
 * DESCRIPTION:
 *     点が線上にあるか
 * 
 * PARAMS: 
 *     double x                        
 *     double y                        
 *     double sx                       
 *     double sy                       
 *     double ex                       
 *     double ey                       
 * 
 * RETURN VALUE:
 *     線上である                       1
 *     線上ではない                     0
 * 
 * USAGE: 
 * 
 * --------------------------------------------------------------------------
 *  1 : "線の始点から終点への角度"と"線の始点から点までの角度"は同じ。
 *  2 : "線の始点から交点までの距離"は、"線の長さ"よりも小さい。
 * (3 : 線と点の最近点が "0")
 * --------------------------------------------------------------------------
 */
int point_on_line(double x, double y, double sx, double sy, double ex, double ey)
{
	struct RtnDat rtn_la;
	double l_angle, p_angle;


	/* -----------------------------------------------------
	 *  1 : "線の始点から終点への角度"と"線の始点から点までの角度"は同じ。
	 */
	/* 線の始点から終点への角度 l_angle を求める。 */
	rtn_la.sx[1] = sx;
	rtn_la.sy[1] = sy;
	rtn_la.ex[1] = ex;
	rtn_la.ey[1] = ey;
	la(&rtn_la);
	l_angle = rtn_la.angle;
	
	/* 線の始点から点までの角度 p_angle を求める。 */
	rtn_la.sx[1] = sx;
	rtn_la.sy[1] = sy;
	rtn_la.ex[1] = x;
	rtn_la.ey[1] = y;
	la(&rtn_la);
	p_angle = rtn_la.angle;

	if ( sg(l_angle, compa_digits) != sg(p_angle, compa_digits) ) return 0;



	struct RtnDat rtn_pp;
	double l_length, p_length;

	/* -----------------------------------------------------
	 *  2 : "線の始点から交点までの距離"は、"線の長さ"よりも小さい。
	 */
	/* 線の長さを求める。 */
	rtn_pp.sx[1] = sx;
	rtn_pp.sy[1] = sy;
	rtn_pp.ex[1] = ex;
	rtn_pp.ey[1] = ey;
	pp(&rtn_pp);
	l_length = rtn_pp.l;
	
	/* 線の始点から点までの距離を求める。 */
	rtn_pp.sx[1] = sx;
	rtn_pp.sy[1] = sy;
	rtn_pp.ex[1] = x;
	rtn_pp.ey[1] = y;
	pp(&rtn_pp);
	p_length = rtn_pp.l;
	
	if (l_length < p_length) return 0;
	
	return 1;
}





/* --------------------------------------------------------------------------
 * NAME: 
 *     point_on_arc
 * 
 * DESCRIPTION:
 *     点が円弧上にあるか
 * 
 * PARAMS: 
 *     double x                        
 *     double y                        
 *     double cx                       
 *     double cy                       
 *     double r                        
 *     double sx                       
 *     double sy                       
 *     double ex                       
 *     double ey                       
 * 
 * RETURN VALUE:
 *     線上である                       1
 *     線上ではない                     0
 * 
 * USAGE: 
 * 
 * --------------------------------------------------------------------------
 */
int point_on_arc(double x, double y, double cx, double cy, double r, double sx, double sy, double ex, double ey)
{
	struct RtnDat rtn_la, rtn_pp;
	double sa, ea, pa;


	/* -----------------------------------------------------
	 * 距離による検査
	 */
	
	/* 円弧の中心から点までの距離を求め、円弧の半径と同じになるかを確認 */
	rtn_pp.sx[1] = cx;
	rtn_pp.sy[1] = cy;
	rtn_pp.ex[1] = x;
	rtn_pp.ey[1] = y;
	pp(&rtn_pp);
	if ( sg(r, compa_digits) != sg(rtn_pp.l, compa_digits) ) {
		//g_print ("culcfunc.c : point_on_arc() : 円弧の中心から点までの距離を求め、円弧の半径と同じにならない。\n");
		return 0;
	}


	/* -----------------------------------------------------
	 * 角度による検査
	 */
	
	/* 円弧の中心から点までの角度を求める。 */
	rtn_la.sx[1] = cx;
	rtn_la.sy[1] = cy;
	rtn_la.ex[1] = x;
	rtn_la.ey[1] = y;
	la(&rtn_la);
	pa = rtn_la.angle;

	/* 円弧の開始点(sa) の角度を求める。 */
	rtn_la.sx[1] = cx;
	rtn_la.sy[1] = cy;
	rtn_la.ex[1] = sx;
	rtn_la.ey[1] = sy;
	la(&rtn_la);
	sa = rtn_la.angle;

	/* 円弧の終了点(ea) の角度を求める。 */
	rtn_la.sx[1] = cx;
	rtn_la.sy[1] = cy;
	rtn_la.ex[1] = ex;
	rtn_la.ey[1] = ey;
	la(&rtn_la);
	ea = rtn_la.angle;

	/* sa と ea が同じなら円 */
	if (sg(sa, compa_digits) == sg(ea, compa_digits)) {
		sa = 0;
		ea = 360;
	}

//	g_print("culcfunc.c : point_on_arc() : sa = %f  ea = %f  pa = %f\n", sa, ea, pa);

	/* sa < ea で０度をまたがない。 */
	if (sa < ea) {
		/* 点は、sa 以上 ea 以下 */
		if (sg(pa, compa_digits) >= sg(sa, compa_digits) 
			&& 
			sg(pa, compa_digits) <= sg(ea, compa_digits) 
		) {
			//g_print ("culcfunc.c : point_on_arc() : (sa < ea) -> (pa >= sa && pa <= ea) OK !\n");
			return 1;
		}
		else {
			//g_print ("culcfunc.c : point_on_arc() : (sa < ea) -> (pa >= sa && pa <= ea) NO !\n");
			return 0;
		}
	}
	/* sa > ea で０度をまたぐ。 */
	else if (sa > ea) {
		/* 点は、sa 以上 または ea 以下 */
		if (sg(pa, compa_digits) >= sg(sa, compa_digits) 
			|| 
			sg(pa, compa_digits) <= sg(ea, compa_digits) 
		) {
			//g_print ("culcfunc.c : point_on_arc() : (sa > ea) -> (pa >= sa || pa <= ea) OK !\n");
			return 1;
		}
		else {
			//g_print ("culcfunc.c : point_on_arc() : (sa > ea) -> (pa >= sa || pa <= ea) NO !\n");
			return 0;
		}
	}

	return 0;
}





/* -------------------------------------------------------------------
 * 説明 : ２直線の実際の交点
 * 関数 : Act_llp
 * 引数 : RtnDat
 *		:	(sx[1] , sy[1]) (ex[1] , ey[1])
 *		:	(sx[2] , sy[2]) (ex[2] , ey[2])
 * 戻値 : int	 0:実際の交点なし	 1:実際の交点ある)
 *		:	(sx[3] , sy[3]) , (type =
 * 外部 : 無し
 * 内部 : 
 */
int Act_llp(struct RtnDat *rtn)
{
	struct RtnDat rtn_llp;
	double px, py;

	rtn->type = 0;
	rtn_llp = *rtn;

	llp(&rtn_llp);
	if (rtn_llp.type == 0) return 0;
	/* rtn_llp.sx[3] , rtn_llp.sy[3] */
	px = rtn_llp.sx[3];
	py = rtn_llp.sy[3];


	if ( point_on_line(px, py, rtn->sx[1], rtn->sy[1], rtn->ex[1], rtn->ey[1]) == 0 ) return 0;
	if ( point_on_line(px, py, rtn->sx[2], rtn->sy[2], rtn->ex[2], rtn->ey[2]) == 0 ) return 0;

	rtn->type = 1;
	rtn->sx[3] = px;
	rtn->sy[3] = py;
	return 1;
}





/* -------------------------------------------------------------------
 * 説明 : 直線と円弧の実際の交点
 * 関数 : Act_lcp
 * 引数 : RtnDat
 *		:	線 1  (sx[1],sy[1]) - (ex[1],ey[1])
 *		:	円 1  (cx[1],cy[1]) ,r[1], (sx[4],sy[4]) - (ex[4],ey[4])
 * 戻値 : RtnDat
 *		:	type   0 , 1 , 2
 *		:	点 1  (sx[2],sy[2])
 *		:	点 2  (sx[3],sy[3])
 * 外部 : 無し
 * 内部 : RtnDat plph,pph
 */
int Act_lcp(struct RtnDat *rtn)
{
	struct RtnDat rtn_lcp;
	double px1, py1, px2, py2;
	int i = 0;

	rtn->type = 0;
	rtn_lcp = *rtn;

	/* -------------------------------------------------------------------
	 * 説明 : [11] 直線と円の交点  初めてのグラフィックス P126
	 * 関数 : lcp
	 * 引数 : RtnDat
	 *		:	線 1	(sx[1],sy[1]) - (ex[1],ey[1])
	 *		:	円 1	(cx[1],cy[1]) ,r[1]
	 * 戻値 : RtnDat
	 *		:	type	0 , 1 , 2
	 *		:	点 1	(sx[2],sy[2])
	 *		:	点 2	(sx[3],sy[3])
	 * 外部 : 無し
	 * 内部 : RtnDat plph,pph
	 */
	lcp(&rtn_lcp);
	/* 直線と円の交点がなければ */
	if (rtn_lcp.type == 0) {
		//g_print ("交点 NO by lcp()\n");
		return 0;
	}
	//g_print ("culcfunc.c : Alt_lcp() : return lcp type = %d\n", rtn_lcp.type);


	i = 2;
	/* 交点 1 */
	if (rtn_lcp.type > 0) {
		px1 = rtn_lcp.sx[2];
		py1 = rtn_lcp.sy[2];

		if ( point_on_line(px1, py1, rtn->sx[1], rtn->sy[1], rtn->ex[1], rtn->ey[1]) == 1 ) {
			if ( point_on_arc(px1, py1, rtn->cx[1], rtn->cy[1], rtn->r[1], rtn->sx[4], rtn->sy[4], rtn->ex[4], rtn->ey[4]) == 1 ) {
				rtn->type++;
				rtn->sx[i] = px1;
				rtn->sy[i] = py1;
				i++;

				//g_print ("交点(%f,%f) (%f,%f),R%f (%f,%f)-(%f,%f)\n", 
				//		 px1, py1, rtn->cx[1], rtn->cy[1], rtn->r[1], 
				//		 rtn->sx[4], rtn->sy[4], rtn->ex[4], rtn->ey[4]);
			}
			else {
				//g_print ("交点 1 NO by point_on_arc()\n");
				//g_print ("交点(%f,%f) (%f,%f),R%f (%f,%f)-(%f,%f)\n", 
				//		 px1, py1, rtn->cx[1], rtn->cy[1], rtn->r[1], 
				//		 rtn->sx[4], rtn->sy[4], rtn->ex[4], rtn->ey[4]);
			}
		}
		else {
			//g_print ("交点 1 NO by point_on_line()\n");
		}
	}

	/* 交点 2 */
	if (rtn_lcp.type == 2) {
		px2 = rtn_lcp.sx[3];
		py2 = rtn_lcp.sy[3];

		if ( point_on_line(px2, py2, rtn->sx[1], rtn->sy[1], rtn->ex[1], rtn->ey[1]) == 1 ) {
			if ( point_on_arc(px2, py2, rtn->cx[1], rtn->cy[1], rtn->r[1], rtn->sx[4], rtn->sy[4], rtn->ex[4], rtn->ey[4]) == 1 ) {
				rtn->type++;
				rtn->sx[i] = px2;
				rtn->sy[i] = py2;

				//g_print ("交点(%f,%f) (%f,%f),R%f (%f,%f)-(%f,%f)\n", 
				//		 px2, py2, rtn->cx[1], rtn->cy[1], rtn->r[1], 
				//		 rtn->sx[4], rtn->sy[4], rtn->ex[4], rtn->ey[4]);
			}
			else {
				//g_print ("交点 2 NO by point_on_arc()\n");
				//g_print ("交点(%f,%f) (%f,%f),R%f (%f,%f)-(%f,%f)\n", 
				//		 px2, py2, rtn->cx[1], rtn->cy[1], rtn->r[1], 
				//		 rtn->sx[4], rtn->sy[4], rtn->ex[4], rtn->ey[4]);
			}
		}
		else {
			//g_print ("交点 2 NO by point_on_line()\n");
		}
	}

	return rtn->type;
}





/* -------------------------------------------------------------------
 * 説明 : ２円弧の実際の交点
 * 関数 : Act_ccp
 * 引数 : rtndat
 *		:	円 1	  (cx[1] , cy[1]) , r[1], (sx[3] , sy[3]) - (ex[3] , ey[3])
 *		:	円 2	  (cx[2] , cy[2]) , r[2], (sx[4] , sy[4]) - (ex[4] , ey[4])
 * 戻値 : rtndat
 *		:	type	   0 , 1 , 2
 *		:	交点 1	  (sx[1] , sy[1])
 *		:	交点 2	  (sx[2] , sy[2])
 * 外部 : 無し
 * 内部 : RtnDat pph,lah,paph,lcph,llah
 */
int Act_ccp(struct RtnDat *rtn)
{
	struct RtnDat rtn_ccp;
	double px1, py1, px2, py2;


	rtn->type = 0;
	rtn_ccp = *rtn;

	ccp(&rtn_ccp);
	/* (rtn_ccp.sx[1] , rtn_ccp.sy[1]) */
	/* (rtn_ccp.sx[2] , rtn_ccp.sy[2]) */

	/* 円と円の交点がなければ */
	if (rtn_ccp.type == 0) {
		//g_print ("交点 NO by ccp()\n");
		return 0;
	}
	if (rtn_ccp.type > 0) {
		px1 = rtn_ccp.sx[1];
		py1 = rtn_ccp.sy[1];
		px2 = 0;
		py2 = 0;
	}
	if (rtn_ccp.type == 2) {
		px2 = rtn_ccp.sx[2];
		py2 = rtn_ccp.sy[2];
	}


	/* 交点 1 */
	if (rtn_ccp.type > 0) {
		if ( point_on_arc(px1, py1, rtn->cx[1], rtn->cy[1], rtn->r[1], rtn->sx[3], rtn->sy[3], rtn->ex[3], rtn->ey[3]) == 0 ) {
			//g_print ("交点 1 NO by point_on_arc()\n");
			return 0;
		}
		if ( point_on_arc(px1, py1, rtn->cx[2], rtn->cy[2], rtn->r[2], rtn->sx[4], rtn->sy[4], rtn->ex[4], rtn->ey[4]) == 0 ) {
			//g_print ("交点 1 NO by point_on_arc()\n");
			return 0;
		}
		rtn->type++;
		rtn->sx[1] = px1;
		rtn->sy[1] = py1;
	}
	
	/* 交点 2 */
	if (rtn_ccp.type == 2) {
		if ( point_on_arc(px2, py2, rtn->cx[1], rtn->cy[1], rtn->r[1], rtn->sx[3], rtn->sy[3], rtn->ex[3], rtn->ey[3]) == 0 ) {
			//g_print ("交点 2 NO by point_on_arc()\n");
			return 0;
		}
		if ( point_on_arc(px2, py2, rtn->cx[2], rtn->cy[2], rtn->r[2], rtn->sx[4], rtn->sy[4], rtn->ex[4], rtn->ey[4]) == 0 ) {
			//g_print ("交点 2 NO by point_on_arc()\n");
			return 0;
		}
		rtn->type++;
		rtn->sx[2] = px2;
		rtn->sy[2] = py2;
	}
	return rtn->type;
}





/* -------------------------------------------------------------------
 * データの入れ替え
 * -------------------------------------
 * a と b のデータを入れ換える
 *	
 * -------------------------------------
 * a	   : 数値
 * b	   : 数値
 * -------------------------------------------------------------------
 */
int Swap_double(double *a, double *b)
{
	double dumy;

	dumy = *a;
	*a = *b;
	*b = dumy;
	return 1;
}



/* -------------------------------------------------------------------
 * 数値(小数点以下 3 桁)出力用
 * -------------------------------------
 * dbl を 文字列にして str いれて返す
 *	
 * -------------------------------------
 * dbl	   : 数値
 * str	   : このポインタに結果をいれて返す
 * Dot = 0 : 整数の場合小数点  無
 *	   = 1 : 整数の場合も小数点  有
 * -------------------------------------------------------------------
 */
int FloatOut(char *str, double dbl, int Dot)
{
	int i = 1, j;
	double dumy, dumy2, Dumy_dbl;


	if (dbl < 0) {
		i = -1;
		dbl = dbl * i;
	}
	dbl = sg(dbl, 6);

	/* 0 */
	Dumy_dbl = dbl;
	dumy = floor(Dumy_dbl);
	dumy2 = Dumy_dbl - dumy;
	if (dumy2 == 0) {
		if (Dot == 0) sprintf(str, "%d", (int)dbl*i);
		else		  sprintf(str, "%d.", (int)dbl*i);
		return 1;
	}

	/* 1-3 */
	for(j=1 ; j<=3 ; j++) {
		Dumy_dbl = dbl;
		Dumy_dbl = Dumy_dbl * pow( 10, (double) j);  /* pow は、 10 の j乗 */
		dumy = floor( sg(Dumy_dbl, 3) );			 /* floor は、値の切り捨てを計算 */
		dumy2 = Dumy_dbl - dumy;
		if (sg(dumy2, 3) == 0) {
			if (j == 1) 	 sprintf(str, "%.1f", dbl*i);
			else if (j == 2) sprintf(str, "%.2f", dbl*i);
			else if (j == 3) sprintf(str, "%.3f", dbl*i);
			else			 sprintf(str, "%.3f", dbl*i);
			return 1;
		}
	}
	sprintf(str, "%.3f", dbl*i);
	return 1;
}



/* -------------------------------------------------------------------
 * 数値(小数点以下 6 桁)出力用
 * -------------------------------------
 * dbl を 文字列にして str いれて返す
 *	
 * -------------------------------------
 * dbl	   : 数値
 * str	   : このポインタに結果をいれて返す
 * Dot = 0 : 整数の場合小数点  無
 *	   = 1 : 整数の場合も小数点  有
 * -------------------------------------------------------------------
 */
int FloatOut6(char *str, double dbl, int Dot)
{
	int i = 1, j;
	double dumy, dumy2, Dumy_dbl;

	/* -----------------------------------------------------
	 * まず符号をチェックし、 i に符号を入力
	 */
	if (dbl < 0) {
		i = -1;
		dbl = dbl * i;
	}
	/* 小数点以下 6 桁に四捨五入 */
	dbl = sg(dbl, 6);

	/* 0 */
	Dumy_dbl = dbl;
	dumy = floor( sg(Dumy_dbl, 6) );  /* 小数点以下を切り捨て、dumy に */
	dumy2 = Dumy_dbl - dumy;		  /* 少数点以下のみを求め、dumy2 に */
	/* 小数点以下のみを少数点以下 6 桁に四捨五入して 0 になれば、
	 * 求める値は、整数である。
	 */
	if (sg(dumy2, 6) == 0) {
		if (Dot == 0) sprintf(str, "%d", (int) dbl*i);
		else sprintf(str, "%d.", (int) dbl*i);
		return 1;
	}

	/* 1-6 */
	for(j=1 ; j<=6 ; j++) {
		Dumy_dbl = dbl;
		Dumy_dbl = Dumy_dbl * pow( 10, (double) j);
		dumy = floor( sg(Dumy_dbl, 6) );
		dumy2 = Dumy_dbl - dumy;
		if (sg(dumy2, 6) == 0) {
			if (j ==1 ) 	 sprintf(str, "%.1f", dbl*i);
			else if (j == 2) sprintf(str, "%.2f", dbl*i);
			else if (j == 3) sprintf(str, "%.3f", dbl*i);
			else if (j == 4) sprintf(str, "%.4f", dbl*i);
			else if (j == 5) sprintf(str, "%.5f", dbl*i);
			else			 sprintf(str, "%.6f", dbl*i);
			return 1;
		}
	}
	sprintf(str, "%.6f", dbl*i);
	return 1;
}


/* -------------------------------------------------------------------
 * 数値(小数点以下 n 桁)出力用
 * -------------------------------------
 * dbl を 文字列にして str いれて返す
 *	
 * -------------------------------------
 * dbl	   : 数値
 * str	   : このポインタに結果をいれて返す
 * n	   : 四捨五入で求める小数点以下の桁数
 * Dot = 0 : 整数の場合小数点	 無
 *	   = 1 : 整数の場合も小数点  有
 * -------------------------------------------------------------------
 */
int FloatOut_n(char *str, double dbl, int n, int Dot)
{
	int i = 1, j;
	double dumy, dumy2, Dumy_dbl;
	char out_n[256];

	/* -----------------------------------------------------
	 * まず符号をチェックし、 i に符号を入力
	 */
	if (dbl < 0) {
		i = -1;
		dbl = dbl * i;
	}
	/* 小数点以下 n 桁に四捨五入 */
	dbl = sg(dbl, n);

	/* -----------------------------------------------------
	 * 整数かどうかのチェック 0 
	 */
	Dumy_dbl = dbl;
	dumy = floor( sg(Dumy_dbl, n) );  /* 小数点以下を切り捨て、dumy に */
	dumy2 = Dumy_dbl - dumy;		  /* 少数点以下のみを求め、dumy2 に */
	/* 小数点以下のみを少数点以下 n 桁に四捨五入して 0 になれば、
	 * 求める値は、整数である。
	 */
	if (sg(dumy2, n) == 0) {
		if (Dot == 0) sprintf(str, "%d", (int) dbl*i);
		else sprintf(str, "%d.", (int) dbl*i);
		return 1;
	}

	/* -----------------------------------------------------
	 * 小数点以下何桁かをチェック 1-n	
	 */
	for(j = n-1 ; j >= 0 ; j--) {
		Dumy_dbl = dbl;
		Dumy_dbl = Dumy_dbl * pow(10, (double) j);
		dumy = floor( sg(Dumy_dbl, n) );
		dumy2 = Dumy_dbl - dumy;
		if (sg(dumy2, n) != 0) {
			sprintf(out_n, "%%.%df", j+1);
			sprintf(str, out_n, dbl*i);
			return 1;
		}
	}

	return 0;
}





/* -------------------------------------------------------------------
 * 回転    点(px,py) を 原点(x,y) を中心に Angle 度回転移動
 */
int rotation(double *px, double *py, double x, double y, double Angle)
{
	double RAD, DX, DY;

	/* 移動ベクトル */
	RAD = degrad(Angle);

	DX = *px - x;
	DY = *py - y;
	*px = sg((DX * cos(RAD) - DY * sin(RAD)) + x, calcu_digits);
	*py = sg((DY * cos(RAD) + DX * sin(RAD)) + y, calcu_digits);

	return 1;
}

/* -------------------------------------------------------------------
 * 平行    点(px,py) を ベクトル(x,y) で平行移動
 */
int parallel(double *px, double *py, double x, double y)
{
	*px = sg(*px + x, calcu_digits);
	*py = sg(*py + y, calcu_digits);
	return 1;
}

/* -------------------------------------------------------------------
 * 対称    点(x,y) を 線(sx,sy)-(ex,ey) に対して対称移動
 */
int symmetry(double sx, double sy, double ex, double ey, double *x, double *y)
{
	struct RtnDat PLPH;

	/* -----------------------------------------------------
	 * 線に対してのみ 直線
	 */
	PLPH.sx[1] = sx; PLPH.sy[1] = sy; PLPH.ex[1] = ex; PLPH.ey[1] = ey;
	PLPH.sx[2] = *x; PLPH.sy[2] = *y;
	plp(&PLPH);

	*x = sg(*x + 2 * (sg(PLPH.ex[2], calcu_digits) - sg(*x, calcu_digits)), calcu_digits);
	*y = sg(*y + 2 * (sg(PLPH.ey[2], calcu_digits) - sg(*y, calcu_digits)), calcu_digits);
	return 1;
}

/* -------------------------------------------------------------------
 * スケーリング    点(x,y) を 原点(x,y) を中心に k 倍した図形に変換
 */
int scaling(double *px, double *py, double x, double y, double k)
{
	*px = sg((k * *px) + x * (1 - k), calcu_digits);
	*py = sg((k * *py) + y * (1 - k), calcu_digits);
	return 1;
}

/* -------------------------------------------------------------------
 * スケーリング    点(px,py) を 原点(x,y) を中心に 
 * Ⅹ軸方向に kx倍し、Ｙ軸方向に ky倍した図形に変換
 */
int scaling_xy(double *px, double *py, double x, double y, double kx, double ky)
{
	*px = sg((kx * *px) + x * (1 - kx), calcu_digits);
	*py = sg((ky * *py) + y * (1 - ky), calcu_digits);
	return 1;
}





/* ====================================================================
 * ===	Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 *	  Project			   : SagCAD
 *	  Source			   : culcfunc.c
 * ====================================================================
 */





/* -------------------------------------------------------------------
 * 角度を 0 から 360 の間にする
 */
int angle_check(double *angle)
{
	while (*angle < 0 || *angle > 360) {
		if (*angle > 360) *angle = *angle - 360;
		if (*angle < 0) *angle = *angle + 360;
	}
	return 1;
}





/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 *    Project              : SagCAD
 *    Source               : culcfunc.c
 * ====================================================================
 */
