---
layout: default
title: x86
parent: Single & Double Precision
grand_parent: References
permalink: /2-references/libm/x86
---

<h1>Single & Double Math library reference (x86)</h1>

<h2>Table of contents</h2>

* [Function naming convention](#naming)
* [Data types](#datatypes)
* [Trigonometric functions](#trig)
* [Power, exponential, and logarithmic functions](#pow)
* [Inverse trigonometric functions](#invtrig)
* [Hyperbolic functions and inverse hyperbolic functions](#hyp)
* [Error and gamma functions](#eg)
* [Nearest integer functions](#nearint)
* [Other functions](#other)
* [Supplementary functions](#supplementary)

<h2 id="naming">Function naming convention</h2>

The naming convention for the vectorized math functions is shown in
Fig. 3.1 and 3.2. The function name is a concatenation of the
following items, in this order.

* String "Sleef_"
* Name of the corresponding double precision function in math.h
* Data type specifier of a vector element, "d" and "f" for double and single
  precision functions, respectively
* The number of elements in a vector
* Accuracy specifier, which is a concatenation of "_u" and 10
  times the maximum error for typical input domain in
  [ULP](../../3-extra#ulp) (two digits). If the function is expected to always
  return the correctly rounded value, its name has no accuracy specifier.
* Vector extension specifier
  * (Nothing) : [Dispatcher](../../3-extra#dispatcher) automatically
    chooses the fastest available vector extension
  * sse2 : SSE2
  * sse4 : SSE4.1
  * avx2128 : AVX2+FMA3 instructions utilized for 128 bit computation
  * avx : AVX
  * avx2 : AVX2+FMA3
  * avx512f : AVX512F
* Attributes
  * c : The function gives bit-wise consistent results across all platforms
  * f : The function requires
    [FMA](https://en.wikipedia.org/wiki/Multiply%E2%80%93accumulate_operation#Fused_multiply%E2%80%93add)
    support and gives bit-wise consistent results across all platforms
  * i : Infinity is correctly handled
  * n : NaN is correctly handled
  * z : Negative zero is correctly handled

<p style="text-align:center; margin-top:1cm;">
  <img src="../../img/convention.png" alt="naming convention" width="50%" height="50%" />
  <br />
  Fig. 3.1: Naming convention of vectorized functions
</p>

<p style="text-align:center; margin-top:1cm;">
  <img src="../../img/convention2.png" alt="naming convention with attributes" width="50%" height="50%" />
  <br />
  Fig. 3.2: Naming convention of vectorized functions with attributes
</p>

<h2 id="datatypes">Data types for x86 architecture</h2>

### Sleef___m128_2

`Sleef___m128_2` is a data type for storing two `__m128` values,
which is defined in sleef.h as follows:

```c
typedef struct {
  __m128 x, y;
} Sleef___m128_2;
```

### Sleef___m128d_2

`Sleef___m128d_2` is a data type for storing two `__m128d` values,
which is defined in sleef.h as follows:

```c
typedef struct {
  __m128d x, y;
} Sleef___m128d_2;
```

### Sleef___m256_2

`Sleef___m256_2` is a data type for storing two `__m256` values,
which is defined in sleef.h as follows:

```c
typedef struct {
  __m256 x, y;
} Sleef___m256_2;
```

### Sleef___m256d_2

`Sleef___m256d_2` is a data type for storing two `__m256d` values,
which is defined in sleef.h as follows:

```c
typedef struct {
  __m256d x, y;
} Sleef___m256d_2;
```

### Sleef___m512_2

`Sleef___m512_2` is a data type for storing two `__m512` values,
which is defined in sleef.h as follows:

```c
typedef struct {
  __m512 x, y;
} Sleef___m512_2;
```

### Sleef___m512d_2

`Sleef___m512d_2` is a data type for storing two `__m512d` values,
which is defined in sleef.h as follows:

```c
typedef struct {
  __m512d x, y;
} Sleef___m512d_2;
```

<h2 id="trig">Trigonometric Functions</h2>

### Vectorized double precision sine functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_sind2_u10(__m128d a);
__m128d Sleef_sind2_u10sse2(__m128d a);
__m128d Sleef_sind2_u10sse4(__m128d a);
__m128d Sleef_sind2_u10avx2128(__m128d a);
__m128d Sleef_cinz_sind2_u10sse2(__m128d a);
__m128d Sleef_cinz_sind2_u10sse4(__m128d a);
__m128d Sleef_finz_sind2_u10avx2128(__m128d a);

__m256d Sleef_sind4_u10(__m256d a);
__m256d Sleef_sind4_u10avx(__m256d a);
__m256d Sleef_sind4_u10avx2(__m256d a);
__m256d Sleef_sind4_u10fma4(__m256d a);
__m256d Sleef_cinz_sind4_u10avx(__m256d a);
__m256d Sleef_finz_sind4_u10avx2(__m256d a);
__m256d Sleef_finz_sind4_u10fma4(__m256d a);

__m512d Sleef_sind8_u10(__m512d a);
__m512d Sleef_sind8_u10avx512f(__m512d a);
__m512d Sleef_cinz_sind8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_sind8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sin_u10](../libm#sleef_sin_u10) with the same accuracy specification.

### Vectorized single precision sine functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_sinf4_u10(__m128 a);
__m128 Sleef_sinf4_u10sse2(__m128 a);
__m128 Sleef_sinf4_u10sse4(__m128 a);
__m128 Sleef_sinf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_sinf4_u10sse2(__m128 a);
__m128 Sleef_cinz_sinf4_u10sse4(__m128 a);
__m128 Sleef_finz_sinf4_u10avx2128(__m128 a);

__m256 Sleef_sinf8_u10(__m256 a);
__m256 Sleef_sinf8_u10avx(__m256 a);
__m256 Sleef_sinf8_u10avx2(__m256 a);
__m256 Sleef_sinf8_u10fma4(__m256 a);
__m256 Sleef_cinz_sinf8_u10avx(__m256 a);
__m256 Sleef_finz_sinf8_u10avx2(__m256 a);
__m256 Sleef_finz_sinf8_u10fma4(__m256 a);

__m512 Sleef_sinf16_u10(__m512 a);
__m512 Sleef_sinf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_sinf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_sinf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sinf_u10](../libm#sleef_sinf_u10) with the same accuracy specification.

### Vectorized double precision sine functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_sind2_u35(__m128d a);
__m128d Sleef_sind2_u35sse2(__m128d a);
__m128d Sleef_sind2_u35sse4(__m128d a);
__m128d Sleef_sind2_u35avx2128(__m128d a);
__m128d Sleef_cinz_sind2_u35sse2(__m128d a);
__m128d Sleef_cinz_sind2_u35sse4(__m128d a);
__m128d Sleef_finz_sind2_u35avx2128(__m128d a);

__m256d Sleef_sind4_u35(__m256d a);
__m256d Sleef_sind4_u35avx(__m256d a);
__m256d Sleef_sind4_u35avx2(__m256d a);
__m256d Sleef_sind4_u35fma4(__m256d a);
__m256d Sleef_cinz_sind4_u35avx(__m256d a);
__m256d Sleef_finz_sind4_u35avx2(__m256d a);
__m256d Sleef_finz_sind4_u35fma4(__m256d a);

__m512d Sleef_sind8_u35(__m512d a);
__m512d Sleef_sind8_u35avx512f(__m512d a);
__m512d Sleef_cinz_sind8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_sind8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sin_u35](../libm#sleef_sin_u35) with the same accuracy specification.

### Vectorized single precision sine functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_sinf4_u35(__m128 a);
__m128 Sleef_sinf4_u35sse2(__m128 a);
__m128 Sleef_sinf4_u35sse4(__m128 a);
__m128 Sleef_sinf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_sinf4_u35sse2(__m128 a);
__m128 Sleef_cinz_sinf4_u35sse4(__m128 a);
__m128 Sleef_finz_sinf4_u35avx2128(__m128 a);

__m256 Sleef_sinf8_u35(__m256 a);
__m256 Sleef_sinf8_u35avx(__m256 a);
__m256 Sleef_sinf8_u35avx2(__m256 a);
__m256 Sleef_sinf8_u35fma4(__m256 a);
__m256 Sleef_cinz_sinf8_u35avx(__m256 a);
__m256 Sleef_finz_sinf8_u35avx2(__m256 a);
__m256 Sleef_finz_sinf8_u35fma4(__m256 a);

__m512 Sleef_sinf16_u35(__m512 a);
__m512 Sleef_sinf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_sinf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_sinf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sinf_u35](../libm#sleef_sinf_u35) with the same accuracy specification.

### Vectorized double precision cosine functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_cosd2_u10(__m128d a);
__m128d Sleef_cosd2_u10sse2(__m128d a);
__m128d Sleef_cosd2_u10sse4(__m128d a);
__m128d Sleef_cosd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_cosd2_u10sse2(__m128d a);
__m128d Sleef_cinz_cosd2_u10sse4(__m128d a);
__m128d Sleef_finz_cosd2_u10avx2128(__m128d a);

__m256d Sleef_cosd4_u10(__m256d a);
__m256d Sleef_cosd4_u10avx(__m256d a);
__m256d Sleef_cosd4_u10avx2(__m256d a);
__m256d Sleef_cosd4_u10fma4(__m256d a);
__m256d Sleef_cinz_cosd4_u10avx(__m256d a);
__m256d Sleef_finz_cosd4_u10avx2(__m256d a);
__m256d Sleef_finz_cosd4_u10fma4(__m256d a);

__m512d Sleef_cosd8_u10(__m512d a);
__m512d Sleef_cosd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_cosd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_cosd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cos_u10](../libm#sleef_cos_u10) with the same accuracy specification.

### Vectorized single precision cosine functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_cosf4_u10(__m128 a);
__m128 Sleef_cosf4_u10sse2(__m128 a);
__m128 Sleef_cosf4_u10sse4(__m128 a);
__m128 Sleef_cosf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_cosf4_u10sse2(__m128 a);
__m128 Sleef_cinz_cosf4_u10sse4(__m128 a);
__m128 Sleef_finz_cosf4_u10avx2128(__m128 a);

__m256 Sleef_cosf8_u10(__m256 a);
__m256 Sleef_cosf8_u10avx(__m256 a);
__m256 Sleef_cosf8_u10avx2(__m256 a);
__m256 Sleef_cosf8_u10fma4(__m256 a);
__m256 Sleef_cinz_cosf8_u10avx(__m256 a);
__m256 Sleef_finz_cosf8_u10avx2(__m256 a);
__m256 Sleef_finz_cosf8_u10fma4(__m256 a);

__m512 Sleef_cosf16_u10(__m512 a);
__m512 Sleef_cosf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_cosf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_cosf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cosf_u10](../libm#sleef_cosf_u10) with the same accuracy specification.

### Vectorized double precision cosine functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_cosd2_u35(__m128d a);
__m128d Sleef_cosd2_u35sse2(__m128d a);
__m128d Sleef_cosd2_u35sse4(__m128d a);
__m128d Sleef_cosd2_u35avx2128(__m128d a);
__m128d Sleef_cinz_cosd2_u35sse2(__m128d a);
__m128d Sleef_cinz_cosd2_u35sse4(__m128d a);
__m128d Sleef_finz_cosd2_u35avx2128(__m128d a);

__m256d Sleef_cosd4_u35(__m256d a);
__m256d Sleef_cosd4_u35avx(__m256d a);
__m256d Sleef_cosd4_u35avx2(__m256d a);
__m256d Sleef_cosd4_u35fma4(__m256d a);
__m256d Sleef_cinz_cosd4_u35avx(__m256d a);
__m256d Sleef_finz_cosd4_u35avx2(__m256d a);
__m256d Sleef_finz_cosd4_u35fma4(__m256d a);

__m512d Sleef_cosd8_u35(__m512d a);
__m512d Sleef_cosd8_u35avx512f(__m512d a);
__m512d Sleef_cinz_cosd8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_cosd8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cos_u35](../libm#sleef_cos_u35) with the same accuracy specification.

### Vectorized single precision cosine functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_cosf4_u35(__m128 a);
__m128 Sleef_cosf4_u35sse2(__m128 a);
__m128 Sleef_cosf4_u35sse4(__m128 a);
__m128 Sleef_cosf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_cosf4_u35sse2(__m128 a);
__m128 Sleef_cinz_cosf4_u35sse4(__m128 a);
__m128 Sleef_finz_cosf4_u35avx2128(__m128 a);

__m256 Sleef_cosf8_u35(__m256 a);
__m256 Sleef_cosf8_u35avx(__m256 a);
__m256 Sleef_cosf8_u35avx2(__m256 a);
__m256 Sleef_cosf8_u35fma4(__m256 a);
__m256 Sleef_cinz_cosf8_u35avx(__m256 a);
__m256 Sleef_finz_cosf8_u35avx2(__m256 a);
__m256 Sleef_finz_cosf8_u35fma4(__m256 a);

__m512 Sleef_cosf16_u35(__m512 a);
__m512 Sleef_cosf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_cosf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_cosf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cosf_u35](../libm#sleef_cosf_u35) with the same accuracy specification.

### Vectorized double precision combined sine and cosine functions with 1.0 ULP error bound

```c
#include <sleef.h>

Sleef___m128d_2 Sleef_sincosd2_u10(__m128d a);
Sleef___m128d_2 Sleef_sincosd2_u10sse2(__m128d a);
Sleef___m128d_2 Sleef_sincosd2_u10sse4(__m128d a);
Sleef___m128d_2 Sleef_sincosd2_u10avx2128(__m128d a);
Sleef___m128d_2 Sleef_cinz_sincosd2_u10sse2(__m128d a);
Sleef___m128d_2 Sleef_cinz_sincosd2_u10sse4(__m128d a);
Sleef___m128d_2 Sleef_finz_sincosd2_u10avx2128(__m128d a);

Sleef___m256d_2 Sleef_sincosd4_u10(__m256d a);
Sleef___m256d_2 Sleef_sincosd4_u10avx(__m256d a);
Sleef___m256d_2 Sleef_sincosd4_u10avx2(__m256d a);
Sleef___m256d_2 Sleef_sincosd4_u10fma4(__m256d a);
Sleef___m256d_2 Sleef_cinz_sincosd4_u10avx(__m256d a);
Sleef___m256d_2 Sleef_finz_sincosd4_u10avx2(__m256d a);
Sleef___m256d_2 Sleef_finz_sincosd4_u10fma4(__m256d a);

Sleef___m512d_2 Sleef_sincosd8_u10(__m512d a);
Sleef___m512d_2 Sleef_sincosd8_u10avx512f(__m512d a);
Sleef___m512d_2 Sleef_cinz_sincosd8_u10avx512fnofma(__m512d a);
Sleef___m512d_2 Sleef_finz_sincosd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sincos_u10](../libm#sleef_sincos_u10) with the same accuracy specification.

### Vectorized single precision combined sine and cosine functions with 1.0 ULP error bound

```c
#include <sleef.h>

Sleef___m128_2 Sleef_sincosf4_u10(__m128 a);
Sleef___m128_2 Sleef_sincosf4_u10sse2(__m128 a);
Sleef___m128_2 Sleef_sincosf4_u10sse4(__m128 a);
Sleef___m128_2 Sleef_sincosf4_u10avx2128(__m128 a);
Sleef___m128_2 Sleef_cinz_sincosf4_u10sse2(__m128 a);
Sleef___m128_2 Sleef_cinz_sincosf4_u10sse4(__m128 a);
Sleef___m128_2 Sleef_finz_sincosf4_u10avx2128(__m128 a);

Sleef___m256_2 Sleef_sincosf8_u10(__m256 a);
Sleef___m256_2 Sleef_sincosf8_u10avx(__m256 a);
Sleef___m256_2 Sleef_sincosf8_u10avx2(__m256 a);
Sleef___m256_2 Sleef_sincosf8_u10fma4(__m256 a);
Sleef___m256_2 Sleef_cinz_sincosf8_u10avx(__m256 a);
Sleef___m256_2 Sleef_finz_sincosf8_u10avx2(__m256 a);
Sleef___m256_2 Sleef_finz_sincosf8_u10fma4(__m256 a);

Sleef___m512_2 Sleef_sincosf16_u10(__m512 a);
Sleef___m512_2 Sleef_sincosf16_u10avx512f(__m512 a);
Sleef___m512_2 Sleef_cinz_sincosf16_u10avx512fnofma(__m512 a);
Sleef___m512_2 Sleef_finz_sincosf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sincosf_u10](../libm#sleef_sincosf_u10) with the same accuracy specification.

### Vectorized double precision combined sine and cosine functions with 3.5 ULP error bound

```c
#include <sleef.h>

Sleef___m128d_2 Sleef_sincosd2_u35(__m128d a);
Sleef___m128d_2 Sleef_sincosd2_u35sse2(__m128d a);
Sleef___m128d_2 Sleef_sincosd2_u35sse4(__m128d a);
Sleef___m128d_2 Sleef_sincosd2_u35avx2128(__m128d a);
Sleef___m128d_2 Sleef_cinz_sincosd2_u35sse2(__m128d a);
Sleef___m128d_2 Sleef_cinz_sincosd2_u35sse4(__m128d a);
Sleef___m128d_2 Sleef_finz_sincosd2_u35avx2128(__m128d a);

Sleef___m256d_2 Sleef_sincosd4_u35(__m256d a);
Sleef___m256d_2 Sleef_sincosd4_u35avx(__m256d a);
Sleef___m256d_2 Sleef_sincosd4_u35avx2(__m256d a);
Sleef___m256d_2 Sleef_sincosd4_u35fma4(__m256d a);
Sleef___m256d_2 Sleef_cinz_sincosd4_u35avx(__m256d a);
Sleef___m256d_2 Sleef_finz_sincosd4_u35avx2(__m256d a);
Sleef___m256d_2 Sleef_finz_sincosd4_u35fma4(__m256d a);

Sleef___m512d_2 Sleef_sincosd8_u35(__m512d a);
Sleef___m512d_2 Sleef_sincosd8_u35avx512f(__m512d a);
Sleef___m512d_2 Sleef_cinz_sincosd8_u35avx512fnofma(__m512d a);
Sleef___m512d_2 Sleef_finz_sincosd8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sincos_u35](../libm#sleef_sincos_u35) with the same accuracy specification.

### Vectorized single precision combined sine and cosine functions with 3.5 ULP error bound

```c
#include <sleef.h>

Sleef___m128_2 Sleef_sincosf4_u35(__m128 a);
Sleef___m128_2 Sleef_sincosf4_u35sse2(__m128 a);
Sleef___m128_2 Sleef_sincosf4_u35sse4(__m128 a);
Sleef___m128_2 Sleef_sincosf4_u35avx2128(__m128 a);
Sleef___m128_2 Sleef_cinz_sincosf4_u35sse2(__m128 a);
Sleef___m128_2 Sleef_cinz_sincosf4_u35sse4(__m128 a);
Sleef___m128_2 Sleef_finz_sincosf4_u35avx2128(__m128 a);

Sleef___m256_2 Sleef_sincosf8_u35(__m256 a);
Sleef___m256_2 Sleef_sincosf8_u35avx(__m256 a);
Sleef___m256_2 Sleef_sincosf8_u35avx2(__m256 a);
Sleef___m256_2 Sleef_sincosf8_u35fma4(__m256 a);
Sleef___m256_2 Sleef_cinz_sincosf8_u35avx(__m256 a);
Sleef___m256_2 Sleef_finz_sincosf8_u35avx2(__m256 a);
Sleef___m256_2 Sleef_finz_sincosf8_u35fma4(__m256 a);

Sleef___m512_2 Sleef_sincosf16_u35(__m512 a);
Sleef___m512_2 Sleef_sincosf16_u35avx512f(__m512 a);
Sleef___m512_2 Sleef_cinz_sincosf16_u35avx512fnofma(__m512 a);
Sleef___m512_2 Sleef_finz_sincosf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sincosf_u35](../libm#sleef_sincosf_u35) with the same accuracy specification.

### Vectorized double precision sine functions with 0.506 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_sinpid2_u05(__m128d a);
__m128d Sleef_sinpid2_u05sse2(__m128d a);
__m128d Sleef_sinpid2_u05sse4(__m128d a);
__m128d Sleef_sinpid2_u05avx2128(__m128d a);
__m128d Sleef_cinz_sinpid2_u05sse2(__m128d a);
__m128d Sleef_cinz_sinpid2_u05sse4(__m128d a);
__m128d Sleef_finz_sinpid2_u05avx2128(__m128d a);

__m256d Sleef_sinpid4_u05(__m256d a);
__m256d Sleef_sinpid4_u05avx(__m256d a);
__m256d Sleef_sinpid4_u05avx2(__m256d a);
__m256d Sleef_sinpid4_u05fma4(__m256d a);
__m256d Sleef_cinz_sinpid4_u05avx(__m256d a);
__m256d Sleef_finz_sinpid4_u05avx2(__m256d a);
__m256d Sleef_finz_sinpid4_u05fma4(__m256d a);

__m512d Sleef_sinpid8_u05(__m512d a);
__m512d Sleef_sinpid8_u05avx512f(__m512d a);
__m512d Sleef_cinz_sinpid8_u05avx512fnofma(__m512d a);
__m512d Sleef_finz_sinpid8_u05avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sinpi_u05](../libm#sleef_sinpi_u05) with the same accuracy specification.

### Vectorized single precision sine functions with 0.506 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_sinpif4_u05(__m128 a);
__m128 Sleef_sinpif4_u05sse2(__m128 a);
__m128 Sleef_sinpif4_u05sse4(__m128 a);
__m128 Sleef_sinpif4_u05avx2128(__m128 a);
__m128 Sleef_cinz_sinpif4_u05sse2(__m128 a);
__m128 Sleef_cinz_sinpif4_u05sse4(__m128 a);
__m128 Sleef_finz_sinpif4_u05avx2128(__m128 a);

__m256 Sleef_sinpif8_u05(__m256 a);
__m256 Sleef_sinpif8_u05avx(__m256 a);
__m256 Sleef_sinpif8_u05avx2(__m256 a);
__m256 Sleef_sinpif8_u05fma4(__m256 a);
__m256 Sleef_cinz_sinpif8_u05avx(__m256 a);
__m256 Sleef_finz_sinpif8_u05avx2(__m256 a);
__m256 Sleef_finz_sinpif8_u05fma4(__m256 a);

__m512 Sleef_sinpif16_u05(__m512 a);
__m512 Sleef_sinpif16_u05avx512f(__m512 a);
__m512 Sleef_cinz_sinpif16_u05avx512fnofma(__m512 a);
__m512 Sleef_finz_sinpif16_u05avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sinpif_u05](../libm#sleef_sinpif_u05) with the same accuracy specification.

### Vectorized double precision cosine functions with 0.506 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_cospid2_u05(__m128d a);
__m128d Sleef_cospid2_u05sse2(__m128d a);
__m128d Sleef_cospid2_u05sse4(__m128d a);
__m128d Sleef_cospid2_u05avx2128(__m128d a);
__m128d Sleef_cinz_cospid2_u05sse2(__m128d a);
__m128d Sleef_cinz_cospid2_u05sse4(__m128d a);
__m128d Sleef_finz_cospid2_u05avx2128(__m128d a);

__m256d Sleef_cospid4_u05(__m256d a);
__m256d Sleef_cospid4_u05avx(__m256d a);
__m256d Sleef_cospid4_u05avx2(__m256d a);
__m256d Sleef_cospid4_u05fma4(__m256d a);
__m256d Sleef_cinz_cospid4_u05avx(__m256d a);
__m256d Sleef_finz_cospid4_u05avx2(__m256d a);
__m256d Sleef_finz_cospid4_u05fma4(__m256d a);

__m512d Sleef_cospid8_u05(__m512d a);
__m512d Sleef_cospid8_u05avx512f(__m512d a);
__m512d Sleef_cinz_cospid8_u05avx512fnofma(__m512d a);
__m512d Sleef_finz_cospid8_u05avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cospi_u05](../libm#sleef_cospi_u05) with the same accuracy specification.

### Vectorized single precision cosine functions with 0.506 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_cospif4_u05(__m128 a);
__m128 Sleef_cospif4_u05sse2(__m128 a);
__m128 Sleef_cospif4_u05sse4(__m128 a);
__m128 Sleef_cospif4_u05avx2128(__m128 a);
__m128 Sleef_cinz_cospif4_u05sse2(__m128 a);
__m128 Sleef_cinz_cospif4_u05sse4(__m128 a);
__m128 Sleef_finz_cospif4_u05avx2128(__m128 a);

__m256 Sleef_cospif8_u05(__m256 a);
__m256 Sleef_cospif8_u05avx(__m256 a);
__m256 Sleef_cospif8_u05avx2(__m256 a);
__m256 Sleef_cospif8_u05fma4(__m256 a);
__m256 Sleef_cinz_cospif8_u05avx(__m256 a);
__m256 Sleef_finz_cospif8_u05avx2(__m256 a);
__m256 Sleef_finz_cospif8_u05fma4(__m256 a);

__m512 Sleef_cospif16_u05(__m512 a);
__m512 Sleef_cospif16_u05avx512f(__m512 a);
__m512 Sleef_cinz_cospif16_u05avx512fnofma(__m512 a);
__m512 Sleef_finz_cospif16_u05avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cospif_u05](../libm#sleef_cospif_u05) with the same accuracy specification.

### Vectorized double precision combined sine and cosine functions with 0.506 ULP error bound

```c
#include <sleef.h>

Sleef___m128d_2 Sleef_sincospid2_u05(__m128d a);
Sleef___m128d_2 Sleef_sincospid2_u05sse2(__m128d a);
Sleef___m128d_2 Sleef_sincospid2_u05sse4(__m128d a);
Sleef___m128d_2 Sleef_sincospid2_u05avx2128(__m128d a);
Sleef___m128d_2 Sleef_cinz_sincospid2_u05sse2(__m128d a);
Sleef___m128d_2 Sleef_cinz_sincospid2_u05sse4(__m128d a);
Sleef___m128d_2 Sleef_finz_sincospid2_u05avx2128(__m128d a);

Sleef___m256d_2 Sleef_sincospid4_u05(__m256d a);
Sleef___m256d_2 Sleef_sincospid4_u05avx(__m256d a);
Sleef___m256d_2 Sleef_sincospid4_u05avx2(__m256d a);
Sleef___m256d_2 Sleef_sincospid4_u05fma4(__m256d a);
Sleef___m256d_2 Sleef_cinz_sincospid4_u05avx(__m256d a);
Sleef___m256d_2 Sleef_finz_sincospid4_u05avx2(__m256d a);
Sleef___m256d_2 Sleef_finz_sincospid4_u05fma4(__m256d a);

Sleef___m512d_2 Sleef_sincospid8_u05(__m512d a);
Sleef___m512d_2 Sleef_sincospid8_u05avx512f(__m512d a);
Sleef___m512d_2 Sleef_cinz_sincospid8_u05avx512fnofma(__m512d a);
Sleef___m512d_2 Sleef_finz_sincospid8_u05avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sincospi_u05](../libm#sleef_sincospi_u05) with the same accuracy specification.

### Vectorized single precision combined sine and cosine functions with 0.506 ULP error bound

```c
#include <sleef.h>

Sleef___m128_2 Sleef_sincospif4_u05(__m128 a);
Sleef___m128_2 Sleef_sincospif4_u05sse2(__m128 a);
Sleef___m128_2 Sleef_sincospif4_u05sse4(__m128 a);
Sleef___m128_2 Sleef_sincospif4_u05avx2128(__m128 a);
Sleef___m128_2 Sleef_cinz_sincospif4_u05sse2(__m128 a);
Sleef___m128_2 Sleef_cinz_sincospif4_u05sse4(__m128 a);
Sleef___m128_2 Sleef_finz_sincospif4_u05avx2128(__m128 a);

Sleef___m256_2 Sleef_sincospif8_u05(__m256 a);
Sleef___m256_2 Sleef_sincospif8_u05avx(__m256 a);
Sleef___m256_2 Sleef_sincospif8_u05avx2(__m256 a);
Sleef___m256_2 Sleef_sincospif8_u05fma4(__m256 a);
Sleef___m256_2 Sleef_cinz_sincospif8_u05avx(__m256 a);
Sleef___m256_2 Sleef_finz_sincospif8_u05avx2(__m256 a);
Sleef___m256_2 Sleef_finz_sincospif8_u05fma4(__m256 a);

Sleef___m512_2 Sleef_sincospif16_u05(__m512 a);
Sleef___m512_2 Sleef_sincospif16_u05avx512f(__m512 a);
Sleef___m512_2 Sleef_cinz_sincospif16_u05avx512fnofma(__m512 a);
Sleef___m512_2 Sleef_finz_sincospif16_u05avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sincospif_u05](../libm#sleef_sincospif_u05) with the same accuracy specification.

### Vectorized double precision combined sine and cosine functions with 3.5 ULP error bound

```c
#include <sleef.h>

Sleef___m128d_2 Sleef_sincospid2_u35(__m128d a);
Sleef___m128d_2 Sleef_sincospid2_u35sse2(__m128d a);
Sleef___m128d_2 Sleef_sincospid2_u35sse4(__m128d a);
Sleef___m128d_2 Sleef_sincospid2_u35avx2128(__m128d a);
Sleef___m128d_2 Sleef_cinz_sincospid2_u35sse2(__m128d a);
Sleef___m128d_2 Sleef_cinz_sincospid2_u35sse4(__m128d a);
Sleef___m128d_2 Sleef_finz_sincospid2_u35avx2128(__m128d a);

Sleef___m256d_2 Sleef_sincospid4_u35(__m256d a);
Sleef___m256d_2 Sleef_sincospid4_u35avx(__m256d a);
Sleef___m256d_2 Sleef_sincospid4_u35avx2(__m256d a);
Sleef___m256d_2 Sleef_sincospid4_u35fma4(__m256d a);
Sleef___m256d_2 Sleef_cinz_sincospid4_u35avx(__m256d a);
Sleef___m256d_2 Sleef_finz_sincospid4_u35avx2(__m256d a);
Sleef___m256d_2 Sleef_finz_sincospid4_u35fma4(__m256d a);

Sleef___m512d_2 Sleef_sincospid8_u35(__m512d a);
Sleef___m512d_2 Sleef_sincospid8_u35avx512f(__m512d a);
Sleef___m512d_2 Sleef_cinz_sincospid8_u35avx512fnofma(__m512d a);
Sleef___m512d_2 Sleef_finz_sincospid8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sincospi_u35](../libm#sleef_sincospi_u35) with the same accuracy specification.

### Vectorized single precision combined sine and cosine functions with 3.5 ULP error bound

```c
#include <sleef.h>

Sleef___m128_2 Sleef_sincospif4_u35(__m128 a);
Sleef___m128_2 Sleef_sincospif4_u35sse2(__m128 a);
Sleef___m128_2 Sleef_sincospif4_u35sse4(__m128 a);
Sleef___m128_2 Sleef_sincospif4_u35avx2128(__m128 a);
Sleef___m128_2 Sleef_cinz_sincospif4_u35sse2(__m128 a);
Sleef___m128_2 Sleef_cinz_sincospif4_u35sse4(__m128 a);
Sleef___m128_2 Sleef_finz_sincospif4_u35avx2128(__m128 a);

Sleef___m256_2 Sleef_sincospif8_u35(__m256 a);
Sleef___m256_2 Sleef_sincospif8_u35avx(__m256 a);
Sleef___m256_2 Sleef_sincospif8_u35avx2(__m256 a);
Sleef___m256_2 Sleef_sincospif8_u35fma4(__m256 a);
Sleef___m256_2 Sleef_cinz_sincospif8_u35avx(__m256 a);
Sleef___m256_2 Sleef_finz_sincospif8_u35avx2(__m256 a);
Sleef___m256_2 Sleef_finz_sincospif8_u35fma4(__m256 a);

Sleef___m512_2 Sleef_sincospif16_u35(__m512 a);
Sleef___m512_2 Sleef_sincospif16_u35avx512f(__m512 a);
Sleef___m512_2 Sleef_cinz_sincospif16_u35avx512fnofma(__m512 a);
Sleef___m512_2 Sleef_finz_sincospif16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sincospif_u35](../libm#sleef_sincospif_u35) with the same accuracy specification.

### Vectorized double precision tangent functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_tand2_u10(__m128d a);
__m128d Sleef_tand2_u10sse2(__m128d a);
__m128d Sleef_tand2_u10sse4(__m128d a);
__m128d Sleef_tand2_u10avx2128(__m128d a);
__m128d Sleef_cinz_tand2_u10sse2(__m128d a);
__m128d Sleef_cinz_tand2_u10sse4(__m128d a);
__m128d Sleef_finz_tand2_u10avx2128(__m128d a);

__m256d Sleef_tand4_u10(__m256d a);
__m256d Sleef_tand4_u10avx(__m256d a);
__m256d Sleef_tand4_u10avx2(__m256d a);
__m256d Sleef_tand4_u10fma4(__m256d a);
__m256d Sleef_cinz_tand4_u10avx(__m256d a);
__m256d Sleef_finz_tand4_u10avx2(__m256d a);
__m256d Sleef_finz_tand4_u10fma4(__m256d a);

__m512d Sleef_tand8_u10(__m512d a);
__m512d Sleef_tand8_u10avx512f(__m512d a);
__m512d Sleef_cinz_tand8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_tand8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_tan_u10](../libm#sleef_tan_u10) with the same accuracy specification.

### Vectorized single precision tangent functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_tanf4_u10(__m128 a);
__m128 Sleef_tanf4_u10sse2(__m128 a);
__m128 Sleef_tanf4_u10sse4(__m128 a);
__m128 Sleef_tanf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_tanf4_u10sse2(__m128 a);
__m128 Sleef_cinz_tanf4_u10sse4(__m128 a);
__m128 Sleef_finz_tanf4_u10avx2128(__m128 a);

__m256 Sleef_tanf8_u10(__m256 a);
__m256 Sleef_tanf8_u10avx(__m256 a);
__m256 Sleef_tanf8_u10avx2(__m256 a);
__m256 Sleef_tanf8_u10fma4(__m256 a);
__m256 Sleef_cinz_tanf8_u10avx(__m256 a);
__m256 Sleef_finz_tanf8_u10avx2(__m256 a);
__m256 Sleef_finz_tanf8_u10fma4(__m256 a);

__m512 Sleef_tanf16_u10(__m512 a);
__m512 Sleef_tanf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_tanf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_tanf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_tanf_u10](../libm#sleef_tanf_u10) with the same accuracy specification.

### Vectorized double precision tangent functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_tand2_u35(__m128d a);
__m128d Sleef_tand2_u35sse2(__m128d a);
__m128d Sleef_tand2_u35sse4(__m128d a);
__m128d Sleef_tand2_u35avx2128(__m128d a);
__m128d Sleef_cinz_tand2_u35sse2(__m128d a);
__m128d Sleef_cinz_tand2_u35sse4(__m128d a);
__m128d Sleef_finz_tand2_u35avx2128(__m128d a);

__m256d Sleef_tand4_u35(__m256d a);
__m256d Sleef_tand4_u35avx(__m256d a);
__m256d Sleef_tand4_u35avx2(__m256d a);
__m256d Sleef_tand4_u35fma4(__m256d a);
__m256d Sleef_cinz_tand4_u35avx(__m256d a);
__m256d Sleef_finz_tand4_u35avx2(__m256d a);
__m256d Sleef_finz_tand4_u35fma4(__m256d a);

__m512d Sleef_tand8_u35(__m512d a);
__m512d Sleef_tand8_u35avx512f(__m512d a);
__m512d Sleef_cinz_tand8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_tand8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_tan_u35](../libm#sleef_tan_u35) with the same accuracy specification.

### Vectorized single precision tangent functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_tanf4_u35(__m128 a);
__m128 Sleef_tanf4_u35sse2(__m128 a);
__m128 Sleef_tanf4_u35sse4(__m128 a);
__m128 Sleef_tanf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_tanf4_u35sse2(__m128 a);
__m128 Sleef_cinz_tanf4_u35sse4(__m128 a);
__m128 Sleef_finz_tanf4_u35avx2128(__m128 a);

__m256 Sleef_tanf8_u35(__m256 a);
__m256 Sleef_tanf8_u35avx(__m256 a);
__m256 Sleef_tanf8_u35avx2(__m256 a);
__m256 Sleef_tanf8_u35fma4(__m256 a);
__m256 Sleef_cinz_tanf8_u35avx(__m256 a);
__m256 Sleef_finz_tanf8_u35avx2(__m256 a);
__m256 Sleef_finz_tanf8_u35fma4(__m256 a);

__m512 Sleef_tanf16_u35(__m512 a);
__m512 Sleef_tanf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_tanf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_tanf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_tanf_u35](../libm#sleef_tanf_u35) with the same accuracy specification.

<h2 id="pow">Power, exponential, and logarithmic functions</h2>

### Vectorized double precision power functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_powd2_u10(__m128d a, __m128d b);
__m128d Sleef_powd2_u10sse2(__m128d a, __m128d b);
__m128d Sleef_powd2_u10sse4(__m128d a, __m128d b);
__m128d Sleef_powd2_u10avx2128(__m128d a, __m128d b);
__m128d Sleef_cinz_powd2_u10sse2(__m128d a, __m128d b);
__m128d Sleef_cinz_powd2_u10sse4(__m128d a, __m128d b);
__m128d Sleef_finz_powd2_u10avx2128(__m128d a, __m128d b);

__m256d Sleef_powd4_u10(__m256d a, __m256d b);
__m256d Sleef_powd4_u10avx(__m256d a, __m256d b);
__m256d Sleef_powd4_u10avx2(__m256d a, __m256d b);
__m256d Sleef_powd4_u10fma4(__m256d a, __m256d b);
__m256d Sleef_cinz_powd4_u10avx(__m256d a, __m256d b);
__m256d Sleef_finz_powd4_u10avx2(__m256d a, __m256d b);
__m256d Sleef_finz_powd4_u10fma4(__m256d a, __m256d b);

__m512d Sleef_powd8_u10(__m512d a, __m512d b);
__m512d Sleef_powd8_u10avx512f(__m512d a, __m512d b);
__m512d Sleef_cinz_powd8_u10avx512fnofma(__m512d a, __m512d b);
__m512d Sleef_finz_powd8_u10avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_pow_u10](../libm#sleef_pow_u10) with the same accuracy specification.

### Vectorized single precision power functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_powf4_u10(__m128 a, __m128 b);
__m128 Sleef_powf4_u10sse2(__m128 a, __m128 b);
__m128 Sleef_powf4_u10sse4(__m128 a, __m128 b);
__m128 Sleef_powf4_u10avx2128(__m128 a, __m128 b);
__m128 Sleef_cinz_powf4_u10sse2(__m128 a, __m128 b);
__m128 Sleef_cinz_powf4_u10sse4(__m128 a, __m128 b);
__m128 Sleef_finz_powf4_u10avx2128(__m128 a, __m128 b);

__m256 Sleef_powf8_u10(__m256 a, __m256 b);
__m256 Sleef_powf8_u10avx(__m256 a, __m256 b);
__m256 Sleef_powf8_u10avx2(__m256 a, __m256 b);
__m256 Sleef_powf8_u10fma4(__m256 a, __m256 b);
__m256 Sleef_cinz_powf8_u10avx(__m256 a, __m256 b);
__m256 Sleef_finz_powf8_u10avx2(__m256 a, __m256 b);
__m256 Sleef_finz_powf8_u10fma4(__m256 a, __m256 b);

__m512 Sleef_powf16_u10(__m512 a, __m512 b);
__m512 Sleef_powf16_u10avx512f(__m512 a, __m512 b);
__m512 Sleef_cinz_powf16_u10avx512fnofma(__m512 a, __m512 b);
__m512 Sleef_finz_powf16_u10avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_powf_u10](../libm#sleef_powf_u10) with the same accuracy specification.

### Vectorized double precision natural logarithmic functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_logd2_u10(__m128d a);
__m128d Sleef_logd2_u10sse2(__m128d a);
__m128d Sleef_logd2_u10sse4(__m128d a);
__m128d Sleef_logd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_logd2_u10sse2(__m128d a);
__m128d Sleef_cinz_logd2_u10sse4(__m128d a);
__m128d Sleef_finz_logd2_u10avx2128(__m128d a);

__m256d Sleef_logd4_u10(__m256d a);
__m256d Sleef_logd4_u10avx(__m256d a);
__m256d Sleef_logd4_u10avx2(__m256d a);
__m256d Sleef_logd4_u10fma4(__m256d a);
__m256d Sleef_cinz_logd4_u10avx(__m256d a);
__m256d Sleef_finz_logd4_u10avx2(__m256d a);
__m256d Sleef_finz_logd4_u10fma4(__m256d a);

__m512d Sleef_logd8_u10(__m512d a);
__m512d Sleef_logd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_logd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_logd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_log_u10](../libm#sleef_log_u10) with the same accuracy specification.

### Vectorized single precision natural logarithmic functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_logf4_u10(__m128 a);
__m128 Sleef_logf4_u10sse2(__m128 a);
__m128 Sleef_logf4_u10sse4(__m128 a);
__m128 Sleef_logf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_logf4_u10sse2(__m128 a);
__m128 Sleef_cinz_logf4_u10sse4(__m128 a);
__m128 Sleef_finz_logf4_u10avx2128(__m128 a);

__m256 Sleef_logf8_u10(__m256 a);
__m256 Sleef_logf8_u10avx(__m256 a);
__m256 Sleef_logf8_u10avx2(__m256 a);
__m256 Sleef_logf8_u10fma4(__m256 a);
__m256 Sleef_cinz_logf8_u10avx(__m256 a);
__m256 Sleef_finz_logf8_u10avx2(__m256 a);
__m256 Sleef_finz_logf8_u10fma4(__m256 a);

__m512 Sleef_logf16_u10(__m512 a);
__m512 Sleef_logf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_logf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_logf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_logf_u10](../libm#sleef_logf_u10) with the same accuracy specification.

### Vectorized double precision natural logarithmic functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_logd2_u35(__m128d a);
__m128d Sleef_logd2_u35sse2(__m128d a);
__m128d Sleef_logd2_u35sse4(__m128d a);
__m128d Sleef_logd2_u35avx2128(__m128d a);
__m128d Sleef_cinz_logd2_u35sse2(__m128d a);
__m128d Sleef_cinz_logd2_u35sse4(__m128d a);
__m128d Sleef_finz_logd2_u35avx2128(__m128d a);

__m256d Sleef_logd4_u35(__m256d a);
__m256d Sleef_logd4_u35avx(__m256d a);
__m256d Sleef_logd4_u35avx2(__m256d a);
__m256d Sleef_logd4_u35fma4(__m256d a);
__m256d Sleef_cinz_logd4_u35avx(__m256d a);
__m256d Sleef_finz_logd4_u35avx2(__m256d a);
__m256d Sleef_finz_logd4_u35fma4(__m256d a);

__m512d Sleef_logd8_u35(__m512d a);
__m512d Sleef_logd8_u35avx512f(__m512d a);
__m512d Sleef_cinz_logd8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_logd8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_log_u35](../libm#sleef_log_u35) with the same accuracy specification.

### Vectorized single precision natural logarithmic functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_logf4_u35(__m128 a);
__m128 Sleef_logf4_u35sse2(__m128 a);
__m128 Sleef_logf4_u35sse4(__m128 a);
__m128 Sleef_logf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_logf4_u35sse2(__m128 a);
__m128 Sleef_cinz_logf4_u35sse4(__m128 a);
__m128 Sleef_finz_logf4_u35avx2128(__m128 a);

__m256 Sleef_logf8_u35(__m256 a);
__m256 Sleef_logf8_u35avx(__m256 a);
__m256 Sleef_logf8_u35avx2(__m256 a);
__m256 Sleef_logf8_u35fma4(__m256 a);
__m256 Sleef_cinz_logf8_u35avx(__m256 a);
__m256 Sleef_finz_logf8_u35avx2(__m256 a);
__m256 Sleef_finz_logf8_u35fma4(__m256 a);

__m512 Sleef_logf16_u35(__m512 a);
__m512 Sleef_logf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_logf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_logf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_logf_u35](../libm#sleef_logf_u35) with the same accuracy specification.

### Vectorized double precision base-10 logarithmic functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_log10d2_u10(__m128d a);
__m128d Sleef_log10d2_u10sse2(__m128d a);
__m128d Sleef_log10d2_u10sse4(__m128d a);
__m128d Sleef_log10d2_u10avx2128(__m128d a);
__m128d Sleef_cinz_log10d2_u10sse2(__m128d a);
__m128d Sleef_cinz_log10d2_u10sse4(__m128d a);
__m128d Sleef_finz_log10d2_u10avx2128(__m128d a);

__m256d Sleef_log10d4_u10(__m256d a);
__m256d Sleef_log10d4_u10avx(__m256d a);
__m256d Sleef_log10d4_u10avx2(__m256d a);
__m256d Sleef_log10d4_u10fma4(__m256d a);
__m256d Sleef_cinz_log10d4_u10avx(__m256d a);
__m256d Sleef_finz_log10d4_u10avx2(__m256d a);
__m256d Sleef_finz_log10d4_u10fma4(__m256d a);

__m512d Sleef_log10d8_u10(__m512d a);
__m512d Sleef_log10d8_u10avx512f(__m512d a);
__m512d Sleef_cinz_log10d8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_log10d8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_log10_u10](../libm#sleef_log10_u10) with the same accuracy specification.

### Vectorized single precision base-10 logarithmic functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_log10f4_u10(__m128 a);
__m128 Sleef_log10f4_u10sse2(__m128 a);
__m128 Sleef_log10f4_u10sse4(__m128 a);
__m128 Sleef_log10f4_u10avx2128(__m128 a);
__m128 Sleef_cinz_log10f4_u10sse2(__m128 a);
__m128 Sleef_cinz_log10f4_u10sse4(__m128 a);
__m128 Sleef_finz_log10f4_u10avx2128(__m128 a);

__m256 Sleef_log10f8_u10(__m256 a);
__m256 Sleef_log10f8_u10avx(__m256 a);
__m256 Sleef_log10f8_u10avx2(__m256 a);
__m256 Sleef_log10f8_u10fma4(__m256 a);
__m256 Sleef_cinz_log10f8_u10avx(__m256 a);
__m256 Sleef_finz_log10f8_u10avx2(__m256 a);
__m256 Sleef_finz_log10f8_u10fma4(__m256 a);

__m512 Sleef_log10f16_u10(__m512 a);
__m512 Sleef_log10f16_u10avx512f(__m512 a);
__m512 Sleef_cinz_log10f16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_log10f16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_log10f_u10](../libm#sleef_log10f_u10) with the same accuracy specification.

### Vectorized double precision base-2 logarithmic functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_log2d2_u10(__m128d a);
__m128d Sleef_log2d2_u10sse2(__m128d a);
__m128d Sleef_log2d2_u10sse4(__m128d a);
__m128d Sleef_log2d2_u10avx2128(__m128d a);
__m128d Sleef_cinz_log2d2_u10sse2(__m128d a);
__m128d Sleef_cinz_log2d2_u10sse4(__m128d a);
__m128d Sleef_finz_log2d2_u10avx2128(__m128d a);

__m256d Sleef_log2d4_u10(__m256d a);
__m256d Sleef_log2d4_u10avx(__m256d a);
__m256d Sleef_log2d4_u10avx2(__m256d a);
__m256d Sleef_log2d4_u10fma4(__m256d a);
__m256d Sleef_cinz_log2d4_u10avx(__m256d a);
__m256d Sleef_finz_log2d4_u10avx2(__m256d a);
__m256d Sleef_finz_log2d4_u10fma4(__m256d a);

__m512d Sleef_log2d8_u10(__m512d a);
__m512d Sleef_log2d8_u10avx512f(__m512d a);
__m512d Sleef_cinz_log2d8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_log2d8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_log2_u10](../libm#sleef_log2_u10) with the same accuracy specification.

### Vectorized single precision base-2 logarithmic functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_log2f4_u10(__m128 a);
__m128 Sleef_log2f4_u10sse2(__m128 a);
__m128 Sleef_log2f4_u10sse4(__m128 a);
__m128 Sleef_log2f4_u10avx2128(__m128 a);
__m128 Sleef_cinz_log2f4_u10sse2(__m128 a);
__m128 Sleef_cinz_log2f4_u10sse4(__m128 a);
__m128 Sleef_finz_log2f4_u10avx2128(__m128 a);

__m256 Sleef_log2f8_u10(__m256 a);
__m256 Sleef_log2f8_u10avx(__m256 a);
__m256 Sleef_log2f8_u10avx2(__m256 a);
__m256 Sleef_log2f8_u10fma4(__m256 a);
__m256 Sleef_cinz_log2f8_u10avx(__m256 a);
__m256 Sleef_finz_log2f8_u10avx2(__m256 a);
__m256 Sleef_finz_log2f8_u10fma4(__m256 a);

__m512 Sleef_log2f16_u10(__m512 a);
__m512 Sleef_log2f16_u10avx512f(__m512 a);
__m512 Sleef_cinz_log2f16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_log2f16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_log2f_u10](../libm#sleef_log2f_u10) with the same accuracy specification.

### Vectorized double precision logarithm of one plus argument with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_log1pd2_u10(__m128d a);
__m128d Sleef_log1pd2_u10sse2(__m128d a);
__m128d Sleef_log1pd2_u10sse4(__m128d a);
__m128d Sleef_log1pd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_log1pd2_u10sse2(__m128d a);
__m128d Sleef_cinz_log1pd2_u10sse4(__m128d a);
__m128d Sleef_finz_log1pd2_u10avx2128(__m128d a);

__m256d Sleef_log1pd4_u10(__m256d a);
__m256d Sleef_log1pd4_u10avx(__m256d a);
__m256d Sleef_log1pd4_u10avx2(__m256d a);
__m256d Sleef_log1pd4_u10fma4(__m256d a);
__m256d Sleef_cinz_log1pd4_u10avx(__m256d a);
__m256d Sleef_finz_log1pd4_u10avx2(__m256d a);
__m256d Sleef_finz_log1pd4_u10fma4(__m256d a);

__m512d Sleef_log1pd8_u10(__m512d a);
__m512d Sleef_log1pd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_log1pd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_log1pd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_log1p_u10](../libm#sleef_log1p_u10) with the same accuracy specification.

### Vectorized single precision logarithm of one plus argument with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_log1pf4_u10(__m128 a);
__m128 Sleef_log1pf4_u10sse2(__m128 a);
__m128 Sleef_log1pf4_u10sse4(__m128 a);
__m128 Sleef_log1pf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_log1pf4_u10sse2(__m128 a);
__m128 Sleef_cinz_log1pf4_u10sse4(__m128 a);
__m128 Sleef_finz_log1pf4_u10avx2128(__m128 a);

__m256 Sleef_log1pf8_u10(__m256 a);
__m256 Sleef_log1pf8_u10avx(__m256 a);
__m256 Sleef_log1pf8_u10avx2(__m256 a);
__m256 Sleef_log1pf8_u10fma4(__m256 a);
__m256 Sleef_cinz_log1pf8_u10avx(__m256 a);
__m256 Sleef_finz_log1pf8_u10avx2(__m256 a);
__m256 Sleef_finz_log1pf8_u10fma4(__m256 a);

__m512 Sleef_log1pf16_u10(__m512 a);
__m512 Sleef_log1pf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_log1pf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_log1pf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_log1pf_u10](../libm#sleef_log1pf_u10) with the same accuracy specification.

### Vectorized double precision base-<i>e</i> exponential functions functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_expd2_u10(__m128d a);
__m128d Sleef_expd2_u10sse2(__m128d a);
__m128d Sleef_expd2_u10sse4(__m128d a);
__m128d Sleef_expd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_expd2_u10sse2(__m128d a);
__m128d Sleef_cinz_expd2_u10sse4(__m128d a);
__m128d Sleef_finz_expd2_u10avx2128(__m128d a);

__m256d Sleef_expd4_u10(__m256d a);
__m256d Sleef_expd4_u10avx(__m256d a);
__m256d Sleef_expd4_u10avx2(__m256d a);
__m256d Sleef_expd4_u10fma4(__m256d a);
__m256d Sleef_cinz_expd4_u10avx(__m256d a);
__m256d Sleef_finz_expd4_u10avx2(__m256d a);
__m256d Sleef_finz_expd4_u10fma4(__m256d a);

__m512d Sleef_expd8_u10(__m512d a);
__m512d Sleef_expd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_expd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_expd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_exp_u10](../libm#sleef_exp_u10) with the same accuracy specification.

### Vectorized single precision base-<i>e</i> exponential functions functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_expf4_u10(__m128 a);
__m128 Sleef_expf4_u10sse2(__m128 a);
__m128 Sleef_expf4_u10sse4(__m128 a);
__m128 Sleef_expf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_expf4_u10sse2(__m128 a);
__m128 Sleef_cinz_expf4_u10sse4(__m128 a);
__m128 Sleef_finz_expf4_u10avx2128(__m128 a);

__m256 Sleef_expf8_u10(__m256 a);
__m256 Sleef_expf8_u10avx(__m256 a);
__m256 Sleef_expf8_u10avx2(__m256 a);
__m256 Sleef_expf8_u10fma4(__m256 a);
__m256 Sleef_cinz_expf8_u10avx(__m256 a);
__m256 Sleef_finz_expf8_u10avx2(__m256 a);
__m256 Sleef_finz_expf8_u10fma4(__m256 a);

__m512 Sleef_expf16_u10(__m512 a);
__m512 Sleef_expf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_expf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_expf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_expf_u10](../libm#sleef_expf_u10) with the same accuracy specification.

### Vectorized double precision base-<i>2</i> exponential functions functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_exp2d2_u10(__m128d a);
__m128d Sleef_exp2d2_u10sse2(__m128d a);
__m128d Sleef_exp2d2_u10sse4(__m128d a);
__m128d Sleef_exp2d2_u10avx2128(__m128d a);
__m128d Sleef_cinz_exp2d2_u10sse2(__m128d a);
__m128d Sleef_cinz_exp2d2_u10sse4(__m128d a);
__m128d Sleef_finz_exp2d2_u10avx2128(__m128d a);

__m256d Sleef_exp2d4_u10(__m256d a);
__m256d Sleef_exp2d4_u10avx(__m256d a);
__m256d Sleef_exp2d4_u10avx2(__m256d a);
__m256d Sleef_exp2d4_u10fma4(__m256d a);
__m256d Sleef_cinz_exp2d4_u10avx(__m256d a);
__m256d Sleef_finz_exp2d4_u10avx2(__m256d a);
__m256d Sleef_finz_exp2d4_u10fma4(__m256d a);

__m512d Sleef_exp2d8_u10(__m512d a);
__m512d Sleef_exp2d8_u10avx512f(__m512d a);
__m512d Sleef_cinz_exp2d8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_exp2d8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_exp2_u10](../libm#sleef_exp2_u10) with the same accuracy specification.

### Vectorized single precision base-<i>2</i> exponential functions functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_exp2f4_u10(__m128 a);
__m128 Sleef_exp2f4_u10sse2(__m128 a);
__m128 Sleef_exp2f4_u10sse4(__m128 a);
__m128 Sleef_exp2f4_u10avx2128(__m128 a);
__m128 Sleef_cinz_exp2f4_u10sse2(__m128 a);
__m128 Sleef_cinz_exp2f4_u10sse4(__m128 a);
__m128 Sleef_finz_exp2f4_u10avx2128(__m128 a);

__m256 Sleef_exp2f8_u10(__m256 a);
__m256 Sleef_exp2f8_u10avx(__m256 a);
__m256 Sleef_exp2f8_u10avx2(__m256 a);
__m256 Sleef_exp2f8_u10fma4(__m256 a);
__m256 Sleef_cinz_exp2f8_u10avx(__m256 a);
__m256 Sleef_finz_exp2f8_u10avx2(__m256 a);
__m256 Sleef_finz_exp2f8_u10fma4(__m256 a);

__m512 Sleef_exp2f16_u10(__m512 a);
__m512 Sleef_exp2f16_u10avx512f(__m512 a);
__m512 Sleef_cinz_exp2f16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_exp2f16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_exp2f_u10](../libm#sleef_exp2f_u10) with the same accuracy specification.

### Vectorized double precision base-10 exponential functions functions with 1.09 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_exp10d2_u10(__m128d a);
__m128d Sleef_exp10d2_u10sse2(__m128d a);
__m128d Sleef_exp10d2_u10sse4(__m128d a);
__m128d Sleef_exp10d2_u10avx2128(__m128d a);
__m128d Sleef_cinz_exp10d2_u10sse2(__m128d a);
__m128d Sleef_cinz_exp10d2_u10sse4(__m128d a);
__m128d Sleef_finz_exp10d2_u10avx2128(__m128d a);

__m256d Sleef_exp10d4_u10(__m256d a);
__m256d Sleef_exp10d4_u10avx(__m256d a);
__m256d Sleef_exp10d4_u10avx2(__m256d a);
__m256d Sleef_exp10d4_u10fma4(__m256d a);
__m256d Sleef_cinz_exp10d4_u10avx(__m256d a);
__m256d Sleef_finz_exp10d4_u10avx2(__m256d a);
__m256d Sleef_finz_exp10d4_u10fma4(__m256d a);

__m512d Sleef_exp10d8_u10(__m512d a);
__m512d Sleef_exp10d8_u10avx512f(__m512d a);
__m512d Sleef_cinz_exp10d8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_exp10d8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_exp10_u10](../libm#sleef_exp10_u10) with the same accuracy specification.

### Vectorized single precision base-10 exponential functions functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_exp10f4_u10(__m128 a);
__m128 Sleef_exp10f4_u10sse2(__m128 a);
__m128 Sleef_exp10f4_u10sse4(__m128 a);
__m128 Sleef_exp10f4_u10avx2128(__m128 a);
__m128 Sleef_cinz_exp10f4_u10sse2(__m128 a);
__m128 Sleef_cinz_exp10f4_u10sse4(__m128 a);
__m128 Sleef_finz_exp10f4_u10avx2128(__m128 a);

__m256 Sleef_exp10f8_u10(__m256 a);
__m256 Sleef_exp10f8_u10avx(__m256 a);
__m256 Sleef_exp10f8_u10avx2(__m256 a);
__m256 Sleef_exp10f8_u10fma4(__m256 a);
__m256 Sleef_cinz_exp10f8_u10avx(__m256 a);
__m256 Sleef_finz_exp10f8_u10avx2(__m256 a);
__m256 Sleef_finz_exp10f8_u10fma4(__m256 a);

__m512 Sleef_exp10f16_u10(__m512 a);
__m512 Sleef_exp10f16_u10avx512f(__m512 a);
__m512 Sleef_cinz_exp10f16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_exp10f16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_exp10f_u10](../libm#sleef_exp10f_u10) with the same accuracy specification.

### Vectorized double precision base-<i>e</i> exponential functions minus 1 with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_expm1d2_u10(__m128d a);
__m128d Sleef_expm1d2_u10sse2(__m128d a);
__m128d Sleef_expm1d2_u10sse4(__m128d a);
__m128d Sleef_expm1d2_u10avx2128(__m128d a);
__m128d Sleef_cinz_expm1d2_u10sse2(__m128d a);
__m128d Sleef_cinz_expm1d2_u10sse4(__m128d a);
__m128d Sleef_finz_expm1d2_u10avx2128(__m128d a);

__m256d Sleef_expm1d4_u10(__m256d a);
__m256d Sleef_expm1d4_u10avx(__m256d a);
__m256d Sleef_expm1d4_u10avx2(__m256d a);
__m256d Sleef_expm1d4_u10fma4(__m256d a);
__m256d Sleef_cinz_expm1d4_u10avx(__m256d a);
__m256d Sleef_finz_expm1d4_u10avx2(__m256d a);
__m256d Sleef_finz_expm1d4_u10fma4(__m256d a);

__m512d Sleef_expm1d8_u10(__m512d a);
__m512d Sleef_expm1d8_u10avx512f(__m512d a);
__m512d Sleef_cinz_expm1d8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_expm1d8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_expm1_u10](../libm#sleef_expm1_u10) with the same accuracy specification.

### Vectorized single precision base-<i>e</i> exponential functions minus 1 with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_expm1f4_u10(__m128 a);
__m128 Sleef_expm1f4_u10sse2(__m128 a);
__m128 Sleef_expm1f4_u10sse4(__m128 a);
__m128 Sleef_expm1f4_u10avx2128(__m128 a);
__m128 Sleef_cinz_expm1f4_u10sse2(__m128 a);
__m128 Sleef_cinz_expm1f4_u10sse4(__m128 a);
__m128 Sleef_finz_expm1f4_u10avx2128(__m128 a);

__m256 Sleef_expm1f8_u10(__m256 a);
__m256 Sleef_expm1f8_u10avx(__m256 a);
__m256 Sleef_expm1f8_u10avx2(__m256 a);
__m256 Sleef_expm1f8_u10fma4(__m256 a);
__m256 Sleef_cinz_expm1f8_u10avx(__m256 a);
__m256 Sleef_finz_expm1f8_u10avx2(__m256 a);
__m256 Sleef_finz_expm1f8_u10fma4(__m256 a);

__m512 Sleef_expm1f16_u10(__m512 a);
__m512 Sleef_expm1f16_u10avx512f(__m512 a);
__m512 Sleef_cinz_expm1f16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_expm1f16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_expm1f_u10](../libm#sleef_expm1f_u10) with the same accuracy specification.

### Vectorized double precision square root functions with 0.5001 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_sqrtd2_u05(__m128d a);
__m128d Sleef_sqrtd2_u05sse2(__m128d a);
__m128d Sleef_sqrtd2_u05sse4(__m128d a);
__m128d Sleef_sqrtd2_u05avx2128(__m128d a);
__m128d Sleef_cinz_sqrtd2_u05sse2(__m128d a);
__m128d Sleef_cinz_sqrtd2_u05sse4(__m128d a);
__m128d Sleef_finz_sqrtd2_u05avx2128(__m128d a);

__m256d Sleef_sqrtd4_u05(__m256d a);
__m256d Sleef_sqrtd4_u05avx(__m256d a);
__m256d Sleef_sqrtd4_u05avx2(__m256d a);
__m256d Sleef_sqrtd4_u05fma4(__m256d a);
__m256d Sleef_cinz_sqrtd4_u05avx(__m256d a);
__m256d Sleef_finz_sqrtd4_u05avx2(__m256d a);
__m256d Sleef_finz_sqrtd4_u05fma4(__m256d a);

__m512d Sleef_sqrtd8_u05(__m512d a);
__m512d Sleef_sqrtd8_u05avx512f(__m512d a);
__m512d Sleef_cinz_sqrtd8_u05avx512fnofma(__m512d a);
__m512d Sleef_finz_sqrtd8_u05avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sqrt_u05](../libm#sleef_sqrt_u05) with the same accuracy specification.

### Vectorized single precision square root functions with 0.5001 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_sqrtf4_u05(__m128 a);
__m128 Sleef_sqrtf4_u05sse2(__m128 a);
__m128 Sleef_sqrtf4_u05sse4(__m128 a);
__m128 Sleef_sqrtf4_u05avx2128(__m128 a);
__m128 Sleef_cinz_sqrtf4_u05sse2(__m128 a);
__m128 Sleef_cinz_sqrtf4_u05sse4(__m128 a);
__m128 Sleef_finz_sqrtf4_u05avx2128(__m128 a);

__m256 Sleef_sqrtf8_u05(__m256 a);
__m256 Sleef_sqrtf8_u05avx(__m256 a);
__m256 Sleef_sqrtf8_u05avx2(__m256 a);
__m256 Sleef_sqrtf8_u05fma4(__m256 a);
__m256 Sleef_cinz_sqrtf8_u05avx(__m256 a);
__m256 Sleef_finz_sqrtf8_u05avx2(__m256 a);
__m256 Sleef_finz_sqrtf8_u05fma4(__m256 a);

__m512 Sleef_sqrtf16_u05(__m512 a);
__m512 Sleef_sqrtf16_u05avx512f(__m512 a);
__m512 Sleef_cinz_sqrtf16_u05avx512fnofma(__m512 a);
__m512 Sleef_finz_sqrtf16_u05avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sqrtf_u05](../libm#sleef_sqrtf_u05) with the same accuracy specification.

### Vectorized double precision square root functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_sqrtd2_u35(__m128d a);
__m128d Sleef_sqrtd2_u35sse2(__m128d a);
__m128d Sleef_sqrtd2_u35sse4(__m128d a);
__m128d Sleef_sqrtd2_u35avx2128(__m128d a);
__m128d Sleef_cinz_sqrtd2_u35sse2(__m128d a);
__m128d Sleef_cinz_sqrtd2_u35sse4(__m128d a);
__m128d Sleef_finz_sqrtd2_u35avx2128(__m128d a);

__m256d Sleef_sqrtd4_u35(__m256d a);
__m256d Sleef_sqrtd4_u35avx(__m256d a);
__m256d Sleef_sqrtd4_u35avx2(__m256d a);
__m256d Sleef_sqrtd4_u35fma4(__m256d a);
__m256d Sleef_cinz_sqrtd4_u35avx(__m256d a);
__m256d Sleef_finz_sqrtd4_u35avx2(__m256d a);
__m256d Sleef_finz_sqrtd4_u35fma4(__m256d a);

__m512d Sleef_sqrtd8_u35(__m512d a);
__m512d Sleef_sqrtd8_u35avx512f(__m512d a);
__m512d Sleef_cinz_sqrtd8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_sqrtd8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sqrt_u35](../libm#sleef_sqrt_u35) with the same accuracy specification.

### Vectorized single precision square root functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_sqrtf4_u35(__m128 a);
__m128 Sleef_sqrtf4_u35sse2(__m128 a);
__m128 Sleef_sqrtf4_u35sse4(__m128 a);
__m128 Sleef_sqrtf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_sqrtf4_u35sse2(__m128 a);
__m128 Sleef_cinz_sqrtf4_u35sse4(__m128 a);
__m128 Sleef_finz_sqrtf4_u35avx2128(__m128 a);

__m256 Sleef_sqrtf8_u35(__m256 a);
__m256 Sleef_sqrtf8_u35avx(__m256 a);
__m256 Sleef_sqrtf8_u35avx2(__m256 a);
__m256 Sleef_sqrtf8_u35fma4(__m256 a);
__m256 Sleef_cinz_sqrtf8_u35avx(__m256 a);
__m256 Sleef_finz_sqrtf8_u35avx2(__m256 a);
__m256 Sleef_finz_sqrtf8_u35fma4(__m256 a);

__m512 Sleef_sqrtf16_u35(__m512 a);
__m512 Sleef_sqrtf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_sqrtf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_sqrtf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sqrtf_u35](../libm#sleef_sqrtf_u35) with the same accuracy specification.

### Vectorized double precision cubic root functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_cbrtd2_u10(__m128d a);
__m128d Sleef_cbrtd2_u10sse2(__m128d a);
__m128d Sleef_cbrtd2_u10sse4(__m128d a);
__m128d Sleef_cbrtd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_cbrtd2_u10sse2(__m128d a);
__m128d Sleef_cinz_cbrtd2_u10sse4(__m128d a);
__m128d Sleef_finz_cbrtd2_u10avx2128(__m128d a);

__m256d Sleef_cbrtd4_u10(__m256d a);
__m256d Sleef_cbrtd4_u10avx(__m256d a);
__m256d Sleef_cbrtd4_u10avx2(__m256d a);
__m256d Sleef_cbrtd4_u10fma4(__m256d a);
__m256d Sleef_cinz_cbrtd4_u10avx(__m256d a);
__m256d Sleef_finz_cbrtd4_u10avx2(__m256d a);
__m256d Sleef_finz_cbrtd4_u10fma4(__m256d a);

__m512d Sleef_cbrtd8_u10(__m512d a);
__m512d Sleef_cbrtd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_cbrtd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_cbrtd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cbrt_u10](../libm#sleef_cbrt_u10) with the same accuracy specification.

### Vectorized single precision cubic root functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_cbrtf4_u10(__m128 a);
__m128 Sleef_cbrtf4_u10sse2(__m128 a);
__m128 Sleef_cbrtf4_u10sse4(__m128 a);
__m128 Sleef_cbrtf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_cbrtf4_u10sse2(__m128 a);
__m128 Sleef_cinz_cbrtf4_u10sse4(__m128 a);
__m128 Sleef_finz_cbrtf4_u10avx2128(__m128 a);

__m256 Sleef_cbrtf8_u10(__m256 a);
__m256 Sleef_cbrtf8_u10avx(__m256 a);
__m256 Sleef_cbrtf8_u10avx2(__m256 a);
__m256 Sleef_cbrtf8_u10fma4(__m256 a);
__m256 Sleef_cinz_cbrtf8_u10avx(__m256 a);
__m256 Sleef_finz_cbrtf8_u10avx2(__m256 a);
__m256 Sleef_finz_cbrtf8_u10fma4(__m256 a);

__m512 Sleef_cbrtf16_u10(__m512 a);
__m512 Sleef_cbrtf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_cbrtf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_cbrtf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cbrtf_u10](../libm#sleef_cbrtf_u10) with the same accuracy specification.

### Vectorized double precision cubic root functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_cbrtd2_u35(__m128d a);
__m128d Sleef_cbrtd2_u35sse2(__m128d a);
__m128d Sleef_cbrtd2_u35sse4(__m128d a);
__m128d Sleef_cbrtd2_u35avx2128(__m128d a);
__m128d Sleef_cinz_cbrtd2_u35sse2(__m128d a);
__m128d Sleef_cinz_cbrtd2_u35sse4(__m128d a);
__m128d Sleef_finz_cbrtd2_u35avx2128(__m128d a);

__m256d Sleef_cbrtd4_u35(__m256d a);
__m256d Sleef_cbrtd4_u35avx(__m256d a);
__m256d Sleef_cbrtd4_u35avx2(__m256d a);
__m256d Sleef_cbrtd4_u35fma4(__m256d a);
__m256d Sleef_cinz_cbrtd4_u35avx(__m256d a);
__m256d Sleef_finz_cbrtd4_u35avx2(__m256d a);
__m256d Sleef_finz_cbrtd4_u35fma4(__m256d a);

__m512d Sleef_cbrtd8_u35(__m512d a);
__m512d Sleef_cbrtd8_u35avx512f(__m512d a);
__m512d Sleef_cinz_cbrtd8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_cbrtd8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cbrt_u35](../libm#sleef_cbrt_u35) with the same accuracy specification.

### Vectorized single precision cubic root functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_cbrtf4_u35(__m128 a);
__m128 Sleef_cbrtf4_u35sse2(__m128 a);
__m128 Sleef_cbrtf4_u35sse4(__m128 a);
__m128 Sleef_cbrtf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_cbrtf4_u35sse2(__m128 a);
__m128 Sleef_cinz_cbrtf4_u35sse4(__m128 a);
__m128 Sleef_finz_cbrtf4_u35avx2128(__m128 a);

__m256 Sleef_cbrtf8_u35(__m256 a);
__m256 Sleef_cbrtf8_u35avx(__m256 a);
__m256 Sleef_cbrtf8_u35avx2(__m256 a);
__m256 Sleef_cbrtf8_u35fma4(__m256 a);
__m256 Sleef_cinz_cbrtf8_u35avx(__m256 a);
__m256 Sleef_finz_cbrtf8_u35avx2(__m256 a);
__m256 Sleef_finz_cbrtf8_u35fma4(__m256 a);

__m512 Sleef_cbrtf16_u35(__m512 a);
__m512 Sleef_cbrtf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_cbrtf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_cbrtf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cbrtf_u35](../libm#sleef_cbrtf_u35) with the same accuracy specification.

### Vectorized double precision 2D Euclidian distance functions with 0.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_hypotd2_u05(__m128d a, __m128d b);
__m128d Sleef_hypotd2_u05sse2(__m128d a, __m128d b);
__m128d Sleef_hypotd2_u05sse4(__m128d a, __m128d b);
__m128d Sleef_hypotd2_u05avx2128(__m128d a, __m128d b);
__m128d Sleef_cinz_hypotd2_u05sse2(__m128d a, __m128d b);
__m128d Sleef_cinz_hypotd2_u05sse4(__m128d a, __m128d b);
__m128d Sleef_finz_hypotd2_u05avx2128(__m128d a, __m128d b);

__m256d Sleef_hypotd4_u05(__m256d a, __m256d b);
__m256d Sleef_hypotd4_u05avx(__m256d a, __m256d b);
__m256d Sleef_hypotd4_u05avx2(__m256d a, __m256d b);
__m256d Sleef_hypotd4_u05fma4(__m256d a, __m256d b);
__m256d Sleef_cinz_hypotd4_u05avx(__m256d a, __m256d b);
__m256d Sleef_finz_hypotd4_u05avx2(__m256d a, __m256d b);
__m256d Sleef_finz_hypotd4_u05fma4(__m256d a, __m256d b);

__m512d Sleef_hypotd8_u05(__m512d a, __m512d b);
__m512d Sleef_hypotd8_u05avx512f(__m512d a, __m512d b);
__m512d Sleef_cinz_hypotd8_u05avx512fnofma(__m512d a, __m512d b);
__m512d Sleef_finz_hypotd8_u05avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_hypot_u05](../libm#sleef_hypot_u05) with the same accuracy specification.

### Vectorized single precision 2D Euclidian distance functions with 0.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_hypotf4_u05(__m128 a, __m128 b);
__m128 Sleef_hypotf4_u05sse2(__m128 a, __m128 b);
__m128 Sleef_hypotf4_u05sse4(__m128 a, __m128 b);
__m128 Sleef_hypotf4_u05avx2128(__m128 a, __m128 b);
__m128 Sleef_cinz_hypotf4_u05sse2(__m128 a, __m128 b);
__m128 Sleef_cinz_hypotf4_u05sse4(__m128 a, __m128 b);
__m128 Sleef_finz_hypotf4_u05avx2128(__m128 a, __m128 b);

__m256 Sleef_hypotf8_u05(__m256 a, __m256 b);
__m256 Sleef_hypotf8_u05avx(__m256 a, __m256 b);
__m256 Sleef_hypotf8_u05avx2(__m256 a, __m256 b);
__m256 Sleef_hypotf8_u05fma4(__m256 a, __m256 b);
__m256 Sleef_cinz_hypotf8_u05avx(__m256 a, __m256 b);
__m256 Sleef_finz_hypotf8_u05avx2(__m256 a, __m256 b);
__m256 Sleef_finz_hypotf8_u05fma4(__m256 a, __m256 b);

__m512 Sleef_hypotf16_u05(__m512 a, __m512 b);
__m512 Sleef_hypotf16_u05avx512f(__m512 a, __m512 b);
__m512 Sleef_cinz_hypotf16_u05avx512fnofma(__m512 a, __m512 b);
__m512 Sleef_finz_hypotf16_u05avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_hypotf_u05](../libm#sleef_hypotf_u05) with the same accuracy specification.

### Vectorized double precision 2D Euclidian distance functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_hypotd2_u35(__m128d a, __m128d b);
__m128d Sleef_hypotd2_u35sse2(__m128d a, __m128d b);
__m128d Sleef_hypotd2_u35sse4(__m128d a, __m128d b);
__m128d Sleef_hypotd2_u35avx2128(__m128d a, __m128d b);
__m128d Sleef_cinz_hypotd2_u35sse2(__m128d a, __m128d b);
__m128d Sleef_cinz_hypotd2_u35sse4(__m128d a, __m128d b);
__m128d Sleef_finz_hypotd2_u35avx2128(__m128d a, __m128d b);

__m256d Sleef_hypotd4_u35(__m256d a, __m256d b);
__m256d Sleef_hypotd4_u35avx(__m256d a, __m256d b);
__m256d Sleef_hypotd4_u35avx2(__m256d a, __m256d b);
__m256d Sleef_hypotd4_u35fma4(__m256d a, __m256d b);
__m256d Sleef_cinz_hypotd4_u35avx(__m256d a, __m256d b);
__m256d Sleef_finz_hypotd4_u35avx2(__m256d a, __m256d b);
__m256d Sleef_finz_hypotd4_u35fma4(__m256d a, __m256d b);

__m512d Sleef_hypotd8_u35(__m512d a, __m512d b);
__m512d Sleef_hypotd8_u35avx512f(__m512d a, __m512d b);
__m512d Sleef_cinz_hypotd8_u35avx512fnofma(__m512d a, __m512d b);
__m512d Sleef_finz_hypotd8_u35avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_hypot_u35](../libm#sleef_hypot_u35) with the same accuracy specification.

### Vectorized single precision 2D Euclidian distance functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_hypotf4_u35(__m128 a, __m128 b);
__m128 Sleef_hypotf4_u35sse2(__m128 a, __m128 b);
__m128 Sleef_hypotf4_u35sse4(__m128 a, __m128 b);
__m128 Sleef_hypotf4_u35avx2128(__m128 a, __m128 b);
__m128 Sleef_cinz_hypotf4_u35sse2(__m128 a, __m128 b);
__m128 Sleef_cinz_hypotf4_u35sse4(__m128 a, __m128 b);
__m128 Sleef_finz_hypotf4_u35avx2128(__m128 a, __m128 b);

__m256 Sleef_hypotf8_u35(__m256 a, __m256 b);
__m256 Sleef_hypotf8_u35avx(__m256 a, __m256 b);
__m256 Sleef_hypotf8_u35avx2(__m256 a, __m256 b);
__m256 Sleef_hypotf8_u35fma4(__m256 a, __m256 b);
__m256 Sleef_cinz_hypotf8_u35avx(__m256 a, __m256 b);
__m256 Sleef_finz_hypotf8_u35avx2(__m256 a, __m256 b);
__m256 Sleef_finz_hypotf8_u35fma4(__m256 a, __m256 b);

__m512 Sleef_hypotf16_u35(__m512 a, __m512 b);
__m512 Sleef_hypotf16_u35avx512f(__m512 a, __m512 b);
__m512 Sleef_cinz_hypotf16_u35avx512fnofma(__m512 a, __m512 b);
__m512 Sleef_finz_hypotf16_u35avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_hypotf_u35](../libm#sleef_hypotf_u35) with the same accuracy specification.

<h2 id="invtrig">Inverse Trigonometric Functions</h2>

### Vectorized double precision arc sine functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_asind2_u10(__m128d a);
__m128d Sleef_asind2_u10sse2(__m128d a);
__m128d Sleef_asind2_u10sse4(__m128d a);
__m128d Sleef_asind2_u10avx2128(__m128d a);
__m128d Sleef_cinz_asind2_u10sse2(__m128d a);
__m128d Sleef_cinz_asind2_u10sse4(__m128d a);
__m128d Sleef_finz_asind2_u10avx2128(__m128d a);

__m256d Sleef_asind4_u10(__m256d a);
__m256d Sleef_asind4_u10avx(__m256d a);
__m256d Sleef_asind4_u10avx2(__m256d a);
__m256d Sleef_asind4_u10fma4(__m256d a);
__m256d Sleef_cinz_asind4_u10avx(__m256d a);
__m256d Sleef_finz_asind4_u10avx2(__m256d a);
__m256d Sleef_finz_asind4_u10fma4(__m256d a);

__m512d Sleef_asind8_u10(__m512d a);
__m512d Sleef_asind8_u10avx512f(__m512d a);
__m512d Sleef_cinz_asind8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_asind8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_asin_u10](../libm#sleef_asin_u10) with the same accuracy specification.

### Vectorized single precision arc sine functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_asinf4_u10(__m128 a);
__m128 Sleef_asinf4_u10sse2(__m128 a);
__m128 Sleef_asinf4_u10sse4(__m128 a);
__m128 Sleef_asinf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_asinf4_u10sse2(__m128 a);
__m128 Sleef_cinz_asinf4_u10sse4(__m128 a);
__m128 Sleef_finz_asinf4_u10avx2128(__m128 a);

__m256 Sleef_asinf8_u10(__m256 a);
__m256 Sleef_asinf8_u10avx(__m256 a);
__m256 Sleef_asinf8_u10avx2(__m256 a);
__m256 Sleef_asinf8_u10fma4(__m256 a);
__m256 Sleef_cinz_asinf8_u10avx(__m256 a);
__m256 Sleef_finz_asinf8_u10avx2(__m256 a);
__m256 Sleef_finz_asinf8_u10fma4(__m256 a);

__m512 Sleef_asinf16_u10(__m512 a);
__m512 Sleef_asinf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_asinf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_asinf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_asinf_u10](../libm#sleef_asinf_u10) with the same accuracy specification.

### Vectorized double precision arc sine functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_asind2_u35(__m128d a);
__m128d Sleef_asind2_u35sse2(__m128d a);
__m128d Sleef_asind2_u35sse4(__m128d a);
__m128d Sleef_asind2_u35avx2128(__m128d a);
__m128d Sleef_cinz_asind2_u35sse2(__m128d a);
__m128d Sleef_cinz_asind2_u35sse4(__m128d a);
__m128d Sleef_finz_asind2_u35avx2128(__m128d a);

__m256d Sleef_asind4_u35(__m256d a);
__m256d Sleef_asind4_u35avx(__m256d a);
__m256d Sleef_asind4_u35avx2(__m256d a);
__m256d Sleef_asind4_u35fma4(__m256d a);
__m256d Sleef_cinz_asind4_u35avx(__m256d a);
__m256d Sleef_finz_asind4_u35avx2(__m256d a);
__m256d Sleef_finz_asind4_u35fma4(__m256d a);

__m512d Sleef_asind8_u35(__m512d a);
__m512d Sleef_asind8_u35avx512f(__m512d a);
__m512d Sleef_cinz_asind8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_asind8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_asin_u35](../libm#sleef_asin_u35) with the same accuracy specification.

### Vectorized single precision arc sine functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_asinf4_u35(__m128 a);
__m128 Sleef_asinf4_u35sse2(__m128 a);
__m128 Sleef_asinf4_u35sse4(__m128 a);
__m128 Sleef_asinf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_asinf4_u35sse2(__m128 a);
__m128 Sleef_cinz_asinf4_u35sse4(__m128 a);
__m128 Sleef_finz_asinf4_u35avx2128(__m128 a);

__m256 Sleef_asinf8_u35(__m256 a);
__m256 Sleef_asinf8_u35avx(__m256 a);
__m256 Sleef_asinf8_u35avx2(__m256 a);
__m256 Sleef_asinf8_u35fma4(__m256 a);
__m256 Sleef_cinz_asinf8_u35avx(__m256 a);
__m256 Sleef_finz_asinf8_u35avx2(__m256 a);
__m256 Sleef_finz_asinf8_u35fma4(__m256 a);

__m512 Sleef_asinf16_u35(__m512 a);
__m512 Sleef_asinf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_asinf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_asinf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_asinf_u35](../libm#sleef_asinf_u35) with the same accuracy specification.

### Vectorized double precision arc cosine functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_acosd2_u10(__m128d a);
__m128d Sleef_acosd2_u10sse2(__m128d a);
__m128d Sleef_acosd2_u10sse4(__m128d a);
__m128d Sleef_acosd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_acosd2_u10sse2(__m128d a);
__m128d Sleef_cinz_acosd2_u10sse4(__m128d a);
__m128d Sleef_finz_acosd2_u10avx2128(__m128d a);

__m256d Sleef_acosd4_u10(__m256d a);
__m256d Sleef_acosd4_u10avx(__m256d a);
__m256d Sleef_acosd4_u10avx2(__m256d a);
__m256d Sleef_acosd4_u10fma4(__m256d a);
__m256d Sleef_cinz_acosd4_u10avx(__m256d a);
__m256d Sleef_finz_acosd4_u10avx2(__m256d a);
__m256d Sleef_finz_acosd4_u10fma4(__m256d a);

__m512d Sleef_acosd8_u10(__m512d a);
__m512d Sleef_acosd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_acosd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_acosd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_acos_u10](../libm#sleef_acos_u10) with the same accuracy specification.

### Vectorized single precision arc cosine functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_acosf4_u10(__m128 a);
__m128 Sleef_acosf4_u10sse2(__m128 a);
__m128 Sleef_acosf4_u10sse4(__m128 a);
__m128 Sleef_acosf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_acosf4_u10sse2(__m128 a);
__m128 Sleef_cinz_acosf4_u10sse4(__m128 a);
__m128 Sleef_finz_acosf4_u10avx2128(__m128 a);

__m256 Sleef_acosf8_u10(__m256 a);
__m256 Sleef_acosf8_u10avx(__m256 a);
__m256 Sleef_acosf8_u10avx2(__m256 a);
__m256 Sleef_acosf8_u10fma4(__m256 a);
__m256 Sleef_cinz_acosf8_u10avx(__m256 a);
__m256 Sleef_finz_acosf8_u10avx2(__m256 a);
__m256 Sleef_finz_acosf8_u10fma4(__m256 a);

__m512 Sleef_acosf16_u10(__m512 a);
__m512 Sleef_acosf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_acosf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_acosf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_acosf_u10](../libm#sleef_acosf_u10) with the same accuracy specification.

### Vectorized double precision arc cosine functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_acosd2_u35(__m128d a);
__m128d Sleef_acosd2_u35sse2(__m128d a);
__m128d Sleef_acosd2_u35sse4(__m128d a);
__m128d Sleef_acosd2_u35avx2128(__m128d a);
__m128d Sleef_cinz_acosd2_u35sse2(__m128d a);
__m128d Sleef_cinz_acosd2_u35sse4(__m128d a);
__m128d Sleef_finz_acosd2_u35avx2128(__m128d a);

__m256d Sleef_acosd4_u35(__m256d a);
__m256d Sleef_acosd4_u35avx(__m256d a);
__m256d Sleef_acosd4_u35avx2(__m256d a);
__m256d Sleef_acosd4_u35fma4(__m256d a);
__m256d Sleef_cinz_acosd4_u35avx(__m256d a);
__m256d Sleef_finz_acosd4_u35avx2(__m256d a);
__m256d Sleef_finz_acosd4_u35fma4(__m256d a);

__m512d Sleef_acosd8_u35(__m512d a);
__m512d Sleef_acosd8_u35avx512f(__m512d a);
__m512d Sleef_cinz_acosd8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_acosd8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_acos_u35](../libm#sleef_acos_u35) with the same accuracy specification.

### Vectorized single precision arc cosine functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_acosf4_u35(__m128 a);
__m128 Sleef_acosf4_u35sse2(__m128 a);
__m128 Sleef_acosf4_u35sse4(__m128 a);
__m128 Sleef_acosf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_acosf4_u35sse2(__m128 a);
__m128 Sleef_cinz_acosf4_u35sse4(__m128 a);
__m128 Sleef_finz_acosf4_u35avx2128(__m128 a);

__m256 Sleef_acosf8_u35(__m256 a);
__m256 Sleef_acosf8_u35avx(__m256 a);
__m256 Sleef_acosf8_u35avx2(__m256 a);
__m256 Sleef_acosf8_u35fma4(__m256 a);
__m256 Sleef_cinz_acosf8_u35avx(__m256 a);
__m256 Sleef_finz_acosf8_u35avx2(__m256 a);
__m256 Sleef_finz_acosf8_u35fma4(__m256 a);

__m512 Sleef_acosf16_u35(__m512 a);
__m512 Sleef_acosf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_acosf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_acosf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_acosf_u35](../libm#sleef_acosf_u35) with the same accuracy specification.

### Vectorized double precision arc tangent functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_atand2_u10(__m128d a);
__m128d Sleef_atand2_u10sse2(__m128d a);
__m128d Sleef_atand2_u10sse4(__m128d a);
__m128d Sleef_atand2_u10avx2128(__m128d a);
__m128d Sleef_cinz_atand2_u10sse2(__m128d a);
__m128d Sleef_cinz_atand2_u10sse4(__m128d a);
__m128d Sleef_finz_atand2_u10avx2128(__m128d a);

__m256d Sleef_atand4_u10(__m256d a);
__m256d Sleef_atand4_u10avx(__m256d a);
__m256d Sleef_atand4_u10avx2(__m256d a);
__m256d Sleef_atand4_u10fma4(__m256d a);
__m256d Sleef_cinz_atand4_u10avx(__m256d a);
__m256d Sleef_finz_atand4_u10avx2(__m256d a);
__m256d Sleef_finz_atand4_u10fma4(__m256d a);

__m512d Sleef_atand8_u10(__m512d a);
__m512d Sleef_atand8_u10avx512f(__m512d a);
__m512d Sleef_cinz_atand8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_atand8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_atan_u10](../libm#sleef_atan_u10) with the same accuracy specification.

### Vectorized single precision arc tangent functions with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_atanf4_u10(__m128 a);
__m128 Sleef_atanf4_u10sse2(__m128 a);
__m128 Sleef_atanf4_u10sse4(__m128 a);
__m128 Sleef_atanf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_atanf4_u10sse2(__m128 a);
__m128 Sleef_cinz_atanf4_u10sse4(__m128 a);
__m128 Sleef_finz_atanf4_u10avx2128(__m128 a);

__m256 Sleef_atanf8_u10(__m256 a);
__m256 Sleef_atanf8_u10avx(__m256 a);
__m256 Sleef_atanf8_u10avx2(__m256 a);
__m256 Sleef_atanf8_u10fma4(__m256 a);
__m256 Sleef_cinz_atanf8_u10avx(__m256 a);
__m256 Sleef_finz_atanf8_u10avx2(__m256 a);
__m256 Sleef_finz_atanf8_u10fma4(__m256 a);

__m512 Sleef_atanf16_u10(__m512 a);
__m512 Sleef_atanf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_atanf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_atanf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_atanf_u10](../libm#sleef_atanf_u10) with the same accuracy specification.

### Vectorized double precision arc tangent functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_atand2_u35(__m128d a);
__m128d Sleef_atand2_u35sse2(__m128d a);
__m128d Sleef_atand2_u35sse4(__m128d a);
__m128d Sleef_atand2_u35avx2128(__m128d a);
__m128d Sleef_cinz_atand2_u35sse2(__m128d a);
__m128d Sleef_cinz_atand2_u35sse4(__m128d a);
__m128d Sleef_finz_atand2_u35avx2128(__m128d a);

__m256d Sleef_atand4_u35(__m256d a);
__m256d Sleef_atand4_u35avx(__m256d a);
__m256d Sleef_atand4_u35avx2(__m256d a);
__m256d Sleef_atand4_u35fma4(__m256d a);
__m256d Sleef_cinz_atand4_u35avx(__m256d a);
__m256d Sleef_finz_atand4_u35avx2(__m256d a);
__m256d Sleef_finz_atand4_u35fma4(__m256d a);

__m512d Sleef_atand8_u35(__m512d a);
__m512d Sleef_atand8_u35avx512f(__m512d a);
__m512d Sleef_cinz_atand8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_atand8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_atan_u35](../libm#sleef_atan_u35) with the same accuracy specification.

### Vectorized single precision arc tangent functions with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_atanf4_u35(__m128 a);
__m128 Sleef_atanf4_u35sse2(__m128 a);
__m128 Sleef_atanf4_u35sse4(__m128 a);
__m128 Sleef_atanf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_atanf4_u35sse2(__m128 a);
__m128 Sleef_cinz_atanf4_u35sse4(__m128 a);
__m128 Sleef_finz_atanf4_u35avx2128(__m128 a);

__m256 Sleef_atanf8_u35(__m256 a);
__m256 Sleef_atanf8_u35avx(__m256 a);
__m256 Sleef_atanf8_u35avx2(__m256 a);
__m256 Sleef_atanf8_u35fma4(__m256 a);
__m256 Sleef_cinz_atanf8_u35avx(__m256 a);
__m256 Sleef_finz_atanf8_u35avx2(__m256 a);
__m256 Sleef_finz_atanf8_u35fma4(__m256 a);

__m512 Sleef_atanf16_u35(__m512 a);
__m512 Sleef_atanf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_atanf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_atanf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_atanf_u35](../libm#sleef_atanf_u35) with the same accuracy specification.

### Vectorized double precision arc tangent functions of two variables with 1.0 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_atan2d2_u10(__m128d a, __m128d b);
__m128d Sleef_atan2d2_u10sse2(__m128d a, __m128d b);
__m128d Sleef_atan2d2_u10sse4(__m128d a, __m128d b);
__m128d Sleef_atan2d2_u10avx2128(__m128d a, __m128d b);
__m128d Sleef_cinz_atan2d2_u10sse2(__m128d a, __m128d b);
__m128d Sleef_cinz_atan2d2_u10sse4(__m128d a, __m128d b);
__m128d Sleef_finz_atan2d2_u10avx2128(__m128d a, __m128d b);

__m256d Sleef_atan2d4_u10(__m256d a, __m256d b);
__m256d Sleef_atan2d4_u10avx(__m256d a, __m256d b);
__m256d Sleef_atan2d4_u10avx2(__m256d a, __m256d b);
__m256d Sleef_atan2d4_u10fma4(__m256d a, __m256d b);
__m256d Sleef_cinz_atan2d4_u10avx(__m256d a, __m256d b);
__m256d Sleef_finz_atan2d4_u10avx2(__m256d a, __m256d b);
__m256d Sleef_finz_atan2d4_u10fma4(__m256d a, __m256d b);

__m512d Sleef_atan2d8_u10(__m512d a, __m512d b);
__m512d Sleef_atan2d8_u10avx512f(__m512d a, __m512d b);
__m512d Sleef_cinz_atan2d8_u10avx512fnofma(__m512d a, __m512d b);
__m512d Sleef_finz_atan2d8_u10avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_atan2_u10](../libm#sleef_atan2_u10) with the same accuracy specification.

### Vectorized single precision arc tangent functions of two variables with 1.0 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_atan2f4_u10(__m128 a, __m128 b);
__m128 Sleef_atan2f4_u10sse2(__m128 a, __m128 b);
__m128 Sleef_atan2f4_u10sse4(__m128 a, __m128 b);
__m128 Sleef_atan2f4_u10avx2128(__m128 a, __m128 b);
__m128 Sleef_cinz_atan2f4_u10sse2(__m128 a, __m128 b);
__m128 Sleef_cinz_atan2f4_u10sse4(__m128 a, __m128 b);
__m128 Sleef_finz_atan2f4_u10avx2128(__m128 a, __m128 b);

__m256 Sleef_atan2f8_u10(__m256 a, __m256 b);
__m256 Sleef_atan2f8_u10avx(__m256 a, __m256 b);
__m256 Sleef_atan2f8_u10avx2(__m256 a, __m256 b);
__m256 Sleef_atan2f8_u10fma4(__m256 a, __m256 b);
__m256 Sleef_cinz_atan2f8_u10avx(__m256 a, __m256 b);
__m256 Sleef_finz_atan2f8_u10avx2(__m256 a, __m256 b);
__m256 Sleef_finz_atan2f8_u10fma4(__m256 a, __m256 b);

__m512 Sleef_atan2f16_u10(__m512 a, __m512 b);
__m512 Sleef_atan2f16_u10avx512f(__m512 a, __m512 b);
__m512 Sleef_cinz_atan2f16_u10avx512fnofma(__m512 a, __m512 b);
__m512 Sleef_finz_atan2f16_u10avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_atan2f_u10](../libm#sleef_atan2f_u10) with the same accuracy specification.

### Vectorized double precision arc tangent functions of two variables with 3.5 ULP error bound

```c
#include <sleef.h>

__m128d Sleef_atan2d2_u35(__m128d a, __m128d b);
__m128d Sleef_atan2d2_u35sse2(__m128d a, __m128d b);
__m128d Sleef_atan2d2_u35sse4(__m128d a, __m128d b);
__m128d Sleef_atan2d2_u35avx2128(__m128d a, __m128d b);
__m128d Sleef_cinz_atan2d2_u35sse2(__m128d a, __m128d b);
__m128d Sleef_cinz_atan2d2_u35sse4(__m128d a, __m128d b);
__m128d Sleef_finz_atan2d2_u35avx2128(__m128d a, __m128d b);

__m256d Sleef_atan2d4_u35(__m256d a, __m256d b);
__m256d Sleef_atan2d4_u35avx(__m256d a, __m256d b);
__m256d Sleef_atan2d4_u35avx2(__m256d a, __m256d b);
__m256d Sleef_atan2d4_u35fma4(__m256d a, __m256d b);
__m256d Sleef_cinz_atan2d4_u35avx(__m256d a, __m256d b);
__m256d Sleef_finz_atan2d4_u35avx2(__m256d a, __m256d b);
__m256d Sleef_finz_atan2d4_u35fma4(__m256d a, __m256d b);

__m512d Sleef_atan2d8_u35(__m512d a, __m512d b);
__m512d Sleef_atan2d8_u35avx512f(__m512d a, __m512d b);
__m512d Sleef_cinz_atan2d8_u35avx512fnofma(__m512d a, __m512d b);
__m512d Sleef_finz_atan2d8_u35avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_atan2_u35](../libm#sleef_atan2_u35) with the same accuracy specification.

### Vectorized single precision arc tangent functions of two variables with 3.5 ULP error bound

```c
#include <sleef.h>

__m128 Sleef_atan2f4_u35(__m128 a, __m128 b);
__m128 Sleef_atan2f4_u35sse2(__m128 a, __m128 b);
__m128 Sleef_atan2f4_u35sse4(__m128 a, __m128 b);
__m128 Sleef_atan2f4_u35avx2128(__m128 a, __m128 b);
__m128 Sleef_cinz_atan2f4_u35sse2(__m128 a, __m128 b);
__m128 Sleef_cinz_atan2f4_u35sse4(__m128 a, __m128 b);
__m128 Sleef_finz_atan2f4_u35avx2128(__m128 a, __m128 b);

__m256 Sleef_atan2f8_u35(__m256 a, __m256 b);
__m256 Sleef_atan2f8_u35avx(__m256 a, __m256 b);
__m256 Sleef_atan2f8_u35avx2(__m256 a, __m256 b);
__m256 Sleef_atan2f8_u35fma4(__m256 a, __m256 b);
__m256 Sleef_cinz_atan2f8_u35avx(__m256 a, __m256 b);
__m256 Sleef_finz_atan2f8_u35avx2(__m256 a, __m256 b);
__m256 Sleef_finz_atan2f8_u35fma4(__m256 a, __m256 b);

__m512 Sleef_atan2f16_u35(__m512 a, __m512 b);
__m512 Sleef_atan2f16_u35avx512f(__m512 a, __m512 b);
__m512 Sleef_cinz_atan2f16_u35avx512fnofma(__m512 a, __m512 b);
__m512 Sleef_finz_atan2f16_u35avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_atan2f_u35](../libm#sleef_atan2f_u35) with the same accuracy specification.

<h2 id="hyp">Hyperbolic functions and inverse hyperbolic functions</h2>

### Vectorized double precision hyperbolic sine functions

```c
#include <sleef.h>

__m128d Sleef_sinhd2_u10(__m128d a);
__m128d Sleef_sinhd2_u10sse2(__m128d a);
__m128d Sleef_sinhd2_u10sse4(__m128d a);
__m128d Sleef_sinhd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_sinhd2_u10sse2(__m128d a);
__m128d Sleef_cinz_sinhd2_u10sse4(__m128d a);
__m128d Sleef_finz_sinhd2_u10avx2128(__m128d a);

__m256d Sleef_sinhd4_u10(__m256d a);
__m256d Sleef_sinhd4_u10avx(__m256d a);
__m256d Sleef_sinhd4_u10avx2(__m256d a);
__m256d Sleef_sinhd4_u10fma4(__m256d a);
__m256d Sleef_cinz_sinhd4_u10avx(__m256d a);
__m256d Sleef_finz_sinhd4_u10avx2(__m256d a);
__m256d Sleef_finz_sinhd4_u10fma4(__m256d a);

__m512d Sleef_sinhd8_u10(__m512d a);
__m512d Sleef_sinhd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_sinhd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_sinhd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sinh_u10](../libm#sleef_sinh_u10) with the same accuracy specification.

### Vectorized single precision hyperbolic sine functions

```c
#include <sleef.h>

__m128 Sleef_sinhf4_u10(__m128 a);
__m128 Sleef_sinhf4_u10sse2(__m128 a);
__m128 Sleef_sinhf4_u10sse4(__m128 a);
__m128 Sleef_sinhf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_sinhf4_u10sse2(__m128 a);
__m128 Sleef_cinz_sinhf4_u10sse4(__m128 a);
__m128 Sleef_finz_sinhf4_u10avx2128(__m128 a);

__m256 Sleef_sinhf8_u10(__m256 a);
__m256 Sleef_sinhf8_u10avx(__m256 a);
__m256 Sleef_sinhf8_u10avx2(__m256 a);
__m256 Sleef_sinhf8_u10fma4(__m256 a);
__m256 Sleef_cinz_sinhf8_u10avx(__m256 a);
__m256 Sleef_finz_sinhf8_u10avx2(__m256 a);
__m256 Sleef_finz_sinhf8_u10fma4(__m256 a);

__m512 Sleef_sinhf16_u10(__m512 a);
__m512 Sleef_sinhf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_sinhf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_sinhf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sinhf_u10](../libm#sleef_sinhf_u10) with the same accuracy specification.

### Vectorized double precision hyperbolic sine functions

```c
#include <sleef.h>

__m128d Sleef_sinhd2_u35(__m128d a);
__m128d Sleef_sinhd2_u35sse2(__m128d a);
__m128d Sleef_sinhd2_u35sse4(__m128d a);
__m128d Sleef_sinhd2_u35avx2128(__m128d a);
__m128d Sleef_cinz_sinhd2_u35sse2(__m128d a);
__m128d Sleef_cinz_sinhd2_u35sse4(__m128d a);
__m128d Sleef_finz_sinhd2_u35avx2128(__m128d a);

__m256d Sleef_sinhd4_u35(__m256d a);
__m256d Sleef_sinhd4_u35avx(__m256d a);
__m256d Sleef_sinhd4_u35avx2(__m256d a);
__m256d Sleef_sinhd4_u35fma4(__m256d a);
__m256d Sleef_cinz_sinhd4_u35avx(__m256d a);
__m256d Sleef_finz_sinhd4_u35avx2(__m256d a);
__m256d Sleef_finz_sinhd4_u35fma4(__m256d a);

__m512d Sleef_sinhd8_u35(__m512d a);
__m512d Sleef_sinhd8_u35avx512f(__m512d a);
__m512d Sleef_cinz_sinhd8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_sinhd8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sinh_u35](../libm#sleef_sinh_u35) with the same accuracy specification.

### Vectorized single precision hyperbolic sine functions

```c
#include <sleef.h>

__m128 Sleef_sinhf4_u35(__m128 a);
__m128 Sleef_sinhf4_u35sse2(__m128 a);
__m128 Sleef_sinhf4_u35sse4(__m128 a);
__m128 Sleef_sinhf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_sinhf4_u35sse2(__m128 a);
__m128 Sleef_cinz_sinhf4_u35sse4(__m128 a);
__m128 Sleef_finz_sinhf4_u35avx2128(__m128 a);

__m256 Sleef_sinhf8_u35(__m256 a);
__m256 Sleef_sinhf8_u35avx(__m256 a);
__m256 Sleef_sinhf8_u35avx2(__m256 a);
__m256 Sleef_sinhf8_u35fma4(__m256 a);
__m256 Sleef_cinz_sinhf8_u35avx(__m256 a);
__m256 Sleef_finz_sinhf8_u35avx2(__m256 a);
__m256 Sleef_finz_sinhf8_u35fma4(__m256 a);

__m512 Sleef_sinhf16_u35(__m512 a);
__m512 Sleef_sinhf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_sinhf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_sinhf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_sinhf_u35](../libm#sleef_sinhf_u35) with the same accuracy specification.

### Vectorized double precision hyperbolic cosine functions

```c
#include <sleef.h>

__m128d Sleef_coshd2_u10(__m128d a);
__m128d Sleef_coshd2_u10sse2(__m128d a);
__m128d Sleef_coshd2_u10sse4(__m128d a);
__m128d Sleef_coshd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_coshd2_u10sse2(__m128d a);
__m128d Sleef_cinz_coshd2_u10sse4(__m128d a);
__m128d Sleef_finz_coshd2_u10avx2128(__m128d a);

__m256d Sleef_coshd4_u10(__m256d a);
__m256d Sleef_coshd4_u10avx(__m256d a);
__m256d Sleef_coshd4_u10avx2(__m256d a);
__m256d Sleef_coshd4_u10fma4(__m256d a);
__m256d Sleef_cinz_coshd4_u10avx(__m256d a);
__m256d Sleef_finz_coshd4_u10avx2(__m256d a);
__m256d Sleef_finz_coshd4_u10fma4(__m256d a);

__m512d Sleef_coshd8_u10(__m512d a);
__m512d Sleef_coshd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_coshd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_coshd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cosh_u10](../libm#sleef_cosh_u10) with the same accuracy specification.

### Vectorized single precision hyperbolic cosine functions

```c
#include <sleef.h>

__m128 Sleef_coshf4_u10(__m128 a);
__m128 Sleef_coshf4_u10sse2(__m128 a);
__m128 Sleef_coshf4_u10sse4(__m128 a);
__m128 Sleef_coshf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_coshf4_u10sse2(__m128 a);
__m128 Sleef_cinz_coshf4_u10sse4(__m128 a);
__m128 Sleef_finz_coshf4_u10avx2128(__m128 a);

__m256 Sleef_coshf8_u10(__m256 a);
__m256 Sleef_coshf8_u10avx(__m256 a);
__m256 Sleef_coshf8_u10avx2(__m256 a);
__m256 Sleef_coshf8_u10fma4(__m256 a);
__m256 Sleef_cinz_coshf8_u10avx(__m256 a);
__m256 Sleef_finz_coshf8_u10avx2(__m256 a);
__m256 Sleef_finz_coshf8_u10fma4(__m256 a);

__m512 Sleef_coshf16_u10(__m512 a);
__m512 Sleef_coshf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_coshf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_coshf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_coshf_u10](../libm#sleef_coshf_u10) with the same accuracy specification.

### Vectorized double precision hyperbolic cosine functions

```c
#include <sleef.h>

__m128d Sleef_coshd2_u35(__m128d a);
__m128d Sleef_coshd2_u35sse2(__m128d a);
__m128d Sleef_coshd2_u35sse4(__m128d a);
__m128d Sleef_coshd2_u35avx2128(__m128d a);
__m128d Sleef_cinz_coshd2_u35sse2(__m128d a);
__m128d Sleef_cinz_coshd2_u35sse4(__m128d a);
__m128d Sleef_finz_coshd2_u35avx2128(__m128d a);

__m256d Sleef_coshd4_u35(__m256d a);
__m256d Sleef_coshd4_u35avx(__m256d a);
__m256d Sleef_coshd4_u35avx2(__m256d a);
__m256d Sleef_coshd4_u35fma4(__m256d a);
__m256d Sleef_cinz_coshd4_u35avx(__m256d a);
__m256d Sleef_finz_coshd4_u35avx2(__m256d a);
__m256d Sleef_finz_coshd4_u35fma4(__m256d a);

__m512d Sleef_coshd8_u35(__m512d a);
__m512d Sleef_coshd8_u35avx512f(__m512d a);
__m512d Sleef_cinz_coshd8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_coshd8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_cosh_u35](../libm#sleef_cosh_u35) with the same accuracy specification.

### Vectorized single precision hyperbolic cosine functions

```c
#include <sleef.h>

__m128 Sleef_coshf4_u35(__m128 a);
__m128 Sleef_coshf4_u35sse2(__m128 a);
__m128 Sleef_coshf4_u35sse4(__m128 a);
__m128 Sleef_coshf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_coshf4_u35sse2(__m128 a);
__m128 Sleef_cinz_coshf4_u35sse4(__m128 a);
__m128 Sleef_finz_coshf4_u35avx2128(__m128 a);

__m256 Sleef_coshf8_u35(__m256 a);
__m256 Sleef_coshf8_u35avx(__m256 a);
__m256 Sleef_coshf8_u35avx2(__m256 a);
__m256 Sleef_coshf8_u35fma4(__m256 a);
__m256 Sleef_cinz_coshf8_u35avx(__m256 a);
__m256 Sleef_finz_coshf8_u35avx2(__m256 a);
__m256 Sleef_finz_coshf8_u35fma4(__m256 a);

__m512 Sleef_coshf16_u35(__m512 a);
__m512 Sleef_coshf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_coshf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_coshf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_coshf_u35](../libm#sleef_coshf_u35) with the same accuracy specification.

### Vectorized double precision hyperbolic tangent functions

```c
#include <sleef.h>

__m128d Sleef_tanhd2_u10(__m128d a);
__m128d Sleef_tanhd2_u10sse2(__m128d a);
__m128d Sleef_tanhd2_u10sse4(__m128d a);
__m128d Sleef_tanhd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_tanhd2_u10sse2(__m128d a);
__m128d Sleef_cinz_tanhd2_u10sse4(__m128d a);
__m128d Sleef_finz_tanhd2_u10avx2128(__m128d a);

__m256d Sleef_tanhd4_u10(__m256d a);
__m256d Sleef_tanhd4_u10avx(__m256d a);
__m256d Sleef_tanhd4_u10avx2(__m256d a);
__m256d Sleef_tanhd4_u10fma4(__m256d a);
__m256d Sleef_cinz_tanhd4_u10avx(__m256d a);
__m256d Sleef_finz_tanhd4_u10avx2(__m256d a);
__m256d Sleef_finz_tanhd4_u10fma4(__m256d a);

__m512d Sleef_tanhd8_u10(__m512d a);
__m512d Sleef_tanhd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_tanhd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_tanhd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_tanh_u10](../libm#sleef_tanh_u10) with the same accuracy specification.

### Vectorized single precision hyperbolic tangent functions

```c
#include <sleef.h>

__m128 Sleef_tanhf4_u10(__m128 a);
__m128 Sleef_tanhf4_u10sse2(__m128 a);
__m128 Sleef_tanhf4_u10sse4(__m128 a);
__m128 Sleef_tanhf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_tanhf4_u10sse2(__m128 a);
__m128 Sleef_cinz_tanhf4_u10sse4(__m128 a);
__m128 Sleef_finz_tanhf4_u10avx2128(__m128 a);

__m256 Sleef_tanhf8_u10(__m256 a);
__m256 Sleef_tanhf8_u10avx(__m256 a);
__m256 Sleef_tanhf8_u10avx2(__m256 a);
__m256 Sleef_tanhf8_u10fma4(__m256 a);
__m256 Sleef_cinz_tanhf8_u10avx(__m256 a);
__m256 Sleef_finz_tanhf8_u10avx2(__m256 a);
__m256 Sleef_finz_tanhf8_u10fma4(__m256 a);

__m512 Sleef_tanhf16_u10(__m512 a);
__m512 Sleef_tanhf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_tanhf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_tanhf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_tanhf_u10](../libm#sleef_tanhf_u10) with the same accuracy specification.

### Vectorized double precision hyperbolic tangent functions

```c
#include <sleef.h>

__m128d Sleef_tanhd2_u35(__m128d a);
__m128d Sleef_tanhd2_u35sse2(__m128d a);
__m128d Sleef_tanhd2_u35sse4(__m128d a);
__m128d Sleef_tanhd2_u35avx2128(__m128d a);
__m128d Sleef_cinz_tanhd2_u35sse2(__m128d a);
__m128d Sleef_cinz_tanhd2_u35sse4(__m128d a);
__m128d Sleef_finz_tanhd2_u35avx2128(__m128d a);

__m256d Sleef_tanhd4_u35(__m256d a);
__m256d Sleef_tanhd4_u35avx(__m256d a);
__m256d Sleef_tanhd4_u35avx2(__m256d a);
__m256d Sleef_tanhd4_u35fma4(__m256d a);
__m256d Sleef_cinz_tanhd4_u35avx(__m256d a);
__m256d Sleef_finz_tanhd4_u35avx2(__m256d a);
__m256d Sleef_finz_tanhd4_u35fma4(__m256d a);

__m512d Sleef_tanhd8_u35(__m512d a);
__m512d Sleef_tanhd8_u35avx512f(__m512d a);
__m512d Sleef_cinz_tanhd8_u35avx512fnofma(__m512d a);
__m512d Sleef_finz_tanhd8_u35avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_tanh_u35](../libm#sleef_tanh_u35) with the same accuracy specification.

### Vectorized single precision hyperbolic tangent functions

```c
#include <sleef.h>

__m128 Sleef_tanhf4_u35(__m128 a);
__m128 Sleef_tanhf4_u35sse2(__m128 a);
__m128 Sleef_tanhf4_u35sse4(__m128 a);
__m128 Sleef_tanhf4_u35avx2128(__m128 a);
__m128 Sleef_cinz_tanhf4_u35sse2(__m128 a);
__m128 Sleef_cinz_tanhf4_u35sse4(__m128 a);
__m128 Sleef_finz_tanhf4_u35avx2128(__m128 a);

__m256 Sleef_tanhf8_u35(__m256 a);
__m256 Sleef_tanhf8_u35avx(__m256 a);
__m256 Sleef_tanhf8_u35avx2(__m256 a);
__m256 Sleef_tanhf8_u35fma4(__m256 a);
__m256 Sleef_cinz_tanhf8_u35avx(__m256 a);
__m256 Sleef_finz_tanhf8_u35avx2(__m256 a);
__m256 Sleef_finz_tanhf8_u35fma4(__m256 a);

__m512 Sleef_tanhf16_u35(__m512 a);
__m512 Sleef_tanhf16_u35avx512f(__m512 a);
__m512 Sleef_cinz_tanhf16_u35avx512fnofma(__m512 a);
__m512 Sleef_finz_tanhf16_u35avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_tanhf_u35](../libm#sleef_tanhf_u35) with the same accuracy specification.

### Vectorized double precision inverse hyperbolic sine functions

```c
#include <sleef.h>

__m128d Sleef_asinhd2_u10(__m128d a);
__m128d Sleef_asinhd2_u10sse2(__m128d a);
__m128d Sleef_asinhd2_u10sse4(__m128d a);
__m128d Sleef_asinhd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_asinhd2_u10sse2(__m128d a);
__m128d Sleef_cinz_asinhd2_u10sse4(__m128d a);
__m128d Sleef_finz_asinhd2_u10avx2128(__m128d a);

__m256d Sleef_asinhd4_u10(__m256d a);
__m256d Sleef_asinhd4_u10avx(__m256d a);
__m256d Sleef_asinhd4_u10avx2(__m256d a);
__m256d Sleef_asinhd4_u10fma4(__m256d a);
__m256d Sleef_cinz_asinhd4_u10avx(__m256d a);
__m256d Sleef_finz_asinhd4_u10avx2(__m256d a);
__m256d Sleef_finz_asinhd4_u10fma4(__m256d a);

__m512d Sleef_asinhd8_u10(__m512d a);
__m512d Sleef_asinhd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_asinhd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_asinhd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_asinh_u10](../libm#sleef_asinh_u10) with the same accuracy specification.

### Vectorized single precision inverse hyperbolic sine functions

```c
#include <sleef.h>

__m128 Sleef_asinhf4_u10(__m128 a);
__m128 Sleef_asinhf4_u10sse2(__m128 a);
__m128 Sleef_asinhf4_u10sse4(__m128 a);
__m128 Sleef_asinhf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_asinhf4_u10sse2(__m128 a);
__m128 Sleef_cinz_asinhf4_u10sse4(__m128 a);
__m128 Sleef_finz_asinhf4_u10avx2128(__m128 a);

__m256 Sleef_asinhf8_u10(__m256 a);
__m256 Sleef_asinhf8_u10avx(__m256 a);
__m256 Sleef_asinhf8_u10avx2(__m256 a);
__m256 Sleef_asinhf8_u10fma4(__m256 a);
__m256 Sleef_cinz_asinhf8_u10avx(__m256 a);
__m256 Sleef_finz_asinhf8_u10avx2(__m256 a);
__m256 Sleef_finz_asinhf8_u10fma4(__m256 a);

__m512 Sleef_asinhf16_u10(__m512 a);
__m512 Sleef_asinhf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_asinhf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_asinhf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_asinhf_u10](../libm#sleef_asinhf_u10) with the same accuracy specification.

### Vectorized double precision inverse hyperbolic cosine functions

```c
#include <sleef.h>

__m128d Sleef_acoshd2_u10(__m128d a);
__m128d Sleef_acoshd2_u10sse2(__m128d a);
__m128d Sleef_acoshd2_u10sse4(__m128d a);
__m128d Sleef_acoshd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_acoshd2_u10sse2(__m128d a);
__m128d Sleef_cinz_acoshd2_u10sse4(__m128d a);
__m128d Sleef_finz_acoshd2_u10avx2128(__m128d a);

__m256d Sleef_acoshd4_u10(__m256d a);
__m256d Sleef_acoshd4_u10avx(__m256d a);
__m256d Sleef_acoshd4_u10avx2(__m256d a);
__m256d Sleef_acoshd4_u10fma4(__m256d a);
__m256d Sleef_cinz_acoshd4_u10avx(__m256d a);
__m256d Sleef_finz_acoshd4_u10avx2(__m256d a);
__m256d Sleef_finz_acoshd4_u10fma4(__m256d a);

__m512d Sleef_acoshd8_u10(__m512d a);
__m512d Sleef_acoshd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_acoshd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_acoshd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_acosh_u10](../libm#sleef_acosh_u10) with the same accuracy specification.

### Vectorized single precision inverse hyperbolic cosine functions

```c
#include <sleef.h>

__m128 Sleef_acoshf4_u10(__m128 a);
__m128 Sleef_acoshf4_u10sse2(__m128 a);
__m128 Sleef_acoshf4_u10sse4(__m128 a);
__m128 Sleef_acoshf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_acoshf4_u10sse2(__m128 a);
__m128 Sleef_cinz_acoshf4_u10sse4(__m128 a);
__m128 Sleef_finz_acoshf4_u10avx2128(__m128 a);

__m256 Sleef_acoshf8_u10(__m256 a);
__m256 Sleef_acoshf8_u10avx(__m256 a);
__m256 Sleef_acoshf8_u10avx2(__m256 a);
__m256 Sleef_acoshf8_u10fma4(__m256 a);
__m256 Sleef_cinz_acoshf8_u10avx(__m256 a);
__m256 Sleef_finz_acoshf8_u10avx2(__m256 a);
__m256 Sleef_finz_acoshf8_u10fma4(__m256 a);

__m512 Sleef_acoshf16_u10(__m512 a);
__m512 Sleef_acoshf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_acoshf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_acoshf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_acoshf_u10](../libm#sleef_acoshf_u10) with the same accuracy specification.

### Vectorized double precision inverse hyperbolic tangent functions

```c
#include <sleef.h>

__m128d Sleef_atanhd2_u10(__m128d a);
__m128d Sleef_atanhd2_u10sse2(__m128d a);
__m128d Sleef_atanhd2_u10sse4(__m128d a);
__m128d Sleef_atanhd2_u10avx2128(__m128d a);
__m128d Sleef_cinz_atanhd2_u10sse2(__m128d a);
__m128d Sleef_cinz_atanhd2_u10sse4(__m128d a);
__m128d Sleef_finz_atanhd2_u10avx2128(__m128d a);

__m256d Sleef_atanhd4_u10(__m256d a);
__m256d Sleef_atanhd4_u10avx(__m256d a);
__m256d Sleef_atanhd4_u10avx2(__m256d a);
__m256d Sleef_atanhd4_u10fma4(__m256d a);
__m256d Sleef_cinz_atanhd4_u10avx(__m256d a);
__m256d Sleef_finz_atanhd4_u10avx2(__m256d a);
__m256d Sleef_finz_atanhd4_u10fma4(__m256d a);

__m512d Sleef_atanhd8_u10(__m512d a);
__m512d Sleef_atanhd8_u10avx512f(__m512d a);
__m512d Sleef_cinz_atanhd8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_atanhd8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_atanh_u10](../libm#sleef_atanh_u10) with the same accuracy specification.

### Vectorized single precision inverse hyperbolic tangent functions

```c
#include <sleef.h>

__m128 Sleef_atanhf4_u10(__m128 a);
__m128 Sleef_atanhf4_u10sse2(__m128 a);
__m128 Sleef_atanhf4_u10sse4(__m128 a);
__m128 Sleef_atanhf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_atanhf4_u10sse2(__m128 a);
__m128 Sleef_cinz_atanhf4_u10sse4(__m128 a);
__m128 Sleef_finz_atanhf4_u10avx2128(__m128 a);

__m256 Sleef_atanhf8_u10(__m256 a);
__m256 Sleef_atanhf8_u10avx(__m256 a);
__m256 Sleef_atanhf8_u10avx2(__m256 a);
__m256 Sleef_atanhf8_u10fma4(__m256 a);
__m256 Sleef_cinz_atanhf8_u10avx(__m256 a);
__m256 Sleef_finz_atanhf8_u10avx2(__m256 a);
__m256 Sleef_finz_atanhf8_u10fma4(__m256 a);

__m512 Sleef_atanhf16_u10(__m512 a);
__m512 Sleef_atanhf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_atanhf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_atanhf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_atanhf_u10](../libm#sleef_atanhf_u10) with the same accuracy specification.

<h2 id="eg">Error and gamma functions</h2>

### Vectorized double precision error functions

```c
#include <sleef.h>

__m128 Sleef_erfd2_u10(__m128 a);
__m128 Sleef_erfd2_u10sse2(__m128 a);
__m128 Sleef_erfd2_u10sse4(__m128 a);
__m128 Sleef_erfd2_u10avx2128(__m128 a);
__m128 Sleef_cinz_erfd2_u10sse2(__m128 a);
__m128 Sleef_cinz_erfd2_u10sse4(__m128 a);
__m128 Sleef_finz_erfd2_u10avx2128(__m128 a);

__m128 Sleef_erfd4_u10(__m128 a);
__m128 Sleef_erfd4_u10avx(__m128 a);
__m128 Sleef_erfd4_u10avx2(__m128 a);
__m128 Sleef_erfd4_u10fma4(__m128 a);
__m128 Sleef_cinz_erfd4_u10avx(__m128 a);
__m128 Sleef_finz_erfd4_u10avx2(__m128 a);
__m128 Sleef_finz_erfd4_u10fma4(__m128 a);

__m256 Sleef_erfd8_u10(__m256 a);
__m256 Sleef_erfd8_u10avx512f(__m256 a);
__m256 Sleef_cinz_erfd8_u10avx512fnofma(__m256 a);
__m256 Sleef_finz_erfd8_u10avx512f(__m256 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_erf_u10](../libm#sleef_erf_u10) with the same accuracy specification.

### Vectorized single precision error functions

```c
#include <sleef.h>

__m128 Sleef_erff4_u10(__m128 a);
__m128 Sleef_erff4_u10sse2(__m128 a);
__m128 Sleef_erff4_u10sse4(__m128 a);
__m128 Sleef_erff4_u10avx2128(__m128 a);
__m128 Sleef_cinz_erff4_u10sse2(__m128 a);
__m128 Sleef_cinz_erff4_u10sse4(__m128 a);
__m128 Sleef_finz_erff4_u10avx2128(__m128 a);

__m256 Sleef_erff8_u10(__m256 a);
__m256 Sleef_erff8_u10avx(__m256 a);
__m256 Sleef_erff8_u10avx2(__m256 a);
__m256 Sleef_erff8_u10fma4(__m256 a);
__m256 Sleef_cinz_erff8_u10avx(__m256 a);
__m256 Sleef_finz_erff8_u10avx2(__m256 a);
__m256 Sleef_finz_erff8_u10fma4(__m256 a);

__m512 Sleef_erff16_u10(__m512 a);
__m512 Sleef_erff16_u10avx512f(__m512 a);
__m512 Sleef_cinz_erff16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_erff16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_erff_u10](../libm#sleef_erff_u10) with the same accuracy specification.

### Vectorized double precision complementary error functions

```c
#include <sleef.h>

__m128d Sleef_erfcd2_u15(__m128d a);
__m128d Sleef_erfcd2_u15sse2(__m128d a);
__m128d Sleef_erfcd2_u15sse4(__m128d a);
__m128d Sleef_erfcd2_u15avx2128(__m128d a);
__m128d Sleef_cinz_erfcd2_u15sse2(__m128d a);
__m128d Sleef_cinz_erfcd2_u15sse4(__m128d a);
__m128d Sleef_finz_erfcd2_u15avx2128(__m128d a);

__m256d Sleef_erfcd4_u15(__m256d a);
__m256d Sleef_erfcd4_u15avx(__m256d a);
__m256d Sleef_erfcd4_u15avx2(__m256d a);
__m256d Sleef_erfcd4_u15fma4(__m256d a);
__m256d Sleef_cinz_erfcd4_u15avx(__m256d a);
__m256d Sleef_finz_erfcd4_u15avx2(__m256d a);
__m256d Sleef_finz_erfcd4_u15fma4(__m256d a);

__m512d Sleef_erfcd8_u15(__m512d a);
__m512d Sleef_erfcd8_u15avx512f(__m512d a);
__m512d Sleef_cinz_erfcd8_u15avx512fnofma(__m512d a);
__m512d Sleef_finz_erfcd8_u15avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_erfc_u15](../libm#sleef_erfc_u15) with the same accuracy specification.

### Vectorized single precision complementary error functions

```c
#include <sleef.h>

__m128 Sleef_erfcf4_u15(__m128 a);
__m128 Sleef_erfcf4_u15sse2(__m128 a);
__m128 Sleef_erfcf4_u15sse4(__m128 a);
__m128 Sleef_erfcf4_u15avx2128(__m128 a);
__m128 Sleef_cinz_erfcf4_u15sse2(__m128 a);
__m128 Sleef_cinz_erfcf4_u15sse4(__m128 a);
__m128 Sleef_finz_erfcf4_u15avx2128(__m128 a);

__m256 Sleef_erfcf8_u15(__m256 a);
__m256 Sleef_erfcf8_u15avx(__m256 a);
__m256 Sleef_erfcf8_u15avx2(__m256 a);
__m256 Sleef_erfcf8_u15fma4(__m256 a);
__m256 Sleef_cinz_erfcf8_u15avx(__m256 a);
__m256 Sleef_finz_erfcf8_u15avx2(__m256 a);
__m256 Sleef_finz_erfcf8_u15fma4(__m256 a);

__m512 Sleef_erfcf16_u15(__m512 a);
__m512 Sleef_erfcf16_u15avx512f(__m512 a);
__m512 Sleef_cinz_erfcf16_u15avx512fnofma(__m512 a);
__m512 Sleef_finz_erfcf16_u15avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_erfcf_u15](../libm#sleef_erfcf_u15) with the same accuracy specification.

### Vectorized double precision gamma functions

```c
#include <sleef.h>

__m128d Sleef_tgammad2_u10(__m128d a);
__m128d Sleef_tgammad2_u10sse2(__m128d a);
__m128d Sleef_tgammad2_u10sse4(__m128d a);
__m128d Sleef_tgammad2_u10avx2128(__m128d a);
__m128d Sleef_cinz_tgammad2_u10sse2(__m128d a);
__m128d Sleef_cinz_tgammad2_u10sse4(__m128d a);
__m128d Sleef_finz_tgammad2_u10avx2128(__m128d a);

__m256d Sleef_tgammad4_u10(__m256d a);
__m256d Sleef_tgammad4_u10avx(__m256d a);
__m256d Sleef_tgammad4_u10avx2(__m256d a);
__m256d Sleef_tgammad4_u10fma4(__m256d a);
__m256d Sleef_cinz_tgammad4_u10avx(__m256d a);
__m256d Sleef_finz_tgammad4_u10avx2(__m256d a);
__m256d Sleef_finz_tgammad4_u10fma4(__m256d a);

__m512d Sleef_tgammad8_u10(__m512d a);
__m512d Sleef_tgammad8_u10avx512f(__m512d a);
__m512d Sleef_cinz_tgammad8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_tgammad8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_tgamma_u10](../libm#sleef_tgamma_u10) with the same accuracy specification.

### Vectorized single precision gamma functions

```c
#include <sleef.h>

__m128 Sleef_tgammaf4_u10(__m128 a);
__m128 Sleef_tgammaf4_u10sse2(__m128 a);
__m128 Sleef_tgammaf4_u10sse4(__m128 a);
__m128 Sleef_tgammaf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_tgammaf4_u10sse2(__m128 a);
__m128 Sleef_cinz_tgammaf4_u10sse4(__m128 a);
__m128 Sleef_finz_tgammaf4_u10avx2128(__m128 a);

__m256 Sleef_tgammaf8_u10(__m256 a);
__m256 Sleef_tgammaf8_u10avx(__m256 a);
__m256 Sleef_tgammaf8_u10avx2(__m256 a);
__m256 Sleef_tgammaf8_u10fma4(__m256 a);
__m256 Sleef_cinz_tgammaf8_u10avx(__m256 a);
__m256 Sleef_finz_tgammaf8_u10avx2(__m256 a);
__m256 Sleef_finz_tgammaf8_u10fma4(__m256 a);

__m512 Sleef_tgammaf16_u10(__m512 a);
__m512 Sleef_tgammaf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_tgammaf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_tgammaf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_tgammaf_u10](../libm#sleef_tgammaf_u10) with the same accuracy specification.

### Vectorized double precision log gamma functions

```c
#include <sleef.h>

__m128d Sleef_lgammad2_u10(__m128d a);
__m128d Sleef_lgammad2_u10sse2(__m128d a);
__m128d Sleef_lgammad2_u10sse4(__m128d a);
__m128d Sleef_lgammad2_u10avx2128(__m128d a);
__m128d Sleef_cinz_lgammad2_u10sse2(__m128d a);
__m128d Sleef_cinz_lgammad2_u10sse4(__m128d a);
__m128d Sleef_finz_lgammad2_u10avx2128(__m128d a);

__m256d Sleef_lgammad4_u10(__m256d a);
__m256d Sleef_lgammad4_u10avx(__m256d a);
__m256d Sleef_lgammad4_u10avx2(__m256d a);
__m256d Sleef_lgammad4_u10fma4(__m256d a);
__m256d Sleef_cinz_lgammad4_u10avx(__m256d a);
__m256d Sleef_finz_lgammad4_u10avx2(__m256d a);
__m256d Sleef_finz_lgammad4_u10fma4(__m256d a);

__m512d Sleef_lgammad8_u10(__m512d a);
__m512d Sleef_lgammad8_u10avx512f(__m512d a);
__m512d Sleef_cinz_lgammad8_u10avx512fnofma(__m512d a);
__m512d Sleef_finz_lgammad8_u10avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_lgamma_u10](../libm#sleef_lgamma_u10) with the same accuracy specification.

### Vectorized single precision log gamma functions

```c
#include <sleef.h>

__m128 Sleef_lgammaf4_u10(__m128 a);
__m128 Sleef_lgammaf4_u10sse2(__m128 a);
__m128 Sleef_lgammaf4_u10sse4(__m128 a);
__m128 Sleef_lgammaf4_u10avx2128(__m128 a);
__m128 Sleef_cinz_lgammaf4_u10sse2(__m128 a);
__m128 Sleef_cinz_lgammaf4_u10sse4(__m128 a);
__m128 Sleef_finz_lgammaf4_u10avx2128(__m128 a);

__m256 Sleef_lgammaf8_u10(__m256 a);
__m256 Sleef_lgammaf8_u10avx(__m256 a);
__m256 Sleef_lgammaf8_u10avx2(__m256 a);
__m256 Sleef_lgammaf8_u10fma4(__m256 a);
__m256 Sleef_cinz_lgammaf8_u10avx(__m256 a);
__m256 Sleef_finz_lgammaf8_u10avx2(__m256 a);
__m256 Sleef_finz_lgammaf8_u10fma4(__m256 a);

__m512 Sleef_lgammaf16_u10(__m512 a);
__m512 Sleef_lgammaf16_u10avx512f(__m512 a);
__m512 Sleef_cinz_lgammaf16_u10avx512fnofma(__m512 a);
__m512 Sleef_finz_lgammaf16_u10avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_lgammaf_u10](../libm#sleef_lgammaf_u10) with the same accuracy specification.

<h2 id="nearint">Nearest integer functions</h2>

### Vectorized double precision functions for rounding to integer towards zero

```c
#include <sleef.h>

__m128d Sleef_truncd2(__m128d a);
__m128d Sleef_truncd2_sse2(__m128d a);
__m128d Sleef_truncd2_sse4(__m128d a);
__m128d Sleef_truncd2_avx2128(__m128d a);

__m256d Sleef_truncd4(__m256d a);
__m256d Sleef_truncd4_avx(__m256d a);
__m256d Sleef_truncd4_fma4(__m256d a);
__m256d Sleef_truncd4_avx2(__m256d a);

__m512d Sleef_truncd8(__m512d a);
__m512d Sleef_truncd8_avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_trunc](../libm#sleef_trunc) with the same accuracy specification.

### Vectorized single precision functions for rounding to integer towards zero

```c
#include <sleef.h>

__m128 Sleef_truncf4(__m128 a);
__m128 Sleef_truncf4_sse2(__m128 a);
__m128 Sleef_truncf4_sse4(__m128 a);
__m128 Sleef_truncf4_avx2128(__m128 a);

__m256 Sleef_truncf8(__m256 a);
__m256 Sleef_truncf8_avx(__m256 a);
__m256 Sleef_truncf8_fma4(__m256 a);
__m256 Sleef_truncf8_avx2(__m256 a);

__m512 Sleef_truncf16(__m512 a);
__m512 Sleef_truncf16_avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_truncf](../libm#sleef_truncf) with the same accuracy specification.

### Vectorized double precision functions for rounding to integer towards negative infinity

```c
#include <sleef.h>

__m128d Sleef_floord2(__m128d a);
__m128d Sleef_floord2_sse2(__m128d a);
__m128d Sleef_floord2_sse4(__m128d a);
__m128d Sleef_floord2_avx2128(__m128d a);

__m256d Sleef_floord4(__m256d a);
__m256d Sleef_floord4_avx(__m256d a);
__m256d Sleef_floord4_fma4(__m256d a);
__m256d Sleef_floord4_avx2(__m256d a);

__m512d Sleef_floord8(__m512d a);
__m512d Sleef_floord8_avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_floor](../libm#sleef_floor) with the same accuracy specification.

### Vectorized single precision functions for rounding to integer towards negative infinity

```c
#include <sleef.h>

__m128 Sleef_floorf4(__m128 a);
__m128 Sleef_floorf4_sse2(__m128 a);
__m128 Sleef_floorf4_sse4(__m128 a);
__m128 Sleef_floorf4_avx2128(__m128 a);

__m256 Sleef_floorf8(__m256 a);
__m256 Sleef_floorf8_avx(__m256 a);
__m256 Sleef_floorf8_fma4(__m256 a);
__m256 Sleef_floorf8_avx2(__m256 a);

__m512 Sleef_floorf16(__m512 a);
__m512 Sleef_floorf16_avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_floorf](../libm#sleef_floorf) with the same accuracy specification.

### Vectorized double precision functions for rounding to integer towards positive infinity

```c
#include <sleef.h>

__m128d Sleef_ceild2(__m128d a);
__m128d Sleef_ceild2_sse2(__m128d a);
__m128d Sleef_ceild2_sse4(__m128d a);
__m128d Sleef_ceild2_avx2128(__m128d a);

__m256d Sleef_ceild4(__m256d a);
__m256d Sleef_ceild4_avx(__m256d a);
__m256d Sleef_ceild4_fma4(__m256d a);
__m256d Sleef_ceild4_avx2(__m256d a);

__m512d Sleef_ceild8(__m512d a);
__m512d Sleef_ceild8_avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_ceil](../libm#sleef_ceil) with the same accuracy specification.

### Vectorized single precision functions for rounding to integer towards positive infinity

```c
#include <sleef.h>

__m128 Sleef_ceilf4(__m128 a);
__m128 Sleef_ceilf4_sse2(__m128 a);
__m128 Sleef_ceilf4_sse4(__m128 a);
__m128 Sleef_ceilf4_avx2128(__m128 a);

__m256 Sleef_ceilf8(__m256 a);
__m256 Sleef_ceilf8_avx(__m256 a);
__m256 Sleef_ceilf8_fma4(__m256 a);
__m256 Sleef_ceilf8_avx2(__m256 a);

__m512 Sleef_ceilf16(__m512 a);
__m512 Sleef_ceilf16_avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_ceilf](../libm#sleef_ceilf) with the same accuracy specification.

### Vectorized double precision functions for rounding to nearest integer

```c
#include <sleef.h>

__m128d Sleef_roundd2(__m128d a);
__m128d Sleef_roundd2_sse2(__m128d a);
__m128d Sleef_roundd2_sse4(__m128d a);
__m128d Sleef_roundd2_avx2128(__m128d a);

__m256d Sleef_roundd4(__m256d a);
__m256d Sleef_roundd4_avx(__m256d a);
__m256d Sleef_roundd4_fma4(__m256d a);
__m256d Sleef_roundd4_avx2(__m256d a);

__m512d Sleef_roundd8(__m512d a);
__m512d Sleef_roundd8_avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_round](../libm#sleef_round) with the same accuracy specification.

### Vectorized single precision functions for rounding to nearest integer

```c
#include <sleef.h>

__m128 Sleef_roundf4(__m128 a);
__m128 Sleef_roundf4_sse2(__m128 a);
__m128 Sleef_roundf4_sse4(__m128 a);
__m128 Sleef_roundf4_avx2128(__m128 a);

__m256 Sleef_roundf8(__m256 a);
__m256 Sleef_roundf8_avx(__m256 a);
__m256 Sleef_roundf8_fma4(__m256 a);
__m256 Sleef_roundf8_avx2(__m256 a);

__m512 Sleef_roundf16(__m512 a);
__m512 Sleef_roundf16_avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_roundf](../libm#sleef_roundf) with the same accuracy specification.

### Vectorized double precision functions for rounding to nearest integer

```c
#include <sleef.h>

__m128d Sleef_rintd2(__m128d a);
__m128d Sleef_rintd2_sse2(__m128d a);
__m128d Sleef_rintd2_sse4(__m128d a);
__m128d Sleef_rintd2_avx2128(__m128d a);

__m256d Sleef_rintd4(__m256d a);
__m256d Sleef_rintd4_avx(__m256d a);
__m256d Sleef_rintd4_fma4(__m256d a);
__m256d Sleef_rintd4_avx2(__m256d a);

__m512d Sleef_rintd8(__m512d a);
__m512d Sleef_rintd8_avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_rint](../libm#sleef_rint) with the same accuracy specification.

### Vectorized single precision functions for rounding to nearest integer

```c
#include <sleef.h>

__m128 Sleef_rintf4(__m128 a);
__m128 Sleef_rintf4_sse2(__m128 a);
__m128 Sleef_rintf4_sse4(__m128 a);
__m128 Sleef_rintf4_avx2128(__m128 a);

__m256 Sleef_rintf8(__m256 a);
__m256 Sleef_rintf8_avx(__m256 a);
__m256 Sleef_rintf8_fma4(__m256 a);
__m256 Sleef_rintf8_avx2(__m256 a);

__m512 Sleef_rintf16(__m512 a);
__m512 Sleef_rintf16_avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_rintf](../libm#sleef_rintf) with the same accuracy specification.

<h2 id="other">Other functions</h2>

### Vectorized double precision functions for fused multiply-accumulation

```c
#include <sleef.h>

__m128d Sleef_fmad2(__m128d a, __m128d b, __m128d c);
__m128d Sleef_fmad2_sse2(__m128d a, __m128d b, __m128d c);
__m128d Sleef_fmad2_sse4(__m128d a, __m128d b, __m128d c);
__m128d Sleef_fmad2_avx2128(__m128d a, __m128d b, __m128d c);

__m256d Sleef_fmad4(__m256d a, __m256d b, __m256d c);
__m256d Sleef_fmad4_avx(__m256d a, __m256d b, __m256d c);
__m256d Sleef_fmad4_fma4(__m256d a, __m256d b, __m256d c);
__m256d Sleef_fmad4_avx2(__m256d a, __m256d b, __m256d c);

__m512d Sleef_fmad8(__m512d a, __m512d b, __m512d c);
__m512d Sleef_fmad8_avx512f(__m512d a, __m512d b, __m512d c);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fma](../libm#sleef_fma) with the same accuracy specification.

### Vectorized single precision functions for fused multiply-accumulation

```c
#include <sleef.h>

__m128 Sleef_fmaf4(__m128 a, __m128 b, __m128 c);
__m128 Sleef_fmaf4_sse2(__m128 a, __m128 b, __m128 c);
__m128 Sleef_fmaf4_sse4(__m128 a, __m128 b, __m128 c);
__m128 Sleef_fmaf4_avx2128(__m128 a, __m128 b, __m128 c);

__m256 Sleef_fmaf8(__m256 a, __m256 b, __m256 c);
__m256 Sleef_fmaf8_avx(__m256 a, __m256 b, __m256 c);
__m256 Sleef_fmaf8_fma4(__m256 a, __m256 b, __m256 c);
__m256 Sleef_fmaf8_avx2(__m256 a, __m256 b, __m256 c);

__m512 Sleef_fmaf16(__m512 a, __m512 b, __m512 c);
__m512 Sleef_fmaf16_avx512f(__m512 a, __m512 b, __m512 c);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fmaf](../libm#sleef_fmaf) with the same accuracy specification.

### Vectorized double precision FP remainder

```c
#include <sleef.h>

__m128d Sleef_fmodd2(__m128d a, __m128d b);
__m128d Sleef_fmodd2_sse2(__m128d a, __m128d b);
__m128d Sleef_fmodd2_sse4(__m128d a, __m128d b);
__m128d Sleef_fmodd2_avx2128(__m128d a, __m128d b);

__m256d Sleef_fmodd4(__m256d a, __m256d b);
__m256d Sleef_fmodd4_avx(__m256d a, __m256d b);
__m256d Sleef_fmodd4_fma4(__m256d a, __m256d b);
__m256d Sleef_fmodd4_avx2(__m256d a, __m256d b);

__m512d Sleef_fmodd8(__m512d a, __m512d b);
__m512d Sleef_fmodd8_avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fmod](../libm#sleef_fmod) with the same accuracy specification.

### Vectorized single precision FP remainder

```c
#include <sleef.h>

__m128 Sleef_fmodf4(__m128 a, __m128 b);
__m128 Sleef_fmodf4_sse2(__m128 a, __m128 b);
__m128 Sleef_fmodf4_sse4(__m128 a, __m128 b);
__m128 Sleef_fmodf4_avx2128(__m128 a, __m128 b);

__m256 Sleef_fmodf8(__m256 a, __m256 b);
__m256 Sleef_fmodf8_avx(__m256 a, __m256 b);
__m256 Sleef_fmodf8_fma4(__m256 a, __m256 b);
__m256 Sleef_fmodf8_avx2(__m256 a, __m256 b);

__m512 Sleef_fmodf16(__m512 a, __m512 b);
__m512 Sleef_fmodf16_avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fmodf](../libm#sleef_fmodf) with the same accuracy specification.

### Vectorized double precision FP remainder

```c
#include <sleef.h>

__m128d Sleef_remainderd2(__m128d a, __m128d b);
__m128d Sleef_remainderd2_sse2(__m128d a, __m128d b);
__m128d Sleef_remainderd2_sse4(__m128d a, __m128d b);
__m128d Sleef_remainderd2_avx2128(__m128d a, __m128d b);

__m256d Sleef_remainderd4(__m256d a, __m256d b);
__m256d Sleef_remainderd4_avx(__m256d a, __m256d b);
__m256d Sleef_remainderd4_fma4(__m256d a, __m256d b);
__m256d Sleef_remainderd4_avx2(__m256d a, __m256d b);

__m512d Sleef_remainderd8(__m512d a, __m512d b);
__m512d Sleef_remainderd8_avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_remainder](../libm#sleef_remainder) with the same accuracy specification.

### Vectorized single precision FP remainder

```c
#include <sleef.h>

__m128 Sleef_remainderf4(__m128 a, __m128 b);
__m128 Sleef_remainderf4_sse2(__m128 a, __m128 b);
__m128 Sleef_remainderf4_sse4(__m128 a, __m128 b);
__m128 Sleef_remainderf4_avx2128(__m128 a, __m128 b);

__m256 Sleef_remainderf8(__m256 a, __m256 b);
__m256 Sleef_remainderf8_avx(__m256 a, __m256 b);
__m256 Sleef_remainderf8_fma4(__m256 a, __m256 b);
__m256 Sleef_remainderf8_avx2(__m256 a, __m256 b);

__m512 Sleef_remainderf16(__m512 a, __m512 b);
__m512 Sleef_remainderf16_avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_remainderf](../libm#sleef_remainderf) with the same accuracy specification.

### Vectorized double precision functions for multiplying by integral power of 2

```c
#include <sleef.h>

__m128d Sleef_ldexpd2(__m128d a, __m128i b);
__m128d Sleef_ldexpd2_sse2(__m128d a, __m128i b);
__m128d Sleef_ldexpd2_sse4(__m128d a, __m128i b);
__m128d Sleef_ldexpd2_avx2128(__m128d a, __m128i b);

__m256d Sleef_ldexpd4(__m256d a, __m128i b);
__m256d Sleef_ldexpd4_avx(__m256d a, __m128i b);
__m256d Sleef_ldexpd4_fma4(__m256d a, __m128i b);
__m256d Sleef_ldexpd4_avx2(__m256d a, __m128i b);

__m512d Sleef_ldexpd8(__m512d a, __m256i b);
__m512d Sleef_ldexpd8_avx512f(__m512d a, __m256i b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_ldexp](../libm#sleef_ldexp) with the same accuracy specification.

### Vectorized double precision functions for obtaining fractional component of an FP number

```c
#include <sleef.h>

__m128d Sleef_frfrexpd2(__m128d a);
__m128d Sleef_frfrexpd2_sse2(__m128d a);
__m128d Sleef_frfrexpd2_sse4(__m128d a);
__m128d Sleef_frfrexpd2_avx2128(__m128d a);

__m256d Sleef_frfrexpd4(__m256d a);
__m256d Sleef_frfrexpd4_avx(__m256d a);
__m256d Sleef_frfrexpd4_fma4(__m256d a);
__m256d Sleef_frfrexpd4_avx2(__m256d a);

__m512d Sleef_frfrexpd8(__m512d a);
__m512d Sleef_frfrexpd8_avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_frfrexp](../libm#sleef_frfrexp) with the same accuracy specification.

### Vectorized single precision functions for obtaining fractional component of an FP number

```c
#include <sleef.h>

__m128 Sleef_frfrexpf4(__m128 a);
__m128 Sleef_frfrexpf4_sse2(__m128 a);
__m128 Sleef_frfrexpf4_sse4(__m128 a);
__m128 Sleef_frfrexpf4_avx2128(__m128 a);

__m256 Sleef_frfrexpf8(__m256 a);
__m256 Sleef_frfrexpf8_avx(__m256 a);
__m256 Sleef_frfrexpf8_fma4(__m256 a);
__m256 Sleef_frfrexpf8_avx2(__m256 a);

__m512 Sleef_frfrexpf16(__m512 a);
__m512 Sleef_frfrexpf16_avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_frfrexpf](../libm#sleef_frfrexpf) with the same accuracy specification.

### Vectorized double precision function for obtaining integral component of an FP number

```c
#include <sleef.h>

__m128i Sleef_expfrexpd2(__m128d a);
__m128i Sleef_expfrexpd2_sse2(__m128d a);
__m128i Sleef_expfrexpd2_sse4(__m128d a);
__m128i Sleef_expfrexpd2_avx2128(__m128d a);

__m128i Sleef_expfrexpd4(__m256d a);
__m128i Sleef_expfrexpd4_avx(__m256d a);
__m128i Sleef_expfrexpd4_fma4(__m256d a);
__m128i Sleef_expfrexpd4_avx2(__m256d a);

__m256i Sleef_expfrexpd8(__m512d a);
__m256i Sleef_expfrexpd8_avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_expfrexp](../libm#sleef_expfrexp) with the same accuracy specification.

### Vectorized double precision functions for getting integer exponent

```c
#include <sleef.h>

__m128i Sleef_ilogbd2(__m128d a);
__m128i Sleef_ilogbd2_sse2(__m128d a);
__m128i Sleef_ilogbd2_sse4(__m128d a);
__m128i Sleef_ilogbd2_avx2128(__m128d a);

__m128i Sleef_ilogbd4(__m256d a);
__m128i Sleef_ilogbd4_avx(__m256d a);
__m128i Sleef_ilogbd4_fma4(__m256d a);
__m128i Sleef_ilogbd4_avx2(__m256d a);

__m256i Sleef_ilogbd8(__m512d a);
__m256i Sleef_ilogbd8_avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_ilogb](../libm#sleef_ilogb) with the same accuracy specification.

### Vectorized double precision signed integral and fractional values

```c
#include <sleef.h>

Sleef___m128d_2 Sleef_modfd2(__m128d a);
Sleef___m128d_2 Sleef_modfd2_sse2(__m128d a);
Sleef___m128d_2 Sleef_modfd2_sse4(__m128d a);
Sleef___m128d_2 Sleef_modfd2_avx2128(__m128d a);

Sleef___m256d_2 Sleef_modfd4(__m256d a);
Sleef___m256d_2 Sleef_modfd4_avx(__m256d a);
Sleef___m256d_2 Sleef_modfd4_fma4(__m256d a);
Sleef___m256d_2 Sleef_modfd4_avx2(__m256d a);

Sleef___m512d_2 Sleef_modfd8(__m512d a);
Sleef___m512d_2 Sleef_modfd8_avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_modf](../libm#sleef_modf) with the same accuracy specification.

### Vectorized single precision signed integral and fractional values

```c
#include <sleef.h>

Sleef___m128_2 Sleef_modff4(__m128 a);
Sleef___m128_2 Sleef_modff4_sse2(__m128 a);
Sleef___m128_2 Sleef_modff4_sse4(__m128 a);
Sleef___m128_2 Sleef_modff4_avx2128(__m128 a);

Sleef___m256_2 Sleef_modff8(__m256 a);
Sleef___m256_2 Sleef_modff8_avx(__m256 a);
Sleef___m256_2 Sleef_modff8_fma4(__m256 a);
Sleef___m256_2 Sleef_modff8_avx2(__m256 a);

Sleef___m512_2 Sleef_modff16(__m512 a);
Sleef___m512_2 Sleef_modff16_avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_modff](../libm#sleef_modff) with the same accuracy specification.

### Vectorized double precision functions for calculating the absolute value

```c
#include <sleef.h>

__m128d Sleef_fabsd2(__m128d a);
__m128d Sleef_fabsd2_sse2(__m128d a);
__m128d Sleef_fabsd2_sse4(__m128d a);
__m128d Sleef_fabsd2_avx2128(__m128d a);

__m256d Sleef_fabsd4(__m256d a);
__m256d Sleef_fabsd4_avx(__m256d a);
__m256d Sleef_fabsd4_fma4(__m256d a);
__m256d Sleef_fabsd4_avx2(__m256d a);

__m512d Sleef_fabsd8(__m512d a);
__m512d Sleef_fabsd8_avx512f(__m512d a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fabs](../libm#sleef_fabs) with the same accuracy specification.

### Vectorized single precision functions for calculating the absolute value

```c
#include <sleef.h>

__m128 Sleef_fabsf4(__m128 a);
__m128 Sleef_fabsf4_sse2(__m128 a);
__m128 Sleef_fabsf4_sse4(__m128 a);
__m128 Sleef_fabsf4_avx2128(__m128 a);

__m256 Sleef_fabsf8(__m256 a);
__m256 Sleef_fabsf8_avx(__m256 a);
__m256 Sleef_fabsf8_fma4(__m256 a);
__m256 Sleef_fabsf8_avx2(__m256 a);

__m512 Sleef_fabsf16(__m512 a);
__m512 Sleef_fabsf16_avx512f(__m512 a);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fabsf](../libm#sleef_fabsf) with the same accuracy specification.

### Vectorized double precision functions for copying signs

```c
#include <sleef.h>

__m128d Sleef_copysignd2(__m128d a, __m128d b);
__m128d Sleef_copysignd2_sse2(__m128d a, __m128d b);
__m128d Sleef_copysignd2_sse4(__m128d a, __m128d b);
__m128d Sleef_copysignd2_avx2128(__m128d a, __m128d b);

__m256d Sleef_copysignd4(__m256d a, __m256d b);
__m256d Sleef_copysignd4_avx(__m256d a, __m256d b);
__m256d Sleef_copysignd4_fma4(__m256d a, __m256d b);
__m256d Sleef_copysignd4_avx2(__m256d a, __m256d b);

__m512d Sleef_copysignd8(__m512d a, __m512d b);
__m512d Sleef_copysignd8_avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_copysign](../libm#sleef_copysign) with the same accuracy specification.

### Vectorized single precision functions for copying signs

```c
#include <sleef.h>

__m128 Sleef_copysignf4(__m128 a, __m128 b);
__m128 Sleef_copysignf4_sse2(__m128 a, __m128 b);
__m128 Sleef_copysignf4_sse4(__m128 a, __m128 b);
__m128 Sleef_copysignf4_avx2128(__m128 a, __m128 b);

__m256 Sleef_copysignf8(__m256 a, __m256 b);
__m256 Sleef_copysignf8_avx(__m256 a, __m256 b);
__m256 Sleef_copysignf8_fma4(__m256 a, __m256 b);
__m256 Sleef_copysignf8_avx2(__m256 a, __m256 b);

__m512 Sleef_copysignf16(__m512 a, __m512 b);
__m512 Sleef_copysignf16_avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_copysignf](../libm#sleef_copysignf) with the same accuracy specification.

### Vectorized double precision functions for determining maximum of two values

```c
#include <sleef.h>

__m128d Sleef_fmaxd2(__m128d a, __m128d b);
__m128d Sleef_fmaxd2_sse2(__m128d a, __m128d b);
__m128d Sleef_fmaxd2_sse4(__m128d a, __m128d b);
__m128d Sleef_fmaxd2_avx2128(__m128d a, __m128d b);

__m256d Sleef_fmaxd4(__m256d a, __m256d b);
__m256d Sleef_fmaxd4_avx(__m256d a, __m256d b);
__m256d Sleef_fmaxd4_fma4(__m256d a, __m256d b);
__m256d Sleef_fmaxd4_avx2(__m256d a, __m256d b);

__m512d Sleef_fmaxd8(__m512d a, __m512d b);
__m512d Sleef_fmaxd8_avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fmax](../libm#sleef_fmax) with the same accuracy specification.

### Vectorized single precision functions for determining maximum of two values

```c
#include <sleef.h>

__m128 Sleef_fmaxf4(__m128 a, __m128 b);
__m128 Sleef_fmaxf4_sse2(__m128 a, __m128 b);
__m128 Sleef_fmaxf4_sse4(__m128 a, __m128 b);
__m128 Sleef_fmaxf4_avx2128(__m128 a, __m128 b);

__m256 Sleef_fmaxf8(__m256 a, __m256 b);
__m256 Sleef_fmaxf8_avx(__m256 a, __m256 b);
__m256 Sleef_fmaxf8_fma4(__m256 a, __m256 b);
__m256 Sleef_fmaxf8_avx2(__m256 a, __m256 b);

__m512 Sleef_fmaxf16(__m512 a, __m512 b);
__m512 Sleef_fmaxf16_avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fmaxf](../libm#sleef_fmaxf) with the same accuracy specification.

### Vectorized double precision functions for determining minimum of two values

```c
#include <sleef.h>

__m128d Sleef_fmind2(__m128d a, __m128d b);
__m128d Sleef_fmind2_sse2(__m128d a, __m128d b);
__m128d Sleef_fmind2_sse4(__m128d a, __m128d b);
__m128d Sleef_fmind2_avx2128(__m128d a, __m128d b);

__m256d Sleef_fmind4(__m256d a, __m256d b);
__m256d Sleef_fmind4_avx(__m256d a, __m256d b);
__m256d Sleef_fmind4_fma4(__m256d a, __m256d b);
__m256d Sleef_fmind4_avx2(__m256d a, __m256d b);

__m512d Sleef_fmind8(__m512d a, __m512d b);
__m512d Sleef_fmind8_avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fmin](../libm#sleef_fmin) with the same accuracy specification.

### Vectorized single precision functions for determining minimum of two values

```c
#include <sleef.h>

__m128 Sleef_fminf4(__m128 a, __m128 b);
__m128 Sleef_fminf4_sse2(__m128 a, __m128 b);
__m128 Sleef_fminf4_sse4(__m128 a, __m128 b);
__m128 Sleef_fminf4_avx2128(__m128 a, __m128 b);

__m256 Sleef_fminf8(__m256 a, __m256 b);
__m256 Sleef_fminf8_avx(__m256 a, __m256 b);
__m256 Sleef_fminf8_fma4(__m256 a, __m256 b);
__m256 Sleef_fminf8_avx2(__m256 a, __m256 b);

__m512 Sleef_fminf16(__m512 a, __m512 b);
__m512 Sleef_fminf16_avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fminf](../libm#sleef_fminf) with the same accuracy specification.

### Vectorized double precision functions to calculate positive difference of two values

```c
#include <sleef.h>

__m128d Sleef_fdimd2(__m128d a, __m128d b);
__m128d Sleef_fdimd2_sse2(__m128d a, __m128d b);
__m128d Sleef_fdimd2_sse4(__m128d a, __m128d b);
__m128d Sleef_fdimd2_avx2128(__m128d a, __m128d b);

__m256d Sleef_fdimd4(__m256d a, __m256d b);
__m256d Sleef_fdimd4_avx(__m256d a, __m256d b);
__m256d Sleef_fdimd4_fma4(__m256d a, __m256d b);
__m256d Sleef_fdimd4_avx2(__m256d a, __m256d b);

__m512d Sleef_fdimd8(__m512d a, __m512d b);
__m512d Sleef_fdimd8_avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fdim](../libm#sleef_fdim) with the same accuracy specification.

### Vectorized single precision functions to calculate positive difference of two values

```c
#include <sleef.h>

__m128 Sleef_fdimf4(__m128 a, __m128 b);
__m128 Sleef_fdimf4_sse2(__m128 a, __m128 b);
__m128 Sleef_fdimf4_sse4(__m128 a, __m128 b);
__m128 Sleef_fdimf4_avx2128(__m128 a, __m128 b);

__m256 Sleef_fdimf8(__m256 a, __m256 b);
__m256 Sleef_fdimf8_avx(__m256 a, __m256 b);
__m256 Sleef_fdimf8_fma4(__m256 a, __m256 b);
__m256 Sleef_fdimf8_avx2(__m256 a, __m256 b);

__m512 Sleef_fdimf16(__m512 a, __m512 b);
__m512 Sleef_fdimf16_avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_fdimf](../libm#sleef_fdimf) with the same accuracy specification.

### Vectorized double precision functions for obtaining the next representable FP value

```c
#include <sleef.h>

__m128d Sleef_nextafterd2(__m128d a, __m128d b);
__m128d Sleef_nextafterd2_sse2(__m128d a, __m128d b);
__m128d Sleef_nextafterd2_sse4(__m128d a, __m128d b);
__m128d Sleef_nextafterd2_avx2128(__m128d a, __m128d b);

__m256d Sleef_nextafterd4(__m256d a, __m256d b);
__m256d Sleef_nextafterd4_avx(__m256d a, __m256d b);
__m256d Sleef_nextafterd4_fma4(__m256d a, __m256d b);
__m256d Sleef_nextafterd4_avx2(__m256d a, __m256d b);

__m512d Sleef_nextafterd8(__m512d a, __m512d b);
__m512d Sleef_nextafterd8_avx512f(__m512d a, __m512d b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_nextafter](../libm#sleef_nextafter) with the same accuracy specification.

### Vectorized single precision functions for obtaining the next representable FP value

```c
#include <sleef.h>

__m128 Sleef_nextafterf4(__m128 a, __m128 b);
__m128 Sleef_nextafterf4_sse2(__m128 a, __m128 b);
__m128 Sleef_nextafterf4_sse4(__m128 a, __m128 b);
__m128 Sleef_nextafterf4_avx2128(__m128 a, __m128 b);

__m256 Sleef_nextafterf8(__m256 a, __m256 b);
__m256 Sleef_nextafterf8_avx(__m256 a, __m256 b);
__m256 Sleef_nextafterf8_fma4(__m256 a, __m256 b);
__m256 Sleef_nextafterf8_avx2(__m256 a, __m256 b);

__m512 Sleef_nextafterf16(__m512 a, __m512 b);
__m512 Sleef_nextafterf16_avx512f(__m512 a, __m512 b);
```
Link with `-lsleef`.

These are the vectorized functions of [Sleef_nextafterf](../libm#sleef_nextafterf) with the same accuracy specification.

<h2 id="supplementary">Supplementary Functions</h2>

### CPU feature identification function

```c
#include <sleef.h>

void Sleef_x86CpuID(int32_t out[4], uint32_t eax, uint32_t ecx);
```
Link with `-lsleef`.

This is a function for obtaining CPU features by the cpuid instruction available on x86 architecture. This function executes cpuid instruction by passing the values `eax` and `ecx` to the corresponding registers. The information returned in EAX, EBX, ECX, and EDX registers will be stored in array `out`.
