/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* $Id: DSCHandler.java 1345683 2012-06-03 14:50:33Z gadams $ */

package org.apache.xmlgraphics.ps.dsc;

import java.io.IOException;

import org.apache.xmlgraphics.ps.dsc.events.DSCComment;

/**
 * Interface containing events generated by the DSCParser. Applications can implement this
 * interface to react to certain events.
 */
public interface DSCHandler {

    /**
     * Called as a new PostScript file starts.
     * @param header the first line of the DSC-compliant file
     * @throws IOException In case of an I/O error
     */
    void startDocument(String header) throws IOException;

    /**
     * Called when the PostScript file is fully processed, i.e. after the %%EOF comment.
     * @throws IOException In case of an I/O error
     */
    void endDocument() throws IOException;

    /**
     * Called for each standard DSC comment. The classes passed to this method may be simple
     * DSCComment classes or special subclasses for some of the DSC comments.
     * @param comment the DSC comment
     * @throws IOException In case of an I/O error
     */
    void handleDSCComment(DSCComment comment) throws IOException;

    /**
     * Called for a normal line of PostScript code.
     * @param line the line of code
     * @throws IOException In case of an I/O error
     */
    void line(String line) throws IOException;

    /**
     * Called for any line containing a full-line PostScript comment. This is also called for
     * custom comments following the extension mechanism of the DSC specification.
     * @param comment the comment line
     * @throws IOException In case of an I/O error
     */
    void comment(String comment) throws IOException;

}
