/*******************************************************************************
* parsers.cpp: Command line and configuration files parsers
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: parsers.cpp,v 1.1 2001/10/06 21:23:36 bozo Exp $
*
* Authors: Benoit Steiner <benny@via.ecp.fr>
*          Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "defs.h"

#include <stdio.h>
#include <dirent.h>
#include <sys/types.h>
#ifdef HAVE_REGEX_H
#include <regex.h>
#endif

#include "common.h"
#include "debug.h"
#include "reflect.h"
#include "serialization.h"
#include "string.h"
#include "vector.h"
#include "hashtable.h"
#include "stack.h"
#include "buffers.h"
#include "exception.h"
#include "regexp.h"
#include "file.h"
#include "log.h"
#include "stream.h"

#include "parsers.h"

#include "stack.cpp"
#include "stream.cpp"


//******************************************************************************
// class C_ParserException
//******************************************************************************
//
//******************************************************************************

//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
E_Parser::E_Parser(int iCode, const C_String& strMsg) :
                        E_Exception(iCode, strMsg)
{
  // Nothing to do
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
E_Parser::E_Parser(int iCode, const C_String& strMsg, E_Exception e) :
                        E_Exception(iCode, strMsg, e)
{
  // Nothing to do
}





//******************************************************************************
// class C_CfgFileParser
//******************************************************************************
//
//******************************************************************************

//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// regexp match : |^ .... = "...." #....$|
//                   ---- -  ----  -----
//                   sub1 |  sub4   sub5
//                        | ------
//                 sub2 <-+  sub3
//------------------------------------------------------------------------------
C_CfgFileParser::C_CfgFileParser(C_ParserHandler* pCallBack) :
     m_cParser("^[ \t]*([^ \t=#]*)[ \t]*(=?)[ \t]*(\"([^\"]*)\")?[ \t]*(#.*)*$"),
     m_cContextStack(10)
{
  ASSERT(pCallBack);
  m_pCallBack = pCallBack;

  ASSERT(m_cParser.IsValid());

  // Init the file position indicators
  m_iLineNumber = 0;

}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_CfgFileParser::Parse(const C_String& strFileName, bool bCompletePath)
{
  ASSERT(m_pCallBack);

  try
  {
    // Open the cfg file
    C_File* pCfgFile = new C_File(strFileName);
    pCfgFile->Open("r");

    // Build a stream to read strings more easily
    C_Stream<C_File>* pStream = new C_Stream<C_File>(pCfgFile);
    ASSERT(pStream);

    // Set the marker to '\n' so that we will parse the stream line by line
    pStream->SetMarker('\n');

    while(pStream->GetState() == STREAM_READY)
    {
      // Read the next line of the file
      C_String strLine;
      *pStream >> strLine;
      m_iLineNumber++;

      // Remove the comments and skip the empty lines
      ASSERT(m_cParser.GetNbMatches(strLine) <= 1);
      C_RegexpMatch* pMatch = m_cParser.GetFirstMatch(strLine);
      if(pMatch)
      {
        ASSERT(pMatch->GetNbSubExpr() == 5);
        C_String strFirstToken = pMatch->GetSubExpr(1);
        C_String strLowerFirstToken = strFirstToken.ToLower();

        // Parse the line
        if(strLowerFirstToken == "end")
        {
          // Unstack the section
          if(m_cContextStack.Size() == 0)
          {
            throw E_Parser(GEN_ERR, strFirstToken + "tag at line " +
                           m_iLineNumber + "in excess");
          }
          // There should be only the "end"
          else if((pMatch->GetSubExpr(2) != "") ||
                  (pMatch->GetSubExpr(3) != ""))
          {
            throw E_Parser(GEN_ERR,
                        C_String("Error in the configuration file at line ") +
                        m_iLineNumber);
          }
          else
          {
            C_String* pstrSection = m_cContextStack.Pop();
            m_pCallBack->OnEndSection(*pstrSection);
            delete pstrSection;
          }  
        }
        else if(strLowerFirstToken == "begin")
        {
          C_String strSection = pMatch->GetSubExpr(4);

          // There should be no '=' and a not empty section name
          if((pMatch->GetSubExpr(2) != "") || (strSection == ""))
          {
            throw E_Parser(GEN_ERR,
                        C_String("Error in the configuration file at line ") +
                        m_iLineNumber);
          }
          else if(m_cContextStack.Size() == m_cContextStack.Capacity())
          {
            throw E_Parser(GEN_ERR, C_String("Error line ") + m_iLineNumber +
                           ": only " + m_cContextStack.Capacity() +
                           " section levels allowed");
          }
          else
          {
            m_cContextStack.Push(new C_String(strSection));
            m_pCallBack->OnStartSection(strSection);
          }
        }
        else
        {
          bool bName = (strFirstToken != "");
          bool bEqual = (pMatch->GetSubExpr(2) == "=");
          bool bValue = (pMatch->GetSubExpr(3) != "");

          // There should be a '=' and a value (which may be empty) in ""
          // except if there is only a comment
          if((bValue && (!bName || !bEqual)) ||
             (!bValue && (bName || bEqual)))
          {
            throw E_Parser(GEN_ERR,
                        C_String("Error in the configuration file at line ") +
                        m_iLineNumber);
          }
          else if(strFirstToken != "")
          {
            C_String strValue = pMatch->GetSubExpr(4);
            m_pCallBack->OnProperty(strFirstToken, strValue);
          }
        }

        delete pMatch;
      }
      else
      {
        throw E_Parser(GEN_ERR,
                       C_String("Error in the configuration file at line ") +
                       m_iLineNumber);
      }
    }
  
    // Close the stream
    pStream->Close();
    delete pStream;
  }
  catch(E_Exception e)
  {
    throw E_Parser(GEN_ERR, "Parsing of file '" + strFileName + "' failed", e);
    
  }
}


